
#include "PlayerItem.hpp"
#include "MapItem.hpp"
#include "mixr/models/player/Player.hpp"

#include <QPainter>
#include <sstream>

PlayerItem::PlayerItem(MapView* viewer, MapItem* map, QGraphicsItem* parent) : QGraphicsItem(parent)
{
   view = viewer;
   myMap = map;
   plyId = 0;
   defaultImage = nullptr;
}

PlayerItem::~PlayerItem()
{
   myMap = nullptr;
   view = nullptr;
}

// ---
// boundingRect() - returns the bounding rectangle of this PlayerItem
// ---
QRectF PlayerItem::boundingRect() const
{
   return QRectF();
}

// ---
// paint() - draw stuff here
// ---
void PlayerItem::paint(QPainter* painter, const QStyleOptionGraphicsItem*, QWidget*)
{
   // get our width and height from our icon
   const int w = size.width();
   const int h = size.height();
   if (defaultImage != nullptr) {
      painter->drawImage(QRectF(QPointF(-w/2, -h/2), QSizeF(w, h)), *defaultImage);
   }
   else painter->drawEllipse(0, 0, 30, 30);

   // draw the player's name
   QFontMetrics fm = painter->fontMetrics();
   QPen pen(Qt::white, 2.0f);
   QRectF fontRect = fm.boundingRect(name + "X");
   painter->save();

   double hdg = rotation();
   painter->rotate(-hdg);

   if (hdg < 270 && hdg > 90)    { painter->translate(-fontRect.width()/2, -size.height() + fontRect.height() - 5);  }
   else                          { painter->translate(-fontRect.width()/2, size.height() + 5);                       }

   QColor sideColor(qRgba(0, 0, 0, 255));
   sideColor.setAlpha(150);
   painter->setBrush(sideColor);
   painter->setPen(pen);
   painter->matrix().mapRect(fontRect);
   painter->drawRoundedRect(fontRect, 5, 5);
   painter->drawText(fontRect, Qt::AlignCenter, name);
   painter->restore();
}

// ---
// useDefaultImage() - passes a string with the relative path of the image file you wish to use
// as the PlayerItem's icon.  It will use this image for it's symbol.
// ---
bool PlayerItem::useDefaultImage(QString fullName, QSize x)
{
   if (defaultImage == nullptr) {
      defaultImage = new QImage(fullName);
      if (defaultImage->isNull()) {
         std::cout << "DEFAULT IMAGE NAME NOT LOADED: " << fullName.toLatin1().data() << std::endl;
      }
   } else {
      const bool ok = defaultImage->load(fullName);
      if (!ok) {
         std::cout << "DEFAULT IMAGE NAME NOT LOADED: " << fullName.toLatin1().data() << std::endl;
      }
   }

   // set our size to display
   size = x;
   return true;
}

// ---
// setDefaultImageSize() - simply adjusts the size of our image
// ---
bool PlayerItem::setDefaultImageSize(QSize x)
{
   size = x;
   return true;
}

void PlayerItem::refreshPlayer(const mixr::models::Player* const ownship)
{
   if (ownship != nullptr) {
      plyId = ownship->getID();
      double oLat {};
      double oLon {};
      ownship->getPositionLL(&oLat, &oLon);
      if (myMap != nullptr) {
         double tpLat {};
         double tpLon {};
         myMap->llToPixels(oLat, oLon, tpLat, tpLon);
         setPos(tpLon, tpLat);
      }
      setRotation(ownship->getHeadingD());
      // update the player's name
      if (ownship->getName() != "") {
         name = ownship->getName().c_str();
      } else {
         std::ostringstream stream;
         stream << plyId;
         name = stream.str().c_str();
      }

   } else {
      plyId = 0;
   }
}
