
#include "CrsPntr.hpp"
#include "mixr/base/numeric/Boolean.hpp"
#include "mixr/base/numeric/Integer.hpp"
#include "mixr/base/numeric/INumber.hpp"
#include "mixr/base/qty/angles.hpp"

IMPLEMENT_SUBCLASS(CrsPntr, "CrsPntr")
EMPTY_DELETEDATA(CrsPntr)

BEGIN_EVENT_HANDLER(CrsPntr)
    ON_EVENT_OBJ(UPDATE_VALUE7, onUpdateCdiDotsCrsPntr, mixr::base::INumber)
    ON_EVENT_OBJ(UPDATE_VALUE8, onUpdateToFromCrsPntr,  mixr::base::INumber)
END_EVENT_HANDLER()

BEGIN_SLOTTABLE(CrsPntr)
    "numDots",      // number of dots to use
    "numInches",    // how many inches to use for the cdi scale
    "showCdi",      // do we display the cdi?
    "showCrsPntr",  // show to course pointer?
    "showToFrom",   // show our to from pointer?
END_SLOTTABLE(CrsPntr)

BEGIN_SLOT_MAP(CrsPntr)
    ON_SLOT(1, setSlotNumCdiDots,  mixr::base::Integer)
    ON_SLOT(2, setSlotNumInches,   mixr::base::INumber)
    ON_SLOT(3, setSlotShowCdi,     mixr::base::Boolean)
    ON_SLOT(4, setSlotShowCrsPntr, mixr::base::Boolean)
    ON_SLOT(5, setSlotShowToFrom,  mixr::base::Boolean)
END_SLOT_MAP()

CrsPntr::CrsPntr()
{
    STANDARD_CONSTRUCTOR()
}

void CrsPntr::copyData(const CrsPntr& org, const bool)
{
    BaseClass::copyData(org);
    cdiDots = org.cdiDots;
    toFrom = org.toFrom;
    numDots = org.numDots;
    inchesPerDot = org.inchesPerDot;
    inches = org.inches;
    showCdi = org.showCdi;
    showCrsPtr = org.showCrsPtr;
    showToFrom = org.showToFrom;
}

//------------------------------------------------------------------------------
// set functions
//------------------------------------------------------------------------------
bool CrsPntr::setCdiDots(const double newCDD)
{
    cdiDots = newCDD;
    return true;
}

bool CrsPntr::setNumCdiDots(const int newNCD)
{
    if (newNCD != 0) numDots = newNCD;
    return true;
}

bool CrsPntr::setNumInches(const double newNI)
{
    inches = newNI;
    return true;
}

bool CrsPntr::setShowCdi(const bool newSCDI)
{
    showCdi = newSCDI;
    return true;
}

bool CrsPntr::setShowCrsPntr(const bool newSCP)
{
    showCrsPtr = newSCP;
    return true;
}

bool CrsPntr::setToFrom(const double newTF)
{
    toFrom = newTF;
    return true;
}
bool CrsPntr::setShowToFrom(const bool newTF)
{
    showToFrom = newTF;
    return true;
}

// Event functions
bool CrsPntr::onUpdateCdiDotsCrsPntr(const mixr::base::INumber* const x)
{
    bool ok{};
    if (x != nullptr) ok = setCdiDots(x->asDouble());
    return ok;
}
bool CrsPntr::onUpdateToFromCrsPntr(const mixr::base::INumber* const x)
{
    bool ok{};
    if (x != nullptr) ok = setToFrom(x->asDouble());
    return ok;
}

void CrsPntr::drawFunc()
{
    GLfloat ocolor[4]{};
    GLfloat lw{};
    glGetFloatv(GL_CURRENT_COLOR, ocolor);
    glGetFloatv(GL_LINE_WIDTH, &lw);

    glColor3f(1, 1, 1);
    glLineWidth(2);

    if (showCrsPtr) {
        // this is the course pointer
        glPushMatrix();
            glBegin(GL_LINE_STRIP);
                glVertex2f(0.044f, 0.49f);
                glVertex2f(0.044f, 0.865f);
                glVertex2f(0, 1.065f);
                glVertex2f(-0.044f, 0.865f);
                glVertex2f(-0.044f, 0.49f);
                glVertex2f(0.044f, 0.49f);
            glEnd();
            glBegin(GL_LINE_STRIP);
                glVertex2f(0, -0.6f);
                glVertex2f(-0.044f, -0.676f);
                glVertex2f(-0.044f, -1.1f);
                glVertex2f( 0.044f, -1.1f);
                glVertex2f( 0.044f, -0.676f);
                glVertex2f(0, -0.6f);
            glEnd();
        glPopMatrix();
    }

    if (showCdi) {
        // figure how far to translate
        double scale{inchesPerDot * cdiDots};
        // limit ourself to the size of the scale
        if (scale > 0.88) scale = 0.88f;
        if (scale < -0.88) scale = -0.88f;

        // cdi
        glPushMatrix();
            glTranslatef(static_cast<GLfloat>(scale), 0, 0);
            glBegin(GL_LINE_STRIP);
                glVertex2f(-0.044f, -0.45f);
                glVertex2f(0.044f, -0.45f);
                glVertex2f(0.044f, 0.35f);
                glVertex2f(-0.044f, 0.35f);
                glVertex2f(-0.044f, -0.45f);
            glEnd();
        glPopMatrix();
    }

    // our to from arrow will not be display unless we get a 1 (to) or a 0 (from)
    if (showToFrom) {
        if (toFrom == 1) {
            glPushMatrix();
                glBegin(GL_POLYGON);
                    glVertex2f(0, -0.6f);
                    glVertex2f(-0.044f, -0.475f);
                    glVertex2f(0.044f, -0.475f);
                glEnd();
            glPopMatrix();
        }
        if (toFrom == 0) {
            glPushMatrix();
                glBegin(GL_POLYGON);
                    glVertex2f(0, 0.475f);
                    glVertex2f(-0.044f, 0.37f);
                    glVertex2f(0.044f, 0.37f);
                glEnd();
            glPopMatrix();

        }
    }

    glColor4fv(ocolor);
    glLineWidth(lw);
}

void CrsPntr::updateData(const double dt)
{
    // update base class first
    BaseClass::updateData(dt);

    // figure our scaling per dot
    inchesPerDot = inches / numDots;

}

//------------------------------------------------------------------------------
// setSlotNumCdiDots() - set the number of cdi dots we are using
//------------------------------------------------------------------------------
bool CrsPntr::setSlotNumCdiDots(const mixr::base::Integer* const newCDI)
{
    bool ok{};
    if (newCDI != nullptr) ok = setNumCdiDots(newCDI->asInt());
    return ok;
}
//------------------------------------------------------------------------------
// setSlotNumInches() - set the number of inches we are scaling across the dots
//------------------------------------------------------------------------------
bool CrsPntr::setSlotNumInches(const mixr::base::INumber* const newNI)
{
    bool ok{};
    if (newNI != nullptr) ok = setNumInches(newNI->asDouble());
    return ok;
}
//------------------------------------------------------------------------------
// setSlotShowCdi() - display the cdi?
//------------------------------------------------------------------------------
bool CrsPntr::setSlotShowCdi(const mixr::base::Boolean* const newSCDI)
{
    bool ok{};
    if (newSCDI != nullptr) ok = setShowCdi(newSCDI->asBool());
    return ok;
}
//------------------------------------------------------------------------------
// setSlotShowCrsPntr() - display the course pointer?
//------------------------------------------------------------------------------
bool CrsPntr::setSlotShowCrsPntr(const mixr::base::Boolean* const newSCP)
{
    bool ok{};
    if (newSCP != nullptr) ok = setShowCrsPntr(newSCP->asBool());
    return ok;
}
//------------------------------------------------------------------------------
// setSlotShowToFrom() - display the to from arrow?
//------------------------------------------------------------------------------
bool CrsPntr::setSlotShowToFrom(const mixr::base::Boolean* const newTF)
{
    bool ok{};
    if (newTF != nullptr) ok = setShowToFrom(newTF->asBool());
    return ok;
}

