/*
 * Decompiled with CFR 0.152.
 */
package com.logicaldoc.importfolder;

import com.logicaldoc.core.PersistenceException;
import com.logicaldoc.core.automation.Automation;
import com.logicaldoc.core.automation.AutomationException;
import com.logicaldoc.core.document.AbstractDocument;
import com.logicaldoc.core.document.Document;
import com.logicaldoc.core.document.DocumentDAO;
import com.logicaldoc.core.document.DocumentEvent;
import com.logicaldoc.core.document.DocumentHistory;
import com.logicaldoc.core.document.DocumentStatus;
import com.logicaldoc.core.folder.Folder;
import com.logicaldoc.core.folder.FolderDAO;
import com.logicaldoc.core.folder.FolderHistory;
import com.logicaldoc.core.security.authorization.PermissionException;
import com.logicaldoc.core.security.user.User;
import com.logicaldoc.core.security.user.UserDAO;
import com.logicaldoc.core.sequence.SequenceDAO;
import com.logicaldoc.importfolder.CrawlStats;
import com.logicaldoc.importfolder.ImportFolder;
import com.logicaldoc.importfolder.ImportFolderCache;
import com.logicaldoc.importfolder.ImportFolderCacheManager;
import com.logicaldoc.importfolder.ImportFolderCrawler;
import com.logicaldoc.importfolder.ImportFolderEvent;
import com.logicaldoc.importfolder.ImportFolderHistory;
import com.logicaldoc.importfolder.ImportFolderHistoryDAO;
import com.logicaldoc.importfolder.helper.CrawlerHelper;
import com.logicaldoc.util.exception.SkipException;
import com.logicaldoc.util.io.FileUtil;
import com.logicaldoc.util.spring.Context;
import com.logicaldoc.util.time.Pause;
import java.io.IOException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.Callable;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.RowMapper;

public class Crawler
implements Callable<CrawlStats> {
    public static final String IMPORTFOLDER_USERNAME = "_importfolder";
    private static final String IMPORT_FOLDER = "importFolder";
    public static final String COUNTER_SEQUENCE = "ifcounter";
    private Logger log;
    private List<ImportFolder> importFolders = new ArrayList<ImportFolder>();
    private CrawlStats stats = new CrawlStats();
    private boolean interrupt = false;
    private ImportFolderCacheManager cacheManager;
    private DocumentDAO documentDao;
    private SequenceDAO sequenceDao;
    private ImportFolderHistoryDAO importFolderHistoryDao;
    private UserDAO userDao;
    private ImportFolderCrawler importFolderCrawler;

    public Crawler(List<ImportFolder> importFolders, ImportFolderCrawler crawler, Logger log) {
        this.importFolders = importFolders;
        this.log = log;
        this.importFolderCrawler = crawler;
        this.userDao = Context.get(UserDAO.class);
        this.cacheManager = Context.get(ImportFolderCacheManager.class);
        this.documentDao = Context.get(DocumentDAO.class);
        this.sequenceDao = Context.get(SequenceDAO.class);
        this.importFolderHistoryDao = Context.get(ImportFolderHistoryDAO.class);
    }

    @Override
    public CrawlStats call() throws Exception {
        this.log.info("Crawling {} import folders", (Object)this.importFolders.size());
        for (ImportFolder importFolder : this.importFolders) {
            if (this.interrupt) break;
            this.processImportFolder(importFolder);
        }
        this.log.info("Completed");
        return this.stats;
    }

    public boolean isInterrupt() {
        return this.interrupt;
    }

    public Logger getLog() {
        return this.log;
    }

    public CrawlStats getStats() {
        return this.stats;
    }

    public void interrupt() {
        this.interrupt = true;
    }

    private void processImportFolder(ImportFolder importFolder) {
        this.log.info("Process folder {}", (Object)importFolder.getDisplayUrl());
        try {
            try {
                this.crawlImportFolder(importFolder);
                this.afterCrawlingCompleted(importFolder);
            }
            catch (Exception e) {
                this.log.error("Unable to process folder {}", (Object)importFolder.getDisplayUrl());
                this.log.error(e.getMessage(), e);
                this.importFolderCrawler.onImportFolderCompleted(this.stats, importFolder);
            }
        }
        finally {
            this.importFolderCrawler.onImportFolderCompleted(this.stats, importFolder);
        }
    }

    private void crawlImportFolder(ImportFolder importFolder) throws IOException, PersistenceException {
        this.stats.getImportFolderErrors().clear();
        this.stats.getImportFolderImports().clear();
        CrawlerHelper helper = CrawlerHelper.getInstance(importFolder, this);
        if (helper == null) {
            this.log.error("Implementation not found for provider {}", (Object)importFolder.getProvider());
            return;
        }
        User owner = this.userDao.findByUsername(IMPORTFOLDER_USERNAME);
        boolean importFromIndex = this.checkExistenceIndexFile(helper);
        if (importFromIndex) {
            try {
                helper.importUsingIndex(this);
            }
            catch (Exception e) {
                this.log.error(e.getMessage(), e);
            }
            return;
        }
        ImportFolderCache cache = this.cacheManager.getCache(importFolder.getId());
        ArrayList<Object> sourceFiles = new ArrayList<Object>();
        ArrayList<Object> sourceFolders = new ArrayList<Object>();
        try {
            helper.list(null, importFolder.getDepth(), sourceFolders, sourceFiles, importFolder.getBatch(), cache);
            this.log.info("Listed {} files inside the tree {}", (Object)sourceFiles.size(), (Object)importFolder.getPath());
        }
        catch (Exception t) {
            this.log.error(t.getMessage(), t);
        }
        this.createFoldersSkeleton(importFolder, helper, importFromIndex, sourceFolders);
        this.processSourceFiles(sourceFiles, importFolder, helper, owner, cache);
    }

    private void processSourceFiles(List<Object> sourceFiles, ImportFolder importFolder, CrawlerHelper helper, User owner, ImportFolderCache cache) {
        for (Object sourceFile : sourceFiles) {
            if (this.interrupt || this.stats.getImported() >= importFolder.getBatch()) {
                this.log.info("Reached {} processed entries in import folder {}", (Object)importFolder.getBatch(), (Object)importFolder.getDisplayUrl());
                break;
            }
            try {
                this.log.debug("Inspecting file {}", sourceFile);
                this.processSourceFile(sourceFile, importFolder, helper, owner, cache);
            }
            catch (Exception t) {
                this.log.error(t.getMessage(), t);
            }
        }
    }

    private void increaseImported() {
        this.stats.setImported(this.stats.getImported() + 1L);
        this.importFolderCrawler.onFileImported();
    }

    private void increaseUpdated() {
        this.stats.setUpdated(this.stats.getUpdated() + 1L);
        this.importFolderCrawler.onFileImported();
    }

    private void increaseErrors() {
        this.stats.setErrors(this.stats.getErrors() + 1L);
    }

    private void processSourceFile(Object sourceFile, ImportFolder importFolder, CrawlerHelper helper, User owner, ImportFolderCache cache) throws PersistenceException, IOException {
        String sourceRelativePath = helper.getPath(sourceFile);
        Folder parentFolder = this.prepareDocumentTargetFolder(importFolder, sourceRelativePath);
        if (cache.get(sourceRelativePath) != null) {
            ImportFolderCache.CacheEntry entry = cache.get(sourceRelativePath);
            Date lastImported = entry.getDate();
            if (lastImported.compareTo(helper.getLastModified(sourceFile)) < 0) {
                this.reimportExistingDocument(sourceFile, sourceRelativePath, parentFolder, importFolder, helper, owner, cache, entry);
            } else {
                this.log.debug("File {} skipped because already in cache", (Object)sourceRelativePath);
            }
        } else {
            this.processSourceFileNotInChache(sourceFile, sourceRelativePath, parentFolder, importFolder, helper, owner, cache);
        }
    }

    private Document reimportExistingDocument(Object sourceFile, String sourceRelativePath, Folder parentFolder, ImportFolder importFolder, CrawlerHelper helper, User owner, ImportFolderCache cache, ImportFolderCache.CacheEntry entry) throws PersistenceException {
        this.log.debug("Reimporting file {}", (Object)sourceRelativePath);
        Document document = (Document)this.documentDao.findById(entry.getDocId());
        try {
            if (document != null) {
                if (importFolder.getUpdatePolicy() == 0) {
                    if (document.getStatus() != DocumentStatus.UNLOCKED) {
                        this.onError(importFolder, document, sourceRelativePath, new PermissionException(String.format("The document %s is locked or checked out", document.getFileName())));
                    }
                    helper.checkinFile(sourceFile, entry.getDocId(), owner);
                } else {
                    document = helper.importFile(sourceFile, parentFolder, owner, true);
                }
            } else {
                document = helper.importFile(sourceFile, parentFolder, owner, false);
            }
            this.deleteImportedFile(sourceFile, sourceRelativePath, helper, importFolder, owner, document);
            cache.put(new ImportFolderCache.CacheEntry(sourceRelativePath, new Date(), document.getId()));
        }
        catch (Exception e) {
            this.onError(importFolder, document, sourceRelativePath, e);
            document = null;
        }
        return document;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Document processSourceFileNotInChache(Object sourceFile, String sourceRelativePath, Folder parentFolder, ImportFolder importFolder, CrawlerHelper helper, User owner, ImportFolderCache cache) {
        Document document = null;
        try {
            HashMap<String, Object> params = new HashMap<String, Object>();
            params.put("folderId", parentFolder.getId());
            params.put("fileName", helper.getName(sourceFile));
            List<Document> docs = this.documentDao.query("select ld_id, ld_date, ld_status, ld_filename from ld_document where ld_deleted=0 and ld_folderid = :folderId and ld_filename = :fileName ", params, new RowMapper<Document>(){

                public Document mapRow(ResultSet rs, int column) throws SQLException {
                    Document doc = new Document();
                    doc.setId(rs.getLong(1));
                    doc.setDate(rs.getTimestamp(2));
                    int statusCode = rs.getInt(3);
                    doc.setStatus(DocumentStatus.values()[statusCode]);
                    doc.setFileName(rs.getString(4));
                    return doc;
                }
            }, null);
            if (!docs.isEmpty()) {
                document = docs.iterator().next();
                if (document.getStatus() != DocumentStatus.UNLOCKED) {
                    this.onError(importFolder, document, sourceRelativePath, new PermissionException(String.format("The document %s is locked or checked out", document.getFileName())));
                }
                GregorianCalendar cal = new GregorianCalendar();
                cal.setTime(document.getDate());
                cal.set(14, 0);
                Date documentDate = cal.getTime();
                cal.setTime(helper.getLastModified(sourceFile));
                cal.set(14, 0);
                Date fileDate = cal.getTime();
                if (!documentDate.before(fileDate)) {
                    SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                    throw new SkipException(String.format("The document %s dated %s is newer than file %s dated %s", document.getFileName(), df.format(documentDate), sourceRelativePath, df.format(fileDate)));
                }
                if (importFolder.getUpdatePolicy() == 0) {
                    helper.checkinFile(sourceFile, document.getId(), owner);
                } else {
                    document = helper.importFile(sourceFile, parentFolder, owner, true);
                }
            } else {
                document = helper.importFile(sourceFile, parentFolder, owner, false);
            }
            cache.put(new ImportFolderCache.CacheEntry(sourceRelativePath, new Date(), document.getId()));
            this.deleteImportedFile(sourceFile, sourceRelativePath, helper, importFolder, owner, document);
            return document;
        }
        catch (SkipException e) {
            this.log.info(e.getMessage());
            return document;
        }
        catch (Exception e) {
            this.onError(importFolder, document, sourceRelativePath, e);
        }
        return document;
    }

    void deleteImportedFile(Object sourceFile, String sourceRelativePath, CrawlerHelper helper, ImportFolder importFolder, User owner, AbstractDocument document) {
        if (importFolder.getDelImport() == 1) {
            int maxDeletionTrials = 20;
            int currentTrial = 1;
            boolean deleted = false;
            while (!deleted && currentTrial <= maxDeletionTrials) {
                this.log.debug("Trying to delete source file {}", (Object)sourceRelativePath);
                deleted = helper.deleteFile(sourceFile);
                ++currentTrial;
                try {
                    Pause.doPause(1000L);
                }
                catch (InterruptedException e) {
                    this.log.warn("Current thread got interrupted");
                    Thread.currentThread().interrupt();
                }
            }
            if (deleted) {
                this.log.warn("Deleted file: {}", (Object)sourceRelativePath);
            } else {
                this.log.warn("Unable to delete file: {}", (Object)sourceRelativePath);
                this.recordSourceFileDeletionFailure(sourceRelativePath, importFolder, owner, document);
            }
        }
    }

    private void recordSourceFileDeletionFailure(String sourceRelativePath, ImportFolder importFolder, User owner, AbstractDocument document) {
        ImportFolderHistory history = new ImportFolderHistory();
        history.setTenantId(importFolder.getTenantId());
        history.setImportFolderId(importFolder.getId());
        history.setUser(owner);
        history.setDocument(document);
        history.setEvent(ImportFolderEvent.FILE_NOT_DELETED);
        history.setComment("Cannot delete " + sourceRelativePath);
        history.setPathOld(FileUtil.getPath(sourceRelativePath));
        history.setFilenameOld(FileUtil.getName(sourceRelativePath));
        this.saveHistory(history, document, importFolder);
    }

    private Folder prepareDocumentTargetFolder(ImportFolder importFolder, String sourceRelativePath) throws PersistenceException {
        Object dir = "/";
        int lastSlash = sourceRelativePath.lastIndexOf("/");
        if (lastSlash > 0) {
            dir = (String)dir + sourceRelativePath.substring(0, lastSlash);
        }
        return this.prepareFolder(importFolder, (String)dir);
    }

    private void createFoldersSkeleton(ImportFolder importFolder, CrawlerHelper helper, boolean importFromIndex, List<Object> folders) {
        if (importFolder.getImportEmpty() == 1 && !importFromIndex) {
            for (Object fld : folders) {
                try {
                    this.prepareFolder(importFolder, helper.getPath(fld));
                    if (!this.interrupt && this.stats.getImported() < importFolder.getBatch()) continue;
                    this.log.info("Reached {} processed entries in import folder {}", (Object)importFolder.getBatch(), (Object)importFolder.getDisplayUrl());
                    break;
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        }
    }

    private boolean checkExistenceIndexFile(CrawlerHelper helper) {
        boolean importFromIndex = false;
        int indexCount = 0;
        try {
            indexCount = helper.importDocumentsCount();
            if (indexCount >= 0) {
                importFromIndex = true;
                this.log.info("Index file index.xml or index.csv found");
                this.log.info("{} documents declared in the header of the index file", (Object)indexCount);
            }
        }
        catch (Exception e) {
            this.log.error(e.getMessage(), e);
        }
        return importFromIndex;
    }

    public void afterCrawlingCompleted(ImportFolder importFolder) {
        if (StringUtils.isNotEmpty(importFolder.getAutomationEnd())) {
            HashMap<String, Object> dictionary = new HashMap<String, Object>();
            dictionary.put(IMPORT_FOLDER, importFolder);
            dictionary.put("imports", new ArrayList<DocumentHistory>(this.stats.getImportFolderImports()));
            dictionary.put("errors", new ArrayList<ImportFolderHistory>(this.stats.getImportFolderErrors()));
            Automation script = new Automation("importfolder-end", importFolder.getLocale(), importFolder.getTenantId());
            try {
                script.evaluate(importFolder.getAutomationEnd(), dictionary);
            }
            catch (AutomationException e) {
                LoggerFactory.getLogger(ImportFolderCrawler.class).warn(e.getMessage(), e);
            }
        }
    }

    public static void beforeDocumentImported(ImportFolder importFolder, DocumentHistory transaction) {
        if (StringUtils.isNotEmpty(importFolder.getAutomation())) {
            HashMap<String, Object> dictionary = new HashMap<String, Object>();
            dictionary.put(IMPORT_FOLDER, importFolder);
            dictionary.put("document", transaction.getDocument());
            dictionary.put("path", transaction.getPathOld());
            dictionary.put("event", transaction);
            Automation script = new Automation("importfolder", importFolder.getLocale(), importFolder.getTenantId());
            try {
                script.evaluate(importFolder.getAutomation(), dictionary);
            }
            catch (AutomationException e) {
                LoggerFactory.getLogger(ImportFolderCrawler.class).warn(e.getMessage(), e);
            }
        }
    }

    public void afterDocumentImported(ImportFolder importFolder, DocumentHistory transaction) {
        boolean newFile = DocumentEvent.STORED.toString().equals(transaction.getEvent());
        if (newFile) {
            this.increaseImported();
        } else {
            this.increaseUpdated();
        }
        this.sequenceDao.next(COUNTER_SEQUENCE, importFolder.getId(), importFolder.getTenantId());
        if (StringUtils.isNotEmpty(importFolder.getAutomationEnd())) {
            this.stats.getImportFolderImports().add(transaction);
        }
        String originalPath = transaction.getPathOld();
        this.log.debug("Imported/Updated document {} from {}", (Object)transaction.getDocument(), (Object)originalPath);
        ImportFolderHistory history = new ImportFolderHistory();
        history.setTenantId(importFolder.getTenantId());
        history.setImportFolderId(importFolder.getId());
        history.setEvent(newFile ? ImportFolderEvent.IMPORTED : ImportFolderEvent.UPDATED);
        history.setPathOld(FileUtil.getPath(originalPath));
        history.setFilenameOld(FileUtil.getName(originalPath));
        this.saveHistory(history, transaction.getDocument(), importFolder);
        if (StringUtils.isNotEmpty(importFolder.getAutomationAfter())) {
            HashMap<String, Object> dictionary = new HashMap<String, Object>();
            dictionary.put(IMPORT_FOLDER, importFolder);
            dictionary.put("document", transaction.getDocument());
            dictionary.put("path", originalPath);
            dictionary.put("event", history);
            Automation script = new Automation("importfolder-after", importFolder.getLocale(), importFolder.getTenantId());
            try {
                script.evaluate(importFolder.getAutomationAfter(), dictionary);
            }
            catch (AutomationException e) {
                this.log.warn(e.getMessage(), e);
            }
        }
    }

    public void onError(ImportFolder importFolder, Document document, String originalPath, Throwable error) {
        this.increaseErrors();
        this.log.error("Error importing {}", (Object)originalPath);
        this.log.error(error.getMessage(), error);
        ImportFolderHistory history = new ImportFolderHistory();
        history.setTenantId(importFolder.getTenantId());
        history.setImportFolderId(importFolder.getId());
        history.setEvent(ImportFolderEvent.ERROR);
        history.setPathOld(FileUtil.getPath(originalPath));
        history.setFilenameOld(FileUtil.getName(originalPath));
        history.setComment(StringUtils.abbreviate(error.getMessage(), 4000));
        this.saveHistory(history, document, importFolder);
        if (StringUtils.isNotEmpty(importFolder.getAutomationEnd())) {
            this.stats.getImportFolderErrors().add(history);
        }
    }

    private void saveHistory(ImportFolderHistory history, AbstractDocument document, ImportFolder importFolder) {
        if (importFolder.getRecordHistory() == 1 && document != null && document instanceof Document) {
            Document doc = (Document)document;
            try {
                this.importFolderHistoryDao.store(history, doc);
            }
            catch (Exception e) {
                this.log.error("Cannot save history for document {} from import folder {}", document.getFileName(), importFolder.getDisplayUrl(), e);
            }
        }
    }

    protected Folder prepareFolder(ImportFolder importFolder, String dirPath) throws PersistenceException {
        FolderHistory transaction = new FolderHistory();
        User owner = this.userDao.findByUsername(IMPORTFOLDER_USERNAME);
        transaction.setUser(owner);
        FolderDAO folderDao = Context.get(FolderDAO.class);
        Folder folder = folderDao.findFolder(importFolder.getTargetFolderId());
        Folder f = null;
        if (folder != null) {
            f = folderDao.createPath(folder, dirPath, importFolder.getInheritRights() == 1, transaction);
        }
        this.log.debug("Prepared folder {}", (Object)dirPath);
        return f;
    }
}

