package com.moonlit.logfaces.client.gis;

import java.net.InetAddress;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.code.regexp.Matcher;
import com.google.code.regexp.Pattern;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.moonlit.logfaces.server.core.LogEvent;
import com.moonlit.logfaces.server.core.LogFacesPlugin;

public class IpApi implements LogFacesPlugin{
	private static final Logger log = LogManager.getLogger(IpApi.class);
	private Map<String, LocationVO> resolved;
	private static Pattern ipp;
	private Gson gson;
	private HttpClient client;

	public IpApi() {
		try {
			ipp = Pattern.compile("(?<![0-9])(?:(?:25[0-5]|2[0-4][0-9]|[0-1]?[0-9]{1,2})[.](?:25[0-5]|2[0-4][0-9]|[0-1]?[0-9]{1,2})[.](?:25[0-5]|2[0-4][0-9]|[0-1]?[0-9]{1,2})[.](?:25[0-5]|2[0-4][0-9]|[0-1]?[0-9]{1,2}))(?![0-9])");
			resolved = Maps.newHashMap();
			client = HttpClientBuilder.create().build();
			gson = new GsonBuilder().setPrettyPrinting().create();
		}
		catch(Exception e) {
			log.error(e.getMessage(), e);
		}
	}

	@Override
	public List<String> getArgs() {
		return Lists.newArrayList();
	}

	@Override
	public String getName() {
		return "IP2Geolocation converter";
	}

	@Override
	public Object handleEvents(List<LogEvent> events, Map<String, String> args) {
		if(args != null && args.containsKey("refresh"))
			resolved.clear();
		if(args.containsKey("ips"))
			return ips2locations(args.get("ips"));
		if(events == null || events.isEmpty())
			return getDefaultLocation();

		// aggregate IP's from events		
		Set<String> ips = Sets.newHashSet();
		for(LogEvent event : events) {
			if(!event.getProperties().containsKey("gis.ips"))
				ips.addAll(event2ips(event));
		}
			
		if(ips.isEmpty())
			return null;
			
		// convert IP's to locations and store in local memory
		resolve(ips);
		
		// tag events with resolved locations
		for(LogEvent event : events) {
			if(!event.getProperties().containsKey("gis.locations"))
				event2locations(event);
		}
		return null;
	}

	@Override
	public Object validate(Map<String, String> arg0) {
		return true;
	}
	
	private String getDefaultLocation() {
		LocationVO location = resolve("");
		return (location != null) ? location.toString() : null;
	}
	
	private LocationVO resolve(String ip) {
		if(resolved.containsKey(ip))
			return resolved.get(ip);
		
		try {
			String uri = "http://ip-api.com/json/{ip}";
			uri = StringUtils.replace(uri, "{ip}", ip);
			HttpUriRequest request = RequestBuilder.get().setUri(uri).setHeader(HttpHeaders.ACCEPT, "application/json").build();
			HttpResponse response = client.execute(request);
			if(response.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
				log.warn("can't resolve {}, error {}", uri, response.getStatusLine().getStatusCode());
				return null;
			}
			
			HttpEntity entity = response.getEntity();
			String responseString = EntityUtils.toString(entity, "UTF-8");
			Map<String,Object> data = gson.fromJson(responseString, Map.class);
			if(data.get("lon") == null || data.get("lat") == null || data.get("query") == null)
				return null;
			LocationVO location = new LocationVO(data);
			resolved.put(ip, location);
			EntityUtils.consumeQuietly(entity);
			return location;
		} catch (Exception e) {
			log.error(e.getMessage(), e);
			return null;
		}
	}

	private Set<LocationVO> resolve(Collection<String> ips) {
		Set<LocationVO> locations = Sets.newHashSet();
		List<String> list = Lists.newArrayList();
		
		// collected cached locations
		for(String ip : ips) {
			if(resolved.containsKey(ip)) {
				locations.add(resolved.get(ip));
				continue;
			}
			
			list.add(ip);
		}

		// page through the rest
		log.info("resolving {} ips, paging {}, cached {}", ips.size(), list.size(), ips.size() - list.size());
		int pageSize = 100, from = 0, to = from + pageSize;
		while(from < list.size()) {
			if(to >= list.size())
				to = to - (to - list.size());
			
			List<String> temp = list.subList(from, to);
			locations.addAll(page(temp));
			
			from += pageSize;
			to = from + pageSize;
		}
		
		log.info("resolved {} ips to {} locations", ips.size(), locations.size());
		return locations;
	}
	
	private Set<LocationVO> page(List<String> ips) {
		Set<LocationVO> locations = Sets.newHashSet();

		String uri = "http://ip-api.com/batch";
		String body = gson.toJson(ips);
		
		try {
			HttpPost post = new HttpPost(uri);
			post.setHeader("Accept", "*/*");
			post.setHeader("Content-Type", "application/json");
			post.setEntity(new StringEntity(body, "UTF-8"));

			HttpResponse hr = client.execute(post);
			HttpEntity entity = hr.getEntity();
			String content = IOUtils.toString(entity.getContent(), "UTF-8");
			if(hr.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
				log.warn("can't resolve {}, error: {}", uri, hr.getStatusLine().getStatusCode());
				return locations;
			}
			
			List<Map<String,Object>> map = Lists.newArrayList();
			List<Map<String,Object>> response = gson.fromJson(content, map.getClass());
			EntityUtils.consumeQuietly(entity);
			
			for(Map<String,Object> data : response) {
				if(data.get("lon") == null || data.get("lat") == null || data.get("query") == null)
					continue;
				String ip = data.get("query").toString();
				LocationVO location = new LocationVO(data);
				locations.add(location);
				resolved.put(ip, location);
			}
			
		} catch (Exception e) {
			log.error(e.getMessage(), e);
		}
		
		return locations;
	}
	
	private List<String> ips2locations(String csv){
		List<String> ips = Arrays.asList(StringUtils.split(csv,","));
		resolve(ips);
		
		List<String> list = Lists.newArrayList();
		for(String ip : ips) {
			LocationVO location = resolved.get(ip);
			if(location != null)
				list.add(location.toString());
		}

		return list;
	}

	private void event2locations(LogEvent event){
		String ips = event.getProperties().get("gis.ips");
		if(ips == null)
			return;
		List<String> list = Lists.newArrayList();
		for(String ip : StringUtils.split(ips,",")) {
			LocationVO location = resolved.get(ip);
			if(location != null) {
				list.add(location.toString());
			}
		}
		
		if(!list.isEmpty())
			event.getProperties().put("gis.location", list.toString());
	}
	
	private Set<String> event2ips(LogEvent event){
		Set<String> ips = Sets.newHashSet();
		matchIPs(event.getMessage(), ips);
		matchIPs(event.getHostName(), ips);
		
		Map<String, String> props = event.getProperties();
		for(String key : props.keySet())
			matchIPs(props.get(key), ips);
		
		event.getProperties().put("gis.ips", StringUtils.join(ips, ","));
		return ips;
	}

	private void matchIPs(String text, Set<String> set) {
		if(text == null)
			return;
		Matcher matcher = ipp.matcher(text);
		while (matcher.find()) {
			try {
				String ip = text.substring(matcher.start(), matcher.end());
				if(ip.contains("0.0.0"))
					continue;
				InetAddress a = InetAddress.getByName(ip);
				if(a.isSiteLocalAddress() || a.isLoopbackAddress() || a.isAnyLocalAddress() || a.isLinkLocalAddress())
					continue;
				set.add(ip);
			} catch (Exception e) {
			}
		}
	}

	
	class LocationVO {
		private Map<String,Object> data;
		public LocationVO(Map<String,Object> data) {
			this.data = data;
			this.data.remove("status");
			this.data.remove("org");
			Object ip = this.data.remove("query");
			if(ip != null)
				this.data.put("ip", ip);
		}
	
		@Override
		public String toString() {
			return gson.toJson(data);
		}
	}
	
}
