# filedata.py - generate displayable file data
#
# Copyright 2011 Steve Borho <steve@borho.org>
#
# This software may be used and distributed according to the terms of the
# GNU General Public License version 2, incorporated herein by reference.

import os
import cStringIO

from mercurial import error, match, patch, util, mdiff
from mercurial import ui as uimod
from hgext import record

from tortoisehg.util import hglib, patchctx
from tortoisehg.hgqt.i18n import _

forcedisplaymsg = _('Display the file anyway')

def _exceedsMaxLineLength(data, maxlength=100000):
    if len(data) < maxlength:
        return False
    for line in data.splitlines():
        if len(line) > maxlength:
            return True
    return False

class FileData(object):
    def __init__(self, ctx, ctx2, wfile, status=None, changeselect=False, force=False):
        self.contents = None
        self.ucontents = None
        self.error = None
        self.olddata = None
        self.diff = None
        self.flabel = u''
        self.elabel = u''
        self.changes = None
        try:
            self.readStatus(ctx, ctx2, wfile, status, changeselect, force)
        except (EnvironmentError, error.LookupError), e:
            self.error = hglib.tounicode(str(e))

    def checkMaxDiff(self, ctx, wfile, maxdiff, status, force):
        self.error = None
        p = _('File or diffs not displayed: ')
        try:
            fctx = ctx.filectx(wfile)
            if ctx.rev() is None:
                size = fctx.size()
            else:
                # fctx.size() can read all data into memory in rename cases so
                # we read the size directly from the filelog, this is deeper
                # under the API than I prefer to go, but seems necessary
                size = fctx._filelog.rawsize(fctx.filerev())
        except (EnvironmentError, error.LookupError), e:
            self.error = p + hglib.tounicode(str(e))
            return None
        if not force and size > maxdiff:
            self.error = p + _('File is larger than the specified max size.\n'
                               'maxdiff = %s KB') % (maxdiff // 1024)
            self.error += u'\n\n' + forcedisplaymsg
            return None

        try:
            data = fctx.data()
            if not force:
                if '\0' in data or ctx.isStandin(wfile):
                    self.error = p + _('File is binary')
                elif _exceedsMaxLineLength(data):
                    # it's incredibly slow to render long line by QScintilla
                    self.error = p + \
                        _('File may be binary (maximum line length exceeded)')
            if self.error:
                self.error += u'\n\n' + forcedisplaymsg
                if status != 'A':
                    return None

                renamed = fctx.renamed()
                if renamed:
                    oldname, node = renamed
                    fr = hglib.tounicode(oldname)
                    if oldname in ctx:
                        self.flabel += _(' <i>(copied from %s)</i>') % fr
                    else:
                        self.flabel += _(' <i>(renamed from %s)</i>') % fr
                else:
                    self.flabel += _(' <i>(was added)</i>')

                return None
        except (EnvironmentError, util.Abort), e:
            self.error = p + hglib.tounicode(str(e))
            return None
        return fctx, data

    def isValid(self):
        return self.error is None

    def readStatus(self, ctx, ctx2, wfile, status, changeselect, force):
        def getstatus(repo, n1, n2, wfile):
            m = match.exact(repo.root, repo.getcwd(), [wfile])
            modified, added, removed = repo.status(n1, n2, match=m)[:3]
            if wfile in modified:
                return 'M'
            if wfile in added:
                return 'A'
            if wfile in removed:
                return 'R'
            if wfile in ctx:
                return 'C'
            return None

        isbfile = False
        repo = ctx._repo
        maxdiff = repo.maxdiff
        self.flabel += u'<b>%s</b>' % hglib.tounicode(wfile)

        if isinstance(ctx, patchctx.patchctx):
            self.diff = ctx.thgmqpatchdata(wfile)
            flags = ctx.flags(wfile)
            if flags == 'x':
                self.elabel = _("exec mode has been <font color='red'>set</font>")
            elif flags == '-':
                self.elabel = _("exec mode has been <font color='red'>unset</font>")
            elif flags == 'l':
                self.flabel += _(' <i>(is a symlink)</i>')

            # Do not show patches that are too big or may be binary
            if not force:
                p = _('Diff not displayed: ')
                data = self.diff
                size = len(data)
                if (size > maxdiff):
                    self.error = p + _('File is larger than the specified max size.\n'
                                   'maxdiff = %s KB') % (maxdiff // 1024)
                elif '\0' in data:
                    self.error = p + _('File is binary')
                elif _exceedsMaxLineLength(data):
                    # it's incredibly slow to render long line by QScintilla
                    self.error = p + \
                        _('File may be binary (maximum line length exceeded)')
                if self.error:
                    self.error += u'\n\n' + forcedisplaymsg

            return

        if ctx2:
            # If a revision to compare to was provided, we must put it in
            # the context of the subrepo as well
            if ctx2._repo.root != ctx._repo.root:
                wsub2, wfileinsub2, sctx2 = \
                    hglib.getDeepestSubrepoContainingFile(wfile, ctx2)
                if wsub2:
                    ctx2 = sctx2

        absfile = repo.wjoin(wfile)
        if (wfile in ctx and 'l' in ctx.flags(wfile)) or \
           os.path.islink(absfile):
            if wfile in ctx:
                data = ctx[wfile].data()
            else:
                data = os.readlink(absfile)
            self.contents = data
            self.flabel += _(' <i>(is a symlink)</i>')
            return

        if status is None:
            status = getstatus(repo, ctx.p1().node(), ctx.node(), wfile)
        if ctx2 is None:
            ctx2 = ctx.p1()

        if status == 'S':
            try:
                from mercurial import subrepo, commands

                def genSubrepoRevChangedDescription(subrelpath, sfrom, sto, repo):
                    """Generate a subrepository revision change description"""
                    out = []
                    def getLog(_ui, srepo, opts):
                        _ui.pushbuffer()
                        logOutput = ''
                        try:
                            commands.log(_ui, srepo, **opts)
                            logOutput = _ui.popbuffer()
                        except error.ParseError, e:
                            # Some mercurial versions have a bug that results in
                            # saving a subrepo node id in the .hgsubstate file
                            # which ends with a "+" character. If that is the
                            # case, add a warning to the output, but try to
                            # get the revision information anyway
                            for n, rev in enumerate(opts['rev']):
                                if rev.endswith('+'):
                                    logOutput += _('[WARNING] Invalid subrepo '
                                        'revision ID:\n\t%s\n\n') % rev
                                    opts['rev'][n] = rev[:-1]
                            commands.log(_ui, srepo, **opts)
                            logOutput += _ui.popbuffer()
                        return logOutput

                    opts = {'date':None, 'user':None, 'rev':[sfrom]}
                    subabspath = os.path.join(repo.root, subrelpath)
                    missingsub = not os.path.isdir(subabspath)
                    sfromlog = ''
                    def isinitialrevision(rev):
                        return all([el == '0' for el in rev])
                    if isinitialrevision(sfrom):
                        sfrom = ''
                    if isinitialrevision(sto):
                        sto = ''
                    header = ''
                    if not sfrom and not sto:
                        sstatedesc = 'new'
                        out.append(_('Subrepo created and set to initial revision.') + u'\n\n')
                        return out, sstatedesc
                    elif not sfrom:
                        sstatedesc = 'new'
                        header = _('Subrepo initialized to revision:') + u'\n\n'
                    elif not sto:
                        sstatedesc = 'removed'
                        out.append(_('Subrepo removed from repository.') + u'\n\n')
                        return out, sstatedesc
                    elif sfrom == sto:
                        sstatedesc = 'unchanged'
                        header = _('Subrepo was not changed.')
                        slog = _('changeset: %s') % sfrom[:12] + u'\n'
                        if missingsub:
                            header = _('[WARNING] Missing subrepo. '
                                   'Update to this revision to clone it.') \
                                 + u'\n\n' + header
                        else:
                            try:
                                subinfo = getLog(_ui, srepo, opts)
                                slog = hglib.tounicode(subinfo)
                            except error.RepoError:
                                header = _('[WARNING] Incomplete subrepo. '
                                   'Update to this revision to pull it.') \
                                 + u'\n\n' + header
                        out.append(header + u' ')
                        out.append(_('Subrepo state is:') + u'\n\n' + slog)
                        return out, sstatedesc
                    else:
                        sstatedesc = 'changed'

                        header = _('Revision has changed to:') + u'\n\n'
                        sfromlog = _('changeset: %s') % sfrom[:12] + u'\n\n'
                        if not missingsub:
                            try:
                                sfromlog = hglib.tounicode(getLog(_ui, srepo, opts))
                            except error.RepoError:
                                sfromlog = _('changeset: %s '
                                             '(not found on subrepository)') \
                                                % sfrom[:12] + u'\n\n'
                        sfromlog = _('From:') + u'\n' + sfromlog

                    stolog = ''
                    if missingsub:
                        header = _(
                            '[WARNING] Missing changed subrepository. '
                            'Update to this revision to clone it.') \
                            + u'\n\n' + header
                        stolog = _('changeset: %s') % sto + '\n\n'
                        sfromlog += _(
                            'Subrepository not found in the working '
                            'directory.') + '\n'
                    else:
                        try:
                            opts['rev'] = [sto]
                            stolog = getLog(_ui, srepo, opts)
                            if not stolog:
                                stolog = _('Initial revision') + u'\n'
                        except error.RepoError:
                            header = _(
                                '[WARNING] Incomplete changed subrepository. '
                                'Update to this revision to pull it.') \
                                 + u'\n\n' + header
                            stolog = _('changeset: %s '
                                       '(not found on subrepository)') \
                                     % sto[:12] + u'\n\n'
                    out.append(header)
                    out.append(hglib.tounicode(stolog))

                    if sfromlog:
                        out.append(hglib.tounicode(sfromlog))

                    return out, sstatedesc

                srev = ctx.substate.get(wfile, subrepo.nullstate)[1]
                srepo = None
                try:
                    subabspath = os.path.join(ctx._repo.root, wfile)
                    if not os.path.isdir(subabspath):
                        sactual = ''
                    else:
                        sub = ctx.sub(wfile)
                        if isinstance(sub, subrepo.hgsubrepo):
                            srepo = sub._repo
                            sactual = srepo['.'].hex()
                        else:
                            self.error = _('Not a Mercurial subrepo, not previewable')
                            return
                except (util.Abort, KeyError), e:
                    self.error = (_('Error previewing subrepo: %s')
                                  % hglib.tounicode(str(e)))
                    return

                out = []
                _ui = uimod.ui()

                if srepo is None or ctx.rev() is not None:
                    data = []
                else:
                    _ui.pushbuffer()
                    commands.status(_ui, srepo, modified=True, added=True, removed=True, deleted=True)
                    data = _ui.popbuffer()
                    if data:
                        out.append(_('The subrepository is dirty.') + u' ' + _('File Status:') + u'\n')
                        out.append(hglib.tounicode(data))
                        out.append(u'\n')

                sstatedesc = 'changed'
                if ctx.rev() is not None:
                    sparent = ctx2.substate.get(wfile, subrepo.nullstate)[1]
                    subrepochange, sstatedesc = \
                        genSubrepoRevChangedDescription(wfile,
                            sparent, srev, ctx._repo)
                    out += subrepochange
                else:
                    sstatedesc = 'dirty'
                    if srev != sactual:
                        subrepochange, sstatedesc = \
                            genSubrepoRevChangedDescription(wfile,
                                srev, sactual, ctx._repo)
                        out += subrepochange
                        if data:
                            sstatedesc += ' and dirty'
                    elif srev and not sactual:
                        sstatedesc = 'removed'
                self.ucontents = u''.join(out).strip()

                lbl = {
                    'changed':   _('(is a changed sub-repository)'),
                    'unchanged':   _('(is an unchanged sub-repository)'),
                    'dirty':   _('(is a dirty sub-repository)'),
                    'new':   _('(is a new sub-repository)'),
                    'removed':   _('(is a removed sub-repository)'),
                    'changed and dirty':   _('(is a changed and dirty sub-repository)'),
                    'new and dirty':   _('(is a new and dirty sub-repository)'),
                    'removed and dirty':   _('(is a removed sub-repository)')
                }[sstatedesc]
                self.flabel += ' <i>' + lbl + '</i>'
                if sactual:
                    lbl = ' <a href="repo:%%s">%s</a>' % _('open...')
                    self.flabel += lbl % hglib.tounicode(srepo.root)
            except (EnvironmentError, error.RepoError, util.Abort), e:
                self.error = _('Error previewing subrepo: %s') % \
                        hglib.tounicode(str(e))
            return

        # TODO: elif check if a subdirectory (for manifest tool)

        mde = _('File or diffs not displayed: '
                'File is larger than the specified max size.\n'
                'maxdiff = %s KB') % (maxdiff // 1024)

        if status in ('R', '!'):
            if wfile in ctx.p1():
                fctx = ctx.p1()[wfile]
                if fctx._filelog.rawsize(fctx.filerev()) > maxdiff:
                    self.error = mde
                else:
                    olddata = fctx.data()
                    if '\0' in olddata:
                        self.error = 'binary file'
                    else:
                        self.contents = olddata
                self.flabel += _(' <i>(was deleted)</i>')
            elif hasattr(ctx.p1(), 'hasStandin') and ctx.p1().hasStandin(wfile):
                self.error = 'binary file'
                self.flabel += _(' <i>(was deleted)</i>')
            else:
                self.flabel += _(' <i>(was added, now missing)</i>')
            return

        if status in ('I', '?', 'C'):
            if ctx.rev() is None:
                if status in ('I', '?'):
                    self.flabel += _(' <i>(is unversioned)</i>')
                if os.path.getsize(absfile) > maxdiff:
                    self.error = mde
                    return
                else:
                    data = util.posixfile(absfile, 'r').read()
            elif ctx.hasStandin(wfile):
                data = '\0'
            else:
                data = ctx.filectx(wfile).data()
            if not force and '\0' in data:
                self.error = 'binary file'
            else:
                self.contents = data
            return

        if status in ('M', 'A'):
            if ctx.hasStandin(wfile):
                wfile = ctx.findStandin(wfile)
                isbfile = True
            res = self.checkMaxDiff(ctx, wfile, maxdiff, status, force)
            if res is None:
                return
            fctx, newdata = res
            self.contents = newdata
            for pctx in ctx.parents():
                if 'x' in fctx.flags() and 'x' not in pctx.flags(wfile):
                    self.elabel = _("exec mode has been <font color='red'>set</font>")
                elif 'x' not in fctx.flags() and 'x' in pctx.flags(wfile):
                    self.elabel = _("exec mode has been <font color='red'>unset</font>")

        if status == 'A':
            renamed = fctx.renamed()
            if not renamed:
                self.flabel += _(' <i>(was added)</i>')
                return

            oldname, node = renamed
            fr = hglib.tounicode(oldname)
            if oldname in ctx:
                self.flabel += _(' <i>(copied from %s)</i>') % fr
            else:
                self.flabel += _(' <i>(renamed from %s)</i>') % fr
            olddata = repo.filectx(oldname, fileid=node).data()
        elif status == 'M':
            if wfile not in ctx2:
                # merge situation where file was added in other branch
                self.flabel += _(' <i>(was added)</i>')
                return
            oldname = wfile
            olddata = ctx2[wfile].data()
        else:
            return

        self.olddata = olddata
        if changeselect:
            diffopts = patch.diffopts(repo.ui, {})
            diffopts.git = True
            m = match.exact(repo.root, repo.root, [wfile])
            fp = cStringIO.StringIO()
            for c in patch.diff(repo, ctx.node(), None, match=m, opts=diffopts):
                fp.write(c)
            fp.seek(0)

            # feed diffs through record.parsepatch() for more fine grained
            # chunk selection
            filediffs = record.parsepatch(fp)
            if filediffs:
                self.changes = filediffs[0]
            else:
                self.diff = ''
                return
            self.changes.excludecount = 0
            values = []
            lines = 0
            for chunk in self.changes.hunks:
                buf = cStringIO.StringIO()
                chunk.write(buf)
                chunk.excluded = False
                val = buf.getvalue()
                values.append(val)
                chunk.lineno = lines
                chunk.linecount = len(val.splitlines())
                lines += chunk.linecount
            self.diff = ''.join(values)
        else:
            diffopts = patch.diffopts(repo.ui, {})
            diffopts.git = False
            newdate = util.datestr(ctx.date())
            olddate = util.datestr(ctx2.date())
            if isbfile:
                olddata += '\0'
                newdata += '\0'
            difftext = mdiff.unidiff(olddata, olddate, newdata, newdate,
                                     oldname, wfile, opts=diffopts)
            if difftext:
                self.diff = ('diff -r %s -r %s %s\n' % (ctx, ctx2, oldname)
                             + difftext)
            else:
                self.diff = ''
