/*
 *  Remodal - v1.1.0
 *  Responsive, lightweight, fast, synchronized with CSS animations, fully customizable modal window plugin with declarative configuration and hash tracking.
 *  http://vodkabears.github.io/remodal/
 *
 *  Made by Ilya Makarov
 *  Under MIT License
 */

!(function(root, factory) {
  if (typeof define === 'function' && define.amd) {
    define(['jquery'], function($) {
      return factory(root, $);
    });
  } else if (typeof exports === 'object') {
    factory(root, require('jquery'));
  } else {
    factory(root, root.jQuery || root.Zepto);
  }
})(this, function(global, $) {

  'use strict';

  /**
   * Name of the plugin
   * @private
   * @const
   * @type {String}
   */
  var PLUGIN_NAME = 'remodal';

  /**
   * Namespace for CSS and events
   * @private
   * @const
   * @type {String}
   */
  var NAMESPACE = global.REMODAL_GLOBALS && global.REMODAL_GLOBALS.NAMESPACE || PLUGIN_NAME;

  /**
   * Animationstart event with vendor prefixes
   * @private
   * @const
   * @type {String}
   */
  var ANIMATIONSTART_EVENTS = $.map(
    ['animationstart', 'webkitAnimationStart', 'MSAnimationStart', 'oAnimationStart'],

    function(eventName) {
      return eventName + '.' + NAMESPACE;
    }

  ).join(' ');

  /**
   * Animationend event with vendor prefixes
   * @private
   * @const
   * @type {String}
   */
  var ANIMATIONEND_EVENTS = $.map(
    ['animationend', 'webkitAnimationEnd', 'MSAnimationEnd', 'oAnimationEnd'],

    function(eventName) {
      return eventName + '.' + NAMESPACE;
    }

  ).join(' ');

  /**
   * Default settings
   * @private
   * @const
   * @type {Object}
   */
  var DEFAULTS = $.extend({
    hashTracking: true,
    closeOnConfirm: true,
    closeOnCancel: true,
    closeOnEscape: true,
    closeOnOutsideClick: true,
    modifier: '',
    appendTo: null
  }, global.REMODAL_GLOBALS && global.REMODAL_GLOBALS.DEFAULTS);

  /**
   * States of the Remodal
   * @private
   * @const
   * @enum {String}
   */
  var STATES = {
    CLOSING: 'closing',
    CLOSED: 'closed',
    OPENING: 'opening',
    OPENED: 'opened'
  };

  /**
   * Reasons of the state change.
   * @private
   * @const
   * @enum {String}
   */
  var STATE_CHANGE_REASONS = {
    CONFIRMATION: 'confirmation',
    CANCELLATION: 'cancellation'
  };

  /**
   * Is animation supported?
   * @private
   * @const
   * @type {Boolean}
   */
  var IS_ANIMATION = (function() {
    var style = document.createElement('div').style;

    return style.animationName !== undefined ||
      style.WebkitAnimationName !== undefined ||
      style.MozAnimationName !== undefined ||
      style.msAnimationName !== undefined ||
      style.OAnimationName !== undefined;
  })();

  /**
   * Is iOS?
   * @private
   * @const
   * @type {Boolean}
   */
  var IS_IOS = /iPad|iPhone|iPod/.test(navigator.platform);

  /**
   * Current modal
   * @private
   * @type {Remodal}
   */
  var current;

  /**
   * Scrollbar position
   * @private
   * @type {Number}
   */
  var scrollTop;

  /**
   * Returns an animation duration
   * @private
   * @param {jQuery} $elem
   * @returns {Number}
   */
  function getAnimationDuration($elem) {
    if (
      IS_ANIMATION &&
      $elem.css('animation-name') === 'none' &&
      $elem.css('-webkit-animation-name') === 'none' &&
      $elem.css('-moz-animation-name') === 'none' &&
      $elem.css('-o-animation-name') === 'none' &&
      $elem.css('-ms-animation-name') === 'none'
    ) {
      return 0;
    }

    var duration = $elem.css('animation-duration') ||
      $elem.css('-webkit-animation-duration') ||
      $elem.css('-moz-animation-duration') ||
      $elem.css('-o-animation-duration') ||
      $elem.css('-ms-animation-duration') ||
      '0s';

    var delay = $elem.css('animation-delay') ||
      $elem.css('-webkit-animation-delay') ||
      $elem.css('-moz-animation-delay') ||
      $elem.css('-o-animation-delay') ||
      $elem.css('-ms-animation-delay') ||
      '0s';

    var iterationCount = $elem.css('animation-iteration-count') ||
      $elem.css('-webkit-animation-iteration-count') ||
      $elem.css('-moz-animation-iteration-count') ||
      $elem.css('-o-animation-iteration-count') ||
      $elem.css('-ms-animation-iteration-count') ||
      '1';

    var max;
    var len;
    var num;
    var i;

    duration = duration.split(', ');
    delay = delay.split(', ');
    iterationCount = iterationCount.split(', ');

    // The 'duration' size is the same as the 'delay' size
    for (i = 0, len = duration.length, max = Number.NEGATIVE_INFINITY; i < len; i++) {
      num = parseFloat(duration[i]) * parseInt(iterationCount[i], 10) + parseFloat(delay[i]);

      if (num > max) {
        max = num;
      }
    }

    return max;
  }

  /**
   * Returns a scrollbar width
   * @private
   * @returns {Number}
   */
  function getScrollbarWidth() {
    if ($(document.body).height() <= $(window).height()) {
      return 0;
    }

    var outer = document.createElement('div');
    var inner = document.createElement('div');
    var widthNoScroll;
    var widthWithScroll;

    outer.style.visibility = 'hidden';
    outer.style.width = '100px';
    document.body.appendChild(outer);

    widthNoScroll = outer.offsetWidth;

    // Force scrollbars
    outer.style.overflow = 'scroll';

    // Add inner div
    inner.style.width = '100%';
    outer.appendChild(inner);

    widthWithScroll = inner.offsetWidth;

    // Remove divs
    outer.parentNode.removeChild(outer);

    return widthNoScroll - widthWithScroll;
  }

  /**
   * Locks the screen
   * @private
   */
  function lockScreen() {
    if (IS_IOS) {
      return;
    }

    var $html = $('html');
    var lockedClass = namespacify('is-locked');
    var paddingRight;
    var $body;

    if (!$html.hasClass(lockedClass)) {
      $body = $(document.body);

      // Zepto does not support '-=', '+=' in the `css` method
      paddingRight = parseInt($body.css('padding-right'), 10) + getScrollbarWidth();

      $body.css('padding-right', paddingRight + 'px');
      $html.addClass(lockedClass);
    }
  }

  /**
   * Unlocks the screen
   * @private
   */
  function unlockScreen() {
    if (IS_IOS) {
      return;
    }

    var $html = $('html');
    var lockedClass = namespacify('is-locked');
    var paddingRight;
    var $body;

    if ($html.hasClass(lockedClass)) {
      $body = $(document.body);

      // Zepto does not support '-=', '+=' in the `css` method
      paddingRight = parseInt($body.css('padding-right'), 10) - getScrollbarWidth();

      $body.css('padding-right', paddingRight + 'px');
      $html.removeClass(lockedClass);
    }
  }

  /**
   * Sets a state for an instance
   * @private
   * @param {Remodal} instance
   * @param {STATES} state
   * @param {Boolean} isSilent If true, Remodal does not trigger events
   * @param {String} Reason of a state change.
   */
  function setState(instance, state, isSilent, reason) {

    var newState = namespacify('is', state);
    var allStates = [namespacify('is', STATES.CLOSING),
                     namespacify('is', STATES.OPENING),
                     namespacify('is', STATES.CLOSED),
                     namespacify('is', STATES.OPENED)].join(' ');

    instance.$bg
      .removeClass(allStates)
      .addClass(newState);

    instance.$overlay
      .removeClass(allStates)
      .addClass(newState);

    instance.$wrapper
      .removeClass(allStates)
      .addClass(newState);

    instance.$modal
      .removeClass(allStates)
      .addClass(newState);

    instance.state = state;
    !isSilent && instance.$modal.trigger({
      type: state,
      reason: reason
    }, [{ reason: reason }]);
  }

  /**
   * Synchronizes with the animation
   * @param {Function} doBeforeAnimation
   * @param {Function} doAfterAnimation
   * @param {Remodal} instance
   */
  function syncWithAnimation(doBeforeAnimation, doAfterAnimation, instance) {
    var runningAnimationsCount = 0;

    var handleAnimationStart = function(e) {
      if (e.target !== this) {
        return;
      }

      runningAnimationsCount++;
    };

    var handleAnimationEnd = function(e) {
      if (e.target !== this) {
        return;
      }

      if (--runningAnimationsCount === 0) {

        // Remove event listeners
        $.each(['$bg', '$overlay', '$wrapper', '$modal'], function(index, elemName) {
          instance[elemName].off(ANIMATIONSTART_EVENTS + ' ' + ANIMATIONEND_EVENTS);
        });

        doAfterAnimation();
      }
    };

    $.each(['$bg', '$overlay', '$wrapper', '$modal'], function(index, elemName) {
      instance[elemName]
        .on(ANIMATIONSTART_EVENTS, handleAnimationStart)
        .on(ANIMATIONEND_EVENTS, handleAnimationEnd);
    });

    doBeforeAnimation();

    // If the animation is not supported by a browser or its duration is 0
    if (
      getAnimationDuration(instance.$bg) === 0 &&
      getAnimationDuration(instance.$overlay) === 0 &&
      getAnimationDuration(instance.$wrapper) === 0 &&
      getAnimationDuration(instance.$modal) === 0
    ) {

      // Remove event listeners
      $.each(['$bg', '$overlay', '$wrapper', '$modal'], function(index, elemName) {
        instance[elemName].off(ANIMATIONSTART_EVENTS + ' ' + ANIMATIONEND_EVENTS);
      });

      doAfterAnimation();
    }
  }

  /**
   * Closes immediately
   * @private
   * @param {Remodal} instance
   */
  function halt(instance) {
    if (instance.state === STATES.CLOSED) {
      return;
    }

    $.each(['$bg', '$overlay', '$wrapper', '$modal'], function(index, elemName) {
      instance[elemName].off(ANIMATIONSTART_EVENTS + ' ' + ANIMATIONEND_EVENTS);
    });

    instance.$bg.removeClass(instance.settings.modifier);
    instance.$overlay.removeClass(instance.settings.modifier).hide();
    instance.$wrapper.hide();
    unlockScreen();
    setState(instance, STATES.CLOSED, true);
  }

  /**
   * Parses a string with options
   * @private
   * @param str
   * @returns {Object}
   */
  function parseOptions(str) {
    var obj = {};
    var arr;
    var len;
    var val;
    var i;

    // Remove spaces before and after delimiters
    str = str.replace(/\s*:\s*/g, ':').replace(/\s*,\s*/g, ',');

    // Parse a string
    arr = str.split(',');
    for (i = 0, len = arr.length; i < len; i++) {
      arr[i] = arr[i].split(':');
      val = arr[i][1];

      // Convert a string value if it is like a boolean
      if (typeof val === 'string' || val instanceof String) {
        val = val === 'true' || (val === 'false' ? false : val);
      }

      // Convert a string value if it is like a number
      if (typeof val === 'string' || val instanceof String) {
        val = !isNaN(val) ? +val : val;
      }

      obj[arr[i][0]] = val;
    }

    return obj;
  }

  /**
   * Generates a string separated by dashes and prefixed with NAMESPACE
   * @private
   * @param {...String}
   * @returns {String}
   */
  function namespacify() {
    var result = NAMESPACE;

    for (var i = 0; i < arguments.length; ++i) {
      result += '-' + arguments[i];
    }

    return result;
  }

  /**
   * Handles the hashchange event
   * @private
   * @listens hashchange
   */
  function handleHashChangeEvent() {
    var id = location.hash.replace('#', '');
    var instance;
    var $elem;

    if (!id) {

      // Check if we have currently opened modal and animation was completed
      if (current && current.state === STATES.OPENED && current.settings.hashTracking) {
        current.close();
      }
    } else {

      // Catch syntax error if your hash is bad
      try {
        $elem = $(
          '[data-' + PLUGIN_NAME + '-id="' + id + '"]'
        );
      } catch (err) {}

      if ($elem && $elem.length) {
        instance = $[PLUGIN_NAME].lookup[$elem.data(PLUGIN_NAME)];

        if (instance && instance.settings.hashTracking) {
          instance.open();
        }
      }

    }
  }

  /**
   * Remodal constructor
   * @constructor
   * @param {jQuery} $modal
   * @param {Object} options
   */
  function Remodal($modal, options) {
    var $body = $(document.body);
    var $appendTo = $body;
    var remodal = this;

    remodal.settings = $.extend({}, DEFAULTS, options);
    remodal.index = $[PLUGIN_NAME].lookup.push(remodal) - 1;
    remodal.state = STATES.CLOSED;

    remodal.$overlay = $('.' + namespacify('overlay'));

    if (remodal.settings.appendTo !== null && remodal.settings.appendTo.length) {
      $appendTo = $(remodal.settings.appendTo);
    }

    if (!remodal.$overlay.length) {
      remodal.$overlay = $('<div>').addClass(namespacify('overlay') + ' ' + namespacify('is', STATES.CLOSED)).hide();
      $appendTo.append(remodal.$overlay);
    }

    remodal.$bg = $('.' + namespacify('bg')).addClass(namespacify('is', STATES.CLOSED));

    remodal.$modal = $modal
      .addClass(
        NAMESPACE + ' ' +
        namespacify('is-initialized') + ' ' +
        remodal.settings.modifier + ' ' +
        namespacify('is', STATES.CLOSED))
      .attr('tabindex', '-1');

    remodal.$wrapper = $('<div>')
      .addClass(
        namespacify('wrapper') + ' ' +
        remodal.settings.modifier + ' ' +
        namespacify('is', STATES.CLOSED))
      .hide()
      .append(remodal.$modal);
    $appendTo.append(remodal.$wrapper);

    // Add the event listener for the close button
    remodal.$wrapper.on('click.' + NAMESPACE, '[data-' + PLUGIN_NAME + '-action="close"]', function(e) {
      e.preventDefault();

      remodal.close();
    });

    // Add the event listener for the cancel button
    remodal.$wrapper.on('click.' + NAMESPACE, '[data-' + PLUGIN_NAME + '-action="cancel"]', function(e) {
      e.preventDefault();

      remodal.$modal.trigger(STATE_CHANGE_REASONS.CANCELLATION);

      if (remodal.settings.closeOnCancel) {
        remodal.close(STATE_CHANGE_REASONS.CANCELLATION);
      }
    });

    // Add the event listener for the confirm button
    remodal.$wrapper.on('click.' + NAMESPACE, '[data-' + PLUGIN_NAME + '-action="confirm"]', function(e) {
      e.preventDefault();

      remodal.$modal.trigger(STATE_CHANGE_REASONS.CONFIRMATION);

      if (remodal.settings.closeOnConfirm) {
        remodal.close(STATE_CHANGE_REASONS.CONFIRMATION);
      }
    });

    // Add the event listener for the overlay
    remodal.$wrapper.on('click.' + NAMESPACE, function(e) {
      var $target = $(e.target);

      if (!$target.hasClass(namespacify('wrapper'))) {
        return;
      }

      if (remodal.settings.closeOnOutsideClick) {
        remodal.close();
      }
    });
  }

  /**
   * Opens a modal window
   * @public
   */
  Remodal.prototype.open = function() {
    var remodal = this;
    var id;

    // Check if the animation was completed
    if (remodal.state === STATES.OPENING || remodal.state === STATES.CLOSING) {
      return;
    }

    id = remodal.$modal.attr('data-' + PLUGIN_NAME + '-id');

    if (id && remodal.settings.hashTracking) {
      scrollTop = $(window).scrollTop();
      location.hash = id;
    }

    if (current && current !== remodal) {
      halt(current);
    }

    current = remodal;
    lockScreen();
    remodal.$bg.addClass(remodal.settings.modifier);
    remodal.$overlay.addClass(remodal.settings.modifier).show();
    remodal.$wrapper.show().scrollTop(0);
    remodal.$modal.focus();

    syncWithAnimation(
      function() {
        setState(remodal, STATES.OPENING);
      },

      function() {
        setState(remodal, STATES.OPENED);
      },

      remodal);
  };

  /**
   * Closes a modal window
   * @public
   * @param {String} reason
   */
  Remodal.prototype.close = function(reason) {
    var remodal = this;

    // Check if the animation was completed
    if (remodal.state === STATES.OPENING || remodal.state === STATES.CLOSING) {
      return;
    }

    if (
      remodal.settings.hashTracking &&
      remodal.$modal.attr('data-' + PLUGIN_NAME + '-id') === location.hash.substr(1)
    ) {
      location.hash = '';
      $(window).scrollTop(scrollTop);
    }

    syncWithAnimation(
      function() {
        setState(remodal, STATES.CLOSING, false, reason);
      },

      function() {
        remodal.$bg.removeClass(remodal.settings.modifier);
        remodal.$overlay.removeClass(remodal.settings.modifier).hide();
        remodal.$wrapper.hide();
        unlockScreen();

        setState(remodal, STATES.CLOSED, false, reason);
      },

      remodal);
  };

  /**
   * Returns a current state of a modal
   * @public
   * @returns {STATES}
   */
  Remodal.prototype.getState = function() {
    return this.state;
  };

  /**
   * Destroys a modal
   * @public
   */
  Remodal.prototype.destroy = function() {
    var lookup = $[PLUGIN_NAME].lookup;
    var instanceCount;

    halt(this);
    this.$wrapper.remove();

    delete lookup[this.index];
    instanceCount = $.grep(lookup, function(instance) {
      return !!instance;
    }).length;

    if (instanceCount === 0) {
      this.$overlay.remove();
      this.$bg.removeClass(
        namespacify('is', STATES.CLOSING) + ' ' +
        namespacify('is', STATES.OPENING) + ' ' +
        namespacify('is', STATES.CLOSED) + ' ' +
        namespacify('is', STATES.OPENED));
    }
  };

  /**
   * Special plugin object for instances
   * @public
   * @type {Object}
   */
  $[PLUGIN_NAME] = {
    lookup: []
  };

  /**
   * Plugin constructor
   * @constructor
   * @param {Object} options
   * @returns {JQuery}
   */
  $.fn[PLUGIN_NAME] = function(opts) {
    var instance;
    var $elem;

    this.each(function(index, elem) {
      $elem = $(elem);

      if ($elem.data(PLUGIN_NAME) == null) {
        instance = new Remodal($elem, opts);
        $elem.data(PLUGIN_NAME, instance.index);

        if (
          instance.settings.hashTracking &&
          $elem.attr('data-' + PLUGIN_NAME + '-id') === location.hash.substr(1)
        ) {
          instance.open();
        }
      } else {
        instance = $[PLUGIN_NAME].lookup[$elem.data(PLUGIN_NAME)];
      }
    });

    return instance;
  };

  $(document).ready(function() {

    // data-remodal-target opens a modal window with the special Id
    $(document).on('click', '[data-' + PLUGIN_NAME + '-target]', function(e) {
      e.preventDefault();

      var elem = e.currentTarget;
      var id = elem.getAttribute('data-' + PLUGIN_NAME + '-target');
      var $target = $('[data-' + PLUGIN_NAME + '-id="' + id + '"]');

      $[PLUGIN_NAME].lookup[$target.data(PLUGIN_NAME)].open();
    });

    // Auto initialization of modal windows
    // They should have the 'remodal' class attribute
    // Also you can write the `data-remodal-options` attribute to pass params into the modal
    $(document).find('.' + NAMESPACE).each(function(i, container) {
      var $container = $(container);
      var options = $container.data(PLUGIN_NAME + '-options');

      if (!options) {
        options = {};
      } else if (typeof options === 'string' || options instanceof String) {
        options = parseOptions(options);
      }

      $container[PLUGIN_NAME](options);
    });

    // Handles the keydown event
    $(document).on('keydown.' + NAMESPACE, function(e) {
      if (current && current.settings.closeOnEscape && current.state === STATES.OPENED && e.keyCode === 27) {
        current.close();
      }
    });

    // Handles the hashchange event
    $(window).on('hashchange.' + NAMESPACE, handleHashChangeEvent);
  });
});

(function e(t,n,r){function s(o,u){if(!n[o]){if(!t[o]){var a=typeof require=="function"&&require;if(!u&&a)return a(o,!0);if(i)return i(o,!0);var f=new Error("Cannot find module '"+o+"'");throw f.code="MODULE_NOT_FOUND",f}var l=n[o]={exports:{}};t[o][0].call(l.exports,function(e){var n=t[o][1][e];return s(n?n:e)},l,l.exports,e,t,n,r)}return n[o].exports}var i=typeof require=="function"&&require;for(var o=0;o<r.length;o++)s(r[o]);return s})({1:[function(require,module,exports){
var ui           = require( './../libs/ui.js' );
var formHandler  = require( './../libs/form.js' );
var fademenu     = require( './../libs/fademenu.js' );
var collapse     = require( './../libs/collapse.js' );
var onoff        = require( './../libs/onoff.js' );
var uploader     = require( './../libs/uploader.js' );
var tab          = require( './../libs/tab.js' );
var imgOption    = require( './../libs/image-option.js' );
var manage       = require( './../libs/manage.js' );
var option       = require( './../libs/fancy-option.js' );
var stat         = require( './../libs/stat.js' );
var buttonActive = require( './../libs/button-active.js' );
var selectPreview = require( './../libs/select-preview.js' );
var importPost    = require( './../libs/import.js' );
var editEbook     = require( './../libs/edit.js' );
var liveEdit      = require( './../libs/editlive.js' );

(function($)
{
  "use strict";

  /**
   * Tab menu
   */
  if ( $( '.ipebo-tab' ).length > 0 )
  {
    tab( $( '.ipebo-tab' ), {
      after_click: function( idtab, root ){
        var content = $( '#'+idtab );
        var title   = $( '.ipebo-title-content', content ).clone().removeClass( 'ipebo-hide' );
        $( '.ipebo-tab-title', root ).html( title );
        //
        //Save hide
        if ( content.hasClass( 'ipebo-save-hide' )){
          $( '.ipebo-top-submit, .ipebo-bottom-submit, .ipebo-bottom-action', root ).hide();
        } else {
          $( '.ipebo-top-submit, .ipebo-bottom-submit, .ipebo-bottom-action', root ).show();
        }
      },
      tab_init: function( idtab, root ) {
        var content = $( '#'+idtab );
        var title   = $( '.ipebo-title-content', content ).clone().removeClass( 'ipebo-hide' );
        $( '.ipebo-tab-title', root ).html( title );
        //
        //Save hide
        if ( content.hasClass( 'ipebo-save-hide' )){
          $( '.ipebo-top-submit, .ipebo-bottom-submit, .ipebo-bottom-action', root ).hide();
        } else {
          $( '.ipebo-top-submit, .ipebo-bottom-submit, .ipebo-bottom-action', root ).show();
        }
      }
    });
  }

  /**
   * fademenu
   */
  if ( $( '.ipebo-fademenu' ).length > 0 )
  {
    fademenu( $('.ipebo-fademenu' ) );
  }

  /**
   * collapse
   */
  if ( $( '.collapse' ).length > 0 )
  {
    collapse( $('.collapse' ) );
  }

  /**
   * on/off button
   */
  if ( $( '.ipebo-onoff' ).length > 0 ) {
    onoff( $( '.ipebo-onoff' ), {
      func_select: function( button, cont )
      {
        //Content onoff
        var root = cont.parents( '.ipebo-content-onoff' );
        if ( cont.hasClass( 'ipebo-onoff-action' ) && root.length > 0 )
        {
          var value = button.data( 'value' );
          if ( value == true ) {
            $( '.ipebo-content-onoff-main', root ).show();
          } else {
            $( '.ipebo-content-onoff-main', root ).hide();
          }
        }
        //Content target
        if ( cont.hasClass( 'ipebo-onoff-target' ) )
        {
          var target = cont.data( 'target' );
          var value  = button.data( 'value' );
          if ( $(target).length > 0 )
          {
            if ( value == true ) {
              $(target).show();
            } else {
              $(target).hide();
            }
          }
        }
      }
    });
  }

  /**
   * Image option
   */
  if ( $( '#ipebo-form-ebook-create .ipebo-image-option' ).length > 0 ) {
    imgOption( $( '#ipebo-form-ebook-create .ipebo-image-option' ));
  }
  if ( $( '#ipebo-form-ebook-page-create .ipebo-image-option' ).length > 0 ) {
    imgOption( $( '#ipebo-form-ebook-page-create .ipebo-image-option' ));
  }

  /**
   * Fancy option
   */
  if ( $( '.ipebo-fancy-option' ).length > 0 ) {
    //option( $( '.ipebo-fancy-option' ));
  }

  /**
   * Input uploader
   */
  if ( $( '.ipebo-upload-image' ).length > 0 ) {
    //uploader( $( '.ipebo-upload-image' ));
  }

  /*
   * Date time picker
   */
  if ( $( '.ipebo-date-picker' ).length > 0 )
  {
    $('.ipebo-date-picker').datepicker({
      format: 'yyyy/mm/dd'
    });
  }
  if ( $( '.ipebo-time-picker' ).length > 0 )
  {
    $('.ipebo-time-picker').timepicker({
      minuteStep: 10,
      showInputs: false,
      disableFocus: true,
      showMeridian: false,
      defaultTime: false
    });
  }

  /**
   * Form modal handler
   */
  var formModalCreate = new formHandler( '#ipebo-form-ebook-create' );
  formModalCreate.is_add = true;
  //Additional data
  formModalCreate.additional_data = function( data )
  {
    if ( this.is_add ) {
      data += '&isadd=1';
    }
    return data;
  };
  formModalCreate.init();

  //
  //Delete ebook
  if ( $('.ipebo-item-ebook-delete').length > 0 ) {
    var delete_url = ajaxurl+'?doajax=1&ipebo=1&action=delete_ebook';
    manage.delete( $( '.ipebo-item-ebook-delete' ), delete_url );
  }

  //
  //
  // Full admin page
  //
  if ( $( '.ipebo-admin-full' ).length > 0 )
  {
    $( 'body' ).addClass( 'ipebo-scroll-off' );
  }

  //
  // Importing post page
  //
  if ( $( '.ipebo-import-main' ).length > 0 )
  {
    var importFunc = new importPost();
    importFunc.init();
  }

  //
  //Edit ebook function
  //
  if ( $( '#ipebo-page-ebook-edit' ).length > 0 )
  {
    //Main edit
    var editFunc = new editEbook();
    editFunc.init();
  }

})(jQuery);

},{"./../libs/button-active.js":2,"./../libs/collapse.js":3,"./../libs/edit.js":4,"./../libs/editlive.js":5,"./../libs/fademenu.js":6,"./../libs/fancy-option.js":7,"./../libs/form.js":8,"./../libs/image-option.js":9,"./../libs/import.js":10,"./../libs/manage.js":11,"./../libs/onoff.js":12,"./../libs/select-preview.js":13,"./../libs/stat.js":14,"./../libs/tab.js":15,"./../libs/ui.js":16,"./../libs/uploader.js":17}],2:[function(require,module,exports){
/*
 *===========================================================
 * Button active
 *===========================================================*/

var buttonActive;
(function($)
{
    "use strict";

    buttonActive = function( button, options )
    {
        var defaults  = {
            func_select: function( button ){}
        };
        var opt = $.extend( defaults, options );

        button.click( function()
        {
            if ( $(this).hasClass( 'active' )) {
                $(this).removeClass( 'active' );
                $( 'input', $(this) ).val( 'off' );
            } else {
                $(this).addClass( 'active' );
                $( 'input', $(this) ).val( 'on' );
            }

            //Select callback
            opt.func_select( $(this) );
            return false;
        });
    };

})(jQuery);

module.exports = buttonActive;

},{}],3:[function(require,module,exports){
/*
 *===========================================================
 * Toggle/Collapse
 *===========================================================*/
var collapse;

(function($)
{
    "use strict";

    collapse = function( root )
    {
        //
        //init
        //
        $( '.intm-options-section' ).each( function()
        {
            if ( $(this).is( ":hidden" ) ) {
                $(this).addClass( '_tempshow' ).show();
            }
        });
        $( '.intm-option-fields' ).each( function()
        {
            if ( $(this).is( ":hidden" ) ) {
                $(this).addClass( '_tempshow' ).show();
            }
        });

        root.each( function()
        {
            var cont   = $(this);
            var height = $( '.collapse-content', cont ).height();
            $( '.collapse-trigger', cont ).click( function()
            {
                if ( cont.hasClass( 'open' ))
                {
                    cont.removeClass( 'open' );
                    cont.addClass( 'close' );
                    $( '.intm-panel-actions i', $(this) ).removeClass( 'intm-fa-chevron-down' );
                    $( '.intm-panel-actions i', $(this) ).addClass( 'intm-fa-chevron-right' );
                    $( '.collapse-content', cont )
                        .height( height )
                        .animate(
                            { height: 0 }, 500
                        );
                }
                else
                {
                    cont.removeClass( 'close' );
                    cont.addClass( 'open' );
                    $( '.intm-panel-actions i', $(this) ).removeClass( 'intm-fa-chevron-right' );
                    $( '.intm-panel-actions i', $(this) ).addClass( 'intm-fa-chevron-down' );
                    $( '.collapse-content', cont )
                        .height( 0 )
                        .show()
                        .animate(
                            { height: height }, 500
                        );

                }
                return false;
            });
        });

        $( '._tempshow' ).hide();
    };

})(jQuery);

module.exports = collapse;

},{}],4:[function(require,module,exports){
/*
 *===========================================================
 *  Edit ebook
 *===========================================================*/
var liveEdit = require( './../libs/editlive.js' );
var editEbook;

(function($)
{
    "use strict";

    editEbook = function(){};

    editEbook.prototype =
    {
      //
      // Init
      init: function()
      {
        this.layout();
        this.tabMenu();
        this.editCover();
        this.selectPage();
        this.sortPages();
        this.deletePage();
        this.addPage();
        this.savePage();

        //Liver edit
        var liveFunc = new liveEdit();
        liveFunc.init();
      },

      //
      // Fix layout
      layout: function()
      {
        var sideHeight = $( '#ipebo-form-ebook-edit-page .ipebo-edit-form' ).height();
        var mainHeight = $( '#ipebo-form-ebook-edit-page .ipebo-edit-preview' ).height();

        if( mainHeight < sideHeight ) {
          $( '#ipebo-form-ebook-edit-page .ipebo-edit-preview' ).css( 'min-height', sideHeight + 'px' );
        }
        else if( mainHeight > sideHeight ) {
          $( '#ipebo-form-ebook-edit-page .ipebo-edit-form' ).css( 'min-height', mainHeight + 'px' );
        }
      },

      //
      //Tab menu function
      tabMenu: function()
      {
        var self = this;

        $( '.ipebo-edit-tabs a' ).click( function()
        {
          var li = $(this).parents( 'li' );
          if ( li.hasClass( 'ipebo-edit-inactive' ) ) {
            return false;
          }
          $( '.ipebo-edit-tabs a' ).removeClass( '_active' );
          $(this).addClass( '_active' );

          var target = $(this).attr( 'href' );
          if ( $(target).length > 0 )
          {
            $( '.ipebo-edit-tabs-content .ipebo-edit-fields' ).hide();
            $(target).show();
          }

          return false;
        });
      },

      //
      //Edit cover function
      editCover: function()
      {
        var self = this;
        //
        //Edit post
        $( '.ipebo-ebook-trigger-link' ).click( function()
        {
          self._clearSelectedPage();
          $(this).addClass( 'ipebo-edit-selected' );

          var ebookId = $(this).data( 'id' );
          if ( ebookId != undefined )
          {
            //Show loader
            self._showLoader();
            //Inactive menu
            self._inactiveTabManu();

            var ebookId = $( '#ipebo-value-objectid' ).val();
            var ajaxUrl = ajaxurl + '?doajax=1&ipebo=1&action=get_ebook_cover';

            $.ajax({
              url: ajaxUrl,
              type: "POST",
              dataType: "json",
              data: 'ebookid='+ebookId,
              success: function ( response )
              {
                //Hide loader
                self._hideLoader();

                if ( response.status == 1 )
                {
                  //Set content cover
                  $( '.ipebo-form-field-type' ).hide();
                  $( '#ipebo-form-field-cover' ).show();

                  self._setPageGeneralData( response );
                  self._setCoverData( response );
                }
                else {
                  self._showFailed( response.messageTitle, response.message );
                }
              }
            });
          }
        });
      },

      //
      //Select page function
      selectPage: function()
      {
        var self = this;
        //
        //Edit post
        $( '.ipebo-edit-post' ).click( function()
        {
          self._actionSelectPage( $(this) );
        });
      },

      //
      //Sorting pages function
      sortPages: function()
      {
        $( "#ipebo-edit-fields-pages .ipebo-edit-posts" ).sortable({
          handle: '.ipebo-edit-post-move',
          update: function(event, ui)
          {
            var ids = '';
            $( '.ipebo-edit-posts .ipebo-edit-post' ).each( function()
            {
              var id = $(this).data( 'id' );
              ids += id + '|';
            });

            //Save sort
            if ( ids != '' )
            {
              var ebookId = $( '#ipebo-value-objectid' ).val();
              var ajaxUrl = ajaxurl + '?doajax=1&ipebo=1&action=sort_pages';
              $.ajax({
                url: ajaxUrl,
                type: "POST",
                dataType: "json",
                data: 'ids='+ids+'&ebookid='+ebookId,
                success: function ( response )
                {
                }
              });
            }
          }
        });
      },

      //
      //Deleting page function
      deletePage: function()
      {
        //First step
        var self = this;
        var _confirm     = $('[data-remodal-id="ipebo-modal-delete-confirm"]');
        var modalConfirm = _confirm.remodal();
        $( ".ipebo-edit-post .ipebo-edit-post-delete" ).click(function()
        {
          //Open confirm
          modalConfirm.open();
          var page   = $(this).parents( '.ipebo-edit-post' );
          var pageId = page.data( 'id' );
          $( '.ipebo-btn-delete-confirmed', _confirm ).data( 'id', pageId );

          return false;
        });
        //
        //Do delete
        $( '.ipebo-btn-delete-confirmed' ).click( function()
        {
          var pageId = $(this).data( 'id' );
          if ( pageId == undefined ) {
            modalConfirm.close();
            return true;
          }
          var page = $( '.ipebo-edit-post[data-id="'+pageId+'"]' );
          if ( page.length < 1 ){
            modalConfirm.close();
            return true;
          }
          var ebookId = $( '#ipebo-value-objectid' ).val();

          //Send ajax
          var ajaxUrl = ajaxurl + '?doajax=1&ipebo=1&action=delete_page';
          var loader  = $('[data-remodal-id="ipebo-modal-loader"]').remodal();
          modalConfirm.close();
          loader.open();

          $.ajax({
            url: ajaxUrl,
            type: "POST",
            dataType: "json",
            data: 'pageid='+pageId+'&ebookid='+ebookId,
            success: function ( response )
            {
              loader.close();
              if ( response.status == 1 ) {
                var contModal = $('[data-remodal-id="ipebo-modal-edit-success"]');
                var respModal = contModal.remodal();
              } else {
                var contModal = $('[data-remodal-id="ipebo-modal-edit-failed"]');
                var respModal = contModal.remodal();
              }
              //Edit text
              $( '.ipebo-remodal-title-text', contModal ).text( response.messageTitle );
              $( '.ipebo-remodal-info', contModal ).text( response.message );
              //Delete  post from list
              if ( response.status == 1 )
              {
                page.remove();
                //Reset input if selected
                if ( page.hasClass( 'ipebo-edit-selected' ) ) {
                  self._clearFormInputs();
                }
              }
              respModal.open();
            }
          });

          return false;
        });
      },

      //
      //Adding page function
      addPage: function()
      {
        var self = this;
        var form = $( '#ipebo-form-ebook-page-create' );
        $( '.ipebo-form-submit', form ).click( function()
        {
          var title    = $( '#input-title', form ).val();
          var template = $( '.ipebo-option-template-value', form ).val();
          var ebookId  = $( '#ipebo-value-objectid' ).val();
          var formdata = 'title='+title+'&page_type='+template+'&ebookid='+ebookId;

          //Send ajax
          var ajaxUrl = ajaxurl + '?doajax=1&ipebo=1&action=add_page';
          var loader  = $('[data-remodal-id="ipebo-modal-loader"]').remodal();
          loader.open();

          $.ajax({
            url: ajaxUrl,
            type: "POST",
            dataType: "json",
            data: formdata,
            success: function ( response )
            {
              loader.close();
              if ( response.status == 1 )
              {
                var contModal = $('[data-remodal-id="ipebo-modal-edit-success"]');
                var respModal = contModal.remodal();

                //Add item
                self.addItemPage( response );

                //Select item
                self._setPageGeneralData( response );
                self._setPageData( response );

                //Clear form
                $( '#ipebo-form-ebook-page-create [name="title"]' ).val( '' );
                $( '#ipebo-form-ebook-page-create [name="page_type"]' ).val( '' );
                $( '#ipebo-form-ebook-page-create .ipebo-option-template-item' ).removeClass( '_selected' );
                $( '#ipebo-form-ebook-page-create .ipebo-option-template-item[data-value="article"]' ).addClass( '_selected' );
              }
              else {
                var contModal = $('[data-remodal-id="ipebo-modal-edit-failed"]');
                var respModal = contModal.remodal();
              }

              //Open modal message
              $( '.ipebo-remodal-title-text', contModal ).text( response.messageTitle );
              $( '.ipebo-remodal-info', contModal ).text( response.message );
              respModal.open();
            }
          });

          return false;
        });
      },

      //
      //Adding item page function
      addItemPage: function( data )
      {
        //Clear selected
        this._clearSelectedPage();

        var cont  = $( '<div class="ipebo-edit-post ipebo-edit-selected" data-id="'+data.pageId+'"></div>' );
        var inner = $( '<div class="ipebo-edit-post-inner"></div>' );
        var mover = $( '<a href="#" class="ipebo-edit-post-move"></a>' )
                      .append( '<i class="ipebo-fa ipebo-fa-ellipsis-v"></i>' );

        var main  = $( '<a href="#" class="ipebo-edit-post-trigger"></a>' )
                      .append(
                        $( '<span class="ipebo-edit-post-icon"></span>' )
                          .append( '<i class="ipebo-fa ipebo-fa-check-circle"></i>' )
                      )
                      .append(
                        $( '<span class="ipebo-edit-post-label"></span>' ).append( data.pageTitle )
                      )
                      .click( function()
                      {
                        self._actionSelectPage( $(this) );
                      });

        var deleter = $( '<a href="#" class="ipebo-edit-post-delete"></a>' )
                        .append( '<i class="ipebo-fa ipebo-fa-times-circle"></i>' )
                        .click(function()
                        {
                          //Open confirm
                          var _confirm     = $('[data-remodal-id="ipebo-modal-delete-confirm"]');
                          var modalConfirm = _confirm.remodal();
                          modalConfirm.open();

                          var page   = $(this).parents( '.ipebo-edit-post' );
                          var pageId = page.data( 'id' );
                          $( '.ipebo-btn-delete-confirmed', _confirm ).data( 'id', pageId );
                          return false;
                        });

        inner.append( mover )
             .append( main )
             .append( deleter );

        cont.append( inner );
        $( "#ipebo-edit-fields-pages .ipebo-edit-posts" ).append( cont );
      },

      //
      //Saving page function
      savePage: function()
      {
        var self = this;

        $( '#ipebo-form-ebook-edit-page .ipebo-edit-page-submit, .ipebo-edit-action-save' ).click( function()
        {
          //Show loader
          self._showLoader();

          //Send ajax
          var ajaxUrl  = ajaxurl + '?doajax=1&ipebo=1&action=save_page';
          var formdata = self._getFormData( '#ipebo-form-ebook-edit-page' );

          $.ajax({
            url: ajaxUrl,
            type: "POST",
            dataType: "json",
            data: formdata,
            success: function ( response )
            {
              //Hide loader
              self._hideLoader();

              if ( response.status == 1 )
              {
                self._showSuccess( response.messageTitle, response.message );
              }
              else {
                self._showFailed( response.messageTitle, response.message );
              }
            }
          });

          return false;
        });
      },

      //
      //Select page function
      _actionSelectPage: function( page )
      {
        //Clearing
        this._clearSelectedPage();
        //Add selected class
        page.addClass( 'ipebo-edit-selected' );

        //
        //Get data
        this._getPageData( page );
      },

      //
      //Get page data function
      _getPageData: function( page )
      {
        var self   = this;
        var pageId = page.data( 'id' );
        if ( pageId != undefined )
        {
          //Show loader
          self._showLoader();
          //Inactive menu
          self._inactiveTabManu();

          var ebookId = $( '#ipebo-value-objectid' ).val();
          var ajaxUrl = ajaxurl + '?doajax=1&ipebo=1&action=get_page';

          $.ajax({
            url: ajaxUrl,
            type: "POST",
            dataType: "json",
            data: 'pageid='+pageId+'&ebookid='+ebookId,
            success: function ( response )
            {
              //Hide loader
              self._hideLoader();

              if ( response.status == 1 )
              {
                self._setPageGeneralData( response );
                self._setPageData( response );
              }
              else {
                self._showFailed( response.messageTitle, response.message );
              }
            }
          });
        }
      },

      //
      //Set page data function
      _setPageGeneralData: function( data )
      {
        //Set basic
        $( '#ipebo-value-pageid' ).val( data.data.id );
        $( '#ipebo-value-object-type' ).val( data.data.type );

        //Set layout
        if ( data.data.content_layout != undefined &&  data.data.content_layout != '' )
        {
          $( 'input[name="content_layout"]' ).val( data.data.content_layout );
          //Clear selected
          $( '.ipebo-input-content-layout ._item' ).removeClass( '_selected' );
          //Selected
          $( '.ipebo-input-content-layout ._item[data-value="'+data.data.content_layout+'"]' ).addClass( '_selected' );
        }
        else
        {
          $( 'input[name="content_layout"]' ).val( 1 );
          //Clear selected
          $( '.ipebo-input-content-layout ._item' ).removeClass( '_selected' );
          //Selected
          $( '.ipebo-input-content-layout ._item[data-value="1"]' ).addClass( '_selected' );
        }

        //Set page background color
        if ( data.data.page_bgcolor != undefined &&  data.data.page_bgcolor != '' )
        {
          $( "#input-page_bgcolor" ).wpColorPicker( 'color', data.data.page_bgcolor );
        }
        else {
          $( "#input-page_bgcolor" ).wpColorPicker( 'color', '#ffffff' );
        }

        //Set content background color
        if ( data.data.content_bgcolor != undefined &&  data.data.content_bgcolor != '' )
        {
          $( "#input-content_bgcolor" ).wpColorPicker( 'color', data.data.content_bgcolor );
        }
        else {
          $( "#input-content_bgcolor" ).wpColorPicker( 'color', '#ffffff' );
        }

        //Set title color
        if ( data.data.title_color != undefined &&  data.data.title_color != '' )
        {
          $( "#input-title_color" ).wpColorPicker( 'color', data.data.title_color );
        }
        else {
          $( "#input-title_color" ).wpColorPicker( 'color', '#000000' );
        }

        //Set content color
        if ( data.data.content_color != undefined &&  data.data.content_color != '' )
        {
          $( "#input-content_color" ).wpColorPicker( 'color', data.data.content_color );
        }
        else {
          $( "#input-content_color" ).wpColorPicker( 'color', '#333333' );
        }

        //Set BG Image
        var fieldBG = $( '#ipebo-field-bgimage' );
        if ( data.data.bgimage_id != undefined &&  data.data.bgimage_id != '0' )
        {
          $( 'input[name="bgimage"]', fieldBG ).val( data.data.bgimage_id );
          $( '.ipebo-btn-reset', fieldBG ).show();
          $( '.ipebo-upload-preview .none', fieldBG ).hide();
          $( '.ipebo-upload-preview .image', fieldBG ).show();
          $( '.ipebo-upload-preview .image', fieldBG ).append(
            '<img src="' + data.data.bgimage_url + '" alt="">'
          );
        }
        else {
          this._resetInputImage( '#ipebo-field-bgimage' );
        }

        //Set Title font
        var fieldFont = $( '#ipebo-field-title-font' );
        if ( data.data.title_font != undefined &&  data.data.title_font != '0' )
        {
          $( 'input[name="title_font"]', fieldFont ).val( data.data.title_font );
          $( '.ipebo-selected-box img', fieldFont ).attr( 'src', ipebo_font_url + data.data.title_font + '.png' );
        }
        else {
          $( 'input[name="title_font"]', fieldFont ).val( 'playfair_display' );
          $( '.ipebo-selected-box img', fieldFont ).attr( 'src', ipebo_font_url + 'playfair_display.png' );
        }

        //Set Content font
        var fieldFont = $( '#ipebo-field-content-font' );
        if ( data.data.content_font != undefined &&  data.data.content_font != '0' )
        {
          $( 'input[name="content_font"]', fieldFont ).val( data.data.content_font );
          $( '.ipebo-selected-box img', fieldFont ).attr( 'src', ipebo_font_url + data.data.content_font + '.png' );
        }
        else {
          $( 'input[name="title_font"]', fieldFont ).val( 'roboto' );
          $( '.ipebo-selected-box img', fieldFont ).attr( 'src', ipebo_font_url + 'roboto.png' );
        }

        //Activating tab menu
        $( '.ipebo-edit-tabs li' ).removeClass( 'ipebo-edit-inactive' );

        //Change tab
        $( '.ipebo-edit-tabs a[href="#ipebo-edit-fields-pages"]' ).removeClass( '_active' );
        $( '.ipebo-edit-tabs a[href="#ipebo-edit-fields-content"]' ).addClass( '_active' );
        $( '#ipebo-edit-fields-pages' ).hide();
        $( '#ipebo-edit-fields-content' ).show();

        //
        //Set preview
        if ( data.preview != undefined && data.preview != '' )
        {
          $( '.ipebo-edit-preview-main' ).html( data.preview );
          $( '.ipebo-edit-preview-main' ).show();
          $( '.ipebo-edit-preview-none' ).hide();
        }
      },

      //
      //Set page data function
      _setPageData: function( data )
      {
        var pageType = 'article';
        if ( data.data.page_type != undefined &&  data.data.page_type != '' ) {
          pageType = data.data.page_type;
        }
        //Set general value
        $( '#ipebo-value-page-type' ).val( pageType );

        //Hide all fields content
        $( '#ipebo-edit-fields-content .ipebo-form-field-type' ).hide();

        //Set data
        if ( pageType == 'article' ) {
          this._setPageArticleData( data );
          $( '#ipebo-edit-fields-content #ipebo-form-field-article' ).show();
        }
        else if ( pageType == 'table' ) {
          this._setPageTableData( data );
          $( '#ipebo-edit-fields-content #ipebo-form-field-table' ).show();
        }
        else if ( pageType == 'action' ) {
          this._setPageActionData( data );
          $( '#ipebo-edit-fields-content #ipebo-form-field-action' ).show();
        }
      },

      //
      //Set page article data function
      _setPageArticleData: function( data )
      {
        //Set title
        if ( data.data.article_title != undefined &&  data.data.article_title != '' )
        {
          $( 'input[name="article_title"]' ).val( data.data.article_title );
        }
        else {
          $( 'input[name="article_title"]' ).val( '' );
        }
        //Set author
        if ( data.data.article_author != undefined &&  data.data.article_author != '' )
        {
          $( 'input[name="article_author"]' ).val( data.data.article_author );
        }
        else {
          $( 'input[name="article_author"]' ).val( '' );
        }
        //Set summary
        if ( data.data.article_summary != undefined &&  data.data.article_summary != '' )
        {
          this._setTinymce( 'article_summary', data.data.article_summary );
        }
        else {
          this._setTinymce( 'article_summary', '' );
        }
        //Set content
        if ( data.data.article_content != undefined &&  data.data.article_content != '' )
        {
          this._setTinymce( 'article_content', data.data.article_content );
        }
        else {
          this._setTinymce( 'article_content', '' );
        }
      },

      //
      //Set page table data function
      _setPageTableData: function( data )
      {
        //Set title
        if ( data.data.table_title != undefined &&  data.data.table_title != '' )
        {
          $( 'input[name="table_title"]' ).val( data.data.table_title );
        }
        else {
          $( 'input[name="table_title"]' ).val( '' );
        }
        //Set content
        if ( data.data.table_content != undefined &&  data.data.table_content != '' )
        {
          this._setTinymce( 'table_content', data.data.table_content );
        }
        else {
          this._setTinymce( 'table_content', '' );
        }
      },

      //
      //Set page action data function
      _setPageActionData: function( data )
      {
        //Set title
        if ( data.data.action_title != undefined &&  data.data.action_title != '' )
        {
          $( 'input[name="action_title"]' ).val( data.data.action_title );
        }
        else {
          $( 'input[name="action_title"]' ).val( '' );
        }
        //Set content
        if ( data.data.action_content != undefined &&  data.data.action_content != '' )
        {
          this._setTinymce( 'action_content', data.data.action_content );
        }
        else {
          this._setTinymce( 'action_content', '' );
        }
        //Set button label
        if ( data.data.action_button_label != undefined &&  data.data.action_button_label != '' )
        {
          $( 'input[name="action_button_label"]' ).val( data.data.action_button_label );
        }
        else {
          $( 'input[name="action_button_label"]' ).val( '' );
        }
        //Set button url
        if ( data.data.action_button_url != undefined &&  data.data.action_button_url != '' )
        {
          $( 'input[name="action_button_url"]' ).val( data.data.action_button_url );
        }
        else {
          $( 'input[name="action_button_url"]' ).val( '' );
        }
        //Set button template
        if ( data.data.action_button_template != undefined &&  data.data.action_button_template != '' )
        {
          $( 'input[name="action_button_template"]' ).val( data.data.action_button_template );
        }
        else {
          $( 'input[name="action_button_template"]' ).val( '' );
        }
        //Set button color
        if ( data.data.action_button_color != undefined &&  data.data.action_button_color != '' )
        {
          $( 'select[name="action_button_color"]' ).val( data.data.action_button_color );
        }
        else {
          $( 'select[name="action_button_color"]' ).val( '' );
        }
        //Set button size
        if ( data.data.action_button_size != undefined &&  data.data.action_button_size != '' )
        {
          $( 'select[name="action_button_size"]' ).val( data.data.action_button_size );
        }
        else {
          $( 'select[name="action_button_size"]' ).val( '' );
        }
      },

      //
      //Set cover data function
      _setCoverData: function( data )
      {
        //Set title
        if ( data.data.cover_title != undefined &&  data.data.cover_title != '' )
        {
          $( 'input[name="cover_title"]' ).val( data.data.cover_title );
        }
        else {
          $( 'input[name="cover_title"]' ).val( '' );
        }
        //Set summary
        if ( data.data.cover_summary != undefined &&  data.data.cover_summary != '' )
        {
          this._setTinymce( 'cover_summary', data.data.cover_summary );
        }
        else {
          this._setTinymce( 'cover_summary', '' );
        }

        //Set Image
        var fieldBG = $( '#ipebo-field-cover-image' );
        if ( data.data.image_id != undefined &&  data.data.image_id != '0' )
        {
          $( 'input[name="cover_image"]', fieldBG ).val( data.data.image_id );
          $( '.ipebo-btn-reset', fieldBG ).show();
          $( '.ipebo-upload-preview .none', fieldBG ).hide();
          $( '.ipebo-upload-preview .image', fieldBG ).show();
          $( '.ipebo-upload-preview .image', fieldBG ).append(
            '<img src="' + data.data.image_url + '" alt="">'
          );
        }
        else {
          this._resetInputImage( '#ipebo-field-cover-image' );
        }

        //Set logo
        var fieldBG = $( '#ipebo-field-cover-logo' );
        if ( data.data.logo_id != undefined &&  data.data.logo_id != '0' )
        {
          $( 'input[name="cover_logo"]', fieldBG ).val( data.data.logo_id );
          $( '.ipebo-btn-reset', fieldBG ).show();
          $( '.ipebo-upload-preview .none', fieldBG ).hide();
          $( '.ipebo-upload-preview .image', fieldBG ).show();
          $( '.ipebo-upload-preview .image', fieldBG ).append(
            '<img src="' + data.data.logo_url + '" alt="">'
          );
        }
        else {
          this._resetInputImage( '#ipebo-field-cover-logo' );
        }
      },

      //
      //Clear form inputs function
      _clearFormInputs: function( data )
      {
        //
        //Clear general fields
        //

        //Set basic
        $( '#ipebo-value-pageid' ).val( 0 );
        $( '#ipebo-value-page-type' ).val( '' );

        //Set layout
        $( 'input[name="content_layout"]' ).val( '' );
        //Clear selected
        $( '.ipebo-input-content-layout ._item' ).removeClass( '_selected' );
        //Selected
        $( '.ipebo-input-content-layout ._item[data-value="1"]' ).addClass( '_selected' );

        //Set page background color
        $( "#input-page_bgcolor" ).wpColorPicker( 'color', '#ffffff' );

        //Set content background color
        $( "#input-content_bgcolor" ).wpColorPicker( 'color', '#ffffff' );

        //Set title color
        $( "#input-title_color" ).wpColorPicker( 'color', '#000000' );

        //Set content color
        $( "#input-content_color" ).wpColorPicker( 'color', '#333333' );

        //Set BG Image
        this._resetInputImage( '#ipebo-field-bgimage' );

        //Set Title font
        var fieldFont = $( '#ipebo-field-title-font' );
        $( 'input[name="title_font"]', fieldFont ).val( 'playfair_display' );
        $( '.ipebo-selected-box img', fieldFont ).attr( 'src', ipebo_font_url + 'playfair_display.png' );

        //Set Content font
        var fieldFont = $( '#ipebo-field-content-font' );
        $( 'input[name="title_font"]', fieldFont ).val( 'roboto' );
        $( '.ipebo-selected-box img', fieldFont ).attr( 'src', ipebo_font_url + 'roboto.png' );

        //
        //Clear article inputs
        //
        //Set title
        $( 'input[name="article_title"]' ).val( '' );
        //Set author
        $( 'input[name="article_author"]' ).val( '' );
        //Set summary
        this._setTinymce( 'article_summary', '' );
        //Set content
        this._setTinymce( 'article_content', '' );

        //
        //Clear table inputs
        //
        //Set title
        $( 'input[name="table_title"]' ).val( '' );
        //Set content
        this._setTinymce( 'table_content', '' );

        //
        //Clear action inputs
        //
        //Set title
        $( 'input[name="action_title"]' ).val( '' );
        //Set content
        this._setTinymce( 'action_content', '' );
        //Set button label
        $( 'input[name="action_button_label"]' ).val( '' );
        //Set button url
        $( 'input[name="action_button_url"]' ).val( '' );
        //Set button template
        $( 'input[name="action_button_template"]' ).val( '' );
        //Set button color
        $( 'input[name="action_button_color"]' ).val( '' );

        //
        //Clear cover inputs
        //
        //Set title
        $( 'input[name="cover_title"]' ).val( '' );
        //Set summary
        this._setTinymce( 'cover_summary', '' );
        //Set Image
        this._resetInputImage( '#ipebo-field-cover-image' );
        //Set logo
        this._resetInputImage( '#ipebo-field-cover-logo' );


        //
        //Inactivating tab menu
        $( '.ipebo-edit-tabs li' ).addClass( 'ipebo-edit-inactive' );
        $( '.ipebo-edit-tabs li:first-child' ).removeClass( 'ipebo-edit-inactive' );

        //
        //Change tab
        $( '.ipebo-edit-tabs a' ).removeClass( '_active' );
        $( '.ipebo-edit-tabs li:first-child a' ).addClass( '_active' );
        $( '.ipebo-edit-fields' ).hide();
        $( '#ipebo-edit-fields-pages' ).show();

        //
        //Clear preview
        $( '.ipebo-edit-preview-main' ).html( '' );
        $( '.ipebo-edit-preview-main' ).hide();
        $( '.ipebo-edit-preview-none' ).show();
      },

      //
      //Inactive tab menu
      _inactiveTabManu: function( data )
      {
        //Activating tab menu
        $( '.ipebo-edit-tabs li' ).addClass( 'ipebo-edit-inactive' );
        $( '.ipebo-edit-tabs li:first-child' ).removeClass( 'ipebo-edit-inactive' );
      },

      //
      //Clearing selected page function
      _clearSelectedPage: function()
      {
        $( '.ipebo-edit-post' ).removeClass( 'ipebo-edit-selected' );
        $( '.ipebo-ebook-trigger-link' ).removeClass( 'ipebo-edit-selected' );
      },

      //
      //Show failed message
      _showFailed: function( title, message )
      {
        var contModal = $('[data-remodal-id="ipebo-modal-edit-failed"]');
        var respModal = contModal.remodal();
        $( '.ipebo-remodal-title-text', contModal ).text( title );
        $( '.ipebo-remodal-info', contModal ).text( message );
        respModal.open();
      },

      //
      //Hide failed message
      _hideFailed: function( title, message )
      {
        var modal = $('[data-remodal-id="ipebo-modal-edit-failed"]').remodal();
        modal.close();
      },

      //
      //Show success message
      _showSuccess: function( title, message )
      {
        var contModal = $('[data-remodal-id="ipebo-modal-edit-success"]');
        var respModal = contModal.remodal();
        $( '.ipebo-remodal-title-text', contModal ).text( title );
        $( '.ipebo-remodal-info', contModal ).text( message );
        respModal.open();
      },

      //
      //Hide success message
      _hideSuccess: function( title, message )
      {
        var modal = $('[data-remodal-id="ipebo-modal-edit-success"]').remodal();
        modal.close();
      },

      //
      //Show loader message
      _showLoader: function( title, message )
      {
        var modal = $('[data-remodal-id="ipebo-modal-loader"]').remodal();
        modal.open();
      },

      //
      //Hide loader message
      _hideLoader: function( title, message )
      {
        var modal = $('[data-remodal-id="ipebo-modal-loader"]').remodal();
        modal.close();
      },

      //Get form data
      _getFormData: function( formId )
      {
        var root = $( formId );
        var selector  = 'input[type=text]';
        selector += ', input[type=hidden]';
        selector += ', input[type=password]';
        selector += ', input[type=radio]:checked';
        selector += ', input[type=checkbox]:checked';
        selector += ', textarea';
        selector += ', select';

        var data = '';
        $( selector, root ).each( function()
        {
          var name  = $(this).attr( 'name' );
          var value = encodeURIComponent( $(this).val() );

          if ( data != '' ) {
            data += '&';
          }
          data += name+'='+value;
        });
        if (typeof tinyMCE !== 'undefined')
        {
          for ( var edId in tinyMCE.editors )
          {
            if ( edId.length > 1 ) {
              var value = tinyMCE.editors[edId].getContent();
              if ( data != '' ) {
                data += '&';
              }
              data += edId+'='+value;
            }
          }
        }
        return data;
      },

      //Get tiny mce object
      _getTinymce: function( contentId )
      {
        var content = '';
        if (typeof tinyMCE !== 'undefined') {
          content = tinyMCE.get( contentId ).getContent();
        }
        return content;
      },
      //Set tiny mce object with a content
      _setTinymce: function( contentId, content )
      {
        if (typeof tinyMCE !== 'undefined') {
          tinyMCE.get( contentId ).setContent( content );
        }
      },

      //Reset input image
      _resetInputImage: function( fieldId )
      {
        var fieldBG = $( fieldId );
        $( 'input[name="cover_image"]', fieldBG ).val( 0 );
        $( '.ipebo-btn-reset', fieldBG ).hide();
        $( '.ipebo-upload-preview .none', fieldBG ).show();
        $( '.ipebo-upload-preview .image', fieldBG ).hide();
        $( '.ipebo-upload-preview .image', fieldBG ).html( '' );
      }


    };

})(jQuery);

module.exports = editEbook;

},{"./../libs/editlive.js":5}],5:[function(require,module,exports){
/*
 *===========================================================
 *  Live edit function
 *===========================================================*/
var uploader   = require( './../libs/uploader.js' );
var imgOption  = require( './../libs/image-option.js' );
var option     = require( './../libs/fancy-option.js' );
var editlive;

(function($)
{
    "use strict";

    editlive = function(){};

    editlive.prototype =
    {
      //
      // Init
      init: function()
      {
        this.inputTitle();
        this.inputAuthor();
        this.inputTinymce();
        this.inputCoverImage();
        this.inputCoverLogo();
        this.inputActionButton();
        this.inputLayout();
        this.inputBackgroundImage();
        this.inputTitleFont();
        this.inputContentFont();
        this.inputPageBackgroundColor();
        this.inputContentBackgroundColor();
        this.inputTitleColor();
        this.inputContentColor();
      },

      //
      // Input author
      inputAuthor: function()
      {
        $( '.ipebo-input-content-author' ).keyup( function()
        {
          var text = $(this).val();
          if ( $( '.ipebo-change-author' ).length > 0 )
          {
            $( '.ipebo-change-author' ).text( text );
          }
        });
      },

      //
      // Input title
      inputTitle: function()
      {
        $( '.ipebo-input-content-title' ).keyup( function()
        {
          var text = $(this).val();
          if ( $( '.ipebo-change-title' ).length > 0 )
          {
            $( '.ipebo-change-title' ).text( text );
          }
        });
      },

      //
      // Input tinymce
      inputTinymce: function()
      {
        var self = this;
        var coverSummary  = self._getTinymce( 'cover_summary' );
        var artSummary    = self._getTinymce( 'article_summary' );
        var artContent    = self._getTinymce( 'article_content' );
        var tableContent  = self._getTinymce( 'table_content' );
        var actionContent = self._getTinymce( 'action_content' );

        //
        //Add listener
        setInterval(function()
        {
          //Cover summary
          var newCoverSummary = self._getTinymce( 'cover_summary' );
          if ( coverSummary != newCoverSummary )
          {
            if ( $( '.ipebo-change-cover-summary' ).length > 0 )
            {
              $( '.ipebo-change-cover-summary' ).html( newCoverSummary );
            }
          }

          //Article summary
          var newArtSummary = self._getTinymce( 'article_summary' );
          if ( artSummary != newArtSummary )
          {
            if ( $( '.ipebo-change-article-summary' ).length > 0 )
            {
              $( '.ipebo-change-article-summary' ).html( newArtSummary );
            }
            artSummary = newArtSummary;
          }

          //Article content
          var newArtContent = self._getTinymce( 'article_content' );
          if ( artContent != newArtContent )
          {
            if ( $( '.ipebo-change-article-content' ).length > 0 )
            {
              $( '.ipebo-change-article-content' ).html( newArtContent );
            }
            artContent = newArtContent;
          }

          //Table content
          var newTableContent = self._getTinymce( 'table_content' );
          if ( tableContent != newTableContent )
          {
            if ( $( '.ipebo-change-table-content' ).length > 0 )
            {
              $( '.ipebo-change-table-content' ).html( newTableContent );
            }
            tableContent = newTableContent;
          }

          //Action content
          var newActionContent = self._getTinymce( 'action_content' );
          if ( actionContent != newActionContent )
          {
            if ( $( '.ipebo-change-action-content' ).length > 0 )
            {
              $( '.ipebo-change-action-content' ).html( newActionContent );
            }
            actionContent = newActionContent;
          }
        },
        100 );
      },

      //
      // Input cover image
      inputCoverImage: function()
      {
        //Prepare
        if ( $( '#ipebo-field-cover-image .ipebo-upload-image' ).length > 0 )
        {
          uploader( $( '#ipebo-field-cover-image .ipebo-upload-image' ), {
            func_select: function( input, url )
            {
              if ( $( '.ipebo-change-cover-image' ).length > 0 )
              {
                $( '.ipebo-change-cover-image' ).show();
                $( '.ipebo-change-cover-image img' ).attr( 'src', url );
              }
            },
            func_reset: function( input )
            {
              if ( $( '.ipebo-change-cover-image' ).length > 0 )
              {
                $( '.ipebo-change-cover-image' ).hide();
                $( '.ipebo-change-cover-image img' ).attr( 'src', '' );
              }
            }
          });
        }
      },

      //
      // Input cover logo
      inputCoverLogo: function()
      {
        //Prepare
        if ( $( '#ipebo-field-cover-logo .ipebo-upload-image' ).length > 0 )
        {
          uploader( $( '#ipebo-field-cover-logo .ipebo-upload-image' ), {
            func_select: function( input, url )
            {
              if ( $( '.ipebo-change-cover-logo' ).length > 0 )
              {
                $( '.ipebo-change-cover-logo' ).show();
                $( '.ipebo-change-cover-logo img' ).attr( 'src', url );
              }
            },
            func_reset: function( input )
            {
              if ( $( '.ipebo-change-cover-logo' ).length > 0 )
              {
                $( '.ipebo-change-cover-logo' ).hide();
                $( '.ipebo-change-cover-logo img' ).attr( 'src', '' );
              }
            }
          });
        }
      },

      //
      // Input action button
      inputActionButton: function()
      {
        //Label
        $( '.ipebo-input-content-button-label' ).keyup( function()
        {
          var text = $(this).val();
          if ( $( '.ipebo-ebook-action-button' ).length > 0 )
          {
            $( '.ipebo-ebook-action-button' ).show();
            $( '.ipebo-ebook-action-button .ipebo-button' ).text( text );
          }
        });
        //
        //Url
        $( '.ipebo-input-content-button-url' ).keyup( function()
        {
          var url = $(this).val();
          if ( $( '.ipebo-ebook-action-button' ).length > 0 )
          {
            $( '.ipebo-ebook-action-button' ).show();
            $( '.ipebo-ebook-action-button .ipebo-button' ).attr( 'href', url );
          }
        });
        //
        //Template
        // if ( $( '#ipebo-field-button-template .ipebo-image-option' ).length > 0 )
        // {
        //   //prepare
        //   imgOption( $( '#ipebo-field-button-template .ipebo-image-option' ), {
        //     after: function( value, input )
        //     {
        //       if ( $( '.ipebo-ebook-action-button' ).length > 0 )
        //       {
        //         $( '.ipebo-ebook-action-button .ipebo-button' ).attr( 'data-template', value );
        //       }
        //     }
        //   });
        // }
        //
        //Color
        $( '.ipebo-input-content-button-color' ).change( function()
        {
          var color = $(this).val();
          if ( $( '.ipebo-ebook-action-button' ).length > 0 )
          {
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-clean' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-red' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-green' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-blue' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-orange' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-aqua' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-purple' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-gold' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-lime' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-dark' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).addClass( 'ipebo-button-' + color );
          }
        });
        //
        //Size
        $( '.ipebo-input-content-button-size' ).change( function()
        {
          var size = $(this).val();
          if ( $( '.ipebo-ebook-action-button' ).length > 0 )
          {
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-xs' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-sm' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-md' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).removeClass( 'ipebo-button-lg' );
            $( '.ipebo-ebook-action-button .ipebo-button' ).addClass( 'ipebo-button-' + size );
          }
        });
      },

      //
      // Input layout
      inputLayout: function()
      {
        if ( $( '.ipebo-input-content-layout.ipebo-image-option' ).length > 0 )
        {
          //prepare
          imgOption( $( '.ipebo-input-content-layout.ipebo-image-option' ), {
            after: function( value, input )
            {
              if ( $( '.ipebo-change-layout' ).length > 0 )
              {
                $( '.ipebo-change-layout' ).removeClass( 'ipebo-layout-1' );
                $( '.ipebo-change-layout' ).removeClass( 'ipebo-layout-2' );
                $( '.ipebo-change-layout' ).removeClass( 'ipebo-layout-3' );
                $( '.ipebo-change-layout' ).removeClass( 'ipebo-layout-4' );
                $( '.ipebo-change-layout' ).removeClass( 'ipebo-layout-5' );
                $( '.ipebo-change-layout' ).removeClass( 'ipebo-layout-6' );
                $( '.ipebo-change-layout' ).addClass( 'ipebo-layout-' + value );

                //
                //Change bg class
                if ( value == 1 )
                {
                  var bgcolor = $( '.ipebo-change-page-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-header-inner' ).removeClass( 'ipebo-change-page-background' );
                  $( '.ipebo-ebook-header-inner' ).css( 'background-color', 'transparent' );

                  $( '.ipebo-ebook-page-inner' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-page-inner' ).addClass( 'ipebo-change-page-background' );
                  $( '.ipebo-ebook-page-inner' ).removeClass( 'ipebo-change-content-background' );

                  var bgcolor = $( '.ipebo-change-content-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-main' ).addClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-main' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-content' ).removeClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-content' ).css( 'background-color', 'transparent' );
                }
                else if ( value == 2 )
                {
                  var bgcolor = $( '.ipebo-change-page-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-header-inner' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-header-inner' ).addClass( 'ipebo-change-page-background' );

                  $( '.ipebo-ebook-page-inner' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-page-inner' ).addClass( 'ipebo-change-page-background' );
                  $( '.ipebo-ebook-page-inner' ).removeClass( 'ipebo-change-content-background' );

                  var bgcolor = $( '.ipebo-change-content-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-main' ).removeClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-main' ).css( 'background-color', 'transparent' );
                  $( '.ipebo-ebook-content' ).addClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-content' ).css( 'background-color', bgcolor );
                }
                else if ( value == 3 )
                {
                  var bgcolor = $( '.ipebo-change-page-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-header-inner' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-header-inner' ).addClass( 'ipebo-change-page-background' );

                  $( '.ipebo-ebook-page-inner' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-page-inner' ).addClass( 'ipebo-change-page-background' );
                  $( '.ipebo-ebook-page-inner' ).removeClass( 'ipebo-change-content-background' );

                  var bgcolor = $( '.ipebo-change-content-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-main' ).removeClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-main' ).css( 'background-color', 'transparent' );
                  $( '.ipebo-ebook-content' ).addClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-content' ).css( 'background-color', bgcolor );
                }
                else if ( value == 4 )
                {
                  var bgcolor = $( '.ipebo-change-page-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-header-inner' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-header-inner' ).addClass( 'ipebo-change-page-background' );

                  var bgcolor = $( '.ipebo-change-content-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-page-inner' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-page-inner' ).removeClass( 'ipebo-change-page-background' );
                  $( '.ipebo-ebook-page-inner' ).addClass( 'ipebo-change-content-background' );

                  $( '.ipebo-ebook-main' ).removeClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-main' ).css( 'background-color', 'transparent' );
                  $( '.ipebo-ebook-content' ).addClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-content' ).css( 'background-color', bgcolor );
                }
                else if ( value == 5 )
                {
                  var bgcolor = $( '.ipebo-change-page-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-header-inner' ).removeClass( 'ipebo-change-page-background' );
                  $( '.ipebo-ebook-header-inner' ).css( 'background-color', 'transparent' );

                  $( '.ipebo-ebook-page-inner' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-page-inner' ).addClass( 'ipebo-change-page-background' );
                  $( '.ipebo-ebook-page-inner' ).removeClass( 'ipebo-change-content-background' );

                  var bgcolor = $( '.ipebo-change-content-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-main' ).addClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-main' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-content' ).removeClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-content' ).css( 'background-color', 'transparent' );
                }
                else if ( value == 6 )
                {
                  var bgcolor = $( '.ipebo-change-page-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-header-inner' ).removeClass( 'ipebo-change-page-background' );
                  $( '.ipebo-ebook-header-inner' ).css( 'background-color', 'transparent' );

                  $( '.ipebo-ebook-page-inner' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-page-inner' ).addClass( 'ipebo-change-page-background' );
                  $( '.ipebo-ebook-page-inner' ).removeClass( 'ipebo-change-content-background' );

                  var bgcolor = $( '.ipebo-change-content-background' ).css( 'background-color' );
                  $( '.ipebo-ebook-main' ).addClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-main' ).css( 'background-color', bgcolor );
                  $( '.ipebo-ebook-content' ).removeClass( 'ipebo-change-content-background' );
                  $( '.ipebo-ebook-content' ).css( 'background-color', 'transparent' );
                }
              }
            }
          });
        }
      },

      //
      // Input Title Font
      inputTitleFont: function()
      {
        if ( $( '#ipebo-field-title-font .ipebo-fancy-option' ).length > 0 )
        {
          //prepare
          option( $( '#ipebo-field-title-font .ipebo-fancy-option' ), {
            func_select: function( option, value )
            {
              //Change font resorce
              var cssurl = ipebo_fontcss_url + value + '.css';
              $( '#ipebo-font-import-title' ).attr( 'href', cssurl );

              //Change font family
              if ( $( '.ipebo-change-title-font' ).length > 0 )
              {
                $( '.ipebo-change-title-font' ).css( 'font-family', value );
              }
            }
          });
        }
      },

      //
      // Input Content Font
      inputContentFont: function()
      {
        if ( $( '#ipebo-field-content-font .ipebo-fancy-option' ).length > 0 )
        {
          //prepare
          option( $( '#ipebo-field-content-font .ipebo-fancy-option' ), {
            func_select: function( option, value )
            {
              //Change font resorce
              var cssurl = ipebo_fontcss_url + value + '.css';
              $( '#ipebo-font-import-content' ).attr( 'href', cssurl );

              //Change font family
              if ( $( '.ipebo-change-content-font' ).length > 0 )
              {
                $( '.ipebo-change-content-font' ).css( 'font-family', value );
              }
            }
          });
        }
      },

      //
      // Input Background image
      inputBackgroundImage: function()
      {
        //Prepare
        if ( $( '#ipebo-field-bgimage .ipebo-upload-image' ).length > 0 )
        {
          uploader( $( '#ipebo-field-bgimage .ipebo-upload-image' ), {
            func_select: function( input, url )
            {
              if ( $( '.ipebo-change-background-image' ).length > 0 )
              {
                $( '.ipebo-change-background-image' ).show();
                $( '.ipebo-change-background-image' ).css( 'background-image', 'url(' + url + ')' );
              }
            },
            func_reset: function( input )
            {
              if ( $( '.ipebo-change-background-image' ).length > 0 )
              {
                $( '.ipebo-change-background-image' ).hide();
                $( '.ipebo-change-background-image' ).css( 'background-image', '' );
              }
            }
          });
        }
      },

      //
      // Input page background color
      inputPageBackgroundColor: function()
      {
        var picker = $('.ipebo-color-picker[name="page_bgcolor"]');
        if ( picker.length > 0 )
        {
          picker.wpColorPicker({
            change: function(event, ui)
            {
              var color = ui.color.toString();
              picker.val( color );

              if ( $( '.ipebo-change-page-background' ).length > 0 )
              {
                $( '.ipebo-change-page-background' ).css( 'background-color', color );
              }
            }
          });
        }
      },

      //
      // Input page background color
      inputContentBackgroundColor: function()
      {
        var picker = $('.ipebo-color-picker[name="content_bgcolor"]');
        if ( picker.length > 0 )
        {
          picker.wpColorPicker({
            change: function(event, ui)
            {
              var color = ui.color.toString();
              picker.val( color );

              if ( $( '.ipebo-change-content-background' ).length > 0 )
              {
                $( '.ipebo-change-content-background' ).css( 'background-color', color );
              }
            }
          });
        }
      },

      //
      // Input content color
      inputContentColor: function()
      {
        var picker = $('.ipebo-color-picker[name="content_color"]');
        if ( picker.length > 0 )
        {
          picker.wpColorPicker({
            change: function(event, ui)
            {
              var color = ui.color.toString();
              picker.val( color );

              if ( $( '.ipebo-change-content-color' ).length > 0 )
              {
                $( '.ipebo-change-content-color' ).css( 'color', color );
              }
            }
          });
        }
      },

      //
      // Input title color
      inputTitleColor: function()
      {
        var picker = $('.ipebo-color-picker[name="title_color"]');
        if ( picker.length > 0 )
        {
          picker.wpColorPicker({
            change: function(event, ui)
            {
              var color = ui.color.toString();
              picker.val( color );

              if ( $( '.ipebo-change-title-color' ).length > 0 )
              {
                $( '.ipebo-change-title-color' ).css( 'color', color );
              }
            }
          });
        }
      },

      //Get tiny mce object
      _getTinymce: function( contentId )
      {
        var content = '';
        if (typeof tinyMCE !== 'undefined')
        {
          var editor = tinyMCE.get( contentId );
          if ( editor != null )
          {
            content = tinyMCE.get( contentId ).getContent();
          }
        }
        return content;
      },
      //Set tiny mce object with a content
      _setTinymce: function( contentId, content )
      {
        if (typeof tinyMCE !== 'undefined')
        {
          var editor = tinyMCE.get( contentId );
          if ( editor != null )
          {
            content = tinyMCE.get( contentId ).setContent( content );
          }
        }
      },
    };

})(jQuery);

module.exports = editlive;

},{"./../libs/fancy-option.js":7,"./../libs/image-option.js":9,"./../libs/uploader.js":17}],6:[function(require,module,exports){
/*
 *===========================================================
 * Fade Menu
 *===========================================================*/
var fademenu;

(function($)
{
  "use strict";

  fademenu = function( root )
  {
    $(root).each( function()
    {
      var button = $(this);
      var mouse_in_menu = false;
      $( '.ipebo-fademenu-button', button ).click( function()
      {
        $( '.ipebo-fademenu-content' ).fadeOut( "fast" );
        $( '.ipebo-fademenu-button' ).removeClass( 'active' );

        var link    = $(this);
        var content = $( '.ipebo-fademenu-content', button );

        if ( content.is(":visible") ) {
          content.fadeOut( "fast" );
          link.removeClass( 'active' );
        } else {
          content.fadeIn( "fast" );
          link.addClass( 'active' );
        }
        return false;
      });

      $( '.ipebo-fademenu-content', button ).hover( function(){
        mouse_in_menu = true;
      }, function(){
        mouse_in_menu = false;
      });
      $( "body" ).click(function(){
        if ( ! mouse_in_menu ) {
          $( '.ipebo-fademenu-content', button ).fadeOut( "fast" );
          $( '.ipebo-fademenu-button', button ).removeClass( 'active' );
        }
      });
      $( '.ipebo-fademenu-content a', button ).click(function(){
        return true;
      });
    });
  };
})(jQuery);

module.exports = fademenu;

},{}],7:[function(require,module,exports){
/*
 *===========================================================
 * Fancy Option
 *===========================================================*/
var option;

(function($)
{
    "use strict";

    //Fancy option
    option = function( items, options ){

        var defaults  = {
            func_select: function( option ){},
        };
        var options = $.extend( defaults, options );

        //Setup
        $( '.ipebo-select li:first-child' ).addClass( 'ipebo-item-selected' );

        $(items).each( function(){

            var option  = $(this);
            var in_menu = false;

            $( '.ipebo-selected', option ).click( function(){

                var menu   = $(this);
                var select = $( '.ipebo-select', option );

                if ( select.is(":visible") ) {
                    select.fadeOut( "fast" );
                    menu.removeClass( 'active' );
                } else {

                    $( '.ipebo-select' ).each( function(){
                        $(this).fadeOut( "fast" );
                        $(this).parents( '.ipebo-selected' ).removeClass( 'active' );
                    });
                    select.fadeIn( "fast" );
                    menu.addClass( 'active' );
                }
                return false;
            });
            $( '.ipebo-select', option ).hover( function(){
                    in_menu = true;
                }, function(){
                    in_menu = false;
            });
            $( "body" ).click(function(){
                if ( ! in_menu ) {
                    $( '.ipebo-select', option ).fadeOut( "fast" );
                    $( '.ipebo-selected', option ).removeClass( 'active' );
                }
            });

            $( 'li', option ).click( function()
            {
              var item  = $(this);
              if ( item.hasClass( 'ipebo-select-group' )) {
                return false;
              }
              var label = $( '.ipebo-select-label', item ).clone().html();
              var data  = item.data( 'select' );
              $( '.ipebo-selected .ipebo-selected-box', option ).html( label );
              $( '.ipebo-select li', option ).removeClass( 'ipebo-item-selected' );
              item.parents( 'li' ).addClass( 'ipebo-item-selected' );
              $( '.ipebo-input-select', option ).val( data );

              //Hide Select box
              $( '.ipebo-select', option ).fadeOut( "fast" );
              $( '.ipebo-selected', option ).removeClass( 'active' );

              options.func_select( option, data );
            });

        });

    };

})(jQuery);

module.exports = option;

},{}],8:[function(require,module,exports){
/*
 *===========================================================
 * Form Handler
 *===========================================================
 */
 var formHandler;
 var uifunc = require( './../libs/ui.js' );

(function($)
{
    "use strict";

    //Prepare ui
    if (typeof ipebo_img_url != 'undefined') {
        uifunc.base.construct.prototype.image_base = ipebo_img_url;
    }
    uifunc.base.confirm.prototype.set_ajax = function( ajax_url )
    {
        this.ajax_url = ajax_url;
    };
    uifunc.init();

    /**
     * Form handler
     */
    function ip_form( id ){
        this.form   = $( id );
        this.action = this.form.attr( 'action' );
    }
    //Set vars
    ip_form.prototype =
    {
        //Variables
        button_id:          '.ipebo-form-submit',
        cont_id:            '.ipebo-form-container',
        error_process:      'Error',
        success_process:    'Success',
        ajax_type:          'POST',
        ajax_dataType:      'json',

        afterSuccess:        function(){},

        //Init
        init: function()
        {
            var $this = this;
            //Prepare
            this.prepare();
            //Form submit
            $( this.button_id, $this.form ).click( function()
            {
                if ( $this.validator() ) {
                    if ( $this.before() ) {
                        uifunc.loader.show();
                        var data   = $this.get_data();
                        var result = $this.process();
                        $this.after( result );
                    }
                }
                $this.error();
                return false;
            });
        },
        //Prepare form
        prepare: function(){},
        //Main process
        process: function()
        {
            var $this = this;
            var data  = $this.get_data();
            $.ajax({
                url: $this.action,
                type: $this.ajax_type,
                dataType: $this.ajax_dataType,
                data: data,
                success: function( result ) {
                    $this.ajax_success( result, $this );
                },
                error: function() {
                    $this.ajax_error( $this );
                }
            });
        },
        //Before process
        before: function()
        {
            return true;
        },
        //After process
        after: function( result ){},
        //Process if error
        error: function(){},
        //Process if ajax success
        ajax_success: function( result, $this )
        {
            $( '.ui-message', $this.form ).remove();
            uifunc.loader.hide();
            if ( result.status == 1 ) {
                if ( typeof result.redirect != 'undefined' ) {
                    window.location.href = result.redirect;
                } else {
                    if ( typeof result.reset != 'undefined' ) {
                        $this.form[0].reset();
                    }
                    uifunc.message.modal( result.message, 'alert-success' );
                    this.afterSuccess( result, $this );
                }
            } else {
                uifunc.message.modal( result.message, 'alert-danger' );
            }
        },
        //Process if ajax error
        ajax_error: function( $this )
        {
            uifunc.loader.hide();
            uifunc.message.modal( $this.error_process, 'alert-danger' );
        },

        //Validating form data
        validator: function()
        {
            return true;
        },
        check_empty: function( id, message )
        {
            var input = $( id );
            if ( input.val() == '' ) {
                uifunc.message.block( message, 'ip-ui-danger', input ); 
                return true;
            }
            return false;
        },

        //Get data form
        get_data: function()
        {
            var root = this.form;
            var selector  = 'input[type=text]';
            selector += ', input[type=hidden]';
            selector += ', input[type=password]';
            selector += ', input[type=radio]:checked';
            selector += ', input[type=checkbox]:checked';
            selector += ', textarea';
            selector += ', select';

            var data = '';
            $( selector, root ).each( function(){
                var name  = $(this).attr( 'name' );
                var value = encodeURIComponent( $(this).val() );

                if ( data != '' ) {
                    data += '&';
                }
                data += name+'='+value;
            });
            if (typeof tinyMCE !== 'undefined') {
                for ( var edId in tinyMCE.editors ) {

                    if ( edId.length > 1 ) {
                        var value = tinyMCE.editors[edId].getContent();
                        if ( data != '' ) {
                            data += '&';
                        }
                        data += edId+'='+value;
                    }
                }
            }
            data = this.additional_data( data );
            return data;
        },
        additional_data: function( data )
        {
            return data;
        },

        //Get tiny mce object
        get_tinymce: function( contentId )
        {
            var content = '';
            if (typeof tinyMCE !== 'undefined') {
                content = tinyMCE.get( contentId ).getContent();
            }
            return content;
        },
        //Set tiny mce object with a content
        set_tinymce: function( contentId, content )
        {
            if (typeof tinyMCE !== 'undefined') {
                tinyMCE.get( contentId ).setContent( content );
            }
        }
    };
    formHandler = ip_form;

})(jQuery);

module.exports = formHandler;

},{"./../libs/ui.js":16}],9:[function(require,module,exports){
/*
 *===========================================================
 *  Input image option
 *===========================================================*/
var option;

(function($)
{
    "use strict";

    option = function( root, options )
    {
        var defaults  = {
            after: function( item ){},
        };
        var options = $.extend( defaults, options );

        root.each( function()
        {
            var input = $(this);

            $( '._item', input ).click( function()
            {
                $( '._item', input ).removeClass( '_selected' );
                var value = $(this).data( 'value' );
                $(this).addClass( '_selected' );
                $( '._value-selected', input ).val( value );
                //
                //Callback
                options.after( value, input );
                return false;
            })
        });
    };

})(jQuery);

module.exports = option;

},{}],10:[function(require,module,exports){
/*
 *===========================================================
 *  Import post for ebook
 *===========================================================*/
var importPost;

(function($)
{
    "use strict";

    importPost = function(){};

    importPost.prototype =
    {
      //
      // Init
      init: function()
      {
        this.search();
        this.filter();
        this.filterCategory();
        this.selectPost();
        this.submitForm();
      },

      //
      //Search function
      search: function()
      {
        var self = this;
        $( '.ipebo-import-search-input' ).keyup( function()
        {
          self.filterCategory();
        });
      },

      //Do filter function
      doFilterKeyword: function( keyword )
      {
        $( '.ipebo-import-post' ).each( function()
        {
          var title = $(this).text().trim();
          var re    = new RegExp(keyword, 'g');
          var textIndex = title.match( re );
          if ( textIndex && textIndex.length > 0 ) {
            $(this).show();
            $(this).removeClass( 'hidden' );
            $(this).addClass( 'filtered' );
          }
        });
      },

      //Filter function
      filter: function()
      {
        var self = this;
        $( '.ipebo-import-category a' ).click( function()
        {
          if ( $(this).hasClass( 'selected') ) {
            $(this).removeClass( 'selected' );
          } else {
            $(this).addClass( 'selected' );
          }

          self.filterCategory();
          return false;
        });
      },

      //Filter by category function
      filterCategory: function()
      {
        $( '.ipebo-import-post' ).hide();
        $( '.ipebo-import-post' ).addClass( 'hidden' );

        //Get keyword
        var keyword = $( '.ipebo-import-search-input' ).val();

        //Filter by category
        if ( $( '.ipebo-import-category .selected' ).length > 0 )
        {
          $( '.ipebo-import-category .selected' ).each( function()
          {
            var slug = $(this).data( 'slug' );

            if ( slug == 'all' ) {
              $( '.ipebo-import-post' ).show();
              $( '.ipebo-import-post' ).removeClass( 'hidden' );
            }
            else
            {
              var clsearch = 'category-'+slug;
              $( '.ipebo-import-post' ).each( function()
              {
                var title = $(this).text().trim();
                var hasKeyword  = true;
                var hasCategory = $(this).hasClass( clsearch );
                if ( keyword != '' && keyword != ' ' )
                {
                  var re = new RegExp(keyword, 'g');
                  var textIndex = title.match( re );
                  if ( ! textIndex || textIndex.length < 1 ) {
                    hasKeyword = false;
                  }
                }

                if ( hasCategory && hasKeyword )
                {
                  $(this).show();
                  $(this).removeClass( 'hidden' );
                }
              });
            }
          });
        }
        else
        {
          //Filter by search
          if ( keyword != '' && keyword != ' ' ) {
            this.doFilterKeyword( keyword );
          }
          else {
            $( '.ipebo-import-post' ).show();
            $( '.ipebo-import-post' ).removeClass( 'hidden' );
          }
        }
      },

      //Select post function
      selectPost: function()
      {
        $( '.ipebo-import-posts a' ).click( function()
        {
          if ( $(this).hasClass( 'selected') ) {
            $(this).removeClass( 'selected' );
          } else {
            $(this).addClass( 'selected' );
          }
          return false;
        });
      },

      //Submit
      submitForm: function()
      {
        $( '.ipebo-ebook-import-submit' ).click( function()
        {
          var count = 0;
          var ids   = '';
          $( '.ipebo-import-posts .selected' ).each( function()
          {
            if ( ! $(this).parent().hasClass( 'hidden' ) ){
              count++;
              ids += $(this).data( 'id' );
              ids += '|';
            }
          });

          if ( count > 0 )
          {
            var ebookId = $( '#ipebo-value-objectid' ).val();
            var ajaxUrl = ajaxurl + '?doajax=1&ipebo=1&action=import_posts';

            var loader  = $('[data-remodal-id="ipebo-modal-loader"]').remodal();
            loader.open();

            $.ajax({
              url: ajaxUrl,
              type: "POST",
              dataType: "json",
              data: 'ebookid='+ebookId+'&ebook_posts='+ids,
              success: function ( response )
              {
                loader.close();
                if ( response.status == 1 )
                {
                  window.location.href = response.redirect;
                }
                else {
                  var inst  = $('[data-remodal-id="ipebo-import-modal-failed"]').remodal();
                  inst.open();
                }
              }
            });
          }
          else
          {
            var inst  = $('[data-remodal-id="ipebo-import-modal-error"]').remodal();
            inst.open();
          }

          return false;
        });
      },


    };

})(jQuery);

module.exports = importPost;

},{}],11:[function(require,module,exports){
/*
 *===========================================================
 * Manage script
 *===========================================================*/
var uifunc = require( './../libs/ui.js' );
var manage;

(function($)
{
    "use strict";

    var delete_confirm_title = '';
    var delete_confirm_info  = '';

    if (typeof ui_message != 'undefined') {
        if (typeof ui_message.delete_confirm_title != 'undefined') {
            delete_confirm_title = ui_message.delete_confirm_title;
        }
        if (typeof ui_message.delete_confirm_info != 'undefined') {
            delete_confirm_info = ui_message.delete_confirm_info;
        }
    }

    //Prepare ui
    if (typeof ui_base_image != 'undefined') {
        uifunc.base.construct.prototype.image_base = ui_base_image;
    }
    uifunc.base.confirm.prototype.set_ajax = function( ajax_url )
    {
        this.ajax_url = ajax_url;
    };

    uifunc.init();
    uifunc.delete_confirm = new uifunc.base.confirm();

    uifunc.delete_confirm.set_vars({
        title: delete_confirm_title,
        info: delete_confirm_info,
        yes_class:  'ipebo-btn ipebo-btn-md ipebo-btn-danger',
        no_class:   'ipebo-btn ipebo-btn-md ipebo-btn-default',
        action:     function( object, ui )
        {
            var id = object.data( 'id' );
            uifunc.loader.show();
            ui.hide( object );

            $.ajax({
                url: ui.ajax_url + '&id='+id,
                type: "POST",
                dataType: "json",
                success: function (result)
                {
                    if ( object.hasClass( 'ipebo-delete-edit' ) )
                    {
                        window.location.href = result.redirect;
                    } else
                    {
                        if ( result.status == '-1' ) {
                            uifunc.message.modal( result.message, 'alert-danger' );
                        } else {
                            uifunc.message.modal( result.message, 'alert-success' );
                            ui.vars.remove_element( object, ui );
                        }
                        uifunc.loader.hide();
                    }
                },
                error: function () {
                    alert( "Error in deleting item." );
                },
            });
        },
        remove_element: function( object, ui )
        {
            var tr = object.parents( 'tr' );
            tr.hide( 'blind' );
            setTimeout( function(){
                tr.remove();
            }, 600);
        }
    });

    manage = {
        delete: function( items, url )
        {
            $(items).each( function()
            {
                var button = $(this);
                button.click( function()
                {
                    uifunc.delete_confirm.set_ajax( url );
                    uifunc.delete_confirm.set_vars({
                        remove_element: function( object, ui ){
                            var cont   = object.parents( '.ipebo-manage-list' );
                            var item   = object.parents( '.ipebo-list-item' );
                            var length = $( '.ipebo-list-item', cont ).length;

                            item.hide( 'blind' );
                            if ( length == 1 ) {
                                $( '.ipebo-list-empty', cont ).show();
                            }
                            setTimeout( function(){
                                item.remove();
                            }, 600);
                        }
                    });
                    uifunc.delete_confirm.show( button );
                    return false;
                });
            });
        }
    }
})(jQuery);

module.exports = manage;

},{"./../libs/ui.js":16}],12:[function(require,module,exports){
/*
 *===========================================================
 * On Off Button
 *===========================================================*/
var onoff;

(function($)
{
    "use strict";

    /**
     * On Off button
     */
    onoff = function( root, options )
    {
        var defaults  = {
            func_select: function( button ){},
        };
        var options = $.extend( defaults, options );

        $(root).each( function(){

            var cont = $(this);
            $( '.ipebo-btn-onoff', cont ).click( function()
            {
                var value = $(this).data( 'value' );
                if ( ! $(this).hasClass( 'active' ) ) {
                    $( 'button', cont ).removeClass( 'active' );
                    $( 'button', cont ).removeClass( 'ipebo-btn-on' );
                    $( 'button', cont ).removeClass( 'ipebo-btn-off' );
                    if ( value == true ) {
                        $(this).addClass( 'ipebo-btn-on' );
                    } else {
                        $(this).addClass( 'ipebo-btn-off' );
                    }
                    $(this).addClass( 'active' );
                }
                $( '.ipebo-onoff-value', cont ).val(value);

                var field = $(this).parents( '.ipebo-field-row' );
                if ( field.hasClass( 'ipebo-toggle' )) {

                    if ( value == 'false' || value == false ) {
                        $( '.ipebo-toggle-option', field ).hide();
                    } else {
                        $( '.ipebo-toggle-option', field ).show();
                    }
                }
                options.func_select( $(this), cont );
                return false;
            });
        });
    };

})(jQuery);

module.exports = onoff;

},{}],13:[function(require,module,exports){
/*
 *===========================================================
 * Select preview
 *===========================================================*/

var selectPreview;

(function($)
{
    "use strict";

    selectPreview = function( input, options )
    {
        var defaults  = {
            cont_selector:          '.ipebo-field-row',
            cont_preview_selector:  '.ipebo-select-preview',
            item_preview_class:     'ipebo-preview',
            root_image:             '',
            preview_data:           [],
            func_select:            function( input ){}
        };
        var opt  = $.extend( defaults, options );
        var cont = input.parents( opt.cont_selector );

        input.change( function()
        {
            var select  = $(this).val();
            var preview = $( opt.cont_preview_selector, cont );
            var opts    = opt.preview_data;
            preview.html( '' );
            for (var i = 0; i < opts.length; i++)
            {
                var key = opts[i];
                var src = opt.root_image+select+'/'+key+'.png';
                preview.append(
                    '<span class="'+opt.item_preview_class+'"><img src="'+src+'" alt="'+key+'"></span>'
                );
            }
            opt.func_select( $(this) );
        });
    };

})(jQuery);

module.exports = selectPreview;

},{}],14:[function(require,module,exports){
/*
 *===========================================================
 * Tab menu
 *===========================================================*/
var stat;

(function($)
{
    "use strict";

    stat = function( object, options )
    {
        var defaults  = {
            ajax_url: '',
            chart_id:  'ipebo-chart-div'
        };
        var ipebo_opt = $.extend( defaults, options );

        var root = $(object);

        //Date picker
        $( '#date-start',root )
            .datepicker()
            .bind('changeDate', function(ev){
                $( '#date-start',root ).datepicker('hide');
            });

        $( '#date-end', root )
            .datepicker()
            .bind('changeDate', function(ev){
                $( '#date-end',root ).datepicker('hide');
            });

        //Do filter date
        $( '#do-filter-date', root ).click( function()
        {
            var dateStart = $( '#date-start' ).val();
            var dateEnd   = $( '#date-end' ).val();
            var pageId    = $( '#value-pageid' ).val();
            var type      = $( '#value-stat-type' ).val();

            //Show loader
            $( '.ipebo-chart-main', root ).html('').hide();
            $( '.ipebo-chart-load', root ).show();

            $.ajax({
                url: ipebo_opt.ajax_url,
                type: "POST",
                data: 'start='+dateStart+'&end='+dateEnd+'&pageid='+pageId+'&type='+type,
                dataType: "json",
                success: function (result ) {

                    $( '.ipebo-chart-main' ).show();
                    $( '.ipebo-chart-load' ).hide();
                    if ( result.status == 1 && result.data.length > 0 )
                    {
                        var dataTable = new google.visualization.DataTable();
                        dataTable.addColumn( 'date', result.labelDate );
                        dataTable.addColumn( 'number', result.labelValue );

                        var rows = [];
                        for (var i=0; i<result.data.length; i++)
                        {
                            var cur   = result.data[i];
                            var data  = [ new Date( cur.year, cur.mon, cur.day ), cur.value ];
                            rows.push( data );
                        }
                        dataTable.addRows( rows );
                        var options = {
                            series: [{pointSize:6}]
                        };
                        var chart = new google.visualization.LineChart(document.getElementById(ipebo_opt.chart_id));
                        chart.draw(dataTable, options);
                    }
                },
                error: function () {
                    alert( "Error in get statistic data" );
                },
            });

            return false;
        });

        //Stat Nav
        $( '.ipebo-stat-nav a' ).click( function()
        {
            var link   = $(this);
            var root   = link.parents( 'li' );
            var type   = link.attr( 'href' ).replace( '#stat-', '' );
            var pageId = $( '#value-pageid' ).val();

            if ( root.hasClass( 'active' )) {
                return true;
            }
            $( '.ipebo-stat-nav li' ).removeClass( 'active' );
            root.addClass( 'active' );
            $( '#value-stat-type' ).val( type );

            //Show loader
            $( '.ipebo-chart-main' ).html('').hide();
            $( '.ipebo-chart-load' ).show();

            //Load cart
            $.ajax({
                url: ipebo_opt.ajax_url,
                type: "POST",
                data: 'pageid='+pageId+'&type='+type,
                dataType: "json",
                success: function (result ) {

                    $( '.ipebo-chart-main' ).show();
                    $( '.ipebo-chart-load' ).hide();
                    if ( result.status == 1 && result.data.length > 0 )
                    {
                        var dataTable = new google.visualization.DataTable();
                        dataTable.addColumn( 'date', result.labelDate );
                        dataTable.addColumn( 'number', result.labelValue );

                        var rows = [];
                        for (var i=0; i<result.data.length; i++)
                        {
                            var cur   = result.data[i];
                            var data  = [ new Date( cur.year, cur.mon, cur.day ), cur.value ];
                            rows.push( data );
                        }
                        dataTable.addRows( rows );
                        var options = {
                            series: [{pointSize:6}]
                        };
                        var chart = new google.visualization.LineChart(document.getElementById(ipebo_opt.chart_id));
                        chart.draw(dataTable, options);
                    }
                },
                error: function () {
                alert( "Error in get statistic data" );
                },
            });

            return false;
        });
    };

})(jQuery);

module.exports = stat;

},{}],15:[function(require,module,exports){
/*
 *===========================================================
 * Tab menu
 *===========================================================*/
var tab;

(function($)
{
    "use strict";

    tab = function( object, options )
    {
        var defaults  = {
            after_click:     function(){},
            tab_init:        function(){}
        };
        var opt  = $.extend( defaults, options );

        $(object).each( function()
        {
            var root = $(this);

            tab_init();

            $( '.ipebo-tab-menu a', root ).click( function()
            {
                clear_tab();
                $( this ).addClass( 'active' );

                var idtab = $(this).attr( 'href' ).replace( '#', '' );
                $( '#'+idtab ).show();
                $( '#ipebo-tab-selected' ).val( idtab );

                opt.after_click( idtab, root );

                return false;
            });

            function clear_tab()
            {
                $( '.ipebo-tab-menu a', root ).each( function(){
                    $(this).removeClass( 'active' );
                });
                $( '.ipebo-tab-content', root ).hide();
            };

            function tab_init()
            {
                clear_tab();
                $( '#ipebo-tab-selected', root ).hide();
                var current = $( '#ipebo-tab-selected' ).val();
                $( '#'+current ).show();
                $( 'a[href="#'+current+'"]' ).addClass( 'active' );
                $( '#ipebo-tab-selected' ).val( current );

                opt.tab_init( current, root );
            };
        });
    };

})(jQuery);

module.exports = tab;

},{}],16:[function(require,module,exports){
/*
 *===========================================================
 * UI
 *===========================================================
 */
var ui;

(function($)
{
    "use strict";

    //Inherit function
    function inherit( proto )
    {
        function F() {}
        F.prototype = proto
        return new F
    }

    /*
     *===========================================================
     * Ui prototype
     *===========================================================*/
    function ui_construct( options ){};

    ui_construct.prototype =
    {

        vars:               {},
        class_prefix:       'ipebo-',
        class_main:         '-main',
        class_title:        '-title',
        class_content:      '-content',
        class_disabled:     'disabled',
        class_close:        'ui-close',
        label_close:        'x',
        image_base:         '',

        top_margin:         '80px',
        overlay_speed:      600,
        main_speed:         400,
        show_speed:         4000,

        show_after:         function(){},
        hide_after:         function(){},

        popup_body: function( class_root, content, title, close )
        {
            var body = $( '<div class="'+class_root+'"></div>' );
            var main = $( '<div class="'+class_root + this.class_main+'"></div>' );
            if ( title )
            {
                var _title = $( '<div class="'+class_root + this.class_title+'"></div>' )
                                .append( '<h3>'+title+'</h3>' );
                if ( close )
                {
                    var _close = $( '<a href="#">'+this.label_close+'</a>' );
                    this.popup_close( _close, class_root );

                    _title.append(
                        $( '<div class="'+this.class_prefix+this.class_close+'"></div>' ).append( _close )
                    );
                }
                main.append( _title );
            }
            if ( content ) {
                main.append(
                    $( '<div class="'+class_root + this.class_content+'"></div>' ).append( content )
                );
            }
            body.append( main );
            return body;
        },
        popup_show: function( popup )
        {
            var c_root = popup.attr( 'class' );
            var main   = $( '.'+c_root+this.class_main, popup );
            popup.show();
            main.css( 'margin-top', '-40%' )
                .animate({marginTop: this.top_margin}, this.main_speed, this.show_after);
        },
        popup_hide: function( popup )
        {
            var c_root = popup.attr( 'class' );
            var main   = $( '.'+c_root+this.class_main, popup );

            popup.animate( {opacity: 0}, this.overlay_speed, function(){
                $(this).css( 'opacity', 1 )
                       .hide();
            });
            main.animate( {marginTop: '-60%'}, this.main_speed, this.hide_after);
        },
        popup_close: function( button, class_popup )
        {
            var ui = this;
            button.click( function(){
                var popup = $( '.'+class_popup );
                ui.popup_hide( popup );
                return false;
            });
        },
        is_load: function( selector )
        {
            if ( $( selector ).length > 0 ) {
                return true;
            }
            return false;
        },
        set_vars: function( options )
        {
            this.vars = $.extend(this.vars, options);
        },
        set_base_vars: function( options )
        {
            this.vars = $.extend(this, options);
        },
        go_to: function( selector )
        {
            var pos = $( selector ).position();
            $('html, body').animate({scrollTop: ( pos.top )}, 600);
            return false;
        }
    };

    /*
     * Modal Class
     **/
    function modal(){};
    modal.prototype = inherit( ui_construct.prototype );
    //Modal message
    modal.prototype.ajax = function( id, url )
    {
        var ui      = this;
        var c_root  = this.class_prefix + this.class_modal;
        var modal   = this.popup_body( c_root, '', false, false ).appendTo( $( 'body' ) );
        modal.addClass( class_type )
             .attr( 'id', id );

        this.popup_show( modal );
    };

    /*
     * Loader Class
     **/
    function loader(){};
    loader.prototype = inherit( ui_construct.prototype );
    //Properties
    loader.prototype.class_root     = 'ui-loader';
    loader.prototype.loader_file    = 'loading.gif';
    //Show loader
    loader.prototype.show = function()
    {
        var loader = this.create();
        this.popup_show( loader );
    };
    //Hide loader
    loader.prototype.hide = function()
    {
        var loader = this.create();
        this.popup_hide( loader );
    };
    //Create loader
    loader.prototype.create = function()
    {
        var c_root = this.class_prefix + this.class_root;
        if ( this.is_load( '.'+c_root )) {
            return $( '.'+c_root );
        } else
        {
            var image  = '<img src="'+this.image_base+this.loader_file+'" alt="Loading...">';
            var loader = this.popup_body( c_root, image, false, false ).appendTo( $( 'body' ) );
            loader.hide();

            return loader;
        }
    };


    /*
     * Inline loader Class
     **/
    function iloader(){};
    iloader.prototype = inherit( ui_construct.prototype );
    //Properties
    iloader.prototype.class_root   = 'ui-inline-loader';
    iloader.prototype.loader_file  = 'spin.gif';
    //Loader show
    iloader.prototype.show = function( button )
    {
        button.attr( 'disabled', 'disabled' );
        this.create( button ).show();
    };
    //Loader hide
    iloader.prototype.hide = function( button )
    {
        button.removeAttr( 'disabled' );
        this.create( button ).hide();
    };
    //Creating loader
    iloader.prototype.create = function( button )
    {
        var root   = button.parent();
        var lclass = this.class_prefix + this.class_root;
        if ( $( '.'+lclass, root ).length > 0 ) {
            var loader = $( '.'+lclass, root );
        } else {
            var image  = '<img src="'+this.image_base+this.loader_file+'" alt="Loading...">';
            var loader = $( '<span class="'+lclass+'"></span>' )
                            .append( image )
                            .css( 'margin-left', '8px' );
        }
        button.after( loader );
        return loader;
    };


    /*
     * Confirm popup
     **/
    function confirm( options )
    {
        var defaults = {
            title:      'Confirmation',
            info:       '',
            yes_label:  'Yes',
            no_label:   'Cancel',
            yes_class:  'btn btn-danger',
            no_class:   'btn btn-success',
            action:     function( object ){}
        };
        this.vars = $.extend(defaults, options);
    };
    confirm.prototype = inherit( ui_construct.prototype );
    //Properties
    confirm.prototype.class_root = 'ui-confirm';
    //Confirm show
    confirm.prototype.show = function( object )
    {
        var confirm = this.create( object );
        this.popup_show( confirm );
    };
    //Confirm hide
    confirm.prototype.hide = function( object )
    {
        var confirm = this.create( object );
        this.popup_hide( confirm );
    };
    //Creating confirm
    confirm.prototype.create = function( object )
    {
        var ui     = this;
        var c_root = this.class_prefix + this.class_root;
        if ( this.is_load( '.'+c_root )) {
            $( '.'+c_root ).remove();
        }
        //Create confirm
        var yesbtn = $( '<a href="#" class="ip-confirm-yes '+this.vars.yes_class+'">'+this.vars.yes_label+'</a>' );
        var nobtn  = $( '<a href="#" class="ip-confirm-no '+this.vars.no_class+'">'+this.vars.no_label+'</a>' );

        yesbtn.click( function(){
            ui.vars.action( object, ui );
            return false;
        });
        this.popup_close( nobtn, c_root );

        var content = $( '<div class="'+c_root+'-inner"></div>' )
            .append( '<p>'+this.vars.info+'</p>' )
            .append(
                $( '<div class="'+c_root+'-actions"></div>' )
                    .append( yesbtn )
                    .append( nobtn )
            );

        var confirm = this.popup_body( c_root, content, this.vars.title, false ).appendTo( $( 'body' ) );
        confirm.hide();

        return confirm;
    };


    /*
     * Alert popup
     **/
    function alert( options )
    {
        var defaults = {
            message:        '',
            label_button:   'Ok',
            class_root:     '',
            class_button:   'ipebo-btn ipebo-btn-sm ipebo-btn-danger',
            after:          function(){}
        };
        this.vars = $.extend(defaults, options);
    };
    alert.prototype = inherit( ui_construct.prototype );
    //Properties
    alert.prototype.class_root = 'ui-alert';
    //Alert show
    alert.prototype.show = function( message )
    {
        this.vars.message = message;
        var alert = this.create();
        this.popup_show( alert );
    };
    //Alert hide
    alert.prototype.hide = function()
    {
        var alert = this.create();
        this.hide_after = this.vars.after;
        this.popup_hide( alert );
    };
    //Creating alert
    alert.prototype.create = function()
    {
        var ui     = this;
        var c_root = this.class_prefix + this.class_root;
        if ( this.is_load( '.'+c_root )) {
            return $( '.'+c_root );
        } else
        {
            var button = $( '<a href="#" class="'+this.vars.class_button+'">'+this.vars.label_button+'</a>' );
            this.popup_close( button, c_root );

            var c_info  = c_root + '-inner';
            if ( this.vars.class_root != '' ) {
                c_info += ' ' + this.vars.class_root;
            }
            var content = $( '<div class="'+c_info+'"></div>' )
                .append( '<p>'+this.vars.message+'</p>' )
                .append(
                    $( '<div class="'+c_root+'-actions"></div>' )
                        .append( button )
                );

            var popup  = this.popup_body( c_root, content, false, false ).appendTo( $( 'body' ) );
            popup.hide();

            return popup;
        }
    };


    /*
     * Message
     **/
    function message(){};
    message.prototype = inherit( ui_construct.prototype );
    //Properties
    message.prototype.class_modal = 'ui-modal-message';
    message.prototype.class_block = 'ui-message';
    //Modal message
    message.prototype.modal = function( message, class_type )
    {
        var ui      = this;
        var c_root  = this.class_prefix + this.class_modal;
        var content = '<p>'+message+'</p>';
        var msg     = this.popup_body( c_root, content, false, false ).appendTo( $( 'body' ) );
        msg.addClass( class_type )
           .show()
           .animate({top: "0"}, ui.overlay_speed, function(){
                var object = $( this );
                setTimeout( function(){
                    object.animate({top: '-100px'}, ui.overlay_speed, function(){
                        object.remove();
                    });
                }, ui.show_speed);
            });
    };
    //Bloack message
    message.prototype.block = function( message, class_type, block, close )
    {
        var ui      = this;
        var c_root  = this.class_prefix + this.class_block;
        var parent  = block.parent();

        $( '.'+c_root, parent ).remove();

        var msg = $( '<div></div>' );
        msg.addClass( c_root )
           .addClass( class_type )
           .append( '<p>'+message+'</p>' );

        if ( close )
        {
            var _close = $( '<a href="#">'+this.close_label+'</a>' );
            _close.click( function()
            {
                var _msg = $(this).parents( '.'+c_root );
                _msg.animate({opacity: "0"}, ui.overlay_speed, function(){
                    _msg.remove();
                });
                return false;
            });
            msg.append(
                $( '<div class="'+this.class_prefix+this.class_close+'"></div>' ).append( _close )
            );
        }
        msg.css( 'opacity', '0' );
        parent.prepend( msg );
        msg.animate({opacity: "1"}, ui.overlay_speed);
    };

    ui =
    {
        base: {
            construct:  ui_construct,
            loader:     loader,
            iloader:    iloader,
            confirm:    confirm,
            message:    message,
            alert:      alert,
            modal:      modal

        },
        init: function()
        {
            this.loader  = new this.base.loader(),
            this.iloader = new this.base.iloader(),
            this.confirm = new this.base.confirm(),
            this.message = new this.base.message(),
            this.alert   = new this.base.alert(),
            this.modal   = new this.base.modal()
        }
    };

})(jQuery);

module.exports = ui;

},{}],17:[function(require,module,exports){
/*
 *===========================================================
 * Image uploader handle
 *===========================================================*/
var uploader;

(function($)
{
    "use strict";

    uploader = function( element, options )
    {
        var defaults  = {
            func_select: function( input, url ){},
            func_reset: function( input, url ){},
        };
        var options = $.extend( defaults, options );

        $(element).each( function()
        {
            var root = $(this);
            var frame;
            $( '.ipebo-btn-media', root ).click( function(e)
            {
                var $el = $(this);
                e.preventDefault();

                // If the media frame already exists, reopen it.
                if ( frame ) {
                    frame.open();
                    return;
                }
                // Create the media frame.
                frame = wp.media.frames.customHeader = wp.media({
                    // Set the title of the modal.
                    title: $el.data('choose'),

                    // Tell the modal to show only images.
                    library: {
                        type: 'image'
                    },

                    // Customize the submit button.
                    button: {
                        // Set the text of the button.
                        text: $el.data('update'),
                        // Tell the button not to close the modal, since we're
                        // going to refresh the page when the image is selected.
                        close: false
                    }
                });

                // When an image is selected, run a callback.
                frame.on( 'select', function() {
                    // Grab the selected attachment.
                    var attachment = frame.state().get('selection').first();
                    var id         = attachment.id;

                    var url = '';
                    var fullurl = '';
                    if (typeof attachment.attributes.sizes.thumbnail != 'undefined') {
                        url = attachment.attributes.sizes.thumbnail.url;
                    }
                    else if (typeof attachment.attributes.sizes.medium != 'undefined') {
                        url = attachment.attributes.sizes.medium.url;
                    }
                    else if (typeof attachment.attributes.sizes.full != 'undefined') {
                        url = attachment.attributes.sizes.full.url;
                    }
                    if (typeof attachment.attributes.sizes.full != 'undefined') {
                        fullurl = attachment.attributes.sizes.full.url;
                    }

                    $( '.ipebo-upload-preview .none', root ).hide();
                    $( '.ipebo-upload-preview .image', root )
                        .html( '<img src="'+url+'" alt="" />' )
                        .show();
                    $( '.ipebo-image-id', root ).val( id );
                    $( '.ipebo-btn-reset', root ).show();

                    //Select callback
                    options.func_select( element, fullurl );

                    frame.close();
                });
                frame.open();

                return false;
            });
            $( '.ipebo-btn-reset', root ).click( function(e)
            {
                $( '.ipebo-upload-preview .none', root ).show();
                $( '.ipebo-upload-preview .image', root ).html( '' ).hide();
                $( '.ipebo-image-id', root ).val( 0 );
                $( '.ipebo-btn-reset', root ).hide();

                //Select callback
                options.func_reset( root );

                return false;
            });
        });
    };

})(jQuery);

module.exports = uploader;

},{}]},{},[1]);
