<?php
/**
 * Class helper for create html
 *
 * @package wp-ebook-creator
 * @author InstaProduct
 * @link http://insta-product.com/
 */

if ( ! class_exists( 'IPEbo_Html' )) 
{
  class IPEbo_Html
  {
		/**
		 * Take tag name
		 *
		 * @var string
		 */
		protected $_tag;
		
		/**
		 * Take content
		 *
		 * @access protected
		 * @var string
		 */
		protected $_content;
		
		/**
		 * Take config data
		 *
		 * @access protected
		 * @var array
		 */
		protected $_configData;
		
		/**
		 * Take config object
		 *
		 * @access protected
		 * @var array
		 */
		protected $_config;
		
		/**
		 * Take self tag
		 *
		 * @var array
		 */
		protected $_selfTag = array(
			'img',
			'input',
			'br',
			'hr'
		);
		
		/**
		 * Html attribs
		 *
		 * @var array
		 */
		protected $_htmlAttribs = array(
			'id',
			'class',
			'title',
			'href',
			'rel',
			'alt',
			'src',
			'cite',
			'target',
			'media',
			'rev',
			'type',
			'charset',
			'value',
			'accesskey',
			'coords',
			'name',
			'hreflang',
			'tabindex',
			'dir',
			'datetime',
			'rbspan',
			'disabled',
			'accept-charset',
			'action',
			'selected',
			'size',
			'maxlength',
			'rows',
			'cols',
			'enctype',
			'method',
			'accept',
			'checked',
			'for',
			'readonly',
			'usemap',
			'label',
			'multiple',
			'nohref',
			'height',
			'weight',
			'ismap',
			'longdesc',
			'archive',
			'classid',
			'codebase',
			'codetype',
			'declare',
			'data',
			'declare',
			'valuetype',
			'charoff',
			'char',
			'span',
			'valign',
			'align',
			'cellpadding',
			'cellspacing',
			'frame',
			'rules',
			'summary',
			'abbr',
			'axis',
			'colspan',
			'headers',
			'rowspan',
			'scope',
			'dir',
			'lang',
			'style',
			'aria-required'
		);
		
		/**
		 * __construct
		 *
		 * @param string $tag 
		 * @param string $content
		 * @param array  $config
		 * @return void
		 */
		public function __construct( $tag, $content = '', $config = array() )
		{
			$this->setTag( $tag );
			$this->setContent( $content );
			$this->_setConfig( $config );
		}
		
		/**
		 * setTag
		 *
		 * @param string $tag
		 * @return IPEbo_Html
		 */
	    public function setTag( $tag )
		{
			$this->_tag = $tag;
			return $this;
		}
		
		/**
		 * setContent
		 *
		 * @param string $content
		 * @return IPEbo_Html
		 */
		public function setContent( $content )
		{
			$this->_content = $content;
			return $this;
		}	
		
		/**
		 * getContent
		 *
		 * @return string
		 */
		public function getContent()
		{
			return $this->_content;
		}
		
		/**
		 * isSelfTag
		 *
		 * @return boolean
		 */
	    public function isSelfTag()
		{
			return in_array( $this->_tag, $this->_selfTag );
		}	
		
		/**
		 * create
		 *
		 * @return string
		 */
		public function create()
		{
			$result  = $this->getOpeningTag();
			if ( ! empty( $this->_content ))
			{
				$content = $this->_content;

				if ( is_array( $content )) 
				{
					foreach( $content as $subcontent )
					{
						$result .= $this->_renderContent( $subcontent );
					}
				}
				else {
					$result .= $this->_renderContent( $content );
				}
			}
			$result .= $this->getClosingTag();

			return $result;
		}
		
		/**
		 * _renderContent
		 *
		 * @param  string|array|IPEbo_Html $content
		 * @return IPEbo_Html
		 */
		public function _renderContent( $content )
		{
			$result = '';
			if ( $content instanceof IPEbo_Html ) {
				$result .= $content->create();
			} 
			elseif ( is_string( $content )) {
				$result .= $content;
			}
			return $result;
		}
		
		/**
		 * getOpeningTag
		 *
		 * @return string
		 */
		public function getOpeningTag()
		{
			$tag  = "<$this->_tag";
			$atts = $this->getConfig()->toArray();
			if ( isset( $atts['class'] ) ) {
				$tag .= $this->getHtmlClass( $atts['class'] );
				unset( $atts['class'] );
			}
			$tag .= $this->getHtmlAttribs( $atts );
			if ( $this->isSelfTag() ) {
				$tag .= ' /';
			}
			$tag .= '>';
			return $tag;
		}	
		
		/**
		 * getClosingTag
		 *
		 * @return string
		 */
		public function getClosingTag()
		{
			$tag = '';
			if ( ! $this->isSelfTag() ) {
				$tag = "</$this->_tag>";
			}
			return $tag;
		}	
		
		/**
		 * getHtmlClass
		 *
		 * @param array  $classes
		 * @param string $default Default class
		 * @return string
		 */
	  public function getHtmlClass( $classes = '', $default = '' )
		{
			$result = '';
			$class  = array();
			if ( is_string( $classes )) {
				$class = explode( ' ', $classes );
			} elseif ( is_array( $classes )) {
				$class = $classes;
			} elseif( empty( $classes ) && ! empty( $default )) {
				$class = (array)$default;
			}
			if ( ! empty( $class )) {
				foreach( $class as $idx => $string ) {
					$class[$idx] = trim( $string );
				}
				$class  = array_unique( $class );
				$result = ' class="' . implode( ' ', $class ) . '"';
			}
			return $result;
		}
		
		/**
		 * getHtmlAttrib
		 *
		 * @param string $name Html attribute name
		 * @param string $value
		 * @return string
		 */
	  public function getHtmlAttrib( $name, $value = '' )
		{
			$html = '';
			if ( ! empty( $value ) && is_string( $name ) && in_array( $name, $this->_htmlAttribs, true )) {
				$html = " $name=\"$value\"";
			}
			return $html;
		}
		
		/**
		 * getHtmlAttribs
		 *
		 * @param array $value
		 * @return string
		 */
	  public function getHtmlAttribs( $attribs )
		{
			$html = '';
			foreach( $attribs as $name => $value ) {
				$html .= $this->getHtmlAttrib( $name, $value );
			}
			return $html;
		}
		
		/**
		 * Get config object
		 *
		 * @return IPEbo_Config
		 */
	  public function getConfig()
		{
			return $this->_config;
		}
		
		/**
		 * _setConfig
		 *
		 * @param array $config
		 * @return void
		 */
	  protected function _setConfig( $config = array() )
		{
			$this->_configData = $config;
			$this->_config     = new IPEbo_Config( $config );
		}

  }
}