<?php
/**
 * Class helper for create form
 *
 * @package wp-ebook-creator
 * @author InstaProduct
 * @link http://insta-product.com/
 */

if ( ! class_exists( 'IPEbo_Form' ))
{
  class IPEbo_Form
  {
    //{{ input

    /**
     * Build form input
     *
     * @param string $name
     * @param string $type
     * @param string $value
     * @param array  $atts
     *
     * @return IPEbo_Html
     */
    public function input(
      $name, $type = 'text', $value = '', $atts = array()
    )
    {
      if ( isset( $this->setting ) && $this->setting && $this->setting instanceof IPEbo_Config )
      {
        $value = $this->setting->getSetting( $name, $value );
      }
      $value = stripslashes( $value );
      $defaultAtts = array(
        'type'  => $type,
        'value' => $value,
        'id'    => 'input-' . $name,
        'name'  => $name,
        'class' => 'ipebo-input ipebo-input-full'
      );

      if ( ! empty( $atts )) {
        $defaultAtts = array_merge( $defaultAtts, $atts );
      }

      $input = new IPEbo_Html( 'input', '', $defaultAtts );
      return $input;
    }

    //}}
    //{{ renderInput

    /**
     * Render form input
     *
     * @param string $name
     * @param string $type
     * @param string $value
     * @param array  $atts
     *
     * @return void
     */
    public function renderInput(
      $name, $type = 'text', $value = '', $atts = array()
    ) {
      $html = $this->input( $name, $type, $value, $atts );
      echo $html->create();
    }

    //}}
  	//{{ inputSize

  	/**
  	 * Build form input size
  	 *
  	 * @param string $name
     * @param string $value
  	 * @param string $unit
  	 *
  	 * @return IPEbo_Html
  	 */
  	public function inputSize(
  		$name, $value = '', $unit = 'px'
  	)
  	{
  		if ( isset( $this->setting ) && $this->setting && $this->setting instanceof IPEbo_Config )
  		{
  			$value = $this->setting->getSetting( $name, $value );
  		}
  		$atts = array(
  			'type'	=> 'text',
  			'value' => $value,
  			'id'		=> 'input-' . $name,
  			'name'	=> $name,
  			'class' => 'ipebo-input ipebo-input-xs'
  		);

      $input = new IPEbo_Html( 'input', '', $atts );
      $unit  = new IPEbo_Html( 'div', $unit, array(
        'class'   => 'ipebo-input-addon'
      ));

      $group = new IPEbo_Html( 'div', array(
        $input,
        $unit
      ), array(
        'class'   => 'ipebo-input-group mr5'
      ));

  		return $group;
  	}

  	//}}
  	//{{ renderInputSize

  	/**
  	 * Render form input size
  	 *
     * @param string $name
     * @param string $value
     * @param string $unit
  	 *
  	 * @return void
  	 */
  	public function renderInputSize(
  		$name, $value = '', $unit = 'px'
  	) {
  		$html = $this->input( $name, $type, $value, $atts );
  		echo $html->create();
  	}

    //}}
    //{{ textarea

    /**
     * Build form textarea
     *
     * @param string $name
     * @param string $value
     * @param array  $atts
     *
     * @return IPEbo_Html
     */
    public function textarea(
      $name, $value = '', $atts = array()
    )
    {
      if ( isset( $this->setting ) && $this->setting && $this->setting instanceof IPEbo_Config )
      {
        $value = $this->setting->getSetting( $name, $value );
      }
      $value = stripslashes( $value );
      $defaultAtts = array(
        'id'    => 'input-' . $name,
        'name'  => $name,
        'class' => 'ipebo-input ipebo-input-full'
      );

      if ( ! empty( $atts )) {
        $defaultAtts = array_merge( $defaultAtts, $atts );
      }

      $input = new IPEbo_Html( 'textarea', $value, $defaultAtts );
      return $input;
    }

    //}}
    //{{ renderTextarea

    /**
     * Render form textarea
     *
     * @param string $name
     * @param string $value
     * @param array  $atts
     *
     * @return void
     */
    public function renderTextarea(
      $name, $value = '', $atts = array()
    ) {
      $html = $this->textarea( $name, $value, $atts );
      echo $html->create();
    }

    //}}
    //{{ select

    /**
     * Build form select
     *
     * @param string $name
     * @param array  $options
     * @param string $value
     * @param array  $atts
     *
     * @return IPEbo_Html
     */
    public function select(
      $name, $options, $value = '', $atts = array()
    )
    {
      if ( isset( $this->setting ) && $this->setting && $this->setting instanceof IPEbo_Config )
      {
        $value = $this->setting->getSetting( $name, $value );
      }
      $defaultAtts = array(
        'id'    => 'input-' . $name,
        'name'  => $name,
        'class' => 'ipebo-input'
      );

      if ( ! empty( $atts )) {
        $defaultAtts = array_merge( $defaultAtts, $atts );
      }

      $content = '';
      if ( is_array( $options ))
      {
        foreach( $options as $key => $label )
        {
          $sltd = '';
          if ( $key == $value ) {
              $sltd = ' selected="selected"';
          }
          $content .= '<option value="' . $key . '"' . $sltd . '>' . $label . '</option>';
        }
      }

      $input = new IPEbo_Html( 'select', $content, $defaultAtts );
      return $input;
    }

    //}}
    //{{ renderSelect

    /**
     * Render form select
     *
     * @param string $name
     * @param array  $options
     * @param string $value
     * @param array  $atts
     *
     * @return void
     */
    public function renderSelect(
      $name, $options, $value = '', $atts = array()
    ) {
      $html = $this->select( $name, $options, $value, $atts );
      echo $html->create();
    }

    //}}
    //{{ fancyOption

    /**
     * Build form fancy option
     *
     * @param string $name
     * @param string $value
     *
     * @return IPEbo_Html
     */
    public function fancyOption( $name, $value = '' )
    {
      //Check font class
      if ( ! class_exists( 'IPEbo_Font' )) {
        return false;
      }
      //Get value
      if ( isset( $this->setting ) && $this->setting && $this->setting instanceof IPEbo_Config )
      {
        $value = $this->setting->getSetting( $name, $value );
      }

      $fonts    = IPEbo_Font::getOptions();
      $fname    = IPEbo_Font::getFontName( $value );
      $root     = IPEBO_BASE_URL . 'assets/img/font/';
      $preview  = $root . $fname . '.png';
      $labelimg = '<img src="'.$preview.'" alt="'.$fname.'">';

      $control = '<div class="ipebo-menu">' .
                    '<div class="ipebo-selected">' .
                      '<div class="ipebo-selected-box">' .
                        '<span data-select="' . $value . '">' .
                          $labelimg .
                        '</span>' .
                      '</div>' .
                      '<span class="toggle"></span>' .
                    '</div>' .
                    '<div class="ipebo-select" style="display: none;">' .
                      '<div class="ipebo-select-content">' .
                        '<ul class="clearfix">';

      foreach( $fonts as $groupId => $items )
      {
        $control .= '<li class="ipebo-select-group">' . $groupId . '</li>';
        foreach( $items as $fkey => $label )
        {
          $currclass = '';
          if ( $fkey == $value ) {
            $currclass = ' _selected';
          }
          $root     = IPEBO_BASE_URL . 'assets/img/font/';
          $preview  = $root . $fkey . '.png';
          $labelimg = '<img src="'.$preview.'" alt="'.$label.'">';
          $control .= '<li class="ipebo-select-item'.$currclass.'" data-select="' . $fkey . '">' .
                        '<strong class="ipebo-select-label">' . $labelimg . '</strong>' .
                      '</li>';

        }
      }
      $control .=       '</ul>' .
                      '</div>' .
                    '</div>' .
                    '<input type="hidden" name="' . $name . '" class="ipebo-input-select" value="' . $value . '" />' .
                  '</div>';

      $input = new IPEbo_Html( 'div', $control, array(
        'class' => 'ipebo-fancy-option'
      ));
      return $input;
    }

    //}}
    //{{ renderFancyOption

    /**
     * Render form fancy option
     *
     * @param string $name
     * @param string $value
     *
     * @return void
     */
    public function renderFancyOption( $name, $value = '' )
    {
      $html = $this->fancyOption( $name, $options, $value, $atts );
      echo $html->create();
    }

    //}}
    //{{ onOff

    /**
     * Build form on off buttons
     *
     * @param string $name
     * @param string $value
     * @param string $labelOn
     * @param string $labelOff
     *
     * @return IPEbo_Html
     */
    public function onOff(
      $name, $value = '', $labelOn = 'On', $labelOff = 'Off'
    )
    {
      if ( isset( $this->setting ) && $this->setting && $this->setting instanceof IPEbo_Config )
      {
        $value = $this->setting->getSetting( $name, $value );
      }

      if ( $value == 'false' ) {
          $clon  = '';
          $cloff = ' active';
      } else {
          $clon  = ' active';
          $cloff = '';
      }

      $control = '<div class="ipebo-btn-group">' .
                    '<button data-value="true" class="ipebo-btn-onoff ipebo-btn-on' . $clon . '" type="button">' .
                        __( $labelOn, IPEBO_SLUG ) .
                    '</button>' .
                    '<button data-value="false" class="ipebo-btn-onoff ipebo-btn-off' . $cloff . '" type="button">' .
                      __( $labelOff, IPEBO_SLUG ) .
                    '</button>' .
                  '</div>' .
                  '<input type="hidden" class="ipebo-onoff-value" name="' . $name . '" value="' . $value . '"/>';

      $input = new IPEbo_Html( 'div', $control, array(
        'class' => 'ipebo-onoff'
      ));
      return $input;
    }

    //}}
    //{{ renderOnOff

    /**
     * Render On Off buttons
     *
     * @param string $name
     * @param string $value
     * @param string $labelOn
     * @param string $labelOff
     * @param IPEbo_Config $config
     *
     * @return void
     */
    public function renderOnOff(
      $name, $value = '', $labelOn = 'On', $labelOff = 'Off'
    ) {
      $html = $this->onOff( $name, $value, $labelOn, $labelOff );
      echo $html->create();
    }

  	//}}
  	//{{ image

  	/**
  	 * Build form input image
  	 *
  	 * @param string $name
  	 *
  	 * @return IPEbo_Html
  	 */
  	public function image( $name )
    {
      $setimg   = false;
      $attachId = 0;
  		if ( isset( $this->setting ) && $this->setting && $this->setting instanceof IPEbo_Config )
  		{
  			$attachId = $this->setting->getSetting( $name );
        if ( $attachId ) {
          $setimg = true;
          $data   = wp_get_attachment_image_src( $attachId, 'thumbnail' );
        }
  		}

      //Preview
  		$control = '<div class="ipebo-upload-preview">';
      if ( $setimg ) {
          $control .= '<div class="image"><img src="' . $data[0] . '" /></div>';
          $control .='<div class="none ipebo-hide">' . __( 'No image', IPEBO_SLUG ) . '</div>';
          $resetDsp = '';
      } else {
          $control .='<div class="image"></div>';
          $control .='<div class="none">' . __( 'No image', IPEBO_SLUG ) . '</div>';
          $resetDsp = ' style="display:none;"';
      }
      $control .= '</div>';
      $control .= '<div class="ipebo-image-field">' .
                    '<div class="ipebo-select-image">' .
                      '<p class="mt0">' .
                          __( 'Select an image or upload new image using the button below.', IPEBO_SLUG ) .
                      '</p>' .
                      '<a class="ipebo-btn ipebo-btn-secondary ipebo-btn-media ipebo-btn-sm"' .
                          'data-choose="' . __( 'Choose Image', IPEBO_SLUG ) . '"' .
                          'data-update="' . __( 'Choose image', IPEBO_SLUG ) . '">' .
                        '<i class="ipebo-fa ipebo-fa-image"></i>' .
                        __( 'Select Image', IPEBO_SLUG ) .
                      '</a>' .
                      '<a class="ipebo-btn ipebo-btn-reset ipebo-btn-sm ml5" href="#"' . $resetDsp . '>' .
                          '<i class="ipebo-fa ipebo-fa-ban"></i>' .
                          __( 'Clear Image', IPEBO_SLUG ) .
                      '</a>' .
                    '</div>' .
                    '<input type="hidden" name="upload_type" class="ipebo-upload-type" value="' . $name . '" />' .
                    '<input type="hidden" name="' . $name . '" class="ipebo-image-id" value="' . $attachId . '" />' .
                  '</div>';

  		$input = new IPEbo_Html( 'div', $control, array(
        'class' => 'ipebo-upload-image ipebo-clear'
      ));
  		return $input;
  	}

  	//}}
  	//{{ renderImage

  	/**
  	 * Render input image
  	 *
  	 * @param string $name
  	 *
  	 * @return void
  	 */
  	public function renderImage( $name )
    {
  		$html = $this->image( $name );
  		echo $html->create();
  	}

  	//}}
  	//{{ field

  	/**
  	 * Build form field
  	 *
  	 * @param string  $label
  	 * @param string  $name
  	 * @param mixed   $input
  	 * @param string  $info
  	 * @param string  $layout
  	 * @param boolean $helpAbs
  	 * @param array   $options
  	 *
  	 * @return IPEbo_Html
  	 */
  	public function field(
  		$label, $name, $input, $info = '', $layout = 'horizontal', $helpAbs = false, $options = array()
  	)
  	{
  		$contents  = array();

  		//Form label
  		$label = new IPEbo_Html( 'label', __( $label, IPEBO_SLUG ), array(
  			'for' => 'input-' . $name
  		));

  		if ( $layout == 'vertical' ) {
        $labelClass = 'ipebo-form-label';
      } else {
        $labelClass = 'ipebo-form-label col-lg-3 col-md-4 col-sm-12';
      }
      if ( isset( $options['labelClass'] )) {
        $labelClass = $options['labelClass'];
      }
  		$contents[] =  new IPEbo_Html( 'div', $label, array(
  			'class' => $labelClass
  		));

  		//Form input
  		$inputContents   = array();
  		$inputContents[] = $input;
  		if ( ! empty( $info )) {
  			$inputContents[] = $this->help( $info );
  		}

      if ( $layout == 'vertical' ) {
        $inputClass = 'ipebo-field';
      } else {
        $inputClass = 'ipebo-field col-lg-9 col-md-8 col-sm-12';
      }
      if ( isset( $options['inputClass'] )) {
        $inputClass = $options['inputClass'];
      }
  		$contents[] =  new IPEbo_Html( 'div', $inputContents, array(
  			'class' => $inputClass
  		));

  		//Main
  		$class = 'ipebo-field-row';
  		if ( $layout == 'vertical' ) {
  			$class .= ' ipebo-field-y';
  		} else {
  			$class .= ' ipebo-field-x';
  		}
  		if ( $helpAbs ) {
  			$class .= ' ipebo-help-abs';
  		}
      if ( isset( $options['fieldClass'] )) {
        $class .= $options['fieldClass'];
      }
      $tagOptions = array(
        'class' => $class
      );
      if ( isset( $options['fieldId'] )) {
        $tagOptions['id'] = $options['fieldId'];
      }

  		$field = new IPEbo_Html( 'div', $contents, $tagOptions );

  		return $field;
  	}

  	//}}
  	//{{ renderField

  	/**
  	 * Render form field
  	 *
  	 * @param string  $label
  	 * @param string  $name
  	 * @param mixed   $input
  	 * @param string  $info
  	 * @param string  $layout
  	 * @param boolean $helpAbs
  	 * @param array   $options
  	 *
  	 * @return void
  	 */
  	public function renderField(
  		$label, $name, $input, $info = '', $layout = 'horizontal', $helpAbs = false, $options = array()
  	)
  	{
  		$html = $this->field( $label, $name, $input, $info, $layout, $helpAbs );
  		echo $html->create();
  	}

  	//}}
  	//{{ help

  	/**
  	 * Render form help
  	 *
  	 * @param string $info
  	 *
  	 * @return IPEbo_Html
  	 */
  	public function help( $info )
  	{
  		$contents   = array();
  		$contents[] = new IPEbo_Html(
  			'a',
  			'<i class="ipebo-fa ipebo-fa-question"></i>',
  			array(
  				'class' => 'ipebo-btn-help ipebo-btn ipebo-btn-default',
  				'href'  => '#'
  			)
  		);

  		$contents[] = new IPEbo_Html(
  			'div',
  			__( $info, IPEBO_SLUG ),
  			array(
  				'class' => 'ipebo-help-content ipebo-hide'
  			)
  		);

  		$help = new IPEbo_Html(
  			'div',
  			$contents,
  			array(
  				'class' => 'ipebo-help'
  			)
  		);

  		return $help;
  	}

  	//}}
  	//{{ fieldText

  	/**
  	 * Build form field
  	 *
  	 * @param array $config
  	 * @return IPEbo_Html
  	 */
  	public function fieldText( $config )
  	{
  		//Set config
  		if ( is_array( $config )) {
  			$config = new IPEbo_Config( $config );
  		}

  		$name    = $config->getSetting( 'name', 'name' );
  		$label   = $config->getSetting( 'label', 'Name' );
  		$info    = $config->getSetting( 'info', '' );
      $value   = $config->getSetting( 'inputValue', '' );
      $layout  = $config->getSetting( 'layout', 'horizontal' );
  		$helpAbs = $config->getSetting( 'helpAbs', true );
      $fopts   = (array)$config->getSetting( 'fieldOptions', '' );

  		$inputAtts = array();
  		if ( $iatts = $config->getSetting( 'inputAtts' ) ) {
  			$inputAtts = $iatts;
  		}
  		//Create input text
  		$input = $this->input( $name, 'text', $value, $inputAtts );

  		return $this->field($label, $name, $input, $info, $layout, $helpAbs, $fopts );
  	}

  	//}}
  	//{{ renderFieldText

  	/**
  	 * Render form field
  	 *
  	 * @param array $config
  	 * @return void
  	 */
  	public function renderFieldText( $config )
  	{
  		$html = $this->fieldText( $config );
  		echo $html->create();
  	}

  	//}}
  	//{{ fieldName

  	/**
  	 * Build form field name
  	 *
  	 * @param array $config
  	 * @return IPEbo_Html
  	 */
  	public function fieldName( $config )
  	{
  		$default = array(
  			'inputAtts'	=> array(
  				'class' => 'ipebo-input ipebo-input-full ipebo-input-v-lg'
  			)
  		);
  		$config = array_merge( $default, $config );

  		return $this->fieldText( $config );
  	}

  	//}}
  	//{{ renderFieldName

  	/**
  	 * Render form field name
  	 *
  	 * @param array $config
  	 * @return void
  	 */
  	public function renderFieldName( $config )
  	{
  		$html = $this->fieldName( $config );
  		echo $html->create();
  	}

    //}}
    //{{ fieldTextarea

    /**
     * Build form field with textarea input
     *
     * @param array $config
     * @return IPEbo_Html
     */
    public function fieldTextarea( $config )
    {
      //Set config
      if ( is_array( $config )) {
        $config = new IPEbo_Config( $config );
      }

      $name    = $config->getSetting( 'name', 'name' );
      $label   = $config->getSetting( 'label', 'Name' );
      $info    = $config->getSetting( 'info', '' );
      $value   = $config->getSetting( 'inputValue', '' );
      $layout  = $config->getSetting( 'layout', 'horizontal' );
      $helpAbs = $config->getSetting( 'helpAbs', true );
      $fopts   = (array)$config->getSetting( 'fieldOptions', '' );

      $inputAtts = array();
      if ( $iatts = $config->getSetting( 'inputAtts' ) ) {
        $inputAtts = $iatts;
      }
      //Create input text
      $input = $this->textarea( $name, $value, $inputAtts );

      return $this->field($label, $name, $input, $info, $layout, $helpAbs, $fopts );
    }

    //}}
    //{{ renderFieldTextarea

    /**
     * Render form field
     *
     * @param array $config
     * @return void
     */
    public function renderFieldTextarea( $config )
    {
      $html = $this->fieldTextarea( $config );
      echo $html->create();
    }

    //}}
    //{{ fieldSelect

    /**
     * Build form field with select input
     *
     * @param array $config
     * @return IPEbo_Html
     */
    public function fieldSelect( $config )
    {
      //Set config
      if ( is_array( $config )) {
        $config = new IPEbo_Config( $config );
      }

      $name     = $config->getSetting( 'name', 'name' );
      $label    = $config->getSetting( 'label', 'Name' );
      $info     = $config->getSetting( 'info', '' );
      $value    = $config->getSetting( 'inputValue', '' );
      $options  = (array)$config->getSetting( 'inputOptions', '' );
      $fopts    = (array)$config->getSetting( 'fieldOptions', '' );

      //Create input select
      $input = $this->select( $name, $options, $value );

      return $this->field( $label, $name, $input, $info, 'horizontal', false, $fopts );
    }

    //}}
    //{{ renderFieldSelect

    /**
     * Render form field with select input
     *
     * @param array $config
     * @return void
     */
    public function renderFieldSelect( $config )
    {
      $html = $this->fieldSelect( $config );
      echo $html->create();
    }

    //}}
    //{{ fieldOnOff

    /**
     * Build form field with on off buttons
     *
     * @param array $config
     * @return IPEbo_Html
     */
    public function fieldOnOff( $config )
    {
      //Set config
      if ( is_array( $config )) {
        $config = new IPEbo_Config( $config );
      }

      $name     = $config->getSetting( 'name', 'name' );
      $label    = $config->getSetting( 'label', 'Name' );
      $info     = $config->getSetting( 'info', '' );
      $value    = $config->getSetting( 'inputValue', '' );
      $onLabel  = $config->getSetting( 'inputOnLabel', 'On' );
      $offLabel = $config->getSetting( 'inputOffLabel', 'Off' );
      $fopts    = (array)$config->getSetting( 'fieldOptions', '' );

      //Create input on off
      $input = $this->onOff( $name, $value, $onLabel, $offLabel );

      return $this->field( $label, $name, $input, $info, 'horizontal', false, $fopts );
    }

    //}}
    //{{ renderFieldOnOff

    /**
     * Render form field
     *
     * @param array $config
     * @return void
     */
    public function renderFieldOnOff( $config )
    {
      $html = $this->fieldOnOff( $config );
      echo $html->create();
    }

    //}}
    //{{ fieldColor

    /**
     * Build form field with color input
     *
     * @param array $config
     * @return IPEbo_Html
     */
    public function fieldColor( $config )
    {
      $default = array(
        'inputAtts' => array(
          'class'   => 'ipebo-color-picker',
        ),
        'helpAbs' => false
      );
      $config = array_merge( $default, $config );

      return $this->fieldText( $config );
    }

    //}}
    //{{ renderFieldColor

    /**
     * Render form field with color input
     *
     * @param array $config
     * @return void
     */
    public function renderFieldColor( $config )
    {
      $html = $this->fieldColor( $config );
      echo $html->create();
    }

    //}}
    //{{ fieldSize

    /**
     * Build form field with input size
     *
     * @param array $config
     * @return IPEbo_Html
     */
    public function fieldSize( $config )
    {
      //Set config
      if ( is_array( $config )) {
        $config = new IPEbo_Config( $config );
      }

      $name    = $config->getSetting( 'name', 'name' );
      $label   = $config->getSetting( 'label', 'Name' );
      $info    = $config->getSetting( 'info', '' );
      $value   = $config->getSetting( 'inputValue', '' );
      $unit    = $config->getSetting( 'inputUnit', 'px' );
      $fopts   = (array)$config->getSetting( 'fieldOptions', '' );

      //Create input text
      $input = $this->inputSize( $name, $value, $unit );

      return $this->field( $label, $name, $input, $info, 'horizontal', false, $fopts );
    }

    //}}
    //{{ renderFieldSize

    /**
     * Render form field with input size
     *
     * @param array $config
     * @return void
     */
    public function renderFieldSize( $config )
    {
      $html = $this->fieldSize( $config );
      echo $html->create();
    }

    //}}
    //{{ fieldImage

    /**
     * Build form field with input image
     *
     * @param array $config
     * @return IPEbo_Html
     */
    public function fieldImage( $config )
    {
      //Set config
      if ( is_array( $config )) {
        $config = new IPEbo_Config( $config );
      }

      $name    = $config->getSetting( 'name', 'name' );
      $label   = $config->getSetting( 'label', 'Name' );
      $info    = $config->getSetting( 'info', '' );
      $layout  = $config->getSetting( 'layout', 'horizontal' );
      $fopts   = (array)$config->getSetting( 'fieldOptions', '' );

      //Create input image
      $input = $this->image( $name );

      return $this->field( $label, $name, $input, $info, $layout, true, $fopts );
    }

    //}}
    //{{ renderFieldImage

    /**
     * Render form field with input image
     *
     * @param array $config
     * @return void
     */
    public function renderFieldImage( $config )
    {
      $html = $this->fieldImage( $config );
      echo $html->create();
    }

    //}}
    //{{ fieldBackground

    /**
     * Build form field with input background
     *
     * @param array $config
     * @return string
     */
    public function fieldBackground( $config )
    {
      //Set config
      if ( is_array( $config )) {
        $config = new IPEbo_Config( $config );
      }
      $name    = $config->getSetting( 'name', 'name' );
      $info    = $config->getSetting( 'info', 'object' );
      $bgimage = $config->getSetting( 'bgimage', true, true );
      $color   = $config->getSetting( 'color_default', '#ffffff' );
      $repeat  = $config->getSetting( 'repeat_default', 'no-repeat' );
      $opacity = $config->getSetting( 'repeat_opacity', 20 );

      $fields = '';
      //BG Color
      $config = array(
        'name'       => $name . '_background_color',
        'label'      => 'Background Color',
        'inputValue' => $color,
        'info'       => 'Set the background color of the ' . $info . '.'
      );
      $color   = $this->fieldColor( $config );
      $fields .= $color->create();

      if ( $bgimage )
      {
        //Image
        $config = array(
          'name'  => $name . '_background_image',
          'label' => 'Background Image',
          'info'  => 'Set the background image of the ' . $info . '.'
        );
        $image   = $this->fieldImage( $config );
        $fields .= $image->create();

        //Repeat
        $options = array(
          'no-repeat'   => __( 'No Repeat', IPEBO_SLUG ),
          'repeat-x'    => __( 'Repeat Horizontally', IPEBO_SLUG ),
          'repeat-y'    => __( 'Repeat Vertically', IPEBO_SLUG ),
          'repeat'      => __( 'Repeat All', IPEBO_SLUG ),
        );
        $config  = array(
          'name'  => $name . '_background_repeat',
          'label' => 'Background Repeat',
          'info'  => 'Set the repeat method of the background image.',
          'inputOptions' => $options,
          'inputValue'   => $repeat
        );
        $repeat  = $this->fieldSelect( $config );
        $fields .= $repeat->create();

        //Opacity
        $options = array(
          0    => __( 'None', IPEBO_SLUG ),
          10   => __( '10%', IPEBO_SLUG ),
          20   => __( '20%', IPEBO_SLUG ),
          30   => __( '30%', IPEBO_SLUG ),
          40   => __( '40%', IPEBO_SLUG ),
          50   => __( '50%', IPEBO_SLUG ),
          60   => __( '60%', IPEBO_SLUG ),
          70   => __( '70%', IPEBO_SLUG ),
          80   => __( '80%', IPEBO_SLUG ),
          90   => __( '90%', IPEBO_SLUG )
        );
        $config  = array(
          'name'  => $name . '_background_opacity',
          'label' => 'Background Opacity',
          'info'  => 'Make transparency for the background image.',
          'inputOptions' => $options,
          'inputValue'   => $opacity
        );
        $repeat  = $this->fieldSelect( $config );
        $fields .= $repeat->create();
      }

      return $fields;
    }

    //}}
    //{{ renderFieldBackground

    /**
     * Render form field with input background
     *
     * @param array $config
     * @return void
     */
    public function renderFieldBackground( $config )
    {
      $html = $this->fieldBackground( $config );
      echo $html;
    }

    //}}
    //{{ fieldBorder

    /**
     * Build form field with input border
     *
     * @param array $config
     * @return string
     */
    public function fieldBorder( $config )
    {
      //Set config
      if ( is_array( $config )) {
        $config = new IPEbo_Config( $config );
      }
      $name     = $config->getSetting( 'name', 'object' );
      $info     = $config->getSetting( 'info', '' );
      $label    = $config->getSetting( 'label', 'Border' );
      $style    = $config->getSetting( 'border_style', false );
      $color    = $config->getSetting( 'color_default', '#000000' );
      $sdefault = $config->getSetting( 'style_default', 'solid' );
      $fopts    = (array)$config->getSetting( 'fieldOptions', '' );

      $inputs = '';
      //Size
      $input   = $this->inputSize( $name . '_border_size', 0 );
      $inputs .= $input->create();

      if ( $style )
      {
        //Style
        $options = array(
          'solid'    => __( 'Solid', IPEBO_SLUG ),
          'dashed'   => __( 'Dashed', IPEBO_SLUG ),
          'dotted'   => __( 'Dotted', IPEBO_SLUG ),
          'double'   => __( 'Double', IPEBO_SLUG )
        );
        $input   = $this->select( $name . '_border_style', $options, $sdefault );
        $inputs .= $input->create();
      }

      //color
      $input = $this->input( $name . '_border_color', 'text', $color, array(
        'class'   => 'ipebo-color-picker'
      ));
      $inputs .= $input->create();

      return $this->field( $label, $name . '_border_size', $inputs, $info, 'horizontal', false, $fopts );
    }

    //}}
    //{{ renderFieldBorder

    /**
     * Render form field with input border
     *
     * @param array $config
     * @return void
     */
    public function renderFieldBorder( $config )
    {
      $html = $this->fieldBorder( $config );
      echo $html->create();
    }

    //}}
    //{{ fieldFontFamily

    /**
     * Build form field with input font family
     *
     * @param array $config
     * @return string
     */
    public function fieldFontFamily( $config )
    {
      //Set config
      if ( is_array( $config )) {
        $config = new IPEbo_Config( $config );
      }

      $name    = $config->getSetting( 'name', 'name' );
      $label   = $config->getSetting( 'label', 'Name' );
      $info    = $config->getSetting( 'info', '' );
      $value   = $config->getSetting( 'inputValue', 'roboto' );
      $fopts   = (array)$config->getSetting( 'fieldOptions', '' );

      //Create input fancy option
      $option = $this->fancyOption( $name, $value );
      $input  = new IPEbo_Html( 'span', $option, array(
        'class'   => 'ipebo-font-family'
      ));

      return $this->field( $label, $name, $input, $info, 'horizontal', false, $fopts );
    }

    //}}
    //{{ renderFieldFontFamily

    /**
     * Render form field with input font family
     *
     * @param array $config
     * @return void
     */
    public function renderFieldFontFamily( $config )
    {
      $html = $this->fieldFontFamily( $config );
      echo $html->create();
    }

    //}}
    //{{ template

    /**
     * Build form template options
     *
     * @param string $name
     * @param array  $options
     * @param string $value
     *
     * @return IPEbo_Html
     */
    public function template( $name, $options, $value = '' )
    {
      if ( isset( $this->setting ) && $this->setting && $this->setting instanceof IPEbo_Config )
      {
        $value = $this->setting->getSetting( $name, $value );
      }

      $control  = '<ul class="ipebo-option-template-items _items ipebo-clear">';
      if ( ! empty( $options ))
      {
        foreach( $options as $option )
        {
          $class = 'ipebo-option-template-item _item';
          if ( $option['value'] == $value ) {
            $class .= ' ipebo-option-template-selected _selected';
          }
          $control .= '<li id="' . $option['name'] . '" class="' . $class . '" data-value="' . $option['value'] . '">';
          $control .= '<div class="ipebo-option-template-preview">';
          if ( ! empty( $option['image'] )) {
            $control .= '<img src="' . $option['image'] . '" alt="">';
          }
          else if ( ! empty( $option['icon'] )) {
            $control .= '<i class="' . $option['icon'] . '"></i>';
          }
          else if ( ! empty( $option['content'] )) {
            $control .= $option['content'];
          }
          $control .= '</div>';

          if ( ! empty( $option['label'] )) {
            $control .= '<div class="ipebo-option-template-label">';
            $control .= $option['label'];
            $control .= '</div>';
          }
          $control .= '</li>';

        }
      }
      $control .= '</ul>';
      $control .= '<input type="hidden" class="ipebo-option-template-value _value-selected" name="' . $name . '" value="' . $value . '"/>';

      $input = new IPEbo_Html( 'div', $control, array(
        'class' => 'ipebo-option-template ipebo-image-option'
      ));
      return $input;
    }

    //}}
    //{{ renderTemplate

    /**
     * Render template options
     *
     * @param string $name
     * @param array  $options
     * @param string $value
     * @param IPEbo_Config $config
     *
     * @return void
     */
    public function renderTemplate( $name, $options, $value = '' )
    {
      $html = $this->template( $name, $options, $value );
      echo $html->create();
    }

    //}}
    //{{ fieldTemplate

    /**
     * Build form field with template options
     *
     * @param array $config
     * @return IPEbo_Html
     */
    public function fieldTemplate( $config )
    {
      //Set config
      if ( is_array( $config )) {
        $config = new IPEbo_Config( $config );
      }

      $name     = $config->getSetting( 'name', 'name' );
      $label    = $config->getSetting( 'label', 'Name' );
      $info     = $config->getSetting( 'info', '' );
      $value    = $config->getSetting( 'inputValue', '' );
      $options  = $config->getSetting( 'options', array() );

      //Create input template
      $input = $this->template( $name, $options, $value );

      return $this->field( $label, $name, $input, $info, 'horizontal', true );
    }

    //}}
    //{{ renderFieldTemplate

    /**
     * Render form field
     *
     * @param array $config
     * @return void
     */
    public function renderFieldTemplate( $config )
    {
      $html = $this->fieldTemplate( $config );
      echo $html->create();
    }

  	//}}


  }
}
