#!/bin/ksh 
#!/bin/ksh -xv ### Testing
#
# $Id: sdcm.sh,v 1.1 2023/03/27 16:07:05 root Exp $
#
# The following code is Confidential and is covered by the installation license
# (c) Copyright Fortra, LLC. and its group of companies.
#
#&& Sar Data Consolidation Manager (sdcm.sh)
#
#& Controls the consolidation of sar data into the mpgdata.hostname file
#& Script is used normally used once each day via a crontab line
#
#&@ sar.sh is also required.
#&@ sar.awk is also required.
#
#&% Ran daily via crontab on HPUX and SunOS.
#
## Jack Miller 4/18/2009
#       Updated shell to ksh, not sh compliant.

# hpux values untested

# default values
debug=""
invocdir=`dirname $0`  # invocation dir (default for temp/data files, utilities)
sysname=`uname -n | awk -F. '{print $1}'`     # used in mpg sar datafile names
datafile="${invocdir}/mpgdata.$sysname" # default datafile
pndir="${invocdir}"                  # default directory for pn files
tempfile="${invocdir}/mpg.temp"         # default tempfile
# FS=";"                 # default Field separator
# LSEP='\n';             # default to multi-line output
# withlabels="T"         # default add labels (in sar.sh)
# for truncate
daystokeep=365         # default number of days of data to keep
maxfilelen=10000000    # default maximmum datafile size in bytes

### default values:

# os specific default settings
# defsardir = default sar data location - override with -d option
defsardir_AIX="/var/adm/sa" ;
defsardir_HPUX="/var/adm/sa" ;
defsardir_solaris="/var/adm/sa" ;
defsardir_linux_suse="/var/log/sa" ;
defsardir_linux_redhat="/var/log/sa" ;
defsardir_linux_debian="/var/log/sysstat" ;
# defsarfilepattern = default sar file pattern
defsarfilepattern_AIX='sa$checkdd' ;
defsarfilepattern_HPUX='sa$checkdd' ;
defsarfilepattern_solaris='sa$checkdd' ;
defsarfilepattern_linux_suse='sa.${checkyyyy}_${checkmm}_${checkdd}' ;
defsarfilepattern_linux_redhat='sa$checkdd' ;
defsarfilepattern_linux_debian='sa$checkdd' ;
# hl = sar line to get date from
hl_AIX=2
hl_HPUX=2
hl_solaris=2
hl_linux_suse=1
hl_linux_redhat=1
hl_linux_debian=1
# collectfilepattern = sar datafile regex
defcollectfilepattern_AIX="sa[0-9][0-9]" 
defcollectfilepattern_HPUX="sa[0-9][0-9]" 
defcollectfilepattern_solaris="sa[0-9][0-9]" 
defcollectfilepattern_linux_suse='sa.[12][019][0-9][0-9]_[01][0-9]_[0-3][0-9]' ;
defcollectfilepattern_linux_redhat="sa[0-9][0-9]" 
defcollectfilepattern_linux_debian="sa[0-9][0-9]" 


export LC_TIME="C"   # required to standardize date format, consistent across platforms

function usage_ori {
echo "USAGE:
sdcm.sh [options] command [args]
   where 'command' is one of: daily truncate collect
args and opts specific to each command are:
  daily  [today|yesterday|sarpath]	(default yesterday)
	-d sardir	- specify sar data directory (def: OS specific)
	-F sarfilepattern	- specify sar data file pattern (def: OS specific)
	-l		- with labels (see sar.sh)
  truncate
	-s maxfilesize	- specify maximum datafile size (def: $maxfilelen)
	-n daystokeep	- specify maximum days of data to keep (def: $daystokeep)
  collect [all]		("all" includes today)
	-d sardir	- specify sar data directory (def: OS specific)
	-P collectfilepattern	- sar data file pattern (def: sa[0-9][0-9])
	-l		- with labels (see sar.sh)
general options are:
	-D		- debug
	-c configfile	- get settings from \"configfile\"
	-o datafile	- full path to output file (def: $datafile)
	-t tempfile	- full path to tempfile (def: $tempfile)
sar specific options are:
	-S FS		- field separator to use (def: '$FS')
	-m LSEP		- line separator for multi-line output (def: none)
	-C sarcmd	- pass custom sar command to sar.sh

examples:
sdcm.sh daily yesterday
 - process and append yesterdays sar data (using default files)
sdcm.sh -d /usr/local/sa -o newdatafile daily yesterday
 - process and append yesterdays sar data (using custom files)
sdcm.sh -n 90 truncate
 - truncate default datafile to 90 days (or default max size if less)
"
}

function usage {
# This one can be changed.  Saved original copy.
echo "USAGE:
sdcm.sh [options] command [args]
   where 'command' is one of: daily truncate collect
args and opts specific to each command are:
  daily  [today|yesterday|sarpath]	(default yesterday)
	-d sardir	- specify sar data directory (def: OS specific)
	-F sarfilepattern	- specify sar data file pattern (def: OS specific)
	-l		- with labels (see sar.sh)
  truncate
	-s maxfilesize	- specify maximum datafile size (def: $maxfilelen)
	-n daystokeep	- specify maximum days of data to keep (def: $daystokeep)
  collect [all]		("all" includes today)
	-d sardir	- specify sar data directory (def: OS specific)
	-P collectfilepattern	- sar data file pattern (def: sa[0-9][0-9])
	-l		- with labels (see sar.sh)
general options are:
	-D		- debug
	-c configfile	- get settings from \"configfile\"
	-o datafile	- full path to output file (def: $datafile)
	-t tempfile	- full path to tempfile (def: $tempfile)
sar specific options are:
	-S FS		- field separator to use (def: '$FS')
	-m LSEP		- line separator for multi-line output (def: none)
	-C sarcmd	- pass custom sar command to sar.sh

examples:
sdcm.sh daily yesterday
 - process and append yesterdays sar data (using default files)
sdcm.sh -d /usr/local/sa -o newdatafile daily yesterday
 - process and append yesterdays sar data (using custom files)
sdcm.sh -n 90 truncate
 - truncate default datafile to 90 days (or default max size if less)
"
}

##### utility functions

function setdaysinmonths {      ## arg: year
 typeset yts=$1
#ksh# set -A lastdayofmonth 99 31 28 31 30 31 30 31 31 30 31 30 31
#bash# lastdayofmonth=([0]=99 [1]=31 [2]=28 [3]=31 [4]=30 [5]=31 [6]=30 [7]=31 [8]=31 [9]=30 [10]=31 [11]=30 [12]=31)
# both:
 lastdayofmonth[0]=99
 lastdayofmonth[1]=31
 lastdayofmonth[2]=28
 lastdayofmonth[3]=31
 lastdayofmonth[4]=30
 lastdayofmonth[5]=31
 lastdayofmonth[6]=30
 lastdayofmonth[7]=31
 lastdayofmonth[8]=31
 lastdayofmonth[9]=30
 lastdayofmonth[10]=31
 lastdayofmonth[11]=30
 lastdayofmonth[12]=31
 let lyc=yts%4
 if [ "$lyc" = 0 ] ; then
	if [ ! -z "$debug" ] ; then echo "$cy is leap year" ; fi
   lastdayofmonth[4]=31
 fi
	# unimplemented option "extra" debugging
	if [ "X$debug" = "X+" ] ; then
	for i in 1 2 3 4 5 6 7 8 9 10 11 12
	do
	 echo "ldom $i = ${lastdayofmonth[$i]}"
	done
	fi
}


function get_n_days_ago {	### returns date of N days ago in $ndaysago
 typeset rdb=$1
 typeset cy=`date +%Y`
 typeset cmm=`date +%m`
 typeset cm=${cmm##0}
 typeset cdd=`date +%d`
 typeset cd=${cdd##0}
 typeset pdoy=`date +%-j`
 typeset doy=${pdoy##0}

# short deltas (less than days this year)
	if [ ! -z "$debug" ] ; then echo "cy=$cy, doy=$doy, cm=$cm, cd=$cd, rdb=$rdb" ; fi
 if [ "$rdb" -lt "$doy" ] ; then
  setdaysinmonths $cy
  if [ "$rdb" -lt "$cd" ] ; then	# shorter delta (less than days this month)
    let cd=cd-rdb
    dom=${lastdayofmonth[$cm]}
    let rdb=dom-cd	  # set so that "final" calc below gets correct day
	if [ ! -z "$debug" ] ; then echo "cy=$cy, cm=$cm, cd=$cd, rdb=$rdb (rdb<cd)" ; fi
  else
    let rdb=rdb-cd
    let cm=cm-1
    if [ "$cm" -eq 0 ] ; then let cy=cy-1; let cm=12; fi
    cd=${lastdayofmonth[$cm]}
	if [ ! -z "$debug" ] ; then echo "cy=$cy, cm=$cm, cd=$cd, rdb=$rdb (rdb<doy)" ; fi
  fi
 fi

 # subtract #days per year until less than 1 year left
 while [ $rdb -gt $doy ] ;
 do
   let rdb=rdb-doy
   let cy=cy-1
   doy=365
   let lyc=cy%4 ; if [ "$lyc" = 0 ] ; then doy=366 ; fi
   cm=12
	if [ ! -z "$debug" ] ; then echo "cy=$cy, doy=$doy, cm=$cm, rdb=$rdb" ; fi
 done

	if [ ! -z "$debug" ] ; then echo "less than 1 year left ($cy $cm $cd rdb:$rdb)" ; fi
# subtract #days per month until less than 1 month left
 setdaysinmonths $cy
 dom=${lastdayofmonth[$cm]}
	if [ ! -z "$debug" ] ; then echo "rdb=$rdb, cm=$cm, lastday=$dom" ; fi
 while [[ $rdb -gt $dom ]]
 do
   let rdb=rdb-dom
   let cm=cm-1
   dom=${lastdayofmonth[$cm]}
	if [ ! -z "$debug" ] ; then echo "cy=$cy, cm=$cm, dom=$dom, rdb=$rdb" ; fi
 done

 dom=${lastdayofmonth[$cm]}
 let cd=dom-rdb
 if [ "$cm" -lt 10 ] && [ "${#cm}" -lt 2 ] ; then cm="0$cm"; fi
 if [ "$cd" -lt 10 ] && [ "${#cd}" -lt 2 ] ; then cd="0$cd"; fi
	if [ ! -z "$debug" ] ; then echo "final: $cy $cm $cd" ; fi
 ndaysago=$cy$cm$cd
}


function get_yesterday {  ### sets $yesterday $checkyyyy $checkmm $checkdd $checkyy
 typeset yyyy=`date +%Y`
 typeset yy=`date +%y`	# temporary test
 typeset yyyyy=$yyyy
 typeset pmm=`date +%m`
 typeset mm=${pmm##0}
 typeset ymm=$mm
 typeset pdd=`date +%d` ## was +%-d but fails on solaris, not needed?
 typeset dd=${pdd##0}
 typeset ydd=0
	if [ ! -z "$debug" ] ; then echo "dd=$dd, ydd=$ydd" ; fi
 let ydd=dd-1
	if [ ! -z "$debug" ] ; then echo "today: $yyyy$mm$dd" ; fi
 if [ "$dd" -eq 1 ] ; then
   let ymm=$mm-1
   if [ "$ymm" -eq 0 ] ; then let yyyyy=$yyyy-1 ; ymm=12 ; fi
   setdaysinmonths $yyyyy
   ydd=${lastdayofmonth[$ymm]}
 fi
 if [ "$ymm" -lt 10 ] && [ "${#ymm}" -lt 2 ] ; then ymm="0$ymm"; fi
 if [ "$ydd" -lt 10 ] && [ "${#ydd}" -lt 2 ] ; then ydd="0$ydd"; fi
 # results:
	if [ ! -z "$debug" ] ; then echo "yesterday: $yyyyy$ymm$ydd" ; fi
 yesterday=$yyyyy$ymm$ydd
 checkyyyy=$yyyyy
 checkmm=$ymm
 checkdd=$ydd
 checkyy=${checkyyyy##??}
	if [ ! -z "$debug" ] ; then echo "checkyyyy: $checkyyyy  checkyy:$checkyy" ; fi
}


function set_os_specifics {	### set OS specific defaults
# defsardir = default sar data location - override with -l option
# defsarfilepattern = default sar file pattern
# hl = sar line to get date from
# defcollectfilepattern = sar datafile regex
 ostype=`uname -s`
 if [ "$ostype" = "AIX" ] ; then
   defsardir=$defsardir_AIX ;
   hl=$hl_AIX ;
   defsarfilepattern=$defsarfilepattern_AIX ;
   defcollectfilepattern=$defcollectfilepattern_AIX ;
 elif [ "$ostype" = "HP-UX" ] ; then
   defsardir=$defsardir_HPUX ;
   hl=$hl_HPUX ;
   defsarfilepattern=$defsarfilepattern_HPUX ;
   defcollectfilepattern=$defcollectfilepattern_HPUX ;
 elif [ "$ostype" = "SunOS" ] ; then
   defsardir=$defsardir_solaris ;
   hl=$hl_solaris ;
   defsarfilepattern=$defsarfilepattern_solaris ;
   defcollectfilepattern=$defcollectfilepattern_solaris ;
 elif [ "$ostype" = "Linux" ] ; then
   os=`cat /etc/issue`
   case $os in
	*Ubuntu*)	
		defsardir=$defsardir_linux_debian ;
		hl=$hl_linux_debian ;
		defsarfilepattern=$defsarfilepattern_linux_debian ;
		defcollectfilepattern=$defcollectfilepattern_linux_debian ;;
	*Debian*)
		defsardir=$defsardir_linux_debian ;
		hl=$hl_linux_debian ;
		defsarfilepattern=$defsarfilepattern_linux_debian ;
		defcollectfilepattern=$defcollectfilepattern_linux_debian ;;
	*Red\ Hat*)
		defsardir=$defsardir_linux_redhat ;
		hl=$hl_linux_redhat ;
		defsarfilepattern=$defsarfilepattern_linux_redhat ;
		defcollectfilepattern=$defcollectfilepattern_linux_redhat ;;
	*SUSE*)
		defsardir=$defsardir_linux_suse ;
		hl=$hl_linux_suse ;
		defsarfilepattern=$defsarfilepattern_linux_suse ;
		defcollectfilepattern=$defcollectfilepattern_linux_suse ;;
	*)
		echo "unknown linux OS type - using generic linux defaults";
		defsardir=$defsardir_linux_redhat ;
		hl=$hl_linux_redhat ;
		defsarfilepattern=$defsarfilepattern_linux_redhat ;
		defcollectfilepattern=$defcollectfilepattern_linux_redhat ;;
   esac
#	if [ -s "/etc/SuSE-release" ] ; then
#		defsardir=$defsardir_linux_suse ;
#		hl=$hl_linux_suse ;
#		defsarfilepattern=$defsarfilepattern_linux_suse ;
#		defcollectfilepattern=$defcollectfilepattern_linux_suse ;
#	elif [ -s "/etc/redhat-release" ] ; then
#		defsardir=$defsardir_linux_redhat ;
#		hl=$hl_linux_redhat ;
#		defsarfilepattern=$defsarfilepattern_linux_redhat ;
#		defcollectfilepattern=$defcollectfilepattern_linux_redhat ;
#	else
#		echo "unknown linux OS type - using generic linux defaults";
#		defsardir=$defsardir_linux_redhat ;
#		hl=$hl_linux_redhat ;
#		defsarfilepattern=$defsarfilepattern_linux_redhat ;
#		defcollectfilepattern=$defcollectfilepattern_linux_redhat ;
#	fi
 else
	echo "unknown OS type - using generic AIX defaults";
	defsardir=$defsardir_AIX ;
	hl=$hl_AIX ;
	defsarfilepattern=$defsarfilepattern_AIX ;
	defcollectfilepattern=$defcollectfilepattern_AIX ;
 fi
 if [ "X$collectfilepattern" = "X" ] ; then collectfilepattern=$defcollectfilepattern ; fi
 if [ "X$sardir" = "X" ] ; then sardir=$defsardir ; fi
	if [ ! -z "$debug" ] ; then 
	echo "os specific defaults:" ;
	echo "defsardir=$defsardir" ;
	echo "hl=$hl" ;
	echo "defsarfilepattern=$defsarfilepattern" ;
	echo "defcollectfilepattern=$defcollectfilepattern" ;
	fi
}

function get_sar_data_date {
### get the data date from a sar data file ($1), return it in $sardatadate
 typeset f=$1
 typeset fdline=`sar -u -f $f | head -$hl`
	if [ ! -z "$debug" ] ; then echo "fdline=$fdline" ; fi
 typeset lastword=${fdline##?*[	 ]}
	if [ ! -z "$debug" ] ; then echo "lastword=$lastword" ; fi
 typeset mm=${lastword%%/*}
	if [ ! -z "$debug" ] ; then echo "mm=$mm" ; fi
 typeset lwmm=${lastword##??/}
	if [ ! -z "$debug" ] ; then echo "lwmm=$lwmm" ; fi
 typeset dd=${lwmm%%/*}
	if [ ! -z "$debug" ] ; then echo "dd=$dd" ; fi
 typeset yy=${lwmm##??/}
	if [ ! -z "$debug" ] ; then echo "yy=$yy" ; fi
 lenyy=${#yy}
 if [ "$lenyy" = "2" ] ; then yyyy="20$yy" ; else yyyy=$yy; fi
	if [ ! -z "$debug" ] ; then echo "yyyy=$yyyy" ; fi
 # result:
 sardatadate=$yyyy$mm$dd
	if [ ! -z "$debug" ] ; then
	echo "mm=$mm, lwmm=$lwmm, dd=$dd, yyyy=$yyyy, sardatadate=$sardatadate" ;
	fi
}

function append_to_datafile {	
   ### process data from a sar file and append to $datafile
   ##### could output systemname on a separate line in here
   # echo "sardatadate ($sardatadate)"
   # echo "datafile ($datafile)"
   # exit
   typeset f=$1
   fileopt="-f $f";
   
   get_sar_data_date $f
   if [ ! -z "$debug" ] ; then 
      echo "sardatadate ($sardatadate) - today ($today) - datafile ($datafile)"
   fi 
   if [ "$sardatadate" = "$today" ] ; then
      if [ ! -z "$debug" ] ; then 
         echo "Cannot collect today's data until the next day."
         return
      fi
   fi

   if [ -s $datafile ]; then
      ## check for existence of date being processed in the datafile  
      grep $sardatadate $datafile > /dev/null
      jgchk=$?
      if [ $jgchk -eq 0 ]; then
         ## If date being processed exists in the datafile, error, & exit  
         if [ ! -z "$debug" ] ; then 
            echo "## Error ## $sardatadate is already in $datafile, skipping."
         fi
         return
      fi
   fi
   # sar.sh will not allow -C and -f, so put -f inside of sarcmd :
   if [ ! -z "$sarcmd" ] ; then fileopt="-C '$sarcmd -f $f'"; fi
   # echo "./sar.sh ($debug) ($SARSHOPTS) ($fileopt) > $tempfile"
   eval "./sar.sh $debug $SARSHOPTS $fileopt > $tempfile"
   # rc=$?  echo "sar.sh rc = $rc" ###### no rc from sar.sh
   if [ "$rc" = 0 ] ; then
      if [ ! -z "$debug" ] ; then 
         echo "sar.sh $SARSHOPTS $fileopt > $datafile" 
      fi
      cat $tempfile >> $datafile
      if [ -z "$debug" ] ; then rm $tempfile ; fi
   fi
}


##### main operations

function sdcm_truncate {
	if [ ! -z "$debug" ] ; then echo "datafile=$datafile, daystokeep=$daystokeep, tempfile=$tempfile, LSEP=$LSEP, FS=$FS" ; fi
  get_n_days_ago $daystokeep
  cutdate=$ndaysago
	if [ ! -z "$debug" ] ; then echo "cutdate=$cutdate" ; fi
 ostype=`uname -s`
 if [ "$ostype" = "AIX" ] ; then
    datafilesize=`istat $datafile | awk '/Length/ {print $5}'`
    debug=""	# archaic awk on AIX does not understand /dev/stderr
 elif [ "$ostype" = "Linux" ] ; then
    datafilesize=`stat -t $datafile | awk '{print $2}'`
 fi
	if [ ! -z "$debug" ] ; then echo "current size of $datafile is $datafilesize" ; fi

cat $datafile | awk -v debug=$debug -v datafilesize=$datafilesize -v maxfilelen=$maxfilelen -v cutdate=$cutdate -v LSEP=$LSEP '

 BEGIN {
 gotheader="" ;
 gettingheader="" ;
 needfirstheader="T";
 cutoffdate = cutdate "000000"	# move this out to the shell stuff?
 cumfsize=0;
	if (debug == "T") {
	print "cutoffdate=" cutoffdate > "/dev/stderr"
	print "maxfilelen="maxfilelen ", current len=" datafilesize > "/dev/stderr"
	}
 }

# header line
 /^sysname=/ {
  headerline=$0 ;
  gettingheader="T" ;
  cumfsize += length($0); cumfsize += 1 ;
	if (debug == "T") {
	  print "H cum size: " cumfsize ", last date: " time > "/dev/stderr" ;
	}
  next;
 }

# continuation line
$1 == "" {	# first char is FS so $1 is blank
 if ( gettingheader == "T" ) {
   if ( LSEP != "" ) { headerline=headerline LSEP; }
   headerline=headerline $0 ;
   cumfsize += length($0); cumfsize += 1 ;
	if (debug == "T") {
	  print "Hc cum size: " cumfsize ", last date: " time > "/dev/stderr" ;
	}
   next;
 }
 cumfsize += length($0); cumfsize += 1 ;
### if ( curline != "") {
   if ( LSEP != "" ) { curline=curline LSEP; }
   curline=curline $0;
### }
 curcutsize = datafilesize - cumfsize;
	if (debug == "T") {
	  print "C curcutsize: " curcutsize > "/dev/stderr" ;
	}
 next;
}

# "normal" data line or first line of multi-line data
 {
  if (gettingheader == "T") { gotheader = "T" ; gettingheader = ""; }
  if (( time > cutoffdate ) && ( curcutsize < maxfilelen )) {
    if ( needfirstheader == "T" ) {
      print headerline ;
      gotheader=""; needfirstheader=""; headerline="";
	if (debug == "T") {
	  print "K1 curcutsize: " curcutsize ", date: " time > "/dev/stderr" ;
	}
    }
    print curline ;
    curline=""; ## extraneous?
	if (debug == "T") {
	  print "P curcutsize: " curcutsize ", date: " time > "/dev/stderr" ;
	}
    if ( gotheader == "T" ) {
      print headerline ; gotheader=""; headerline="";
	if (debug == "T") {
	  print "K curcutsize: " curcutsize ", date: " time > "/dev/stderr" ;
	}
    }
  }
 cumfsize += length($0); cumfsize += 1 ;
 curcutsize = datafilesize - cumfsize;
 time=$1;
	curline=$0;
	if (debug == "T") {
	  print "cum size: " cumfsize ", curcutsize=" curcutsize > "/dev/stderr" ; 
	  print "time=" time ", cutoffdate= " cutoffdate > "/dev/stderr"
	}
 }

END {
  if (( curline != "" ) &&  (( time > cutoffdate ) && ( curcutsize < maxfilelen ))) {
	if (debug == "T") {
	  print "Pe curcutsize: " curcutsize ", date: " time > "/dev/stderr" ;
	}
    print curline;
  }
}

' FS="$FS" > $tempfile
cp $datafile ${datafile}.org
mv $tempfile $datafile
}


function sdcm_collect_existing {
 if [ "X$1" != "X" ] ; then
   typeset arg=$1
   if [ "$arg" = "all" ] ; then
     include_today="F"
   fi
 fi
	if [ ! -z "$debug" ] ; then echo "datafile=$datafile, sardir=$sardir" ; fi
	if [ ! -z "$debug" ] ; then echo "collectfilepattern=$collectfilepattern" ; fi

 set_os_specifics
 # lastfile=`ls -1tr $sardir/$collectfilepattern| tail -1` # Never gather today's data, it is not complete
 # filelist=`ls -1tr $sardir/$collectfilepattern| grep -v $lastfile | tail -1023` # max size of ksh array is 1023
 filelist=`ls -1tr $sardir/$collectfilepattern| tail -1023` # max size of ksh array is 1023
 ##### make max number of files an option?
	if [ ! -z "$debug" ] ; then echo "filelist = $filelist" ; fi

 today=`date +%Y%m%d`
 i=0
 for f in $filelist
 do
	if [ ! -z "$debug" ] ; then echo " -----  $f  ----- " ; fi
  sar -u -f $f > /dev/null
  rc=$?
  if [ "$rc" -ne 0 ] ; then continue ; fi
  let i=i+1
  get_sar_data_date $f
  if [ "X$include_today" != "XT" ] ; then
    if [ "$sardatadate" = "$today" ] ; then
      let i=i-1
	if [ ! -z "$debug" ] ; then echo "skipping today ($today in $f)"; fi
      continue
    fi
  fi
  fn[$i]=$f
  fd[$i]=$sardatadate
	if [ ! -z "$debug" ] ; then echo "$i: ${fn[$i]} -- ${fd[$i]}" ; fi
 done

 ## sort filelist (fn) by date (fd) (to be sure...)
 i=1
 nfd=${#fd[*]}
 let nfd1=nfd-1
 while [ $i -le $nfd1 ] ;
 do
   let j=i+1
   while [ $j -le $nfd ] ;
   do
     if [ "${fd[$i]}" -gt "${fd[$j]}" ] ; then
	tfd=${fd[$i]}
	fd[$i]=${fd[$j]}
	fd[$j]=$tfd
	tfn=${fn[$i]}
	fn[$i]=${fn[$j]}
	fn[$j]=$tfn
     fi
     let j=j+1
   done
   let i=i+1
 done
	if [ ! -z "$debug" ] ; then  i=0
	  while [ $i -lt ${#fn[*]} ]; do let i=i+1; echo "$i: ${fn[$i]} -- ${fd[$i]}"; done
	fi

 ## run sar.sh on each and append the output to $datafile
 #? check for existence/contents of $datafile ???
 for f in ${fn[*]}
 do
    append_to_datafile $f
 done
}

function sdcm_get_daily {
# add a generic "date" parameter?
#       or just let something else determine the correct file?
# should check datafile and verify the data needs to be collected (date not already there)
#       some sort of database would help...
 if [ "X$1" != "X" ] ; then
   typeset arg=$1
   if [ "$arg" = "yesterday" ] ; then  date_to_get=$1 ; shift
   elif [ "$arg" = "today" ] ; then  date_to_get=$1 ; shift
   else
     if [ -r "$arg" ] ; then
	sarpath=$arg
     else
	echo "unable to access $arg - exiting";  exit
     fi
   fi
 fi
 if [ "X$sarpath" = "X" ] ; then
   if [ "X$date_to_get" = "X" ] ; then
     date_to_get="yesterday"       # default to yesterday
   fi
	if [ ! -z "$debug" ] ; then echo "date to get: $date_to_get" ; fi
   set_os_specifics
   if [ "$date_to_get" = "yesterday" ] ; then
     get_yesterday	# sets $yesterday $checkyyyy $checkmm $checkdd
     checkdate=$yesterday
   else
     checkyyyy=`date +%Y`
     checkmm=`date +%m`
     checkdd=`date +%d`
     checkdate=$checkyyyy$checkmm$checkdd
   fi
   if [ "X$sarfilepattern" = "X" ] ; then
     sarfilepattern=`eval "echo $defsarfilepattern"`
   else
     sarfilepattern=`eval "echo $sarfilepattern"`
   fi
   sarpath="$sardir/$sarfilepattern"
 fi
	if [ ! -z "$debug" ] ; then echo "sarpath=$sarpath, (sarfilepattern=$sarfilepattern)" ; fi
 # verify file exists and contains sar data
 sar -u -f $sarpath > /dev/null
 rc=$?
 if [ "$rc" -ne 0 ] ; then
   echo "unable to get sar data from $sarpath - exiting"
   exit
 else
   get_sar_data_date $sarpath
   if [ ! -z "$checkdate" ] ; then
     if [ "$checkdate" != "$sardatadate" ] ; then
       echo "$sardatadate data in file $sarpath is not from $checkdate - exiting"
       exit
     fi
   fi
 append_to_datafile $sarpath
 fi
}

#MAIN#
while getopts "Do:t:c:d:S:m:n:s:lC:F:P:" opt ; do
  case $opt in
	D)	debug="-D" ;;			#TCD
	o)	opt_datafile=$OPTARG ;;		#TCD
	t)	opt_tempfile=$OPTARG ;;		#TCD
	c)	opt_configfile=$OPTARG ;;	#TCD
	d)	opt_sardir=$OPTARG ;;		#CD
	S)	opt_FS=$OPTARG ;;		#TCD
	m)	opt_LSEP="$OPTARG" ;;		#TCD
	n)	opt_daystokeep=$OPTARG ;;	#T
	s)	opt_maxfilelen=$OPTARG ;;	#T	(size)
	l)	opt_withlabels="T" ;;		#CD
	C)	opt_sarcmd=$OPTARG ;;		#CD
	F)	opt_sarfilepattern=$OPTARG ;;	#D
	P)	opt_collectfilepattern=$OPTARG ;;	#C
	?)	usage;  exit ;;
  esac
done
shift `expr $OPTIND - 1`
##### should verify options - probably in individual command functions ?

if [ ! -z "$opt_configfile" ] ; then configfile=$opt_configfile; fi
##### quick+dirty configfile - unsafe!
if [ ! -z "$configfile" ] ; then
  . $configfile
fi

	if [ ! -z "$debug" ] ; then 
	echo "options after configfile processing:";
	echo "datafile:$datafile"
	echo "tempfile:$tempfile"
	echo "configfile:$configfile"
	echo "sardir:$sardir"
	echo "FS:$FS"
	echo "LSEP:$LSEP"
	echo "daystokeep:$daystokeep"
	echo "maxfilelen:$maxfilelen"
	echo "withlabels:$withlabels"
	echo "sarcmd:$sarcmd"
	echo "sarfilepattern:$sarfilepattern"
	echo "collectfilepattern:$collectfilepattern"
	fi

# commandline options override config file
if [ ! -z "$opt_datafile" ] ; then datafile=$opt_datafile; fi
if [ ! -z "$opt_tempfile" ] ; then tempfile=$opt_tempfile; fi
if [ ! -z "$opt_sardir" ] ; then sardir=$opt_sardir; fi
if [ ! -z "$opt_FS" ] ; then FS=$opt_FS; fi
if [ ! -z "$opt_LSEP" ] ; then LSEP=$opt_LSEP; fi
if [ ! -z "$opt_daystokeep" ] ; then daystokeep=$opt_daystokeep; fi
if [ ! -z "$opt_maxfilelen" ] ; then maxfilelen=$opt_maxfilelen; fi
if [ ! -z "$opt_withlabels" ] ; then withlabels=$opt_withlabels; fi
if [ ! -z "$opt_sarcmd" ] ; then sarcmd=$opt_sarcmd; fi
if [ ! -z "$opt_sarfilepattern" ] ; then sarfilepattern=$opt_sarfilepattern; fi
if [ ! -z "$opt_collectfilepattern" ] ; then collectfilepattern=$opt_collectfilepattern; fi

	if [ ! -z "$debug" ] ; then 
	echo "options after commandline option processing:";
	echo "opt_datafile:$opt_datafile, datafile:$datafile"
	echo "opt_tempfile:$opt_tempfile, tempfile:$tempfile"
	echo "opt_configfile:$opt_configfile, configfile:$configfile"
	echo "opt_sardir:$opt_sardir, sardir:$sardir"
	echo "opt_FS:$opt_FS, FS:$FS"
	echo "opt_LSEP:$opt_LSEP, LSEP:$LSEP"
	echo "opt_daystokeep:$opt_daystokeep, daystokeep:$daystokeep"
	echo "opt_maxfilelen:$opt_maxfilelen, maxfilelen:$maxfilelen"
	echo "opt_withlabels:$opt_withlabels, withlabels:$withlabels"
	echo "opt_sarcmd:$opt_sarcmd, sarcmd:$sarcmd"
	echo "opt_sarfilepattern:$opt_sarfilepattern, sarfilepattern:$sarfilepattern"
	echo "opt_collectfilepattern:$opt_collectfilepattern, collectfilepattern:$collectfilepattern"
	fi

if [ ! -z "$FS" ] ; then SARSHOPTS="$SARSHOPTS -S '$FS'" ; fi
if [ ! -z "$LSEP" ] ; then SARSHOPTS="$SARSHOPTS -m '$LSEP'" ; fi
if [ ! -z "$withlabels" ] ; then SARSHOPTS="$SARSHOPTS -l" ; fi

if [ "X$1" = "X" ] ; then	# if no command:
##### ??? need to do "set_os_specifics" here ???
  if [ -s "$datafile" ] ; then	# if output file exists and is not empty, get daily
    command="daily";
  else				# else collect existing sar data
    command="collect";
  fi
else
  command=$1 ; shift
fi
	if [ ! -z "$debug" ] ; then echo "command is: $command, args: $*" ; fi
 case $command in
	collect)	sdcm_collect_existing $* ;;
	truncate)	sdcm_truncate ;;
	daily)		sdcm_get_daily $* ;;
	*)	echo "unknown command: $command" ; usage ;;
 esac
