#!/bin/ksh
#
# $Id: sar.sh,v 1.1 2023/03/27 16:07:05 root Exp $
#
# The following code is Confidential and is covered by the installation license
# (c) Copyright Fortra, LLC. and its group of companies.
#
#&& sar.sh is used to format sar data with the help of sar.awk 
#
#& sar data is collected by a cron process on each 5 minute interval 
#& on HP and Solaris.
#& The sar.sh file is normally ran on a daily basis on previous days data.
#
#&@ sar.awk is also required.
#
## Jack Miller 4/18/2009
#  Updated shell to ksh, not sh compliant.

debug=""
oslabel=`uname -s`
withlabels="T";	# should default to "T"
sysname=`uname -n | awk -F. '{print $1}'`

while getopts "D:O:Smo:f:C:l:s" opt ; do
   case $opt in
	D)	debug="T" ;;		# passes debug flag to sar.awk, sets debug here
	O)	oslabel=$OPTARG ;;	# override "uname -s" (careful!)
	S)	OSEP="OSEP=\"$OPTARG\"" ;;	# change the output field separator
	m)	LSEP="LSEP=\"$OPTARG\"" ;;	# set the multi-line output separator
	o)	saropts=$OPTARG ;;	# override the default options
	f)	sarfile="-f $OPTARG" ;;	# shortcut to add "-f file"
	C)	sarcmd=$OPTARG ;;	# just enter the whole command
	l)	withlabels="T" ;;	# add labels (see below)
	s)	sysname=$OPTARG ;;	# override systemname
		# see the AIX default below for an example of why -C is needed
   esac
done
shift `expr $OPTIND - 1`
osopt="-v oslabel=$oslabel"
snopt="-v sysname=$sysname"

if [ ! -z "$debug" ] ; then
   echo "debug=$debug, oslabel=$oslabel, osopt=$osopt, snopt=$snopt, sarfile=$sarfile, saropts=$saropts, sarcmd=$sarcmd, OSEP=$OSEP, LSEP=$LSEP, withlabels=$withlabels"
fi
if [ "X$sarcmd" != "X" ] && ([ "X$saropts" != "X" ] || [ "X$sarfile" != "X" ]) ; then
  echo "ERROR -C(sarcmd) option incompatible with -o(saropts) and -f(sarfile) options"
  exit
fi

#required to standardize time format used by sar:
export LC_TIME="C"

# Linux sar options: -u -r -d -w
# AIX sar options: -r -d -u -w -c -PALL
# Solaris sar options: -u -r -d -w -c
# HP-UX sar options: -u -d -w -c
# Test some of the other hp options

if [ "$oslabel" = "AIX" ] ; then
   if [ -z "$sarcmd" ] ; then
      if [ -z "$saropts" ] ; then
         if [ "$withlabels" = "T" ] ; then
            ##### NOTE - this is the required setting for Performance Navigator
            sarcmd='(echo "sarlabel: disk"; sar -d $sarfile; echo "sarlabel: mem"; echo "linelabel: MEM"; sar -r $sarfile; echo "sarlabel: cpu"; sar -uwc -PALL $sarfile)'
         else
            sarcmd="(sar -r -d $sarfile; sar -u -w -c -PALL $sarfile)"
         fi
      else
         sarcmd="sar $saropts $sarfile"
      fi
   fi
   if [ ! -z "$debug" ] ; then echo "sarcmd=$sarcmd" ; fi
   eval "$sarcmd | awk -f sar.awk -v debug=$debug $osopt $snopt $OSEP $LSEP"

elif [ "$oslabel" = "Linux" ] ; then
   if [ -z "$saropts" ] ; then saropts="-urdw" ; fi
   if [ -z "$sarcmd" ] ; then
      if [ "$withlabels" = "T" ] ; then
         ##### NOTE - this is the required setting for Performance Navigator
         sarcmd='(echo "sarlabel: disk"; sar -d $sarfile; echo "sarlabel: mem"; echo "linelabel: MEM"; sar -r $sarfile; echo "sarlabel: cpu"; sar -u $sarfile; echo "linelabel: CSWITCH"; sar -w $sarfile)'
      else
         sarcmd="sar $saropts $sarfile"
      fi
   fi
   if [ ! -z "$debug" ] ; then echo "sarcmd=$sarcmd" ; fi
   eval "$sarcmd | awk -f sar.awk -v debug=$debug $osopt $snopt $OSEP $LSEP"

elif [ "$oslabel" = "SunOS" ] ; then
   awkcmd="/usr/xpg4/bin/awk"	# more standard awk
   if [ -z "$saropts" ] ; then saropts="-urdwc" ; fi
   if [ -z "$sarcmd" ] ; then
      if [ "$withlabels" = "T" ] ; then
      ##### NOTE - this is the required setting for Performance Navigator
      ## sarcmd='(echo "sarlabel: disk"; sar -d $sarfile; echo "sarlabel: mem"; echo "linelabel: MEM"; sar -r $sarfile; echo "sarlabel: cpu"; sar -u $sarfile; echo "linelabel: CSWITCH"; sar -w $sarfile; echo "linelabel: SYSCALLS"; sar -c $sarfile)'
      sarcmd='(echo "sarlabel: disk"; sar -d $sarfile; echo "sarlabel: cpu"; sar -u $sarfile; echo "sarlabel: cswitch"; sar -w $sarfile; echo "sarlabel: syscalls"; sar -c $sarfile; echo "sarlabel: mem"; sar -r $sarfile; )' 
      else
         sarcmd="sar $saropts $sarfile"
      fi
   fi
   if [ ! -z "$debug" ] ; then 
      echo "sarcmd=$sarcmd" 
   fi
   eval "$sarcmd | $awkcmd -f sar.awk -v debug=$debug $osopt $snopt $OSEP $LSEP"

elif [ "$oslabel" = "HP-UX" ] ; then
   uname_r=`uname -r`
   awkcmd="/usr/bin/awk"	# xpg4 not available 
   if [ -z "$saropts" ] ; then saropts="-udwc" ; fi
   if [ -z "$sarcmd" ] ; then
      if [ "$withlabels" = "T" ] ; then
         if [ "x${uname_r}x" = "xB.11.31x" ]; then 
            #### NOTE - this is the required setting for Performance Navigator
            sarcmd='(echo "sarlabel: disk"; sar -Rd $sarfile; echo "sarlabel: cpu"; sar -u $sarfile; echo "sarlabel: cswitch"; sar -w $sarfile; echo "sarlabel: syscalls"; sar -c $sarfile; echo "sarlabel: PerPrcQue"; sar -MPq $sarfile; echo "sarlabel: PerPrcCPUutil"; sar -MPu $sarfile; echo "sarlabel: HBAact"; sar -H $sarfile; echo "sarlabel: LUNact"; sar -L $sarfile; )' 
         elif [ "x${uname_r}x" = "xB.11.23x" ]; then 
            sarcmd='(echo "sarlabel: disk"; sar -d $sarfile; echo "sarlabel: cpu"; sar -u $sarfile; echo "sarlabel: cswitch"; sar -w $sarfile; echo "sarlabel: syscalls"; sar -c $sarfile; echo "sarlabel: PerPrcQue"; sar -MPq $sarfile; echo "sarlabel: PerPrcCPUutil"; sar -MPu $sarfile; )' 
         elif [ "x${uname_r}x" = "xB.11.11x" ]; then 
            sarcmd='(echo "sarlabel: disk"; sar -d $sarfile; echo "sarlabel: cpu"; sar -u $sarfile; echo "sarlabel: cswitch"; sar -w $sarfile; echo "sarlabel: syscalls"; sar -c $sarfile; echo "sarlabel: PerPrcQue"; sar -Mq $sarfile; echo "sarlabel: PerPrcCPUutil"; sar -Mu $sarfile; )' 
         else
            sarcmd='(echo "sarlabel: disk"; sar -d $sarfile; echo "sarlabel: cpu"; sar -u $sarfile; echo "sarlabel: cswitch"; sar -w $sarfile; echo "sarlabel: syscalls"; sar -c $sarfile; echo "sarlabel: PerPrcQue"; sar -MPq $sarfile; echo "sarlabel: PerPrcCPUutil"; sar -MPu $sarfile; )' 
         fi
      else
	 sarcmd="sar $saropts $sarfile"
      fi
   fi
   if [ ! -z "$debug" ] ; then echo "sarcmd=$sarcmd" ; fi
   eval "$sarcmd | $awkcmd -f sar.awk -v debug=$debug $osopt $snopt $OSEP $LSEP"
else
   echo "unknown OS : $oslabel - unable to collect data"
fi
