#!/bin/ksh
#
# $Id: pndchk_sar.sh,v 1.1 2023/03/27 16:07:05 root Exp $
#
# The following code is Confidential and is covered by the installation license
# (c) Copyright Fortra, LLC. and its group of companies.
#
#&& Performance Navigator Data CHecK for SAR  pndchk_sar.sh) 
#& Performs data validation of mpgdata file data, and creates resequenced file
#& if required.
#
#&@ pndcutils.sh is also required
#
#&% Ran on the second of the month in crontab 
#

mkdir -p /tmp/helpsystems_tmp

invocdir=`dirname $0`
if [ -s $invocdir/pndcutils.sh ] ; then
   . $invocdir/pndcutils.sh
fi
cd ${invocdir}

if [ -s $invocdir/pn.config ] ; then
   . $invocdir/pn.config 
else
   echo "pn.config file does not exist at `pwd`"
fi

#tail command different on SunOS systems
jtailchk=`uname -s`
if [ "x${jtailchk}x" != "xSunOSx" ] ; then
   jtailmod='-n'
else
   jtailmod=''
fi

# Use either gawk, nawk, or awk in that order for AWK.
# Use the first one found.  gawk has more options than nawk which has
# more options than awk.
for jawk in /bin/gawk /usr/bin/gawk /bin/nawk /usr/bin/nawk /bin/awk /usr/bin/awk
do
   # Added Spanish message for not found
   jtest=`which $jawk 2>&1 | egrep -v "annot| no | not | No se puede encontrar "`
   jchk=$?
   if [ $jchk -eq 0 ]; then
      AWK="$jtest"
      break
   fi
done
if [ ! -z "$debug" ] ; then
   echo $AWK
fi
# set > /tmp/helpsystems_tmp/set.out
if [ "x${1}x" != "xx" ]; then
   jdatafile=$1
else
   if [ "x${datafile}x" != "xx" ]; then
      jdatafile=$datafile 
   else
      jdatafile=$def_datafile 
   fi
fi

if [ ! -s $jdatafile ] ; then 
   echo "### Error ###  Cannot find file specified ($jdatafile)"
   echo ""
   exit 1
fi

rm -f ${jdatafile}.data_errors 


echo "Validating data $jdatafile. Details in: ${jdatafile}.dv_rpt"  
# echo $invocdir
# wc -l $jdatafile
cd $invocdir

# Added a date reference to ensure the ${jdatafile}'s date/time does not change.
rm -f ${jdatafile}.timeref
touch -r ${jdatafile} ${jdatafile}.timeref


jreccnt=`wc -l ${jdatafile} | $AWK '{print $1}'`


# It would be expected that each day the record count will be near 288 records 
# for one every 5 minutes.
# Need to track the number of fields also, since that would not be expected to 
# change to often or significantly.
# A test of max field, min field, max records, min records, etc
# Leap year can cause the record counto to change up or down by 12.  
# 300 or 276 instead of 288.
# Identify records with problems in structure.  
# Ensure all records are properly sequenced.
#

grep "^20" ${jdatafile} \
   | $AWK '{jFields=gsub(/;/,";")+1}{if(substr($0,1,7) != "sysname") jDate = substr($0,1,8); print jFields, jDate}'  \
   | uniq -c > record_field_date.hsllc

cat record_field_date.hsllc \
   | $AWK '{print $3}' \
   | uniq -c \
   | $AWK '{if($1 != 1)print $0}' > error_dates_w_changing_field_counts.hsllc

if [ -s error_dates_w_changing_field_counts.hsllc ]; then
   while
   read jcnt jdate
   do
      grep $jdate record_field_date.hsllc   
      echo ""
   done <error_dates_w_changing_field_counts.hsllc > error_dates_w_changing_field_counts_detail.hsllc
fi

echo "Checking for bad format lines"
egrep -vn "^(sy|20)" ${jdatafile} > error_bad_format_lines

echo "Checking for bad data lines"
egrep -n "^20" ${jdatafile} | grep -v "##disk##.*##mem##" > error_bad_data_lines
egrep -n "^20" ${jdatafile} | grep "##disk##.*##disk##" >> error_bad_data_lines

echo "Checking for bad header lines"
egrep -n "^sysname\=" ${jdatafile} | grep -v "##disk##.*##mem##" > error_bad_header_lines
egrep -n "^sysname\=" ${jdatafile} | grep "##disk##.*##disk##" >> error_bad_header_lines

cat error_bad* | cut -d: -f1 | sort -nr  > bad_line_numbers_from_errors

# Used to find the list of files that are not size 0, but have potential problems.
find . -name error_\* -size 0 | xargs rm
echo "Creating a list of problem files into pndchk_sar_errors.hsllc"
find . -name 'error*' > pndchk_sar_errors.hsllc

# This should work because the time only changes at 3:00 AM for DST
grep "^20" ${jdatafile} | cut -c1-8 | uniq > pndchk_datechk.tmp
cat pndchk_datechk.tmp | sort -n > pndchk_datechk_sorted.tmp
cmp pndchk_datechk.tmp pndchk_datechk_sorted.tmp 2>&1 > /dev/null
jchk=$?
if [ $jchk -eq 0 ]; then
   echo "Dates are sequential. "
else
   echo "Dates are NOT sequential, continuing to process"
   paste pndchk_datechk.tmp pndchk_datechk_sorted.tmp  | $AWK '{if( $1 != $2 ) print NR, $1, $2}' > pndchk_datechk_error
   echo "Please review pndchk_datechk_error for details"
fi 


# Really large awk process
egrep "^(sysname=|20[0-9][0-9][0-9][0-9][0-9][0-9])" ${jdatafile} | $AWK \
  'BEGIN{lastrec=0
lastdate="00000000"
chkdate="00000000"
errchk="Ok"}
{ if( substr($0,1,8) == "sysname=" ) 
{chkdate="00000000"
sysrec=NR}  
}
{ if( substr($0,1,8) == "sysname=" && lastrec != 0 ) 
{ 
print mydate,firstrec,lastrec,lastrec-firstrec,errchk
# print mydate,firstrec,lastrec,lastrec-firstrec,lastdate,errchk
lastdate=mydate
errchk="Ok"}
}
{if( substr($0,1,2) == "20" ) 
{
lastrec=NR
firstrec=sysrec  
chkrec=lastrec-firstrec
mydate=substr($0,1,8)
   # Check for a duplicate record with same date 
   {if( mydate == lastdate && errchk == "Ok" )
   errchk="Error_dup_date"}
   # If first date in record set chkdate 
   {if( chkdate == "00000000" )
   chkdate=mydate}
   # Records should not change date within a record
   {if( chkdate != mydate && errchk == "Ok" )
   errchk="Error_mixed_date_rec"}
   # Check for High Record Count over 288 for DLS in November
   {if( chkrec > 288 && errchk == "Ok" )
   errchk="Error_high_rec_cnt"}
}
}
END{print mydate,firstrec,lastrec,lastrec-firstrec,errchk}'  | uniq > ${jdatafile}.dv_rpt
# END{print mydate,firstrec,lastrec,lastrec-firstrec,lastdate,errchk}'  > ${jdatafile}.dv_rpt

# Be carefue do not remove this section....
# List DBF records sorted by date
# Linux
uname_s=`uname -s`
if [ "x${uname_s}x" = "xLinuxx" ]
then
   # Sort command for Linux to sort by 4th field
   # grep DBF ${jdatafile}.dv_rpt | sort -n -k +4 > ${jdatafile}.dv_rpt_DBF_sort
   cat ${jdatafile}.dv_rpt | sort -n  > ${jdatafile}.dv_rpt_sort
else
   # Sort command for SunOS or AIX to sort by 4th field
   # grep DBF ${jdatafile}.dv_rpt | sort -n +3 > ${jdatafile}.dv_rpt_DBF_sort
   cat ${jdatafile}.dv_rpt | sort -n  > ${jdatafile}.dv_rpt_sort
fi


echo Please review "${jdatafile}.dv_rpt" 

jdatecnt=`wc -l ${jdatafile}.dv_rpt | $AWK '{print $1}'`
jdatecntuniq=`cat ${jdatafile}.dv_rpt | $AWK '{print $1}' | sort -n | uniq | wc -l | $AWK '{print $1}'`
if [ $jdatecnt -eq $jdatecntuniq ]; then
   echo "Date counts of $jdatecnt are the same as count unique $jdatecntuniq"
else
   echo "Error Date counts of $jdatecnt are NOT the same as count unique $jdatecntuniq"
   echo "There may be duplicate sar entries from the same date."
fi



if [ -s error_bad_format_lines ]; then
   echo  '---------------------------' >> ${jdatafile}.data_errors 
   echo  'Please review error_bad_format_lines ' >> ${jdatafile}.data_errors 
   echo  "There are `wc -l error_bad_format_lines | $AWK '{print $1}'` incorrectly formated lines"
   echo  'These lines are automatically removed if a sort is required. ' >> ${jdatafile}.data_errors 
   echo  '---------------------------' >> ${jdatafile}.data_errors 
fi

if [ -s error_bad_data_lines ]; then
   echo  '---------------------------' >> ${jdatafile}.data_errors 
   echo  'Please review error_bad_data_lines ' >> ${jdatafile}.data_errors 
   echo  "There are `wc -l error_bad_data_lines | $AWK '{print $1}'` bad data lines"
   echo  '---------------------------' >> ${jdatafile}.data_errors 
fi

# Identify errors that are in the record report.
grep Error_high_rec_cnt ${jdatafile}.dv_rpt > ${jdatafile}.data_errors.tmp2
if [ -s ${jdatafile}.data_errors.tmp2 ]; then
   echo  '---------------------------' >> ${jdatafile}.data_errors 
   echo  "Errors were found with the data in the report file $jdatafile.dv_rpt" >> ${jdatafile}.data_errors 
   echo  "Normal counts are 288(12*24). During daylight savings in early " >> ${jdatafile}.data_errors
   echo  "November they could be 300, for one day when time rolls back" >> ${jdatafile}.data_errors
   echo  "Or this could be from causes unknown. " >> ${jdatafile}.data_errors 
   echo  '---------------------------' >> ${jdatafile}.data_errors 
   echo  'Details:' >> ${jdatafile}.data_errors 
   echo  '---------------------------' >> ${jdatafile}.data_errors 
   cat ${jdatafile}.data_errors.tmp2 >> ${jdatafile}.data_errors 
   echo  '' >> ${jdatafile}.data_errors 
   echo  '' >> ${jdatafile}.data_errors 
fi 

grep Error_dup_date ${jdatafile}.dv_rpt > ${jdatafile}.data_errors.tmp2
if [ -s ${jdatafile}.data_errors.tmp2 ]; then
   echo  '---------------------------' >> ${jdatafile}.data_errors 
   echo  "Duplicate date errors were found within the data in the report file" >> ${jdatafile}.data_errors  
   echo  "$jdatafile.dv_rpt " >> ${jdatafile}.data_errors 
   echo  "It is possible that sar reporting was done multiple time in the same" >> ${jdatafile}.data_errors
   echo  "day. " >> ${jdatafile}.data_errors 
   echo  '---------------------------' >> ${jdatafile}.data_errors 
   echo  'Details:' >> ${jdatafile}.data_errors 
   echo  '---------------------------' >> ${jdatafile}.data_errors 
   cat ${jdatafile}.data_errors.tmp2 >> ${jdatafile}.data_errors 
   echo  '' >> ${jdatafile}.data_errors 
   echo  '' >> ${jdatafile}.data_errors 
fi 

grep Error_mixed_date_rec ${jdatafile}.dv_rpt > ${jdatafile}.data_errors.tmp2
if [ -s ${jdatafile}.data_errors.tmp2 ]; then
   echo  '---------------------------' >> ${jdatafile}.data_errors 
   echo  "Mixed date errors were found within the data in the report file" >> ${jdatafile}.data_errors  
   echo  "$jdatafile.dv_rpt " >> ${jdatafile}.data_errors 
   echo  "The cause of this is not known." >> ${jdatafile}.data_errors
   echo  '---------------------------' >> ${jdatafile}.data_errors 
   echo  'Details:' >> ${jdatafile}.data_errors 
   echo  '---------------------------' >> ${jdatafile}.data_errors 
   cat ${jdatafile}.data_errors.tmp2 >> ${jdatafile}.data_errors 
   echo  '' >> ${jdatafile}.data_errors 
   echo  '' >> ${jdatafile}.data_errors 
fi 


# Remove sorted file from previous times this script is ran.
rm -f ${jdatafile}.sorted 

# If both lists are the same, indicate sequenced correctly.
# Otherwise, create a new file with correctly sequenced records.
diff ${jdatafile}.dv_rpt ${jdatafile}.dv_rpt_sort > /dev/null
jdiffchk=$?
# Uncommenting the next line to force a out of date sequence.
# echo "Comment this line later"; jdiffchk=1
if [[ ${jdiffchk} -ne 0 ]] ; then
   echo "${jdatafile} is out of date sequence."
   jsequence="OutOfSequence"
   echo "Creating a correctly indexed file ${jdatafile}.sorted "
   rm -f ${jdatafile}.sorted 
   touch ${jdatafile}.sorted
   # Added reduction process to record list.
   $AWK '{if(NR == 1){jbegrec=$2; jendrec=$3}}
   {if(NR != 1 && $2-jendrec == 1) 
   { jendrec=$3 } 
   else if(NR != 1 && $2-jendrec != 1)
   {print jbegrec, jendrec ; jbegrec=$2; jendrec=$3}}
   END {print jbegrec, jendrec}' ${jdatafile}.dv_rpt_sort > ${jdatafile}.dv_rpt_sort_reduced 

   # End of reduction process
   while
   read jRecBegin jRecEnd
   # read jRecType jRecBegin jRecEnd jRecDate jRecStat
   do
      # Had started with sed, but awk is much faster, and perl just a bit 
      # slower than awk.  All three approaches tested, but awk was faster.
       egrep "^(sysname=|20[0-9][0-9][0-9][0-9][0-9][0-9])" ${jdatafile} \
         | $AWK -v jbegin=${jRecBegin} -v jend=${jRecEnd} '{if(NR >= jbegin && NR <= jend )print $0}{if(NR > jend) exit}' >> ${jdatafile}.sorted
   done < ${jdatafile}.dv_rpt_sort_reduced


   # Previously # done < ${jdatafile}.dv_rpt_sort
   # Ensure the date is the same for the original and corrected file
   find ${jdatafile} -newer ${jdatafile}.timeref > /tmp/helpsystems_tmp/pndchk.timeref
   if [ -s /tmp/helpsystems_tmp/pndchk.timeref ]; then
      echo "The data has changed while it was being processed."
      echo "Results are not correct, do not use the ${jdatafile}.sorted file." 
   else
      if [ -s ${jdatafile}.sorted ]; then
         touch -r ${jdatafile} ${jdatafile}.sorted  
      fi
      echo "If the size is slightly smaller, there may have been invalid data,"
      echo "please check the file ${jdatafile}.dv_rpt for lines without OK"
      echo "Please check the date and size below."
      ls -al ${jdatafile}.sorted ${jdatafile}
      jfdate=`date '+%A %B %d, %Y'`
      echo "The sorted data should be copied over the actual data once confirmed,"
      echo "providing the date is still ${jfdate}."
      echo "Do not do the mv command below unless it is still ${jfdate}."
      echo "Command required:"
      echo "   mv ${jdatafile}.sorted ${jdatafile}"
      echo "If it is not ${jfdate}, please run pndchk.sh again."
   fi
      # Do we want to fix automatically?
      # How do we want to handle totally invalid data that may be in the datafile?
else
   echo "${jdatafile} is sequenced correctly."
   jsequence="Ok"
fi


if [ -s ${jdatafile}.data_errors ] ; then
   echo "***   There are errors in ${jdatafile}"         
   echo "***   Please review ${jdatafile}.data_errors for details"
fi

if [ -s ${jdatafile}.data_errors.tmp ]; then
   echo  "Times were decreasing in the data."
fi

if [ -s ${jdatafile}.data_errors.tmp1 ]; then
   echo  "Header info placed mid line."
fi

if [ -s ${jdatafile}.data_errors.tmp2 ]; then
   echo  "Assorted errors in the data." 
fi

rm -f ${jdatafile}.data_errors.tmp
rm -f ${jdatafile}.data_errors.tmp1
rm -f ${jdatafile}.data_errors.tmp2

truncate_minimum=30
truncate_default=366

if [ "x${truncate_days}x" = "xx" ]; then
   echo "truncate_days not specified (${truncate_days}); setting value to (${truncate_default})."
   truncate_days=${truncate_default}
fi

# jdbf_cnt is count of actual database records
jdbf_cnt=`cat ${jdatafile}.dv_rpt | $AWK '{print $1}' | uniq | wc -l| $AWK '{print $1}'`
echo "There are (${jdbf_cnt}) days of data in ${jdatafile}"


if [ ${truncate_days} -lt ${truncate_minimum} ] ; then
   echo "Truncate days (${truncate_days}) is less than min retention of (${truncate_minimum}).  Exiting!"
   echo ""
   exit 1
fi


if [ "x${jsequence}x" != "xOkx" ] ; then
   echo "Cannot truncate data that is out of sequence."
   echo "Please run the move (mv) specified above to properly order the data,"
   echo "then rerun this script.  Exiting!"
   echo ""
   exit 1
fi

if [ ${truncate_days} -lt ${jdbf_cnt} ] ; then
   echo "(${jdbf_cnt}) is greater than the specified truncate value of (${truncate_days})."

   # jdbf_date_select is the first date of the data retained.
   # Any data before this date will be archived.
   jdbf_date_select=`cat ${jdatafile}.dv_rpt | $AWK '{print $1}' | uniq | tail ${jtailmod} -${truncate_days} | head -1`

   # jdbf_date_line is the full text of the date line.
   jdbf_date_line=`grep ${jdbf_date_select} ${jdatafile}.dv_rpt | head -1`
   
   # jdbf_tail_lineno is the starting line number of the data to be retained.
   jdbf_tail_lineno=`echo ${jdbf_date_line} | $AWK '{print $2}'`

   # jdbf_head_lineno is the number of lines that are being archived.
   jdbf_head_lineno=`echo ${jdbf_date_line} | $AWK '{print ($2 - 1)}'`

   # Only uncomment the lines below if testing.
   # echo "jdbf_date_select=${jdbf_date_select}"
   # echo "jdbf_date_line=${jdbf_date_line}"
   # echo "jdbf_tail_lineno=${jdbf_tail_lineno}"
   # echo "jdbf_head_lineno=${jdbf_head_lineno}"

   if [ -s ${jdatafile}_archive ] && [ -s ${jdatafile}_archive.gz ] ; then
      echo "### error ###"
      echo "Both files ${jdatafile}_archive and ${jdatafile}_archive.gz exist,"
      echo "exiting.  Please investigate the problem."
      echo ""
      exit 1
   fi
   # If someone has gunzipped the archive file, gzip it back.
   if [ -s ${jdatafile}_archive ] && [ ! -s ${jdatafile}_archive.gz ] ; then
      echo "Archive file ${jdatafile}_archive is not gzipped"
      echo "Running - gzip ${jdatafile}_archive"
      gzip -f ${jdatafile}_archive
   fi

   # Start by gzipping the datafile with -f (force)
   gzip -f ${jdatafile} 

   # Add to the archive file.
   echo "Archiving data that is older than (${truncate_days}) days into file ${jdatafile}_archive.gz"
   gunzip -c ${jdatafile}.gz | head -${jdbf_head_lineno} | gzip -c >> ${jdatafile}_archive.gz
   echo "Creating a truncated version of ${jdatafile} retaining (${truncate_days}) days."
   gunzip -c ${jdatafile}.gz | tail ${jtailmod} +${jdbf_tail_lineno} | gzip -c > ${jdatafile}_tmp.gz
#   This is invalid for sar data, but may be able to do something similar.
   jtailcnt=`gunzip -c ${jdatafile}_tmp.gz | grep "^20" | cut -c1-8 | uniq | wc -l | $AWK '{print $1}'`
   # Added check to ensure record count is as expected.
   if [ ${jtailcnt} -ge ${truncate_days} ] ; then
      echo "Confirmed record count (${jtailcnt}) in modified version is greater than or "
      echo "equal to (${truncate_days}).  Building new (${jdatafile})."
      mv ${jdatafile}_tmp.gz ${jdatafile}.gz 
   else
      echo "### Error ### Unexpected result, returning to original datafile"
      echo "${jdatafile}"
   fi

   gunzip ${jdatafile}.gz
else
   echo "(${jdbf_cnt}) days is not greater than truncate value (${truncate_days})."
   echo "No truncation took place."
fi

# Remove temporary files for ensuring sequence is correct.
# They are used in other processes, so need to be deleted later.
if [ "x${jsequence}x" = "xOkx" ] ; then
   rm -f ${jdatafile}.dv_rpt_sort 
fi
# echo "Remove later!!!!!!"; exit
echo "Done!"
echo ""
