#!/bin/ksh
#!/bin/ksh -kv
#
# $Id: pnbmpgd2gzip.sh,v 1.3 2024/04/02 19:15:38 root Exp root $
#
# The following code is Confidential and is covered by the installation license
# (c) Copyright Fortra, LLC. and its group of companies.
#
#&& Performance Navigator Build mpgdata2.hostname file (pnbmpgd2.sh)
#&
#& By default only the last 60 days will be process for each host.
#& To change that, run the following without the quotes example "export days2keep=90"
#& To go back to the default, run the following without quotes "export days2keep="
#&
#& Will build an mpgdata2 file from existing gzipped (.gz) nmon files that 
#& were not created by the Performance Navigator consolidation process.
#& It is intended to be used on existing nmon data and will build an
#& mpgdata2.hostname file on an AIX system.
#&
#& This will allow existing data to be used with the Performance Navigator
#& consolidation process.
#
#&@ Requires files of the nmon data collected from an existing nmon
#&@ installation formerly gzipped (.gz).
#&@
#&@ Will also run "nmonmerge" if it is the same directory as pnbmpgd2gzip.sh
#&@ Will also run "pnbrdf.sh" if it is the same directory as pnbmpgd2gzip.sh
#&@ Will also run "rackdata.sh"
#&
#& All nmon files in the current directory will be processed except mpgdata2
#& files
#&
#& To create the mpgdata2 manually do the following:
#&
#& First test (ls -1tr *nmon*) and confirm that only nmon files are listed.
#& If not, replace the *nmon* wildcard statement with one that lists nmon files.
#&
#&    ls -1tr *nmon*  | xargs cat /tmp/helpsystems_tmp/mpgmpg 2>/dev/null >> mpgdata2.hostname

if [ "xx" == "x${days2keep}x" ] ; then
   days2keep=60
fi

jdate=`date '+%y%m%d_%H%M'`
echo "# Running (${jdate}) `grep "\$\I\d\:" ${0} | awk '{print $3,$4,$5,$6}' ` " 

jsysname=`uname -n`

mkdir -p /tmp/helpsystems_tmp

# Default is netstrip="T" 
# netstrip is used to remove ip addresses
# To keep compliant with the 
# European Union (EU) GDPR (General Data Protection Regulation). 

# To include IPs for analysis, type: 
#     export netstrip="F"
# then
#     ./pnbmpgd2gzip.sh
# 
# If netstrip is blank or "T" remove IPS
if [ "xT" == "x${netstrip}" ] || [ "x" == "x${netstrip}" ] 
then
   # net_remove="|lsconf|ifconfig|netstat|Gateway"
   net_remove="|ifconfig|,netstat,|Gateway|IP\ Address|Sub\ Netmask"
   # to see what is being removed.   
   # egrep "ifconfig|,netstat,|Gateway|IP\ Address|Sub\ Netmask" mpg_*nmon
else
   net_remove=""
fi

rm -f *.tmp *.hsllc *.hsllc
touch bad_files.tmp good_files.tmp

jtest=`uname -a | awk '{print $1}'`

if [ "$jtest" = "AIX" ] ; then 
   nmonmergever="nmonmerge_aix" 
   osarch=`uname -p`    # uname -p on AIX, ?? on linux-power, ix86 on pentium
else
   osarch=`uname -m`    # uname -m on linux-power, ix86 on pentium, etc.
fi

### echo "osarch = $osarch" ## For Testing

#  osarchx86=${osarch%%i?86}      # see if x86
if [ "$osarch" = "s390" ] ; then
    plat=mainframe_32
    nmonmergever="nmonmerge_mainframe_32" # defaults to a 32 bit version
elif [ "$osarch" = "s390x" ] ; then
    plat=mainframe_64
    nmonmergever="nmonmerge_mainframe_64" # defaults to a 64 bit version
elif [ "oo${osarch}oo" = "oox86_64oo" ] ; then
    plat=x86_64
    nmonmergever="nmonmerge_x86_64_linux"
elif [ "oo${osarch}oo" = "ooppc64leoo" ] ; then
    plat=power_64le
    nmonmergever="nmonmerge_power_64le_linux"
elif [ "oo${osarch}oo" = "ooppcoo" ] || [ "oo${osarch}oo" = "ooppc64oo" ] ; then
    plat=power_32
    nmonmergever="nmonmerge_power_32_linux"
    plat=power_64
    nmonmergever="nmonmerge_power_64_linux"
elif [ "oo${osarch}oo" = "oopowerpcoo" ] ; then
    plat=powerpc
    nmonmergever="nmonmerge_aix"
elif [ "oo${osarch}oo" = "ooia64oo" ] ; then
    plat=ia64
    nmonmergever="nmonmerge_ia64_linux"
elif [ "oo${osarch}oo" = "ooarmv7loo" ] || [ "oo${osarch}oo" = "ooarmv7loo" ] ; then
    plat=arm_32
    nmonmergever="nmonmerge_arm_32_linux"
elif [ "oo${osarch}oo" = "ooaarch64oo" ] || [ "oo${osarch}oo" = "ooaarch64oo" ] ; then
    plat=arm_64
    nmonmergever="nmonmerge_arm_64_linux"
else 
    plat=x86
    nmonmergever="nmonmerge_x86_linux"
fi

### echo "osarch=$osarch, plat=$plat, nmonmergever=$nmonmergever" # For Testing

### ls -al /usr/local/mpg/$nmonmergever # For Testing

if [ ! -s $nmonmergever ] ; then
   echo "Extracting appropriate version (${nmonmergever}) from merges.tar!"
   tar xvf merges.tar $nmonmergever
else
   echo "(${nmonmergever}) already exists!"
fi

# Cannot be ran from /usr/local/mpg
jstartdir=`pwd`
if [ "x${jstartdir}x" = "x/usr/local/mpgx" ]; then
   echo "### Error ### You cannot run from (/usr/local/mpg)"
   exit 1
fi

# Cannot be ran from the root file sytem (/) 
jstartdir=`pwd`
if [ "x${jstartdir}x" = "x/x" ]; then
   echo "### Error ### You cannot run from the root file sytem (/)"
   exit 1
fi

# Cannot be ran from the install directory 
jstartdir=`pwd`
if [ -s /etc/pn.conf ]; then
   .   /etc/pn.conf 
   if [ "x${jstartdir}x" = "x${nminst}x" ]; then
      echo "### Error ### You cannot run from install directory (${nminst})"
      exit 1
   fi
fi

# For safety there should be no mpgdata2 files when we start.
find . -name mpgdata2\* > /tmp/helpsystems_tmp/chk4mpgdata2files
if [ -s /tmp/helpsystems_tmp/chk4mpgdata2files ]; then
   echo "### Error ### There should not be any mpgdata2 files when starting, correct!"
   exit 1
fi

# Confirm being ran on an AIX powerpc, Linux ppc64, or Linux ppc64le
jsyschk=`uname -ps`
# jsyschk="Linux ppc64" # Used for testing only 
# if [ "x${jsyschk}" != "xAIX powerpc" ] && [ "x${jsyschk}" != "xLinux ppc64" ] && [ "x${jsyschk}" != "xLinux ppc64le" ] ; then
#    echo "### Error ### needs to be ran on (AIX powerpc)"
#    echo "# This is on (${jsyschk})"
#    exit 1
# fi

function JProgress {      ## args low_number high_number
# low_number ($1),_high_number ($2)
jdate=`date '+%y%m%d_%H%M'`
# 50 # and 50 spaces for moving bar
jpound='##################################################'
j2space='                                                            '

if [ "x" == "x${1}" ] || [ "x" == "x${2}" ]; then
  echo "Two numbers required"
  exit
fi 

if [ ${1}  -gt  ${2} ]; then 
  echo 'First variable should be less than second variable'
  exit
fi 
# Determine percentage based on low_number/high_number
let jper=${1}*100/${2}
jpercent=`echo $jper | awk '{print int($1/100*50)}'`
let jremain=60-${jpercent}
jcursor=`expr "x$jpound" : "x\(.\{$jpercent\}\)"`    # Will output '\n' to
jspace=`expr "x$j2space" : "x\(.\{$jremain\}\)"`    # Will output '\n' to
printf "${jdate} (${jcursor}${jspace})(${jper}%%)($1 of $2)\r"

# Will want to echo "" if you want to leave the last line visible.

}

# gzip any nmon files that are not gzipped, but gunzip them later
find . -type f -mmin +6 -name '*' | egrep -v "\.sh$|gz$"  | xargs grep -c ,host, | grep -v ":0" | awk -F: '{print $1}' | grep -v "^0$"  > /tmp/helpsystems_tmp/filesnotgzipped.tmp
if [ -s /tmp/helpsystems_tmp/filesnotgzipped.tmp ] ; then
   echo "# # gzipping (gzip) any nmon files that are not already gzipped."
   echo "# # They are listed in (/tmp/helpsystems_tmp/filesnotgzipped.tmp)" 
   cat /tmp/helpsystems_tmp/filesnotgzipped.tmp | xargs gzip 
else
   echo "# # All files are gzipped from the beginning."
fi

# This version requires all original files to be gzipped (.gz)
# It was developed when multiple nmon files were used in the same day and 
# requires nmonmerge to correct the file anomallies.
# 
# 
# Requires executable nmonmerge in same directory
if [ ! -s ./${nmonmergever} ]; then
   echo "### Error ### The executable ${nmonmergever} is required in this directory!"
   exit 1
fi 
# 
# Requires executable pnbrdf.sh in same directory
if [ ! -s ./pnbrdf.sh ]; then
   echo "### Error ### The executable pnbrdf.sh is required in this directory!"
   exit 1
fi 
# 
# First identify files to process (must all be gzipped)  

echo "# Making a list of gzipped files into gzippedfiles.tmp"

find . -type f -mmin +6 -name '*.gz' | egrep -v "\.sh$|nmonmerge|tar.gz"   > gzippedfiles.tmp

jnbrfiles=`wc -l gzippedfiles.tmp | awk '{print $1}'`

jdate=`date '+%y%m%d_%H%M'`

echo "# Making a list of files into files2process.tmp (${jnbrfiles}) ${jdate} (${PWD})"

# This adds better messaging. 
if [ $jnbrfiles -eq 0 ] ; then
   echo "# No nmon files were found that are over 6 minutes old."
   exit 1
fi

# echo "Remove later" ; exit 1
jfilecnt=1
# for jfile in `ls -1 *nmon*gz`
while 
read jfile
do
   gunzip -t -q 2>/dev/null ${jfile}
   jchk=$?
   if [ "x0x" != "x${jchk}x" ] ; then
      echo "${jfile} badgz" >> bad_files.tmp  
   else
      echo `gunzip -c $jfile | grep ',host,' | awk -F, '{print $NF}'` `gunzip -c $jfile | grep ',date,' | awk -F, '{ split("JAN FEB MAR APR MAY JUN JUL AUG SEP OCT NOV DEC", month, " ")
      for (i=1; i<=12; i++) mdigit[month[i]]=i
      m=toupper(substr($NF,4,3))
      jdat=substr($NF,8,20) sprintf("%02d",mdigit[m]) substr($NF,1,2) 
      print jdat }'` `gunzip -c $jfile | grep ',time,' | awk -F, '{print substr($NF,1,2)substr($NF,4,2)substr($NF,7,2)}'` $jfile >> files2process1.tmp 
   fi
   JProgress $jfilecnt $jnbrfiles 
   let jfilecnt=$jfilecnt+1
done < gzippedfiles.tmp 

sort files2process1.tmp > files2process.tmp 

echo "" # needed to for the display of last result from JProgress 


# Ensure now file has multiple hosts, and exit if found after displaying.
multi_hosts="F"
cat files2process.tmp | awk '{print $NF}' > filesonly.tmp
while
read jfilename
do
   jhostuniq_check=`gunzip -c $jfilename | grep ',host,' | sort | uniq | wc -l`
   # Should be one, if not print list. 
   if [ $jhostuniq_check -ne 1 ] ; then
     multi_hosts="T"
     echo "# =================================================#"
     echo "#### Error #### Has more or less than one host listed, not processing files"
     echo "#### Review the data listed below, being placed into bad files. "
     echo "# =================================================#"
     echo "# $jfilename " 
     echo "$jfilename bad_hostcnt" >> bad_files.tmp
     echo "# -------------------------------------------------#"
     gunzip -c $jfilename | grep ',host,' | sort | uniq  
     echo "# =================================================#"
   fi
done < filesonly.tmp

if [ "x${multi_hosts}" != "xT" ];then
   echo "# No filenames had mutiple hosts internally, continuing"
fi

cat files2process.tmp | awk '{if(NF != 4)print $0}' > field_cnt_chk.tmp
if [ -s field_cnt_chk.tmp ] ; then
   echo "# =================================================#"
   echo "#### Error #### Field count should be 4 and is not correct for these "
   echo "#### Review the data listed below, being placed into bad files. "
   echo "# =================================================#"
   cat field_cnt_chk.tmp | awk '{print "#",$0}' | awk '{print $NF, "bad_fieldcnt" }'  >> bad_files.tmp
   cat field_cnt_chk.tmp | awk '{print "#",$0}'
   echo "# =================================================#"
else
   echo "# Field counts are correct in files2process.tmp, continuing"
fi

# Sample of files2process.tmp
# jbigserver 20130720 000008 jbigserver.20130720.000005.GMT.72ef3d2B-Efbd-4b2a-8E36-BF27ae2FB230.nmon.gz
# jbigserver 20130720 003005 jbigserver.20130720.003003.GMT.72ef3d2B-Efbd-4b2a-8E36-BF27ae2FB230.nmon.gz
# jbigserver 20130720 010007 jbigserver.20130720.010004.GMT.72ef3d2B-Efbd-4b2a-8E36-BF27ae2FB230.nmon.gz
# jbigserver 20130720 013005 jbigserver.20130720.013003.GMT.72ef3d2B-Efbd-4b2a-8E36-BF27ae2FB230.nmon.gz
# jbigserver 20130720 020014 jbigserver.20130720.020006.GMT.72ef3d2B-Efbd-4b2a-8E36-BF27ae2FB230.nmon.gz


# jlasthost will change on end of each file process, but needs to have a starting point
jlasthost='jhsllc'
jchk_host='jhsllc'
# jlastdate will change on end of each file process, but needs to have a starting point
jlastdate='29999999'
# jlasttime will change on end of each file process, but needs to have a starting point
jlasttime='000000'

if [ -s bad_files.tmp ]; then
   cat bad_files.tmp | awk '{print $1}' | sort > bad_files_only.tmp
   # Remove bad files from those to be processed 
   cat files2process.tmp | grep -vf bad_files_only.tmp  > files2processj.tmp
   cp files2processj.tmp files2process.tmp
   rm -f files2processj.tmp
fi
cat files2process.tmp | awk '{print $NF}' > good_files.tmp

jfiles_cnt=`wc -l files2process.tmp | awk '{print $1}'`
jservers_cnt=`cat files2process.tmp | awk '{print $1}'  | uniq | wc -l | awk '{print $1}'`
jdate=`date '+%y%m%d_%H%M'`
echo "# Will be processing (${jfiles_cnt}) files for (${jservers_cnt}) servers last ${days2keep} days. ${jdate}"

cat files2process.tmp | awk '{print $1}' | uniq > hosts2process.tmp
while read jhosts
do
grep -h "^${jhosts} " files2process.tmp | tail -${days2keep} 
done < hosts2process.tmp > files2process_trimmed.tmp
cp files2process.tmp files2process.tmp.orig
cp files2process_trimmed.tmp files2process.tmp
if [ -s files2process.tmp  ] ; then
   echo "# =================================================#"
   echo "# Processing files in files2process.tmp"
   echo "# =================================================#"
   jservers_progress=1
   while
   read jchk_host jchk_date jchk_time jchk_filename_gz
   do
      gunzip ${jchk_filename_gz}
      jchk_filename=`echo $jchk_filename_gz | sed s,.gz$,,`
      # echo ${jchk_filename}; exit 1 
      if [ "x${jchk_host}" != "x${jlasthost}" ] ; then
         # if $jchk_host does not equal $jlasthost then create temp.$jchk_host
         if [ -s ${jchk_filename} ]  ; then
            jdate=`date '+%y%m%d_%H%M'`
            echo "# Processing $jchk_host (${jservers_progress} of ${jservers_cnt}) ${jdate}." 
            cp $jchk_filename temp.${jchk_host}

            let "jservers_progress=jservers_progress + 1" 
         fi 
         if [ "x${jlasthost}" != "xjhsllc}" ] && [ -s temp.${jlasthost} ] ; then
            # Added stripping of PCUP and SCPU on 3/13/2015
            # Added ensuring all lines commas on 02/26/2019
            cat temp.${jlasthost} | egrep -v "^PCPU|^SCPU${net_remove}"  | egrep ","  | uniq >> mpgdata2.${jlasthost}
            rm temp.${jlasthost}
            # ls -al mpgdata2.${jlasthost} 
            ./pnbrdf.sh mpgdata2.${jlasthost}
            gzip -f mpgdata2.${jlasthost}
         fi 
      else
         if [ -s ${jchk_filename} ] && [ -s temp.${jchk_host} ] && [ "x${jchk_date}" = "x${jlastdate}" ] ; then
            ./${nmonmergever}  -a temp.${jchk_host} $jchk_filename
         else
            if [ "x${jchk_host}" != "xjhsllc}" ] && [ -s temp.${jchk_host} ] ; then
               # Date new date, same host ???? 
               cat temp.${jchk_host} | egrep -v "^PCPU|^SCPU${net_remove}" | uniq >> mpgdata2.${jchk_host}
               cp $jchk_filename temp.${jchk_host} 
            fi 
         fi 
      fi
      gzip ${jchk_filename}
      jlasthost=${jchk_host}
      jlastdate=${jchk_date}
   done < files2process.tmp 


   #### set > set.hsllc ; echo "Remove later" 
   if [ "x${jlasthost}" != "xjhsllc}" ] && [ -s temp.${jlasthost} ] ; then
      cat temp.${jlasthost} | egrep -v "^PCPU|^SCPU${net_remove}" | uniq >> mpgdata2.${jlasthost}
      rm temp.${jlasthost} 
      # ls -al mpgdata2.${jlasthost} 
      # Test mpgdata2.hostname file for dates being equal,
      # if not run ./Add_headers_each_day_mpgd2.sh against it.
      z_date=`grep ZZZZ mpgdata2.${jlasthost} | cut -d, -f4 | uniq -c | wc -l`
      comma_date=`grep ',date,' mpgdata2.${jlasthost} | uniq -c | wc -l`
      if [ ${z_date} -ne ${comma_date} ]; then
         mkdir -p hold
         mv mpgdata2.${jlasthost} hold
         ./Add_headers_each_day_mpgd2.sh
         # mv hold/mpgdata2.${jlasthost} .
         rm hold/mpgdata2.${jlasthost} 
      fi 
      # End of test mpgdata2.hostname file for dates being equal
      ./pnbrdf.sh mpgdata2.${jlasthost}
      gzip -f mpgdata2.${jlasthost}
   fi

   # Create local_rack directory for local running of rackdata.sh
   mkdir -p local_rack
   cd local_rack
   
   cp -p ../rackdata.sh . 
   ./rackdata.sh 
   # Return to original directory after 
   cd .. 

   files2tar1=`ls -1 mpgdata2.* rackdata.* | grep -v rackdata.sh | tee files2tar.list`
   echo local_rack >> files2tar.list 
   files2tar="${files2tar1} local_rack"
   ulimit -f unlimited # Necessary to not run into file size limits that default on root.
   tar -cf pninfo_multiple_${jsysname}.tar -L files2tar.list 
   gzip -f pninfo_multiple_${jsysname}.tar
   echo "#"
   ls -al pninfo_multiple_${jsysname}.tar.gz | sed s,\ \ *,\ ,g | awk '{print "#",$0}'
   echo "#"
   echo "# Please send file pninfo_multiple_${jsysname}.tar.gz to MPG of HelpSystems, LLC."
   #& Please send file pninfo_multiple_${jsysname}.tar.gz to MPG of HelpSystems, LLC. 
   wc -l good_files.tmp bad_files.tmp
else
   echo "### Error ### files2process.tmp does not exist #"
fi
### cat temp.${jchk_host} | uniq >> mpgdata2.${jchk_host}

# gunzip any nmon files that were not gzipped in the beginning.
# Returning them to gunzipped status.
if [ -s /tmp/helpsystems_tmp/filesnotgzipped.tmp ] ; then
   echo "# # In order To (gunzip) any nmon files that started not gzipped (.gz)."
   echo "# Run --- ( cat /tmp/helpsystems_tmp/filesnotgzipped.tmp | xargs gunzip )"
fi
jdate=`date '+%y%m%d_%H%M'`
echo "# Done (${jdate})"
# For final remove temporary files except when debugging
### rm field_cnt_chk.tmp files2process.tmp filesonly.tmp gzippedfiles.tmp

#& # Sample run with all nmon data files gzipped
#& # =================================================#
#& # Running (YYMMDD_hhmm) pnbmpgd2gzip.sh,v 1.1 2013/12/17 17:38:05
#& # # All files are gzipped from the beginning.
#& # Making a list of gzipped files into gzippedfiles.tmp
#& # # gzipping (gzip) any nmon files that are not already gzipped.
#& # # They are listed in (/tmp/helpsystems_tmp/filesnotgzipped.tmp)
#& # Making a list of files into files2process.tmp
#& # Will be processing (960) files for (2) servers.
#& # No filenames had mutiple hosts internally, continuing
#& # Field counts are correct in files2process.tmp, continuing
#& # =================================================#
#& # Processing files in files2process.tmp
#& # =================================================#
#& # Processing server_one
#& # Processing server_two
#& #     rackdata.server_one was created from mpgdata2.server_one
#& #     rackdata.server_two was created from mpgdata2.server_two
#& #
#& # -rw-rw---- 1 root system 18434556 Dec 17 10:20 pninfo_multiple_yourhostname.tar.gz
#& #
#& # # In order To (gunzip) any nmon files that started not gzipped (.gz).
#& # Run --- ( cat /tmp/helpsystems_tmp/filesnotgzipped.tmp | xargs gunzip )
#& # Please send file pninfo_multiple_processingserver.tar.gz to MPG of HelpSystems, LLC.
#& # # In order To (gunzip) any nmon files that started not gzipped (.gz).
#& # Run --- ( cat /tmp/helpsystems_tmp/filesnotgzipped.tmp | xargs gunzip )
#& #
#& # Please send file pninfo_multiple_AIX720.tar.gz to MPG of HelpSystems, LLC.
#& #      960 good_files.tmp
#& #        0 bad_files.tmp
#& #      960 total
#& # # Done (YYMMDD_hhmm) 
#& # =================================================#
#& # In addition, possible errors or messages.
#& # =================================================#
#& ### Error ### You cannot run from (/usr/local/mpg)
#& #
#& # = = = = = = = = = = = = = = = = = = = = = = = = =#
#& ### Error ### You cannot run from the root file sytem (/)
#& #
#& # = = = = = = = = = = = = = = = = = = = = = = = = =#
#& ### Error ### You cannot run from install directory (yourinstalldir)
#& #
#& # = = = = = = = = = = = = = = = = = = = = = = = = =#
#& ### Error ### There should not be any mpgdata2 files when starting, correct!
#& #
#& # = = = = = = = = = = = = = = = = = = = = = = = = =#
#& ### Error ### needs to be ran on (AIX powerpc)
#& # This is on (Linux ppc64)
#& #
#& # = = = = = = = = = = = = = = = = = = = = = = = = =#
#& ### Error ### The executable nmonmerge_systemtype is required in this directory!
#& #
#& # = = = = = = = = = = = = = = = = = = = = = = = = =#
#& ### Error ### The executable pnbrdf.sh is required in this directory!
#& #
#& # = = = = = = = = = = = = = = = = = = = = = = = = =#
#& #### Error #### Has more or less than one host listed, not processing files
#& #### Review the data listed below, being placed into bad files
#& # filename
#& #
#& # = = = = = = = = = = = = = = = = = = = = = = = = =#
#& #### Error #### Field count should be 4 and is not correct for these
#& #### Review the data listed below, being placed into bad files
#& # filename
#& #
#& # = = = = = = = = = = = = = = = = = = = = = = = = =#
#& ### Error ### files2process.tmp does not exist #
#& #
#& # = = = = = = = = = = = = = = = = = = = = = = = = =#
#& # # gzipping (gzip) any nmon files that are not already gzipped.
#& # # They are listed in (filesnotgzipped.tmp)
#& #
#& # =================================================#

