#!/bin/ksh
#!/bin/ksh -xv
#
# $Id: mpgd3_arcgztrim.sh,v 1.1 2023/03/27 16:07:05 root Exp $
#
# The following code is Confidential and is covered by the installation license
# (c) Copyright Fortra, LLC. and its group of companies.
#
#& Trims mpgdata3.hostname_archive.tar.gz if it has more than 
#& remove_old_arc_data lines.
#& 
#&& mpgdata3.hostname_archive.tar.gz must exist.
#&& mpgdata3.hostname_archive.tar.gz more records than remove_old_arc_data?
#&& Is there enough space to create the new mpgdata3.hostname_archive.tar.gz?
#
#&@ gunzip
#&@ sysname needs to come from pn.config or "-s name" option
#
#&% 
#&% Will be ran monthly when pndchk.sh runs 
#&% Trim will happen automatically without prompting. 
#

invocdir=`dirname $0`

if [ -s $invocdir/pn.config ] ; then
   . $invocdir/pn.config
else
   echo "pn.config file does not exist at `pwd`"
fi

function usage {
echo "usage: $0 [options]
  where options may be any of:
        -a Automatically replace new file over old file [ -a ]
        -s Specify system name (sysname=opt_sysname) [ -s name ]
        -r Specify number of days to retain of archive [ -r # ]
           (remove_old_arc_data=opt_remove_old_arc_data)
        -h Display options [ -h ]
"
}

# The colon (:) is how you indicate and option is required.
while getopts "ahs:r:" opt ; do
   case $opt in
        a)      autoreplace="T" ;;
        s)      opt_sysname=$OPTARG ;;
        r)      opt_remove_old_arc_data=$OPTARG ;;
        h)      usage;  exit ;;
        ?)      usage;  exit ;;
   esac
done

if [ ! -z "$opt_remove_old_arc_data" ] ; then remove_old_arc_data=$opt_remove_old_arc_data; fi

if [ ! -z "$opt_sysname" ] ; then sysname=$opt_sysname; fi

# echo "Remove Later ($remove_old_arc_data)($sysname)"; exit

if [ "xx" = "x${sysname}x" ]; then
   sysname=`hostname`
fi

# If remove_old_arc_data blank set to 366
if [ "x${remove_old_arc_data}x" = "xx" ]; then
   remove_old_arc_data="366"
fi

# If remove_old_arc_data blank less than 180, set to 180. 
if [ ${remove_old_arc_data} -lt 180 ]; then
   echo "(${remove_old_arc_data}) is too low setting to 180"
   remove_old_arc_data="180"
fi

# echo "sysname is (${sysname}). Trimming to (${remove_old_arc_data}) days." 
if [ -s ./mpgdata3.${sysname}_archive.gz ];then
   # ./mpgdata3.${sysname}_archive.gz exists and has space.
   echo "./mpgdata3.${sysname}_archive.gz exists, continuing"
else
   # ./mpgdata3.${sysname}_archive.gz does not exist or is zero bytes. 
   echo "./mpgdata3.${sysname}_archive.gz missing, exiting"
   exit
fi

jspaceavailk=`df -k . | tail -1 | awk '{print $3}'`
echo "Space available is (${jspaceavailk} k bytes)"
jmpgd3arch_filesize=`du -k ./mpgdata3.${sysname}_archive.gz | awk '{print $1}'`
# echo "Size of current ./mpgdata3.${sysname}_archive.gz is (${jmpgd3arch_filesize} k bytes)"
if [ ${jspaceavailk} -ge ${jmpgd3arch_filesize} ]; then
   echo "(${jspaceavailk}) is greater than or equal to (${jmpgd3arch_filesize})" 
else
   echo "(${jspaceavailk}) is smaller than (${jmpgd3arch_filesize})" 
fi

# Determine record (day) count of current file
# echo "Creating,mpgd3prognames.tmp"
gunzip -c ./mpgdata3.${sysname}_archive.gz | egrep -n "^MPG,host,|^MPG,time,|^MPG,date," | grep -v "^PS," |  paste - - -  > mpgd3prognames.tmp 
jmpgd3reccnt=`wc -l mpgd3prognames.tmp | awk '{print $1}'`

# echo "${jmpgd3reccnt}"

echo "./mpgdata3.${sysname}_archive.gz is (${jmpgd3reccnt}) days! Reducing to (${remove_old_arc_data}) days!"

# Determine first line of new file
jfirstnewline=`cat mpgd3prognames.tmp | tail -${remove_old_arc_data} |  head -1 | awk -F: '{print $1}'`
# echo "Line number of first line (${jfirstnewline}) for new mpgdata3 archive file."

# Determine size of old and new files
old_mpg3_arch_file_size=`wc -c ./mpgdata3.${sysname}_archive.gz | awk '{print $1}'`

new_mpg3_arch_file_size=`gunzip -c ./mpgdata3.${sysname}_archive.gz | awk -v jbegin=${jfirstnewline} '{if( NR >= jbegin ) print $0}' | gzip -c | wc -c | awk '{print $1}'`

echo "Old file size (${old_mpg3_arch_file_size})  New file size (${new_mpg3_arch_file_size})"

# echo "Remove later" ;  exit 

# Does the file system have enough space to build the new mpgdata3.hostname_archive.gz
jspaceavail=`echo "${jspaceavailk}" | awk '{printf "%5.0f\n", $1 * 1024}'`
# echo "(${jspaceavail}) (${new_mpg3_arch_file_size}) "  

# echo "" | awk -v javailk="${jspaceavailk}" -v jmpgd3fsize="${new_mpg3_arch_file_size}" '{print "test 1", javailk,"test 2", jmpgd3fsize/1024}'

jokbuild=`echo "${jspaceavailk} ${new_mpg3_arch_file_size}" | awk '{if($2/1024 < $1 ) print "Ok"}'`

### jokbuild="notOk"

if [ "x${jokbuild}x" = "xOkx" ]; then
   # Build new mpgd3arch file
   gunzip -c ./mpgdata3.${sysname}_archive.gz | awk -v jbegin=${jfirstnewline} '{if( NR >= jbegin ) print $0}' | gzip -c > ./mpgdata3.${sysname}_arch_new.gz   

   if [ "x${autoreplace}x" != "xTx" ]; then
   # Size of both files.
      ls -altr ./mpgdata3.${sysname}_archive.gz ./mpgdata3.${sysname}_arch_new.gz
   else
      echo "Autoreplacing ./mpgdata3.${sysname} with ./mpgdata3.${sysname}_new"
      cp ./mpgdata3.${sysname}_arch_new.gz ./mpgdata3.${sysname}_archive.gz
      rm -f ./mpgdata3.${sysname}_arch_new.gz
   fi

else
   # Not enough space to build new mpgd3arch file
   echo "Not enough space to build ./mpgdata3.${sysname}_arch_new.gz"
fi

#& #########################
#& Sample run
#& #########################
#& 
#& ./mpgd3_arcgztrim.sh
#& pn.config file does not exist at /nmondata/arctrim_test
#& ./mpgdata3.AIX720_archive.gz exists, continuing
#& Space available is (194794264 k bytes)
#& (194794264) is greater than or equal to (11648)
#& ./mpgdata3.AIX720_archive.gz is (2723) days! Reducing to (366) days!
#& Old file size (11925638)  New file size (1199537)
#& -rw-rw----    1 root     system     11925638 Oct 05 10:49 ./mpgdata3.AIX720_archive.gz
#& -rw-rw----    1 root     system      1199537 Oct 12 14:34 ./mpgdata3.AIX720_arch_new.gz

