#!/bin/ksh

# $Id: change_pn.sh,v 1.1 2023/03/27 16:07:05 root Exp $

# The following code is Confidential and is covered by the installation license
# (c) Copyright Fortra, LLC. and its group of companies.

#&& change_pn.sh is used to make changes to the pn.config file

#&& It can make several changes at the same time.

#&% Will make a back up copy of the pn.config file to pn.config.date_time

mkdir -p /tmp/helpsystems_tmp

# jdate is used later in the script to mark date_time on file
jdate=`date '+%Y%m%d_%H%M'`  ### timestamp variable

echo "Running `grep "\$\I\d\:" ${0} | awk '{print $3,$4,$5,$6}' ` "

# debug="T"

invocdir=`dirname $0`
if [ -s $invocdir/pndcutils.sh ]; then
   . $invocdir/pndcutils.sh
fi

if [ -s $invocdir/pn.config ];  then
   . $invocdir/pn.config
fi

# Three new environmental variables
# installdir
# mpgdatadir
# nmondatadir

# installdir should be defined, but if not use invocdir
if [ -z "$installdir" ] ; then
   installdir=$invocdir
fi

# mpgdatadir should be defined, but if not use invocdir
if [ -z "$mpgdatadir" ] ; then
   mpgdatadir=$invocdir
fi

# nmondatadir should be defined, but if not use invocdir
if [ -z "$nmondatadir" ] ; then
   nmondatadir=$invocdir
fi

# usage routine #
usage () {
echo " Used to make changes in the pn.config file quicker

USAGE: change_pn.sh [options], where options may be any of:
-p                          previous configuration before last install
			    Note: -p ignores any other changes specified
-s                          Display (s)tatus of nmon install
-O <nmonopts>               Nmon options in quotes
-n <auto_start_nmon>        Automatically start nmon T/F (T)
-G <gen_dat_prot_reg>       General Data Protection Regulation T/F (T)
-e <ext_data_collect>       External data collect (T/F)
-a <admin_data_collect>     Admin data collect (T/F)
-S                          Auto Set transferfiles to (cp/scp/ftp/rcp/pull)
                            I.E.  -S scp
-T <transfer_script_cmd>    Transfer script command in quotes
-C <transfer_current_cmd>   Transfer current command in quotes
-d <truncate_days>          Truncate days for mpgdata2 file to archive (def 366)
-E <ext_data_truncate_days> Truncate days for mpgdata3 file to archive (def 90)
-r <remove_old_arc_data>    Remove old archive data mpgdata2/mpgdata3/rdhist (def 366) 
-P <purge_top_days>         Purge TOP data from mpgdata2 file past days(def 730)
-K <nmondaystokeep>         Nmon daily files to keep (def 7)
-R <Replicate>              Makes script to replicate values on other servers
-D                          debug on
-v                          verbose on
-h                          help (display this)
# --- External Process Collection (defaults) mpgdata5 --- #
-X <ext_ps_collect>         External ps collect normal[PSN]&kernel[PSK] T/F (T)
-x <ext_ps_days>            External ps collect days (90)
# --- Admin Data Collection (defaults) mpgdata4 --- #
-i <include_dir_scans>      Include dir scans in admin data collect T/F (F)
-L <TopDFCnt>               Nmbr of Largest Dir File Counts per file system (20)
-l <TopDSpc> (lowercase L)  Nmbr of Largest Dir Space per file system (20)
-B <TopBigFile>             Number of Largest Files per file system (20)
-F <MinFSz>                 Minimum File System Size (in bytes) (50000000)
-f <MinFCnt>                Minimum Directory File Count (100)
-m <MinDSpc>                Minimum Directory Space (Kbytes) (10000)
# --- External Data Collection (defaults)  mpgdata3 --- #
-A <length_args>            Length of Arguments (400)
-M <percent_mem>            Percent of Memory (5)
-c <percent_cpu>            Percent of CPU (5)
-V <vsz_threshold>          Virtual size Threshold (60000)
-I <page_in>                Page In (9)
-t <thread_cnt>             Thread count (100)"
}

# change a line based on option to the left of the equal 
change_line () {
# option 1 is name of option
# option 2 is value of option
# option 3 is name of the file to change, in this case pn.config
# If debug is T then display command line options
if [ "x${debug}x" = "xTx" ]; then
   echo ""
   echo "#### change_line options set ($1) ($2) ($3) ####" 
fi
# If verbose is T then display before and after
if [ "x${verbose}x" = "xTx" ]; then
   echo ""
   awk -F= -v jvalue="$1" -v new_setting="$2" '{
      if ($1 == jvalue) {
        print "Before", $0;
        print "After", jvalue"=""\042"new_setting"\042";
      }
   }' $3 
fi
# Actual fix portion into pn.config.jtmp
awk -F= -v jvalue="$1" -v new_setting="$2" '{
   if ($1 == jvalue) {
     print jvalue"=""\042"new_setting"\042";
 } else {
     print $0;}
 }' $3 > $3.jtmp 
# This will move pn.config.jtmp to pn.config 
mv $3.jtmp $3
}

make_backup () {
if [ "x${verbose}x" = "xTx" ]; then
   echo ""
   echo "#### Copy the pn.config file to pn.config.${jdate} ####"
fi
 
echo "Backing up the pn.config file to pn.config.${jdate} "
cp pn.config pn.config.${jdate}
}


# Return to config prior to last install
return_to_prior_config () {
   # 
   # Use last_pnconfig_install_change.hsllc to return to post install 
   # configuration changes.
   #
   echo "Returning to pn.config values prior to upgrade installation" 
   grep Previous last_pnconfig_install_change.hsllc  | cut -d" " -f3- > /tmp/helpsystems_tmp/change_pn.tmp
   while
   read line
   do
      jitem=`echo $line|cut -d= -f1 | sed s,\",,g `
      jval=`echo $line|cut -d= -f2 | sed s,\",,g `
      change_line "${jitem}" "${jval}" "pn.config"   
done < /tmp/helpsystems_tmp/change_pn.tmp
} 


# If no options are specified, usage and exit
if [ "x${1}x" = "xx" ]; then
  usage; exit
fi

# Get the options specied in the command line
while getopts "pO:n:G:e:a:i:S:T:C:d:E:P:K:X:x:A:M:c:V:I:t:L:l:B:F:f:m:RDvhs" opt ; do
  case $opt in
	  p)      return_prior="T";; 
	  s)      status_check="T";; 
	  O)      opt_nmonopts="$OPTARG"; 
		  fix_nmonopts="T";; 
	  n)      opt_auto_start_nmon="$OPTARG"; 
		  fix_auto_start_nmon="T";; 
	  G)      opt_gen_dat_prot_reg="$OPTARG"; 
		  fix_opt_gen_dat_prot_reg="T";; 
	  e)      opt_ext_data_collect="$OPTARG" ;
		  fix_ext_data_collect="T" ;;
	  a)      opt_admin_data_collect="$OPTARG" ;
		  fix_admin_data_collect="T" ;;
	  i)      opt_include_dir_scans="$OPTARG" ;
		  fix_include_dir_scans="T" ;;
	  S)      jttype="$OPTARG" ;
		  if [ "x${jttype}" != "xftp" ] && [ "x${jttype}" != "xscp" ] && [ "x${jttype}" != "xrcp" ] && [ "x${jttype}" != "xcp" ] && [ "x${jttype}" != "xpull" ] 
		  then 
		     usage
		     exit
		  fi   ;
		  fix_transfer_script_cmd="T" ;
	          opt_transfer_script_cmd="${installdir}/transfer_script.sh -c ${installdir}/transfer_${jttype}.config" ;
		  fix_transfer_current_cmd="T" ;
	          opt_transfer_current_cmd="${installdir}/transfer_current.sh -c ${installdir}/transfer_${jttype}.config" ;;
	  T)      opt_transfer_script_cmd="$OPTARG" ;
		  fix_transfer_script_cmd="T" ;;
	  C)      opt_transfer_current_cmd="$OPTARG" ;
		  fix_transfer_current_cmd="T" ;;
	  d)      opt_truncate_days="$OPTARG" ;
		  fix_truncate_days="T" ;;
	  E)      opt_ext_data_truncate_days="$OPTARG" ;
		  fix_ext_data_truncate_days="T" ;;
	  P)      opt_purge_top_days="$OPTARG" ;
		  fix_purge_top_days="T" ;;
	  K)      opt_nmondaystokeep="$OPTARG" ;
		  fix_nmondaystokeep="T" ;;
	  X)      opt_ext_ps_collect="$OPTARG" ;
		  fix_ext_ps_collect="T" ;;
	  x)      opt_ext_ps_days="$OPTARG" ;
		  fix_ext_ps_days="T" ;;
	  A)      opt_length_args="$OPTARG" ;
		  fix_length_args="T" ;;
	  M)      opt_percent_mem="$OPTARG" ;
		  fix_percent_mem="T" ;;
	  c)      opt_percent_cpu="$OPTARG" ;
		  fix_percent_cpu="T" ;;
	  V)      opt_vsz_threshold="$OPTARG" ;
		  fix_vsz_threshold="T" ;;
	  I)      opt_page_in="$OPTARG" ;
		  fix_page_in="T" ;;
	  t)      opt_thread_cnt="$OPTARG" ;
		  fix_thread_cnt="T" ;;
	  L)      opt_TopDFCnt="$OPTARG" ;
		  fix_TopDFCnt="T" ;;
	  l)      opt_TopDSpc="$OPTARG" ;
		  fix_TopDSpc="T" ;;
	  B)      opt_TopBigFile="$OPTARG" ;
		  fix_TopBigFile="T" ;;
	  F)      opt_MinFSz="$OPTARG" ;
		  fix_MinFSz="T" ;;
	  f)      opt_MinFCnt="$OPTARG" ;
		  fix_MinFCnt="T" ;;
	  m)      opt_MinDSpc="$OPTARG" ;
		  fix_MinDSpc="T" ;;
	  R)      create_replicate_script="T" ;;
	  r)      opt_remove_old_arc_data="$OPTARG" ;
		  fix_remove_old_arc_data="T" ;;
	  D)      debug="T" ;;
	  v)      verbose="T" ;;
	  h)      usage; exit ;;
	  ?)      usage; exit ;;
	  *)      usage; exit ;;
  esac
done

# main

# Puts values to /tmp/helpsystems_tmp/change_pn_sh_set.hsllc
if [ "x${debug}x" = "xTx" ]; then
   if [ "x${verbose}x" = "xTx" ]; then
      echo ""
      echo "#### putting 'set' to /tmp/helpsystems_tmp/change_pn_sh_set.hsllc ####"
   fi
   set > /tmp/helpsystems_tmp/change_pn_sh_set.hsllc
fi

if [ "x${debug}x" = "xTx" ]; then
   if [ "x${verbose}x" = "xTx" ]; then
      echo ""
      echo "#### Displaying fix_ and opt_ variables  ####"
   fi
   set | egrep "fix_|opt_"
fi

display_perf_nav_status () {
   echo "Performance Navigator Install/Data Status"
   echo "==================================="
   echo "Entries in crontab:"
   crontab -l | egrep -i "mpg|start_nmon_now|pn.config|pndchk.sh|pngetinfo.sh"
   echo "==================================="
   cd $mpgdatadir
   jdaycnt=`grep date, mpgdata2.${sysname} mpgdata.${sysname} 2>/dev/null | uniq | wc -l | awk '{print $1}'` 
   echo "Days of nmon data (${jdaycnt}):" 
   grep date, mpgdata2.${sysname} mpgdata.${sysname} 2>/dev/null | head -1 | awk '{print $0,"(to)"}'
   grep date, mpgdata2.${sysname} mpgdata.${sysname} 2>/dev/null | tail -1
   echo "==================================="
   echo "Displayed nmon process:"
   ps -ef | grep nmon | grep -v grep  
   echo "==================================="
   cd $nmondatadir
   ls -latr  mpg*nmon*.gz 2>/dev/null > /tmp/helpsystems_tmp/Existing_nmon_files.tmp
   ls -latr  mpg*nmon 2>/dev/null  >>  /tmp/helpsystems_tmp/Existing_nmon_files.tmp
   jcntnmon=`wc -l /tmp/helpsystems_tmp/Existing_nmon_files.tmp | awk '{print $1}'` 
   echo "Existing nmon files (${jcntnmon}):"
   cat /tmp/helpsystems_tmp/Existing_nmon_files.tmp
   rm -f /tmp/helpsystems_tmp/Existing_nmon_files.tmp 
   exit
}

build_replicate_script () {
   # This creates a temporary script that calls change_pn.sh to allow
   # (R)eplication of settings across multiple servers without requiring
   # manual edits of the pn.config file.

   . ./pn.config
   
   if [ "x${nmonopts}x" != "xx" ]; then
      jnmonopts=" -O \"${nmonopts}\""
   fi
   
   if [ "x${ext_data_collect}x" != "xx" ]; then
      jext_data_collect=" -e \"${ext_data_collect}\""
   fi

   if [ "x${auto_start_nmon}x" != "xx" ]; then
      jauto_start_nmon="-n \"${auto_start_nmon}\""
   fi
   
   if [ "x${gen_dat_prot_reg}x" != "xx" ]; then
      jgen_dat_prot_reg="-G \"${gen_dat_prot_reg}\""
   fi
   
   if [ "x${admin_data_collect}x" != "xx" ]; then
      jadmin_data_collect="-a \"${admin_data_collect}\""
   fi
   
   if [ "x${include_dir_scans}x" != "xx" ]; then
      jinclude_dir_scans="-i \"${include_dir_scans}\""
   fi
   
   if [ "x${transfer_script_cmd}x" != "xx" ]; then
      jtransfer_script_cmd="-T \"${transfer_script_cmd}\""
   fi

   if [ "x${transfer_current_cmd}x" != "xx" ]; then
      jtransfer_current_cmd="-C \"${transfer_current_cmd}\""
   fi

   if [ "x${truncate_days}x" != "xx" ]; then
      jtruncate_days="-d \"${truncate_days}\""
   fi

   if [ "x${ext_data_truncate_days}x" != "xx" ]; then
      jext_data_truncate_days="-E \"${ext_data_truncate_days}\""
   fi
   
   if [ "x${ext_remove_old_arc_data}x" != "xx" ]; then
      jext_remove_old_arc_data="-r \"${ext_remove_old_arc_data}\""
   fi
   
   if [ "x${purge_top_days}x" != "xx" ]; then
      jpurge_top_days="-P \"${purge_top_days}\""
   fi

   if [ "x${nmondaystokeep}x" != "xx" ]; then
      jnmondaystokeep="-K \"${nmondaystokeep}\""
   fi

   if [ "x${ext_ps_collect}x" != "xx" ]; then
      jext_ps_collect="-X \"${ext_ps_collect}\""
   fi

   if [ "x${ext_ps_days}x" != "xx" ]; then
      jext_ps_days="-x \"${ext_ps_days}\""
   fi

   if [ "x${length_args}x" != "xx" ]; then
      jlength_args="-A \"${length_args}\""
   fi

   if [ "x${percent_mem}x" != "xx" ]; then
      jpercent_mem="-M \"${percent_mem}\""
   fi

   if [ "x${percent_cpu}x" != "xx" ]; then
      jpercent_cpu="-c \"${percent_cpu}\""
   fi

   if [ "x${vsz_threshold}x" != "xx" ]; then
      jvsz_threshold="-V \"${vsz_threshold}\""
   fi

   if [ "x${page_in}x" != "xx" ]; then
      jpage_in="-I \"${page_in}\""
   fi

   if [ "x${thread_cnt}x" != "xx" ]; then
      jthread_cnt="-t \"${thread_cnt}\""
   fi

   if [ "x${TopDFCnt}x" != "xx" ]; then
      jTopDFCnt="-L \"${TopDFCnt}\""
   fi

   if [ "x${TopDSpc}x" != "xx" ]; then
      jTopDSpc="-l \"${TopDSpc}\""
   fi

   if [ "x${TopBigFile}x" != "xx" ]; then
      jTopBigFile="-B \"${TopBigFile}\""
   fi

   if [ "x${MinFSz}x" != "xx" ]; then
      jMinFSz="-F \"${MinFSz}\""
   fi

   if [ "x${MinFCnt}x" != "xx" ]; then
      jMinFCnt="-f \"${MinFCnt}\""
   fi

   if [ "x${MinDSpc}x" != "xx" ]; then
      jMinDSpc="-m \"${MinDSpc}\""
   fi

   # First remove prior temporary scripts
   rm -f /tmp/helpsystems_tmp/pn_change-R*

   echo "Creating script /tmp/helpsystems_tmp/pn_change-R_${jdate}.tmp" 
   echo "Place in /tmp on other servers and run:" # helpsystems 
   echo ""
   echo ".  /tmp/pn_change-R_${jdate}.tmp"  # helpsystems 
   echo ""
   
   echo "# This is a temporary script created with by the "-R" option in change_pn.sh to" >  /tmp/helpsystems_tmp/pn_change-R_${jdate}.tmp
   echo "# allow replication of pn.config settings to mulitple servers." >> /tmp/helpsystems_tmp/pn_change-R_${jdate}.tmp

   echo "./change_pn.sh -v ${jnmonopts} ${jext_data_collect} ${jauto_start_nmon} ${jgen_dat_prot_reg} ${jadmin_data_collect} ${jtransfer_script_cmd} ${jtransfer_current_cmd} ${jtruncate_days} ${jext_data_truncate_days} ${jext_remove_old_arc_data} ${jpurge_top_days} ${jnmondaystokeep} ${jext_ps_collect} ${jext_ps_days} ${jinclude_dir_scans} ${jlength_args} ${jpercent_mem} ${jpercent_cpu} ${jvsz_threshold} ${jpage_in} ${jthread_cnt} ${jTopDFCnt} ${jTopDSpc} ${jTopBigFile} ${jMinFSz} ${jMinFCnt} ${jMinDSpc} " >>  /tmp/helpsystems_tmp/pn_change-R_${jdate}.tmp
   ls -al  /tmp/helpsystems_tmp/pn_change-R_${jdate}.tmp

} 

cmd_line_chgs () {
   if [ "x${verbose}x" = "xTx" ]; then
      echo ""
      echo "#### Run change_line on each set of values specified on Command Line. ####"
   fi

   if [ "x${opt_nmonopts}" != "x" ] && [ "x${fix_nmonopts}" = "xT" ] ; then
      change_line "nmonopts" "${opt_nmonopts}" "pn.config"
   fi
   
   if [ "x${opt_ext_data_collect}" != "x" ] && [ "x${fix_ext_data_collect}" = "xT" ] ; then
      change_line "ext_data_collect" "${opt_ext_data_collect}" "pn.config"
   fi
   
   if [ "x${opt_auto_start_nmon}" != "x" ] && [ "x${fix_auto_start_nmon}" = "xT" ] ; then
      change_line "auto_start_nmon" "${opt_auto_start_nmon}" "pn.config"
   fi

   if [ "x${opt_gen_dat_prot_reg}" != "x" ] && [ "x${fix_gen_dat_prot_reg}" = "xT" ] ; then
      change_line "gen_dat_prot_reg" "${opt_gen_dat_prot_reg}" "pn.config"
   fi

   if [ "x${opt_admin_data_collect}" != "x" ] && [ "x${fix_admin_data_collect}" = "xT" ] ; then
      change_line "admin_data_collect" "${opt_admin_data_collect}" "pn.config"
   fi

   if [ "x${opt_transfer_script_cmd}" != "x" ] && [ "x${fix_transfer_script_cmd}" = "xT" ] ; then
      change_line "transfer_script_cmd" "${opt_transfer_script_cmd}" "pn.config"
   fi
   
   if [ "x${opt_transfer_current_cmd}" != "x" ] && [ "x${fix_transfer_current_cmd}" = "xT" ] ; then
      change_line "transfer_current_cmd" "${opt_transfer_current_cmd}" "pn.config"
   fi

   if [ "x${opt_truncate_days}" != "x" ] && [ "x${fix_truncate_days}" = "xT" ] ; then
      change_line "truncate_days" "${opt_truncate_days}" "pn.config"
   fi

   if [ "x${opt_ext_data_truncate_days}" != "x" ] && [ "x${fix_ext_data_truncate_days}" = "xT" ] ; then
      change_line "ext_data_truncate_days" "${opt_ext_data_truncate_days}" "pn.config"
   fi

   if [ "x${opt_remove_old_arc_data}" != "x" ] && [ "x${fix_remove_old_arc_data}" = "xT" ] ; then
      change_line "remove_old_arc_data" "${opt_remove_old_arc_data}" "pn.config"
   fi

   if [ "x${opt_purge_top_days}" != "x" ] && [ "x${fix_purge_top_days}" = "xT" ] ; then
      change_line "purge_top_days" "${opt_purge_top_days}" "pn.config"
   fi

   if [ "x${opt_nmondaystokeep}" != "x" ] && [ "x${fix_nmondaystokeep}" = "xT" ] ; then
      change_line "nmondaystokeep" "${opt_nmondaystokeep}" "pn.config"
   fi

   if [ "x${opt_ext_ps_collect}" != "x" ] && [ "x${fix_ext_ps_collect}" = "xT" ] ; then
      change_line "ext_ps_collect" "${opt_ext_ps_collect}" "pn.config"
   fi

   if [ "x${opt_ext_ps_days}" != "x" ] && [ "x${fix_ext_ps_days}" = "xT" ] ; then
      change_line "ext_ps_days" "${opt_ext_ps_days}" "pn.config"
   fi

   if [ "x${opt_include_dir_scans}" != "x" ] && [ "x${fix_include_dir_scans}" = "xT" ] ; then
      change_line "include_dir_scans" "${opt_include_dir_scans}" "pn.config"
   fi

   if [ "x${opt_length_args}" != "x" ] && [ "x${fix_length_args}" = "xT" ] ; then
      change_line "length_args" "${opt_length_args}" "pn.config"
   fi

   if [ "x${opt_percent_mem}" != "x" ] && [ "x${fix_percent_mem}" = "xT" ] ; then
      change_line "percent_mem" "${opt_percent_mem}" "pn.config"
   fi

   if [ "x${opt_percent_cpu}" != "x" ] && [ "x${fix_percent_cpu}" = "xT" ] ; then
      change_line "percent_cpu" "${opt_percent_cpu}" "pn.config"
   fi

   if [ "x${opt_vsz_threshold}" != "x" ] && [ "x${fix_vsz_threshold}" = "xT" ] ; then
      change_line "vsz_threshold" "${opt_vsz_threshold}" "pn.config"
   fi

   if [ "x${opt_page_in}" != "x" ] && [ "x${fix_page_in}" = "xT" ] ; then
      change_line "page_in" "${opt_page_in}" "pn.config"
   fi

   if [ "x${opt_thread_cnt}" != "x" ] && [ "x${fix_thread_cnt}" = "xT" ] ; then
      change_line "thread_cnt" "${opt_thread_cnt}" "pn.config"
   fi

   if [ "x${opt_TopDFCnt}" != "x" ] && [ "x${fix_TopDFCnt}" = "xT" ] ; then
      change_line "TopDFCnt" "${opt_TopDFCnt}" "pn.config"
   fi

   if [ "x${opt_TopDSpc}" != "x" ] && [ "x${fix_TopDSpc}" = "xT" ] ; then
      change_line "TopDSpc" "${opt_TopDSpc}" "pn.config"
   fi

   if [ "x${opt_TopBigFile}" != "x" ] && [ "x${fix_TopBigFile}" = "xT" ] ; then
      change_line "TopBigFile" "${opt_TopBigFile}" "pn.config"
   fi

   if [ "x${opt_MinFSz}" != "x" ] && [ "x${fix_MinFSz}" = "xT" ] ; then
      change_line "MinFSz" "${opt_MinFSz}" "pn.config"
   fi

   if [ "x${opt_MinFCnt}" != "x" ] && [ "x${fix_MinFCnt}" = "xT" ] ; then
      change_line "MinFCnt" "${opt_MinFCnt}" "pn.config"
   fi

   if [ "x${opt_MinDSpc}" != "x" ] && [ "x${fix_MinDSpc}" = "xT" ] ; then
      change_line "MinDSpc" "${opt_MinDSpc}" "pn.config"
   fi
}

if [ "${status_check}x" = "Tx" ]; then
   display_perf_nav_status
fi


make_backup

#  
if [ "${return_prior}x" = "Tx" ]; then
   return_to_prior_config 
   echo "Returned to previous configuration in pn.config!"
elif [ "${create_replicate_script}x" = "Tx" ]; then
   build_replicate_script
else
   cmd_line_chgs
   echo "All specified changes were made!"
fi


