/*******************************************************************************

Copyright (c) 2001-2006, Intel Corporation 
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright 
    notice, this list of conditions and the following disclaimer in the 
    documentation and/or other materials provided with the distribution.

 3. Neither the name of the Intel Corporation nor the names of its 
    contributors may be used to endorse or promote products derived from 
    this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*******************************************************************************/

#include "AsfAgentApi.h"
#include "UICommon.h"
#include <iomanip.h>
#include <iostream.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "common_std.h"


#define MAX_HB_INTERVAL  594
#define MIN_HB_INTERVAL  11
#define UNIT_HB_INTERVAL 11
#define MAX_MON_INTERVAL 1275
#define MIN_MON_INTERVAL  350
#define UNIT_MON_INTERVAL 5
#define MAX_WD_INTERVAL  5461
#define MIN_WD_INTERVAL  255
#define MAX_PET_INTERVAL 255
#define MAX_PING_INTERVAL  99999
#define MIN_PING_INTERVAL  60
#define MAX_CHANGE_COUNT    500
#define MAX_CHANGE_INTERVAL 60
#define MIN_CHANGE_INTERVAL 10

//All inner functions behave as follows:
//On op == EXEC, functions return ASF API error code (S_OK==0 on success).
//On op != EXEC, functions return 0 on success, -1 otherwise.

enum OPS {
    PARSE, //Parse command
    EXEC,  //Execute command
    PRINT, //Print command output
    USAGE  //Print usage string
};

static const string errString(int err);

struct command {
    string keyword;
    int (*func) (int argc, char* argv[], OPS op);
};

int init(int argc, char* argv[], OPS op) {
    switch(op) {
    case PARSE:
        return (argc != 0);
    case EXEC:
        return AsfInitialize();
    case PRINT:
        return 0;
    case USAGE:
        cerr << "init                   -- Initialize the internal structures of the agent." << endl;
        return 0;
    }
    return -1;
}
        
int version(int argc, char* argv[], OPS op) {
    switch(op) {
    case PARSE:
        return (argc != 0);
    case EXEC:
        return 0;
    case PRINT:
		cout << "Alert Standard Format(ASF) agent, command-line interface for Linux, version " COMPONENT_VERSION "." << endl;
        return 0;
	case USAGE:
		cerr << "version                -- Print version of the utility." << endl;
        return 0;
    }
    return -1;
}
        
int setcom(int argc, char* argv[], OPS op) {
    static char community[100];
    switch(op) {
    case PARSE: 
    {
        if (argc != 1) {
            return -1;
        }
        strncpy (community, argv[0], 100);
        community[100]='\0';
        return 0;
    }
    case EXEC:
        return AsfSetCommunityString(NULL, community);
    case PRINT:
        return 0;
    case USAGE:
        cerr << "setcom <community str> -- Set the SNMP community string for PET packets." << endl;
        return 0;
    }
    return -1;
}
        
int disable(int argc, char* argv[], OPS op) {
    switch(op) {
    case PARSE:
        return (argc != 0);
    case EXEC:
        return DisableAlerting();
    case PRINT:
        return 0;
    case USAGE:
        cerr << "disable                -- Disable alerting on the current adapter." << endl;
        return 0;
    }
    return -1;
}
        
int apply(int argc, char* argv[], OPS op) {
    static char adapter[100], proxy_ip[100];
    switch(op) {
    case PARSE:
        if (argc != 2 && argc != 1) {
            return -1;
        }
		if (strlen(argv[0]) == 0) {
			return -1;
		}
		if (argc == 2 && strlen(argv[1]) == 0) {
			return -1;
		}

		if( argc == 2 ){
            strncpy (adapter, argv[0], 100);
            adapter[100]='\0';
            strncpy (proxy_ip, argv[1], 100);
            proxy_ip[100]='\0';
		}
		else{
			adapter[0] = '\0';
            strncpy (proxy_ip, argv[0], 100);
            proxy_ip[100]='\0';
		}
        return 0;
	case EXEC:
        return ApplyConfigAndEnable(adapter, proxy_ip, 0);
    case PRINT:
        return 0;
    case USAGE:
        cerr << "apply [adapter] <proxy>-- Apply the configuration and enable ASF on adapter." << endl;
        cerr << "                          Send alerts to the indicated host name/IP address." << endl;
		cerr << "                          If [adapter] is missing, an adapter is selected "<<endl;
		cerr << "                          automatically."<<endl;
        return 0;
    }
    return -1;
}
        
int timers(int argc, char* argv[], OPS op) {
    static unsigned char hb_on, wd_on, ping_on;
    static unsigned int hb_inter, hb_min, hb_max, hb_unit;
    static unsigned int wd_inter, wd_min, wd_max, wd_unit;
	static unsigned int ping_inter, ping_min, ping_max, ping_unit;
#ifdef ASF_DEBUG    
	static unsigned int retxcount, retxcount_min, retxcount_max;
    static unsigned int retxinter, retxinter_min, retxinter_max, retx_unit;
#endif
	static unsigned int mon_min, mon_max, mon_inter, mon_unit;
#ifdef SPAN_TREE
    static unsigned char lcping_on;
	static unsigned int lcpingcount, lcpingcount_min, lcpingcount_max;
    static unsigned int lcpinginter, lcpinginter_min, lcpinginter_max, lcping_unit;
#ifdef SPAN_TREE_PET_DELAY
    static unsigned char lcpd_on;
    static unsigned int lcpd_inter, lcpd_min, lcpd_max, lcpd_unit;
#endif // SPAN_TREE_PET_DELAY
#endif 

    int res;
    switch(op) {
    case PARSE:
        return (argc != 0);
    case EXEC:
        if ((res = AsfGetHeartbeatInfo(&hb_on, &hb_inter, &hb_min, &hb_max))) {
            return res;
        } 
        if ((res = AsfGetWatchdogInfo(&wd_on, &wd_inter, &wd_min, &wd_max))) {
            return res;
        } 
        if ((res = AsfGetPingInfo(&ping_on, &ping_inter))) {
            return res;
        }
		if ((res = AsfGetHeartbeatUnit(&hb_unit))) {
            return res;
        } 
        if ((res = AsfGetWatchdogUnit(&wd_unit))) {
            return res;
        } 
        if ((res = AsfGetPingUnit(&ping_unit))) {
            return res;
        }

        if ((res = AsfGetMonitoringPollUnit(&mon_unit))) {
            return res;
        }
        
		if ((res = AsfGetMonitoringInfo(&mon_inter, &mon_min, &mon_max))) {
            return res;
        }

        if ((res = AsfGetPingLimits(&ping_min, &ping_max))) {
            return res;
        }

#ifdef SPAN_TREE
        if ((res = AsfGetLinkReconnectPingUnit(&lcping_unit))) {
            return res;
        }
        if ((res = AsfGetLinkReconnectPingInfo(&lcping_on,&lcpinginter, &lcpingcount))) {
            return res;
        }
        if ((res = AsfGetLinkReconnectPingLimits(&lcpinginter_min, &lcpinginter_max, &lcpingcount_min, &lcpingcount_max))) {
            return res;
        }
#ifdef SPAN_TREE_PET_DELAY
        if ((res = AsfGetLinkReconnectSendPetDelayUnit(&lcpd_unit))) {
            return res;
        }
        if ((res = AsfGetLinkReconnectSendPetDelayInfo(&lcpd_on,&lcpd_inter))) {
            return res;
        }
        if ((res = AsfGetLinkReconnectSendPetDelayLimits(&lcpd_min,&lcpd_max))) {
            return res;
        }
#endif // SPAN_TREE_PET_DELAY
#endif

#ifdef ASF_DEBUG
		if ((res = AsfGetRetransmitIntervalUnit(&retx_unit))) {
            return res;
        } 
        return AsfGetRetransmitInfo(&retxcount, &retxcount_min, &retxcount_max,
                                    &retxinter, &retxinter_min, &retxinter_max);
#else
		return res;
#endif

    case PRINT:
        cout << "                      current value     minimum       maximum   unit" << endl;
        cout << "--------------------------------------------------------------------" << endl;
        cout << "Heartbeat Timer        ";
        if (hb_on) {
            cout << setw(12) << hb_inter;
        } else {
            cout << setw(12) << "disabled";
        }
        cout << setw(12) << hb_min << setw(14) << hb_max << setw(5) << hb_unit <<endl;
        cout << "Watchdog Timer         ";
        if (wd_on) {
            cout << setw(12) << wd_inter;
        } else {
            cout << setw(12) << "disabled";
        }
        cout << setw(12) << wd_min << setw(14) << wd_max << setw(5) << wd_unit <<endl;
        cout << "Destination Ping       ";
        if (ping_on) {
            cout << setw(12) << ping_inter;
        } else {
            cout << setw(12) << "disabled";
        }
        cout << setw(12) << ping_min << setw(14) << ping_max << setw(5) << ping_unit <<endl;


        cout << "ASF Sensors Monitoring "<<setw(12)<<mon_inter;
        cout << setw(12) << mon_min << setw(14) << mon_max << setw(5) << mon_unit <<endl;
#ifdef ASF_DEBUG
        cout << "Retransmission Count   " ;
        cout << setw(12) << retxcount << setw(12) << retxcount_min << setw(14) << retxcount_max << endl;
        cout << "Retransmission Interval" ;
        cout << setw(12) << retxinter << setw(12) << retxinter_min << setw(14) << retxinter_max;
		cout << setw(5) << retx_unit <<endl;
#endif
#ifdef SPAN_TREE
        cout << "Link Change Ping Count   " ;
        if (lcping_on) {
            cout << setw(10) << lcpingcount;
        } else {
            cout << setw(10) << "disabled";
        }
        cout << setw(12) << lcpingcount_min << setw(14) << lcpingcount_max;
		cout << setw(5) << "1" <<endl;
        cout << "Link Change Ping Interval" ;
        if (lcping_on) {
            cout << setw(10) << lcpinginter;
        } else {
            cout << setw(10) << "disabled";
        }
        cout << setw(12) << lcpinginter_min << setw(14) << lcpinginter_max;
		cout << setw(5) << lcping_unit <<endl;
        
#ifdef SPAN_TREE_PET_DELAY
        cout << "Link Change Pet Delay    " ;
        if (lcpd_on) {
            cout << setw(10) << lcpd_inter;
        } else {
            cout << setw(10) << "disabled";
        }
        cout << setw(12) << lcpd_min << setw(14) << lcpd_max << setw(5) << lcpd_unit <<endl;
#endif // SPAN_TREE_PET_DELAY
#endif
        return 0;
    case USAGE:
#ifdef ASF_DEBUG
        cerr << "timers                 -- Show heartbeat, watchdog, proxy-ping, sensors monitoring," << endl;
        cerr << "                          and retransmission timer settings." << endl;
#elif SPAN_TREE
        cerr << "timers                 -- Show heartbeat, watchdog, proxy-ping, sensors monitoring," << endl;
#ifdef SPAN_TREE_PET_DELAY
        cerr << "                          link change ping and PET delay timer settings." << endl;
#else
        cerr << "                          link change ping timer settings." << endl;
#endif // SPAN_TREE_PET_DELAY
#else
        cerr << "timers                 -- Show heartbeat, watchdog, proxy-ping, and sensors monitoring" << endl;
	cerr << "                          settings." << endl;
#endif
        return 0;
    }
    return -1;
}

int adapters(int argc, char* argv[], OPS op) {
    static int count;
    static ASFAGT_ADAPTER_INFO_EX* info;

    switch(op) {
    case PARSE:
        return (argc != 0);
    case EXEC:
        return EnumerateASFAdaptersEx(&count, &info);
    case PRINT:
        for (int i=0; i<count; i++) {
            cout << info[i].adapter_key << ": " << info[i].description << endl;
            cout << "MAC addr: " << info[i].mac_addr << ", adapter type " << info[i].adapter_type << endl;
            cout << "SMBus address: " << info[i].smbus_addr << endl;
            cout << "HW ASIC version: " << info[i].hw_asic_version << endl;
            if (info[i].asf_active) {
                cout << "ASF is enabled" << endl;
            } else {
                cout << "ASF is disabled" << endl;
            }
            cout << endl << endl;
        }
		//if count=0 it means that there is no capable adapter. and it didn't go into for loop
		if(count == 0)
			cout << "No ASF Capable adapters found in the system." <<endl << endl;
        return 0;
    case USAGE:
        cerr << "adapters               -- List the available ASF-compatible adapters." << endl;
        return 0;
    }
    return -1;
}

int sethb(int argc, char* argv[], OPS op) {
    static unsigned char enable = 1;
    static int interval = 0;

    char* endptr;
    switch(op) {
    case PARSE:
        if (argc != 1)
            return -1;
		
		if (strlen(argv[0]) == 0) {
			return -1;
		}

        interval=strtol(argv[0], &endptr,0);
        if (interval < 0) {
            return -1;
        }
        return *endptr; // 0 if string is integer, nonzero otherwise
    case EXEC:
        if (interval == 0) {
            enable = 0;
			return AsfSetHeartbeatInfo(enable, interval);
        }
	
		if(interval > MAX_HB_INTERVAL || interval < MIN_HB_INTERVAL){
			return ASF_ERROR_INVALIDPARAM;
		}
		if (interval % UNIT_HB_INTERVAL) {
			return  ASF_ERROR_INVALIDUNIT;
		}
        return AsfSetHeartbeatInfo(enable, interval);
    case PRINT:
        return 0;
    case USAGE:
        cerr << "sethb <interval>       -- Set the interval for heartbeat messages (in seconds)." << endl;
        cerr << "                          If interval is 0, heartbeat is disabled." << endl;
        return 0;
    }
    return -1;
}
        
int setwd(int argc, char* argv[], OPS op) {
	unsigned int wd_inter, wd_min, wd_max;
	unsigned char wd_on;
    static unsigned char enable = 1;
    static int interval = 0;
    char* endptr;
	int res;
    switch(op) {
    case PARSE:
        if (argc != 1)
            return -1;

		if (strlen(argv[0]) == 0) {
			return -1;
		}

        interval=strtol(argv[0], &endptr,0);
        if (interval < 0)
            return -1;
        return *endptr; // 0 if string is integer, nonzero otherwise
    case EXEC:
		if ((res = AsfGetWatchdogInfo(&wd_on, &wd_inter, &wd_min, &wd_max))) {
            return res;
		}
        if (interval == 0) {
            enable = 0;
        }
		//getting WD minimal value from bios
		else if(interval > MAX_WD_INTERVAL || interval < wd_min){
			return ASF_ERROR_INVALIDPARAM;
		}
        return AsfSetWatchdogInfo(enable, interval);
    case PRINT:
        return 0;
    case USAGE:
        cerr << "setwd <interval>       -- Set the watchdog expiration interval (in seconds). If" << endl;
        cerr << "                          interval is 0, watchdog is disabled." << endl;
        return 0;
    }
    return -1;
}
        
int setping(int argc, char* argv[], OPS op) {
    static unsigned char enable = 1;
    static int interval = 0;
    char* endptr;
    switch(op) {
    case PARSE:
        if (argc != 1)
            return -1;

		if (strlen(argv[0]) == 0) {
			return -1;
		}

        interval=strtol(argv[0], &endptr,0);
        if (interval < 0)
            return -1;
        return *endptr; // 0 if string is integer, nonzero otherwise
    case EXEC:
        if (interval == 0) {
            enable = 0;
        }else if(interval > MAX_PING_INTERVAL || interval < MIN_PING_INTERVAL){
			return ASF_ERROR_INVALIDPARAM;
		}
        return AsfSetPingInfo(enable, interval);
    case PRINT:
        return 0;
    case USAGE:
        cerr << "setping <interval>     -- Set the interval between pings to proxy (proxy" << endl;
        cerr << "                          update). If interval is 0, proxy update is disabled." << endl;
        return 0;
    }
    return -1;
}


#ifdef ASF_DEBUG

int setretx(int argc, char* argv[], OPS op) {
    static int count;
    static int interval;
    char* endptr;
    switch(op) {
    case PARSE:
        if (argc != 2)
            return -1;

		if (strlen(argv[0]) == 0) {
			return -1;
		}

		if (strlen(argv[1]) == 0) {
			return -1;
		}

        count=strtol(argv[0], &endptr,0);
        if (*endptr || count < 0)
            return -1; // 0 if string is integer, nonzero otherwise
        interval=strtol(argv[1], &endptr,0);
        if (interval < 0)
            return -1;
        return *endptr;
    case EXEC:
        return AsfSetRetransmitInfo(count, interval);
    case PRINT:
        return 0;
    case USAGE:
        cerr << "setretx <n> <interval> -- Set parameters for PET retransmission." <<endl;
		cerr << "                          n - the number of alerting packets sent," << endl;
        cerr << "                          interval - the delay between them." << endl;
        return 0;
    }
    return -1;
}

#endif

int setmon(int argc, char* argv[], OPS op) {
	unsigned int mon_min, mon_max, mon_inter;
    static unsigned int interval = 0;
    char* endptr;
	int res;
    switch(op) {
    case PARSE:
        if (argc != 1)
            return -1;

		if (strlen(argv[0]) == 0) {
			return -1;
		}

        interval=strtol(argv[0], &endptr,0);
        if (interval < 0)
            return -1;
        return *endptr; // 0 if string is integer, nonzero otherwise
	case EXEC:
		if ((res = AsfGetMonitoringInfo(&mon_inter, &mon_min, &mon_max))) {
            return res;
        }
		if(interval > MAX_MON_INTERVAL || interval < mon_min){
			return ASF_ERROR_INVALIDPARAM;
		}
		if (interval % UNIT_MON_INTERVAL) {
			return  ASF_ERROR_INVALIDUNIT;
		}
        return AsfSetMonitoringInfo(interval);
    case PRINT:
        return 0;
    case USAGE:
        cerr << "setmon <interval>      -- Set the ASF sensors monitoring interval." << endl;
        return 0;
    }
    return -1;
}

int status(int argc, char* argv[], OPS op) {
    static unsigned char service;
    static unsigned char enabled_by_user, active;
    static char mac[100], ip[100], host[100], sysid[100], uuid[100], adapter[100], reason_str[100];
    static char community[100], proxy_ip[100];
    static unsigned int reason, flags;
    int res;
	bool status_safe_mode(false), status_enabled(false);

    switch(op) {
    case PARSE:
        return (argc != 0);
    case EXEC:
        if (AsfGetServiceStatus(&service)) {
            service=0;
            return 0;
        }
        res = IsAlertingEnabledEx(&enabled_by_user,&active);
        if (res) {
            return res;
        }
        res = AsfGetTrapDestinationInfo(community, proxy_ip);
        if (res) {
            return res;
        }
		res = AsfGetConfigurationStatus(adapter, &reason, reason_str, &flags);
		if (res) {
			return res;
		}
        return AsfGetClientInfo(mac, ip, host, sysid, uuid);
    case PRINT:
        if (!service) {
            cout << "The ASF agent daemon is not active." << endl << endl;
            return 0;
        }
        cout << "The ASF agent daemon is running." << endl;

        if (active) {
			status_enabled=true;
        } else {
			if (reason != S_OK) {
				status_safe_mode=true;
	    	} else {
				status_enabled=false;
			}
		}

		if (status_safe_mode) {
			cout << "Running in safe mode - " << reason_str << endl << endl;
			if (strcmp(adapter,"")!=0) {
				cout << "    CURRENT ADAPTER: " << adapter << endl;
			}
			if (flags & ASF_STATUS_ADAPTER_SEARCH) {
				cout << "    CURRENT ADAPTER: Auto-select - searching for adapter..."<<endl;
			}
			if (strcmp(proxy_ip,"")!=0) {
				cout << "    PROXY IP/HOSTNAME: " << proxy_ip << endl;
			}          
		} else {
			if (status_enabled) {
				cout << "Alerting is enabled." << endl << endl;

				cout << "    CURRENT ADAPTER: " << adapter << endl;
				cout << "    PROXY IP/HOSTNAME: " << proxy_ip << endl;
			} else {
				cout << "Alerting is disabled." << endl << endl;
			}
		}

		cout << "    COMMUNITY STRING: " << community << endl << endl;

		cout << "Local client info:" << endl;
		cout << "    MAC ADDRESS: " << mac << endl;
		cout << "    IP ADDRESS: " << ip << endl;
		cout << "    HOST NAME: " << host << endl;
		cout << "    SYSTEM ID: " << sysid << endl;
		cout << "    UUID: " << uuid << endl << endl;

        return 0;
    case USAGE:
        cerr << "status                 -- Display alerting status information." << endl;
        return 0;
    }
    return -1;
}

int getfuncs(int argc, char* argv[], OPS op) {
    static vector<unsigned int> func_ids;
	static vector<string> func_names;
	static vector<unsigned char> func_statuses;
	unsigned int curr_func_id;
	unsigned int next_func_id;
	unsigned char curr_func_status;
	char curr_func_name[100];
	int ret_val;

	switch(op) {
    case PARSE:
        return (argc != 0);
    case EXEC:
		ret_val = AsfGetFirstControlFunctionInfo
			(&curr_func_id, curr_func_name, &curr_func_status, &next_func_id);
		while (ret_val!=ASF_ERROR_END_OF_FCTLIST) {
			if (ret_val!=S_OK) {
				return ret_val;
			}
			func_ids.push_back(curr_func_id);
			func_names.push_back(curr_func_name);
			func_statuses.push_back(curr_func_status);
			ret_val = AsfGetNextControlFunctionInfo
				(&curr_func_id, curr_func_name, &curr_func_status, &next_func_id);
		}
		return 0;
	case PRINT:
		cout << "Remote Control Functions:" << endl;
		cout << endl;
		cout << "ID   Status    Name" << endl;
		cout << "---------------------------------------------" << endl;
		cout << setiosflags(ios::left);
		for (unsigned int i = 0; i<func_ids.size(); i++) {
            cout << setw(5) << func_ids[i];
			cout << setw(10);
			if (func_statuses[i]!=0) {
				cout << "enabled";
			} else {
				cout << "disabled";
			}
			cout << func_names[i] << endl;
		}
		cout << endl;
		return 0;
    case USAGE:
        cerr << "funcs                  -- Show list of supported remote-control functions and " << endl;
        cerr << "                          their state (enabled / disabled)." << endl;
		return 0;
    }
    return -1;
}

int setfunc(int argc, char* argv[], OPS op) {
    static unsigned int func_id;
	static unsigned char func_status;
	long long_func_id;
	char* end_ptr;

	switch(op) {
    case PARSE:
        if (argc != 2) {
			return -1;
		}
		long_func_id = strtol(argv[0], &end_ptr, 10);
		if (long_func_id < 0 || *end_ptr!='\0') {
			return -1;
		}
		func_id = (unsigned int)(long_func_id);
		if (!(strcmp(argv[1], "enabled"))) {
			func_status = 1;
		} else if (!(strcmp(argv[1], "disabled"))) {
			func_status = 0;
		} else {
			return -1;
		}
		return 0;
    case EXEC:
		return AsfSetControlFunctionStatus(func_id, func_status);
    case PRINT:
        return 0;
    case USAGE:
        cerr << "setfunc <function_id>" << endl;
		cerr << "   enabled | disabled  -- Set the state of a remote-control function." << endl;
		return 0;
    }
    return -1;
}


int secplatform(int argc, char* argv[], OPS op) {
    static bool is_secure;

	switch(op) {
    case PARSE:
        return (argc != 0);
    case EXEC:
        return IsAsf2SecurePlatform((unsigned char*)&is_secure);
	case PRINT:
		if( is_secure ){
			cout << "Platform supports ASF 2.0 security." << endl;
		}
		else{
			cout << "Platform does not support ASF 2.0 security." << endl;
		}
        return 0;
    case USAGE:
        cerr << "platform               -- Check whether the platform supports ASF 2.0 security." << endl;
        return 0;
    }
    return -1;
}

int secadapter(int argc, char* argv[], OPS op) {
    static string adapt;
	static bool capable;
	
	switch(op) {
    case PARSE:
        if (argc != 1){
			return -1;
		}
		if( !strlen(argv[0]) ){
			return -1;
		}
		adapt = argv[0];
		return 0;
    case EXEC:
        return IsAsfAdapterRspCapable(adapt.c_str(), (unsigned char*)&capable);
	case PRINT:
		if( capable ){
			cout << "Adapter "<< adapt << " is ASF 2.0 security capable." << endl;
		}
		else{
			cout << "Adapter "<< adapt << " does not support ASF 2.0 security." << endl;
		}
        return 0;
    case USAGE:
        cerr << "secadapt <adapter>     -- Check whether the adapter is ASF 2.0 security capable." << endl;
        return 0;
    }
    return -1;
}

int secfunc(int argc, char* argv[], OPS op) {
    static unsigned int func_id;
	static bool is_secure;
	long long_func_id;
	char* end_ptr;
	
	switch(op) {
    case PARSE:
        if (argc != 1){
			return -1;
		}
		if( !strlen(argv[0]) ){
			return -1;
		}
		
		long_func_id = strtol(argv[0], &end_ptr, 10);
		if (long_func_id < 0 || *end_ptr!='\0') {
			return -1;
		}
		
		func_id = (unsigned)long_func_id;
		return 0;
    case EXEC:
        return AsfIsControlFunctionSecure(func_id, (unsigned char*)&is_secure);
	case PRINT:
		if( is_secure ){
			cout << "Remote control function " << func_id << " is secure." << endl;
		}
		else{
			cout << "Remote control function " << func_id << " is not secure."<< endl;
		}
        return 0;
    case USAGE:
        cerr << "secfunc <func_id>      -- Check whether the remote control function is secure." << endl;
        return 0;
    }
    return -1;
}


int suppfunc(int argc, char* argv[], OPS op) {
    static unsigned int func_id;
	static bool is_supported;
	static string adapt;
	long long_func_id;
	char* end_ptr;
	
	switch(op) {
    case PARSE:
        if (argc != 2){
			return -1;
		}
		if( !strlen(argv[0]) || !strlen(argv[1]) ){
			return -1;
		}
		
		long_func_id = strtol(argv[1], &end_ptr, 10);
		if (long_func_id < 0 || *end_ptr!='\0') {
			return -1;
		}

		adapt = argv[0];
		func_id = (unsigned)long_func_id;
		return 0;
    case EXEC:
        return AsfIsControlFunctionSupportedOnAdapter(adapt.c_str(), func_id, (unsigned char*)&is_supported);
	case PRINT:
		if( is_supported ){
			cout << "Remote contorl function " << func_id << " is supported on adapter "<< adapt <<"." << endl;
		}
		else{
			cout << "Remote contorl function " << func_id << " is not supported on adapter "<< adapt <<"." << endl;
		}
        return 0;
    case USAGE:
        cerr << "suppfunc <adapter>" << endl 
			 << "   <func_id>           -- Check whether the remote control function is supported"<< endl
			 << "                          on adapter." << endl;
        return 0;
    }
    return -1;
}


int setkeys(int argc, char* argv[], OPS op) {
    static unsigned char admin[20], oper[20], gener[20], rand[20];
	switch(op) {
    case PARSE:
        if( argc != 4 ){
			return -1;
		}
		for( int i = 0; i < 4; i ++ ){
			if (strlen(argv[i])!=40) {
				return -1;
			}
		}
		for( int i = 0; i < 40; i ++ ){
			if( !isxdigit((int)argv[0][i]) || !isxdigit((int)argv[1][i]) ||
				!isxdigit((int)argv[2][i]) || !isxdigit((int)argv[3][i]) ){
				return -1;
			}

		}
		
		unsigned short buf;
		for( int i = 0; i < 20; i ++ ){
			sscanf( &(argv[0][i*2]), "%2hx", &buf );
			admin[i] = 0x00FF & buf;
			sscanf( &(argv[1][i*2]), "%2hx", &buf );
			oper[i] = 0x00FF & buf;
			sscanf( &(argv[2][i*2]), "%2hx", &buf );
			gener[i] = 0x00FF & buf;
			sscanf( &(argv[3][i*2]), "%2hx", &buf );
			rand[i] = 0x00FF & buf;
		}
		return 0;

	case EXEC:
		return AsfSetRspKeys(admin, oper, gener, rand);
	case PRINT:
        return 0;
    case USAGE:
        cerr << "setkeys <admin> <oper>" << endl
			 << "   <gener> <random>    -- Set ASF 2.0 security keys (each should be 40"<< endl
			 << "                          characters long)." << endl;
        return 0;
    }
    return -1;
}


int haskeys(int argc, char* argv[], OPS op) {
    static bool keys_set;

	switch(op) {
    case PARSE:
        return (argc != 0);
    case EXEC:
        return AsfAreRspKeysSet((unsigned char*)&keys_set);
	case PRINT:
		if( keys_set ){
			cout << "ASF 2.0 security keys are set." << endl;
		}
		else{
			cout << "No ASF 2.0 security keys are currently set." << endl;
		}
        return 0;
    case USAGE:
        cerr << "haskeys                -- Check whether ASF 2.0 security keys are set." << endl;
        return 0;
    }
    return -1;
}

#ifdef SPAN_TREE
#ifdef SPAN_TREE_PET_DELAY
int setlcpd(int argc, char* argv[], OPS op) {
    static unsigned char enable = 1;
    static int interval = 0;

    char* endptr;
    switch(op) {
    case PARSE:
        if (argc != 1)
            return -1;
		
		if (strlen(argv[0]) == 0) {
			return -1;
		}

        interval=strtol(argv[0], &endptr,0);
        if (interval < 0) {
            return -1;
        }
        return *endptr; // 0 if string is integer, nonzero otherwise
    case EXEC:
        if (interval == 0) {
            enable = 0;
        }
		else if(interval > MAX_PET_INTERVAL){
			return ASF_ERROR_INVALIDPARAM;
		}
        return AsfSetLinkReconnectSendPetDelayInfo(enable, interval);
    case PRINT:
        return 0;
    case USAGE:
        cerr << "setlcpd <interval>     -- Set the interval for PET messages delay on a" << endl;
        cerr<<  "                          link change event (in seconds)."<<endl;
        cerr << "                          If interval is 0, PET delay timer is disabled." << endl;
        return 0;
    }
    return -1;
}
#endif // SPAN_TREE_PET_DELAY

int setlcping(int argc, char* argv[], OPS op) {
    static unsigned char enable = 1;
    static int interval = 0;
    static int count = 0;
    char* endptr;
    switch(op) {
    case PARSE:
        if (argc != 2)
            return -1;

		if ((strlen(argv[0]) == 0) || (strlen(argv[1]) == 0)) {
			return -1;
		}

		//check that the string contains only numbers without characters
        if (strspn( argv[0], "0123456789" ) != strlen(argv[0]))
            return -1;

        count=strtol(argv[0], &endptr,0);
        if (count < 0)
            return -1;

        interval=strtol(argv[1], &endptr,0);
        if (interval < 0)
            return -1;

        return *endptr; // 0 if string is integer, nonzero otherwise
    case EXEC:
        if (count == 0) {
			//If we get something like: 0 45 and not: 0 0 we write a warning
			if(interval > 0){
				cout << "Warning in setlcping: Link Change Ping interval will be ignored."<<endl;
			}
            enable = 0;
        }
		else if(count > MAX_CHANGE_COUNT && (interval > MAX_CHANGE_INTERVAL || interval < MIN_CHANGE_INTERVAL)){
			return ASF_ERROR_INVALID_TWO_PARAMS;
		}
		else if(count > MAX_CHANGE_COUNT || interval > MAX_CHANGE_INTERVAL || interval < MIN_CHANGE_INTERVAL){
			return ASF_ERROR_INVALIDPARAM;
		}
        return AsfSetLinkReconnectPingInfo(enable, interval, count);
    case PRINT:
        return 0;
    case USAGE:
        cerr << "setlcping <n>"<<endl;
        cerr<<  "     <interval>        -- Set the interval between pings to proxy (proxy" << endl;
        cerr << "                          update) on a link change event, or when proxy is"<<endl;
        cerr << "                          unreachable. n - the number of times to send ping."<<endl;
        cerr<<  "                          interval - the delay between them."<<endl;
        cerr << "                          If n is 0, the timer is disabled." << endl;
        return 0;
    }
    return -1;
}
#endif


command cmds[] = { {"init", init},
                   {"timers", timers},
                   {"adapters", adapters},
                   {"status", status},
                   {"funcs", getfuncs},				   
#ifdef FULL_FUNCTION				  
                   {"setfunc", setfunc}, 
				   {"sethb", sethb},
                   {"setwd", setwd},
                   {"setping", setping},
				   {"setmon", setmon},
#ifdef ASF_DEBUG
                   {"setretx", setretx},
#endif
#ifdef SPAN_TREE
                   {"setlcping", setlcping},
#ifdef SPAN_TREE_PET_DELAY
                   {"setlcpd", setlcpd},
#endif // SPAN_TREE_PET_DELAY
#endif // SPAN_TREE
                   {"setcom", setcom},
				   {"disable", disable},
                   {"apply", apply},
#endif
				   {"version", version},

				   {"platform", secplatform},
				   {"secadapt", secadapter},
				   {"secfunc", secfunc},
				   {"haskeys", haskeys},
				   {"suppfunc", suppfunc},
				   {"setkeys", setkeys}
};

static void usage() {
    cout << "asfcli - configure the alerting agent."<<endl;
    cout << "Usage: asfcli [options]" <<endl << endl;
    cout << "Options:"<<endl;
    for (unsigned int i=0; i < (sizeof(cmds) / sizeof(command)); i++) {
        cmds[i].func(0, 0, USAGE);
    }
}
    
int main(int argc, char* argv[]) { 
    if (argc < 2) {
        cerr << "Missing command keyword." << endl;
        usage();
        return -1;
    }
    
    int res;
    string keyword = argv[1];

	//for commands other than version printing, check if daemon is running
	if (keyword != "version") {
		unsigned char daemon_running;
		if (AsfGetServiceStatus(&daemon_running)!=S_OK) {
			cout << "The ASF agent daemon is not active." << endl << endl;
			return 0;
		}
	}

    for (unsigned int i=0; i < (sizeof(cmds) / sizeof(command)); i++) {
        if (cmds[i].keyword == keyword) {
            argc -= 2;
            argv += 2;
            if (cmds[i].func(argc, argv, PARSE)) {
                cerr << "Incorrect syntax." << endl;
                return cmds[i].func(0, 0, USAGE);
            }
            res = cmds[i].func(0, 0, EXEC);
			
			//ASF error other than S_OK
			if (res) {
                cerr << "Error in " << keyword << ": " << errString(res) << endl;
				
				//print extra error information in case of "apply" error
				if (keyword == "apply") { 
					unsigned char settings_configured;
					int temp_res = WereLastApplySettingsConfigured(&settings_configured);
					if (temp_res) {
						cerr << "Failed to check if apply settings were configured: "<< errString(temp_res) << endl;
					} else {
						if (settings_configured) {
							cerr << "Settings were configured."<<endl;
						} else {
							cerr << "Settings were not configured."<<endl;
						}
					}
				}
                return res;
            }
            return cmds[i].func(0, 0, PRINT);
        }
    }
    cerr << "Unknown keyword " << keyword << endl;
    usage();
    return -1;
}

const string errString(int err) {
	return getErrMessage(err);
}
