/*******************************************************************************

Copyright (c) 2001-2006, Intel Corporation 
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright 
    notice, this list of conditions and the following disclaimer in the 
    documentation and/or other materials provided with the distribution.

 3. Neither the name of the Intel Corporation nor the names of its 
    contributors may be used to endorse or promote products derived from 
    this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*******************************************************************************/

// This file defines a set of interface functions to the asf agent

// Return values:
// All the API functions return an integer as a return value. The return values are 
// defined in the AsfError enum, in AsfConfigDefines.h

// Changing configuration values consists of 2 steps:
// 1. Setting the requested values.
// 2. Calling "ApplyConfigAndEnable".
// The values will be set in the asf agent only after calling "ApplyConfigAndEnable".
//
// Special cases: the "DisableAlerting" and "AsfSetRspKeys" functions are applied immediately.

						   
#ifdef __cplusplus
extern "C" {
#endif

#include "AsfConfigDefines.h"

// AsfInitialze - Initialize the agent resources.
// This must be the first function call to the library.
int AsfInitialize ();

// AsfGetHeartbeatInfo - retrieve the current state of the ASF heartbeat timer.

// enable (out) - indicates whether ASF heartbeats are enabled 
//                (non-zero) or not (zero)
// interval (out) - value of current heartbeat interval in seconds
// min_value (out) - minimum allowed heartbeat interval in seconds
// max_value (out) - maximum allowed heartbeat interval in seconds
int AsfGetHeartbeatInfo (unsigned char* enable, unsigned int* interval, unsigned int* min_value, unsigned int* max_value);

// AsfSetHeartbeatInfo - enable or disable the ASF heartbeat timer using
// the specified timer interval in seconds.

// enable (in) - enable (non-zero) or disable (zero) the ASF heartbeat timer.
// interval (in) - requested heartbeat interval value in seconds. If the
//                 specified interval value is less than the minimum value
//                 allowed, the minimum value will be set. If the specified
//                 interval value is greater than the maximum value allowed,
//                 the maximum value will be set.
int AsfSetHeartbeatInfo (unsigned char enable, unsigned int interval);

// AsfGetWatchdogInfo - retrieve the current state of the ASF watchdog timer.

// enable (out) - indicates whether the ASF watchdog timer is enabled 
//                (non-zero) or not (zero)
// interval (out) - value of current watchdog timer interval in seconds
// min_value (out) - minimum allowed watchdog interval in seconds
// max_value (out) - maximum allowed watchdog interval in seconds
int AsfGetWatchdogInfo (unsigned char* enable, unsigned int* interval, unsigned int* min_value, unsigned int* max_value);

// AsfSetWatchdogInfo - enable or disable the ASF watchdog timer using
// the specified timer interval in seconds.

// enable (in) - enable (non-zero) or disable (zero) the ASF watchdog timer.
// interval (in) - requested watchdog interval value in seconds. If the
//                 specified interval value is less than the minimum value
//                 allowed, the minimum value will be set. If the specified
//                 interval value is greater than the maximum value allowed,
//                 the maximum value will be set.
int AsfSetWatchdogInfo (unsigned char enable, unsigned int interval);

// AsfGetTrapDestinationInfo - retrieve curent trap destination (Proxy) 
// settings. 

// community (out) - the currently defined community string. Should
//                  be the same community string set on the SNMP server
//                  receiving these traps.
// ip_address (out) - the current IP address of the Server system receiving
//                   the traps.
int AsfGetTrapDestinationInfo (char* community, char* ip_address);

// AsfSetCommunityString - set / update the PET SNMP community string.

// adapter_key (in) - should be set to NULL.
// community (in) - The requested community string. Must be less than 23 characters.
int AsfSetCommunityString (const char* adapter_key, const char* community);

// DisableAlerting - disable alerting functionality on the current ASF adapter.
int DisableAlerting();

// ApplyConfigAndEnable - Make the specified adapter the current ASF-enabled
// adapter. Configure and initialize the adapter or valid ASF operation. All
// other adapters will be ASF disabled. The specified adapter must be ASF
// capable and the Proxy must be reachable from the adapter's interface.

// adapter_key (in) - The interface name of the requested ASF-capable adapter.
//                    Must be one of the list returned by EnumerateASFAdapters 
// proxy_ip (in) - IP address of a valid proxy
// reserved (in)- reserved.
int ApplyConfigAndEnable (const char* adapter_key, const char* proxy_ip, unsigned int reserved);

// EnumerateASFAdapters - enumerate all ASF capable adapters in the system.

// adapter_count (out) - returns the count of all ASF-capable adapters in the 
//                       system.
// adapters (out) - returns a pointer to a list of all ASF-capable adapters.
//                  The adapter information is defined in the ASF_ADAPTER_INFO
//                  structure.
// This function is deprecated.  Use EnumerateASFAdaptersEx instead.
int EnumerateASFAdapters (int* adapter_count, ASFAGT_ADAPTER_INFO** adapters);

// EnumerateASFAdaptersEx - enumerate all ASF capable adapters in the system.
// This is an extension of EnumerateASFAdapters

// adapter_count (out) - returns the count of all ASF-capable adapters in the 
//                       system.
// adapters (out) - returns a pointer to a list of all ASF-capable adapters.
//                  The adapter information is defined in the ASF_ADAPTER_INFO
//                  structure.
int EnumerateASFAdaptersEx (int* adapter_count, ASFAGT_ADAPTER_INFO_EX** adapters);


// AsfGetFirstControlFunctionInfo - retrieve information about the status of the first
// remote control function.

// function_id (out) - return function id of the first function
// function_str (out) - string representing the current function
// enabled (out) - function is enabled (non-zero) or disabled (zero)
// next (out) - function id of the next function
int AsfGetFirstControlFunctionInfo(unsigned int* function_id, char* function_str,								 	
									unsigned char* enabled, unsigned int* next);
		 
// AsfGetNextControlFunctionInfo - retrieve information about the status of the next
// remote control function.

// function_id (out) - return function id of the first function
// function_str (out) - string representing the current function
// enabled (out) - function is enabled (non-zero) or disabled (zero)
// next (in/out) - in: "next" value returned from previous calls to
//			       AsfGetNextControlFunctionInfo or AsfGetFirstControlFunctionInfo.
//				   out: function id of the next function
int AsfGetNextControlFunctionInfo(unsigned int* function_id, char* function_str,								 	
			   					  unsigned char* enabled, unsigned int* next);
		 
// AsfGetControlFunctionStatus - retrieve status about a certain
// remote control function

// function_id (in) - remote control function id
// enabled (out) - returned function status - enabled (non-zero) or disabled
//				   (zero)
int AsfGetControlFunctionStatus(unsigned int function_id,
								unsigned char* enabled);

// AsfSetControlFunctionStatus - set a certain remote control function
// to be enabled or disabled

// function_id (in) - remote control function id
// enabled (in) - set function status - enabled (non-zero) or disabled
//				  (zero)
int AsfSetControlFunctionStatus(unsigned int function_id,
								unsigned char enabled);

///// All possible remote control function id's:
//	Reset - 0
//	Power Up - 1
//	Power Down - 2
//	Power Cycle - 3
//	Secure Reset - 4
//	Secure Power Up - 5
//	Secure Power Down - 6
//	Secure Power Cycle - 7


// AsfGetRetransmitInfo - retrieve information about PET retransmission

// retransmit_count, min_count, max_count (out) - current, minimum
// and maximum values for retransmission count.
// interval, min_interval, max_interval (out) - current, minimum
// and maximum values for interval between retransmissions (in seconds)
int AsfGetRetransmitInfo(unsigned int* retransmit_count, unsigned int* min_count, 
						 unsigned int* max_count, unsigned int* interval,
						 unsigned int* min_interval, unsigned int* max_interval);

// AsfSetRetransmitInfo - set PET retransmission information

// retransmit_count (in) - retransmission count.
// interval (in) - interval between retransmissions (in seconds)
int AsfSetRetransmitInfo(unsigned int retransmit_count, unsigned int interval); 

// AsfGetClientInfo - retrieve information about the client.

// mac_addr (out) - a string containing the current mac address of the active ASF adapter used for ASF operations.    
// ip_addr (out) - a string containing the current IP address of the active ASF adapter used for ASF operations.   
// network_name (out) - a string containing the machine network name.
// system_id (out) - returns the machine's system id residing in the ACPI table (ASF_INFO as per ASF standard).
//                   Alternatively this can be read from the ini file. This parameter is also sent in the alerting trap
//				     as described in the ASF standard.
// uuid (out) - returns the machines UUID (or GUID) residing in the SMBIOS tables.
//    			This parameter is also sent in the alerting trap as described in the ASF standard.
int AsfGetClientInfo(char* mac_addr, char* ip_addr, char* network_name,
                     char* system_id, char* uuid);

// AsfGetMonitoringInfo - retrieve the current ASF sensor polling interval settings.

// polling_interval, min_value, max_value (out) - current, minimum and maximum values for ASF sensor polling rate
//                                                (in milliseconds).
int AsfGetMonitoringInfo (unsigned int* polling_interval, unsigned int* min_value, unsigned int* max_value);

// AsfSetMonitoringInfo - set the current ASF sensor polling interval settings.

// polling_interval (out) - value for ASF sensor polling rate (in milliseconds).
int AsfSetMonitoringInfo (unsigned int polling_interval);

// IsAlertingEnabled - indicate whether alerting functionality is enabled on the current ASF adapter.
// Note: if the system is in "safe mode", alerting functionality is considered disabled.

// enabled (out) - is alerting enabled (non-zero) or disabled (zero) on current ASF adapter.
// This function is deprecated.  Use IsAlertingEnabledEx instead.
int IsAlertingEnabled(unsigned char* enabled);

// IsAlertingEnabledEx - This is an extension of IsAlertingEnabled.
// indicate whether alerting functionality is enabled by the user on the current ASF adapter, and if it is currently active.

// enabled_by_user (out) - is alerting enabled (non-zero) or disabled (zero) by the user on current ASF adapter.
// alerting_active (out) - is alerting currently active(non-zero) or disabled (zero).
int IsAlertingEnabledEx(unsigned char* enabled_by_user, unsigned char* alerting_active);

// AsfGetConfigurationStatus - check whether the agent is running in safe mode. Retrieve the reason
// for running in safe mode.

// adapter_key (out) - key of current ASF adapter.
// safe_mode_reason (out) - the reason for running in safe mode. the value for this variable is taken from the
//                          safe mode reasons list given in the AsfConfigDefines.h header file.
// reason_str (out) - string representing the reason for safe mode
// flags (out) - flags relating to current configuration (defined in AsfConfigDefines.h)
int AsfGetConfigurationStatus(char *adapter_key, unsigned int* safe_mode_reason, char* reason_str, unsigned int* flags);

// AsfGetPingInfo - retrieve availability and interval of proxy ping-polling

// enabled (out) - is proxy polling enabled (non-zero) or disabled (zero).
// interval (out) - interval between proxy pings (in seconds).
int AsfGetPingInfo(unsigned char* enabled, unsigned int* interval); 

// AsfSetPingInfo - set availability and interval of proxy ping-polling

// enabled (in) - set whether proxy polling is enabled (non-zero) or not (zero).
// interval (in) - set interval between proxy pings (in seconds).
int AsfSetPingInfo(unsigned char enabled, unsigned int interval); 

// AsfGetServiceStatus - check if the asf agent daemon is running.

// active (out) - true is the daemon is running.
//                if the daemon is not running, and error code is returned (value of active is undefined)
int AsfGetServiceStatus(unsigned char* active);

// AsfGetHeartbeatUnit - retrieve increment/decrement tokens for ASF Heartbeat timer

// unit (out) - minimum incerement/decrement unit (in seconds).
int AsfGetHeartbeatUnit(unsigned int* unit);

// AsfGetWatchdogUnit - retrieve increment/decrement tokens for ASF Watchdog timer

// unit (out) - minimum incerement/decrement unit (in seconds).
int AsfGetWatchdogUnit(unsigned int* unit);

// AsfGetRetransmitIntervalUnit - retrieve increment/decrement tokens for ASF Retransmission Interval

// unit (out) - minimum incerement/decrement unit (in seconds).
int AsfGetRetransmitIntervalUnit(unsigned int* unit);

// AsfGetMonitoringPollUnit - retrieve increment/decrement tokens for ASF Sensors Polling Timer

// unit (out) - minimum incerement/decrement unit (in seconds).
int AsfGetMonitoringPollUnit(unsigned int* unit);

// AsfGetPingUnit - retrieve increment/decrement tokens for ASF Proxy Ping Timer

// unit (out) - minimum incerement/decrement unit (in seconds).
int AsfGetPingUnit(unsigned int* unit);

// AsfGetPingLimits - retrieve minimum and maximum timer boundaries for proxy ping-polling

// min_value (out) - minimum proxy ping timer(in seconds).
// max_value (out) - maximum proxy ping timer(in seconds)..
int AsfGetPingLimits(unsigned int *min_value, unsigned int* max_value);

// GetExtraAdapterInfo - returns extra information about a given adapter

// adapter_key (in) - the requested ASF-capable adapter. Must be one from the list returned by EnumerateASFAdapters().
// connection_name (out) - adapter's connection name
// reserved1 (out) - reserved
// reserved2 (out) - reserved
// reserved3 (out) - reserved
int GetExtraAdapterInfo(char *adapter_key, char *connection_name, 
						char *reserved1, char *reserved2, char *reserved3);

// WereLastApplySettingsConfigured - returns whether or not the last apply settings were configured. 
// This function should be used after an "apply" has failed, to make a distinction between failures
// in which settings are configured in the agent, and failures in which settings are not changed

// configured (out) - whether or not the last apply settings were configured
int WereLastApplySettingsConfigured(unsigned char *configured);

// AsfIsControlFunctionSupportedOnAdapter - returns whether or not an adapter supports a certain remote control function

// adapter_key (in) - the requested ASF-capable adapter. Must be one from the list returned by EnumerateASFAdapters().
// function_id (in) - remote control function id
// supported (out) - whether the function is supported on the adapter
int AsfIsControlFunctionSupportedOnAdapter(const char *adapter_key, unsigned int function_id, unsigned char *supported);

// AsfIsControlFunctionSupportedOnAdapter - returns whether or not a certain remote control function is a secure function

// function_id (in) - remote control function id
// secure (out) - whether the function is secure
int AsfIsControlFunctionSecure(unsigned int function_id, unsigned char *secure);

// IsAsf2SecurePlatform - indicates whether or not the platform supports ASF2.0 Security (BIOS supports ASF2.0 and
// has at least one secure remote control function supported)

// asf2 (out) - whether the platform is a secure ASF2.0 platform
int IsAsf2SecurePlatform(unsigned char *asf2);

// IsAsfAdapterRspCapable - indicates whether or not an adapter supports RSP

// adapter_key (in) - the requested ASF-capable adapter. Must be one from the list returned by EnumerateASFAdapters().
// capable (out) - whether the adapter is RSP capable
int IsAsfAdapterRspCapable(const char *adapter_key, unsigned char *capable);

// AsfSetRspKeys - installs RSP keys in the client

// administrator_key (in) - administrator key
// operator_key (in) - operator key
// generation_key (in) - generation key
// random_seed (in) - random seed
int AsfSetRspKeys(unsigned char administrator_key[20],
				  unsigned char operator_key[20],
				  unsigned char generation_key[20],
				  unsigned char random_seed[20]);

// AsfAreRspKeysSet - checks whether RSP keys are set in the client

// keys_set (out) - whether RSP keys are set
int AsfAreRspKeysSet(unsigned char *keys_set);

// AsfGetLinkReconnectPingInfo - retrieve availability,interval and count of proxy ping-polling
// on link reconnect.

// enabled (out) - is proxy link reconnect polling enabled (non-zero) or disabled (zero).
// interval (out) - interval between link reconnect proxy pings (in seconds).
// retry_count (out) - the number of times to perform link reconnect proxy pings.
int AsfGetLinkReconnectPingInfo(unsigned char* enabled, unsigned int* interval, unsigned int* retry_count); 

// AsfSetLinkReconnectPingInfo - set availability and interval of link reconnect proxy ping-polling

// enabled (in) - set whether proxy link reconnect polling is enabled (non-zero) or not (zero).
// interval (in) - set interval between link reconnect proxy pings (in seconds).
// retry_count (in) - set the number of times to perform link reconnect proxy pings.
int AsfSetLinkReconnectPingInfo(unsigned char enabled, unsigned int interval, unsigned int retry_count); 

// AsfGetLinkReconnectPingUnit - retrieve increment/decrement tokens for link reconnect ASF Proxy Ping Timer

// unit (out) - minimum incerement/decrement unit (in seconds).
int AsfGetLinkReconnectPingUnit(unsigned int* unit);

// AsfGetLinkReconnectPingLimits - retrieve minimum and maximum timer boundaries for 
// link reconnect proxy ping-polling

// interval_min_value (out) - minimum proxy link reconnect ping timer interval (in seconds).
// interval_max_value (out) - maximum proxy link reconnect ping timer interval (in seconds).
// retry_count_min_value (out) - minimum proxy link reconnect ping timer count (in seconds).
// retry_count_max_value (out) - maximum proxy link reconnect ping timer count (in seconds).
int AsfGetLinkReconnectPingLimits(unsigned int *interval_min_value, unsigned int* interval_max_value, 
                                  unsigned int* retry_count_min_value, unsigned int* retry_count_max_value);

// AsfGetLinkReconnectSendPetDelayInfo - retrieve availability and interval of pet delay
// on link reconnect.

// enabled (out) - is link reconnect pet delay enabled (non-zero) or disabled (zero).
// interval (out) - interval between link reconnect to sending pets (in seconds).
int AsfGetLinkReconnectSendPetDelayInfo(unsigned char* enabled, unsigned int* interval); 

// AsfSetLinkReconnectSendPetDelayInfo - set availability and interval of pet delay on link reconnect.

// enabled (in) - set whether link reconnect pet delay is enabled (non-zero) or not (zero).
// interval (in) - set interval between link reconnect to sending pets (in seconds).
int AsfSetLinkReconnectSendPetDelayInfo(unsigned char enabled, unsigned int interval); 

// AsfGetLinkReconnectSendPetDelayUnit - retrieve increment/decrement tokens for pet delay on link reconnect timer.

// unit (out) - minimum incerement/decrement unit (in seconds).
int AsfGetLinkReconnectSendPetDelayUnit(unsigned int* unit);

// AsfGetLinkReconnectSendPetDelayLimits - retrieve minimum and maximum timer boundaries for 
// pet delay on link reconnect timer.

// min_value (out) - minimum pet delay on link reconnect timer (in seconds).
// max_value (out) - maximum pet delay on link reconnect timer (in seconds).
int AsfGetLinkReconnectSendPetDelayLimits(unsigned int *min_value, unsigned int* max_value);


#ifdef __cplusplus
}
#endif

