/*******************************************************************************

  Copyright (c) 2001-2009, Intel Corporation
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.

   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   3. Neither the name of the Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived from
      this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.

*******************************************************************************/

#include "dcb_osdep.h"
#include "tlv.h"
#include "l2_packet.h"
#include "dcb_protocol.h"

int tlv_ok(struct unpacked_tlv *tlv)
{
	if (!tlv || (!tlv->length && tlv->type))
		return 0;
	else
		return 1;
}

struct packed_tlv *pack_tlv(struct unpacked_tlv *tlv)
{
	u16 tl = 0;
	struct packed_tlv *pkd_tlv = NULL;

	if (!tlv_ok(tlv))
		return NULL;

	tl = tlv->type;
	tl = tl << 9;
	tl |= tlv->length & 0x01ff;
	tl = htons(tl);

	pkd_tlv = (struct packed_tlv *)malloc(sizeof(struct packed_tlv));
	if(!pkd_tlv) {
		printf("pack_tlv: Failed to malloc pkd_tlv\n");
		return NULL;
	}
	memset(pkd_tlv,0,sizeof(struct packed_tlv));
	pkd_tlv->size = tlv->length + sizeof(tl);
	pkd_tlv->tlv = (u8 *)malloc(pkd_tlv->size);
	if(pkd_tlv->tlv) {
		memset(pkd_tlv->tlv,0, pkd_tlv->size);
		memcpy(pkd_tlv->tlv, &tl, sizeof(tl));
		if (tlv->length)
			memcpy(&pkd_tlv->tlv[sizeof(tl)], tlv->info,
				tlv->length);
	} else {
		printf("pack_tlv: Failed to malloc tlv\n");
		free(pkd_tlv);
		pkd_tlv = NULL;
		return NULL;
	}
	return pkd_tlv;
}


struct unpacked_tlv *unpack_tlv(struct packed_tlv *tlv)
{
	u16 tl = 0;
	struct unpacked_tlv *upkd_tlv = NULL;

	if(!tlv) {
		return NULL;
	}

	memcpy(&tl,tlv->tlv, sizeof(tl));
	tl = ntohs(tl);

	upkd_tlv = (struct unpacked_tlv *)malloc(sizeof(struct unpacked_tlv));
	if(upkd_tlv) {
		memset(upkd_tlv,0, sizeof(struct unpacked_tlv));
		upkd_tlv->length = tl & 0x01ff;
		upkd_tlv->info = (u8 *)malloc(upkd_tlv->length);
		if(upkd_tlv->info) {
			memset(upkd_tlv->info,0, upkd_tlv->length);
			tl = tl >> 9;
			upkd_tlv->type = (u8)tl;
			memcpy(upkd_tlv->info, &tlv->tlv[sizeof(tl)],
				upkd_tlv->length);
		} else {
			printf("unpack_tlv: Failed to malloc info\n");
			free (upkd_tlv);
			return NULL;
		}
	} else {
		printf("unpack_tlv: Failed to malloc upkd_tlv\n");
		return NULL;
	}
	return upkd_tlv;
}

struct unpacked_tlv *free_unpkd_tlv(struct unpacked_tlv *tlv)
{
	if (tlv != NULL) {
		if (tlv->info != NULL) {
			free(tlv->info);
			tlv->info = NULL;
		}
		free(tlv);
		tlv = NULL;
	}
	return NULL;
}

struct packed_tlv *free_pkd_tlv(struct packed_tlv *tlv)
{
	if (tlv != NULL) {
		if (tlv->tlv != NULL) {
			free(tlv->tlv);
			tlv->tlv = NULL;
		}
		free(tlv);
		tlv = NULL;
	}
	return NULL;
}

struct unpacked_tlv *create_tlv()
{
	struct unpacked_tlv *tlv =
		(struct unpacked_tlv *)malloc(sizeof(struct unpacked_tlv));

	if(tlv) {
		memset(tlv,0, sizeof(struct unpacked_tlv));
		return tlv;
	} else {
		printf("create_tlv: Failed to malloc tlv\n");
		return NULL;
	}
}

struct unpacked_tlv *bld_end_tlv()
{
	struct unpacked_tlv *tlv = create_tlv();

	if(tlv) {
		tlv->type = END_OF_LLDPDU_TLV;
		tlv->length = 0;
		tlv->info = NULL;
	}
	return tlv;
}

struct unpacked_tlv *bld_chassis_tlv(struct port *port)
{
	struct unpacked_tlv *tlv = create_tlv();

	if (!tlv)
		return NULL;

	tlv->type = CHASSIS_ID_TLV;
	tlv->length = (u16)(strlen(port->ifname)) + sizeof(u8);
	tlv->info = (u8 *)malloc(tlv->length);
	if(tlv->info) {
		memset(tlv->info,0, tlv->length);
		tlv->info[0] = CHASSIS_ID_INTERFACE_NAME;
		memcpy(tlv->info+1, port->ifname, tlv->length-1);
	} else {
		printf("bld_chassis_tlv: Failed to malloc info\n");
		free(tlv);
		return NULL;
	}
	return tlv;
}

struct unpacked_tlv *bld_portid_tlv(struct port *port)
{
	struct unpacked_tlv *tlv = create_tlv();
	u8 own_addr[ETH_ALEN], *hwaddr;

	if (!tlv)
		return NULL;

	hwaddr = &own_addr[0];
	tlv->type = PORT_ID_TLV;
	tlv->length = 7;
	tlv->info = (u8 *)malloc(tlv->length);
	if(tlv->info){
		memset(tlv->info,0, tlv->length);
		tlv->info[0] = PORT_ID_MAC_ADDRESS;
		if (l2_packet_get_own_addr(port->l2, hwaddr) >= 0)
			memcpy(&tlv->info[1], &own_addr, ETH_ALEN);
	} else {
		printf("bld_portid_tlv: Failed to malloc info\n");
		free(tlv);
		return NULL;
	}
	return tlv;
}

struct unpacked_tlv *bld_ttl_tlv(struct port *port)
{
	struct unpacked_tlv *tlv = create_tlv();

	if (!tlv)
		return NULL;

	tlv->type = TIME_TO_LIVE_TLV;
	tlv->length = 2;
	tlv->info = (u8 *)malloc(tlv->length);
	if(tlv->info) {
		memset(tlv->info,0, tlv->length);
		memcpy(tlv->info, &(port->tx.txTTL), tlv->length);
	} else {
		printf("bld_ttl_tlv: Failed to malloc info\n");
		free(tlv);
		return NULL;
	}
	return tlv;
}

struct unpacked_tlv *bld_dcbx1_tlv(struct port *port)
{
	struct unpacked_tlv *tlv = create_tlv();
	struct  packed_tlv *ptlv =  NULL;
	u8 oui[DCB_OUI_LEN] = INIT_DCB_OUI;
	u8 subtype = dcbx_subtype1;
	u32 offset = 0;

	if (!tlv)
		return NULL;

	tlv->type = ORG_SPECIFIC_TLV;
	tlv->length = DCB_OUI_LEN + OUI_SUBTYPE_LEN;
	if (port->tlvs.control) {
		tlv->length = tlv->length + port->tlvs.control->length;
		if (port->tlvs.control->length) tlv->length+=2;
	}
	if (port->tlvs.pg1) {
		tlv->length = tlv->length + port->tlvs.pg1->length;
		if (port->tlvs.pg1->length) tlv->length+=2;
	}
	if (port->tlvs.pfc1) {
		tlv->length = tlv->length + port->tlvs.pfc1->length;
		if (port->tlvs.pfc1->length) tlv->length+=2;
	}
	if (port->tlvs.app1) {
		tlv->length = tlv->length + port->tlvs.app1->length;
		tlv->length+=2;
	}
#ifdef _DCB_LLINK_SUPPORT
	if (port->tlvs.llink) {
		tlv->length = tlv->length + port->tlvs.llink->length;
		if (port->tlvs.llink->length) tlv->length+=2;
	}
#endif /*_DCB_LLINK_SUPPORT */

	tlv->info = (u8 *)malloc(tlv->length);
	if (tlv->info == NULL)
		goto error;
	memset(tlv->info,0, tlv->length);

	if ((DCB_OUI_LEN + OUI_SUBTYPE_LEN) > tlv->length)
		goto error;
	memcpy(tlv->info, &oui, DCB_OUI_LEN);
	offset += DCB_OUI_LEN;
	memcpy(&tlv->info[offset], &subtype, OUI_SUBTYPE_LEN);
	offset += OUI_SUBTYPE_LEN;

	if (tlv_ok(port->tlvs.control)) {
		ptlv =  pack_tlv(port->tlvs.control);
		if (!ptlv || ((ptlv->size+offset) > tlv->length))
			goto error;
		memcpy(&tlv->info[offset], ptlv->tlv, ptlv->size);
		offset += ptlv->size;
		ptlv = free_pkd_tlv(ptlv);
	}
	if (tlv_ok(port->tlvs.pg1)) {
		ptlv = pack_tlv(port->tlvs.pg1);
		if (!ptlv || ((ptlv->size+offset) > tlv->length))
			goto error;
		memcpy(&tlv->info[offset], ptlv->tlv, ptlv->size);
		offset += ptlv->size;
		ptlv = free_pkd_tlv(ptlv);
	}
	if (tlv_ok(port->tlvs.pfc1)) {
		ptlv = pack_tlv(port->tlvs.pfc1);
		if (!ptlv || ((ptlv->size+offset) > tlv->length))
			goto error;
		memcpy(&tlv->info[offset], ptlv->tlv, ptlv->size);
		offset += ptlv->size;
		ptlv = free_pkd_tlv(ptlv);
	}
	if (tlv_ok(port->tlvs.app1)) {
		ptlv = pack_tlv(port->tlvs.app1);
		if (!ptlv || ((ptlv->size+offset) > tlv->length))
			goto error;
		memcpy(&tlv->info[offset], ptlv->tlv, ptlv->size);
		offset += ptlv->size;
		ptlv = free_pkd_tlv(ptlv);
	}

#ifdef _DCB_LLINK_SUPPORT
	if (tlv_ok(port->tlvs.llink)) {
		ptlv = pack_tlv(port->tlvs.llink);
		if (!ptlv || ((ptlv->size+offset) > tlv->length))
			goto error;
		memcpy(&tlv->info[offset], ptlv->tlv, ptlv->size);
		offset += ptlv->size;
		ptlv = free_pkd_tlv(ptlv);
	}
#endif /*_DCB_LLINK_SUPPORT */

	if (offset != tlv->length) printf("assert offset == tlv->length\n");
	assert(offset == tlv->length);
	return tlv;

error:
	ptlv = free_pkd_tlv(ptlv);
	if (tlv) {
		if (tlv->info)
			free(tlv->info);
		free(tlv);
	}
	printf("bld_dcbx1_tlv: malloc failure \n");
	return NULL;
}

struct unpacked_tlv *bld_dcbx2_tlv(struct port *port)
{
	struct unpacked_tlv *tlv = create_tlv();
	struct packed_tlv *ptlv =  NULL;
	u8 oui[DCB_OUI_LEN] = INIT_DCB_OUI;
	u8 subtype = dcbx_subtype2;
	u32 offset = 0;

	if (!tlv)
		return NULL;

	tlv->type = ORG_SPECIFIC_TLV;
	tlv->length = DCB_OUI_LEN + OUI_SUBTYPE_LEN;
	if (port->tlvs.control) {
		tlv->length = tlv->length + port->tlvs.control->length;
		if (port->tlvs.control->length) tlv->length+=2;
	}
	if (port->tlvs.pg2) {
		tlv->length = tlv->length + port->tlvs.pg2->length;
		if (port->tlvs.pg2->length) tlv->length+=2;
	}
	if (port->tlvs.pfc2) {
		tlv->length = tlv->length + port->tlvs.pfc2->length;
		if (port->tlvs.pfc2->length) tlv->length+=2;
	}
	if (port->tlvs.app2) {
		tlv->length = tlv->length + port->tlvs.app2->length;
		tlv->length+=2;
	}

	tlv->info = (u8 *)malloc(tlv->length);
	if (tlv->info == NULL)
		goto error;
	memset(tlv->info,0, tlv->length);

	if ((DCB_OUI_LEN + OUI_SUBTYPE_LEN) > tlv->length)
		goto error;
	memcpy(tlv->info, &oui, DCB_OUI_LEN);
	offset += DCB_OUI_LEN;
	memcpy(&tlv->info[offset], &subtype, OUI_SUBTYPE_LEN);
	offset += OUI_SUBTYPE_LEN;

	if (tlv_ok(port->tlvs.control)) {
		ptlv =  pack_tlv(port->tlvs.control);
		if (!ptlv || ((ptlv->size+offset) > tlv->length))
			goto error;
		memcpy(&tlv->info[offset], ptlv->tlv, ptlv->size);
		offset += ptlv->size;
		ptlv = free_pkd_tlv(ptlv);
	}
	if (tlv_ok(port->tlvs.pg2)) {
		ptlv = pack_tlv(port->tlvs.pg2);
		if (!ptlv || ((ptlv->size+offset) > tlv->length))
			goto error;
		memcpy(&tlv->info[offset], ptlv->tlv, ptlv->size);
		offset += ptlv->size;
		ptlv = free_pkd_tlv(ptlv);
	}
	if (tlv_ok(port->tlvs.pfc2)) {
		ptlv = pack_tlv(port->tlvs.pfc2);
		if (!ptlv || ((ptlv->size+offset) > tlv->length))
			goto error;
		memcpy(&tlv->info[offset], ptlv->tlv, ptlv->size);
		offset += ptlv->size;
		ptlv = free_pkd_tlv(ptlv);
	}
	if (tlv_ok(port->tlvs.app2)) {
		ptlv = pack_tlv(port->tlvs.app2);
		if (!ptlv || ((ptlv->size+offset) > tlv->length))
			goto error;
		memcpy(&tlv->info[offset], ptlv->tlv, ptlv->size);
		offset += ptlv->size;
		ptlv = free_pkd_tlv(ptlv);
	}

	if (offset != tlv->length) printf("assert offset == tlv->length\n");
	assert(offset == tlv->length);
	return tlv;

error:
	ptlv = free_pkd_tlv(ptlv);
	if (tlv) {
		if (tlv->info)
			free(tlv->info);
		free(tlv);
	}
	printf("bld_dcbx2_tlv: malloc failure \n");
	return NULL;
}

struct unpacked_tlv *bld_dcbx_ctrl_tlv(struct port *port)
{
	struct unpacked_tlv *tlv = create_tlv();
	control_protocol_attribs ctrl_cfg;
	int result,i;
	u8 oper_version;
	u8 max_version;
	u32 seqno;
	u32 ackno;

	if (!tlv)
		return NULL;

	result = get_control(port->ifname, &ctrl_cfg);
	oper_version = (u8)ctrl_cfg.Oper_version;
	max_version = (u8)ctrl_cfg.Max_version;
	seqno = htonl(ctrl_cfg.SeqNo);
	ackno = htonl(ctrl_cfg.AckNo);

	tlv->type = DCB_CONTROL_TLV;
	tlv->length = DCBX_CTRL_LEN;
	tlv->info = (u8 *)malloc(tlv->length);
	if (tlv->info) {
		memset(tlv->info,0, tlv->length);
		i = 0;
		memcpy(tlv->info, &oper_version, sizeof(oper_version));
		i += sizeof(oper_version);
		memcpy(tlv->info + i, &max_version, sizeof(max_version));
		i += sizeof(max_version);
		memcpy(tlv->info + i, &seqno, sizeof(seqno));
		i += sizeof(seqno);
		memcpy(tlv->info + i, &ackno ,sizeof(ackno));
		i = 0;
	} else {
		printf("bld_dcbx_ctrl_tlv: Failed to malloc info\n");
		free(tlv);
		return NULL;
	}

	return tlv;
}

struct unpacked_tlv *bld_dcbx1_pg_tlv(struct port *port, boolean_t *success)
{
	struct dcbx1_pg_info *pg_info;
	struct unpacked_tlv *tlv = create_tlv();
	pg_attribs  pg_cfg;
	int result, i;
	u8 tmpbyte = 0;

	*success = FALSE;
	if (!tlv) {
		return NULL;
	}
	result = get_pg(port->ifname, &pg_cfg);
	if (result == dcb_success) {
		mark_pg_sent(port->ifname);
		if (!(pg_cfg.protocol.Advertise)) {
			free(tlv);
			*success = TRUE;
			return NULL;
		}
	} else {
		free(tlv);
		return NULL;
	}

	pg_info = (struct dcbx1_pg_info *)malloc(DCBX1_PG_LEN);
	if (pg_info) {
		memset(pg_info, 0, DCBX1_PG_LEN);
		pg_info->hdr.oper_version = (u8)pg_cfg.protocol.Oper_version;
		pg_info->hdr.max_version = (u8)pg_cfg.protocol.Max_version;
		/* ewe Enable Willing Error */
		if (pg_cfg.protocol.Enable == TRUE)
			pg_info->hdr.ewe |= BIT7;
		if (pg_cfg.protocol.Willing == TRUE)
			pg_info->hdr.ewe |= BIT6;
		if (pg_cfg.protocol.Error == TRUE)
			pg_info->hdr.ewe |= BIT5;
		pg_info->hdr.sub_type = DEFAULT_SUBTYPE;

		for (i = 0; i < MAX_BANDWIDTH_GROUPS; i++) {
			pg_info->data.pg_percent[i] = pg_cfg.tx.pg_percent[i];
		}
		for (i = 0; i < MAX_USER_PRIORITIES; i++) {
			tmpbyte = 0;
			tmpbyte = pg_cfg.tx.up[i].pgid;
			tmpbyte = tmpbyte << 5;
			u8 tmpprio = 0;
			tmpprio = (u8)pg_cfg.tx.up[i].strict_priority;
			tmpprio = tmpprio << 3;
			tmpbyte |= tmpprio;
			pg_info->data.up_cfg[i].byte1 =	tmpbyte;
			pg_info->data.up_cfg[i].byte2 =
				pg_cfg.tx.up[i].percent_of_pg_cap;
		}

		tlv->length = DCBX1_PG_LEN;
	} else {
		printf("bld_dcbx1_pg_tlv: Failed to malloc pg_info\n");
		free(tlv);
		return NULL;
	}
	tlv->type = DCB_PRIORITY_GROUPS_TLV;
	tlv->info = (u8 *)pg_info;
	*success = TRUE;
	return tlv;
}

struct unpacked_tlv *bld_dcbx2_pg_tlv(struct port *port, boolean_t *success)
{
	struct dcbx2_pg_info *pg_info;
	struct unpacked_tlv *tlv = create_tlv();
	pg_attribs  pg_cfg;
	int result, i;
	u8 tmpbyte = 0;
	int j, k;

	*success = FALSE;
	if (!tlv) {
		return NULL;
	}
	result = get_pg(port->ifname, &pg_cfg);
	if (result == dcb_success) {
		mark_pg_sent(port->ifname);
		if (!(pg_cfg.protocol.Advertise)) {
			free(tlv);
			*success = TRUE;
			return NULL;
		}
	} else {
		free(tlv);
		return NULL;
	}

	pg_info = (struct dcbx2_pg_info *)malloc(DCBX2_PG_LEN);
	if (pg_info) {
		memset(pg_info, 0, DCBX2_PG_LEN);
		pg_info->hdr.oper_version = (u8)pg_cfg.protocol.Oper_version;
		pg_info->hdr.max_version = (u8)pg_cfg.protocol.Max_version;
		/* ewe Enable Willing Error */
		if (pg_cfg.protocol.Enable == TRUE)
			pg_info->hdr.ewe |= BIT7;
		if (pg_cfg.protocol.Willing == TRUE)
			pg_info->hdr.ewe |= BIT6;
		if (pg_cfg.protocol.Error == TRUE)
			pg_info->hdr.ewe |= BIT5;
		pg_info->hdr.sub_type = DEFAULT_SUBTYPE;

		for (j=0,k=0 ; k < MAX_BANDWIDTH_GROUPS; j++, k=k+2) {
			tmpbyte = 0;
			if (pg_cfg.tx.up[k].strict_priority == dcb_link)
				tmpbyte = 0xf;
			else	tmpbyte = pg_cfg.tx.up[k+1].pgid & 0xf;
			if (pg_cfg.tx.up[k+1].strict_priority == dcb_link)
				tmpbyte |= (0xf) << 4;
			else	tmpbyte |= (pg_cfg.tx.up[k].pgid & 0xf) << 4;
			pg_info->data.pg_ids[j] = tmpbyte;
		}
		for (i = 0; i < MAX_BANDWIDTH_GROUPS; i++) {
			pg_info->data.pg_percent[i] = pg_cfg.tx.pg_percent[i];
		}
		pg_info->data.num_tcs = pg_cfg.num_tcs;

		tlv->length = DCBX2_PG_LEN;
	} else {
		printf("bld_dcbx2_pg_tlv: Failed to malloc pg_info\n");
		free(tlv);
		return NULL;
	}
	tlv->type = DCB_PRIORITY_GROUPS_TLV2;
	tlv->info = (u8 *)pg_info;
	*success = TRUE;
	return tlv;
}

struct unpacked_tlv *bld_dcbx1_pfc_tlv(struct port *port, boolean_t *success)
{
	struct dcbx1_pfc_info *pfc_info;
	struct unpacked_tlv *tlv = create_tlv();
	pfc_attribs pfc_cfg;
	int result,i;

	*success = FALSE;
	if (!tlv)
		return NULL;
	result = get_pfc(port->ifname, &pfc_cfg);
	if (result == dcb_success) {
		mark_pfc_sent(port->ifname);
		if (!(pfc_cfg.protocol.Advertise)) {
			free(tlv);
			*success = TRUE;
			return NULL;
		}
	} else {
		free(tlv);
		return NULL;
	}

	pfc_info = (struct dcbx1_pfc_info *)malloc(DCBX1_PFC_LEN);
	if (pfc_info) {
		memset(pfc_info, 0, DCBX1_PFC_LEN);
		pfc_info->hdr.oper_version = (u8)pfc_cfg.protocol.Oper_version;
		pfc_info->hdr.max_version = (u8)pfc_cfg.protocol.Max_version;
		/* ewe Enable Willing Error */
		if(pfc_cfg.protocol.Enable == TRUE)
			pfc_info->hdr.ewe |= BIT7;
		if(pfc_cfg.protocol.Willing == TRUE)
			pfc_info->hdr.ewe |= BIT6;
		if(pfc_cfg.protocol.Error == TRUE)
			pfc_info->hdr.ewe |= BIT5;
		pfc_info->hdr.sub_type = DEFAULT_SUBTYPE;
		u8 temp = 0;
		for(i = 0; i < MAX_USER_PRIORITIES; i++) {
			temp = (u8)(pfc_cfg.admin[i] << i);
			pfc_info->data.admin_map |= temp;
		}

		tlv->length = DCBX1_PFC_LEN;
	} else {
		printf("bld_dcbx1_pfc_tlv: Failed to malloc pfc_info\n");
		free(tlv);
		return NULL;
	}
	tlv->type = DCB_PRIORITY_FLOW_CONTROL_TLV;
	tlv->info = (u8 *)pfc_info;
	*success = TRUE;
	return tlv;
}

struct unpacked_tlv *bld_dcbx2_pfc_tlv(struct port *port, boolean_t *success)
{
	struct dcbx2_pfc_info *pfc_info;
	struct unpacked_tlv *tlv = create_tlv();
	pfc_attribs pfc_cfg;
	int result,i;

	*success = FALSE;
	if (!tlv)
		return NULL;
	result = get_pfc(port->ifname, &pfc_cfg);
	if (result == dcb_success) {
		mark_pfc_sent(port->ifname);
		if (!(pfc_cfg.protocol.Advertise)) {
			free(tlv);
			*success = TRUE;
			return NULL;
		}
	} else {
		free(tlv);
		return NULL;
	}

	pfc_info = (struct dcbx2_pfc_info *)malloc(DCBX2_PFC_LEN);
	if (pfc_info) {
		memset(pfc_info, 0, DCBX2_PFC_LEN);
		pfc_info->hdr.oper_version = (u8)pfc_cfg.protocol.Oper_version;
		pfc_info->hdr.max_version = (u8)pfc_cfg.protocol.Max_version;
		/* ewe Enable Willing Error */
		if(pfc_cfg.protocol.Enable == TRUE)
			pfc_info->hdr.ewe |= BIT7;
		if(pfc_cfg.protocol.Willing == TRUE)
			pfc_info->hdr.ewe |= BIT6;
		if(pfc_cfg.protocol.Error == TRUE)
			pfc_info->hdr.ewe |= BIT5;
		pfc_info->hdr.sub_type = DEFAULT_SUBTYPE;
		u8 temp = 0;
		for(i = 0; i < MAX_USER_PRIORITIES; i++) {
			temp = (u8)(pfc_cfg.admin[i] << i);
			pfc_info->data.admin_map |= temp;
		}
		pfc_info->data.num_tcs = pfc_cfg.num_tcs;

		tlv->length = DCBX2_PFC_LEN;
	} else {
		printf("bld_dcbx2_pfc_tlv: Failed to malloc pfc_info\n");
		free(tlv);
		return NULL;
	}
	tlv->type = DCB_PRIORITY_FLOW_CONTROL_TLV2;
	tlv->info = (u8 *)pfc_info;
	*success = TRUE;
	return tlv;
}

struct unpacked_tlv *bld_dcbx1_app_tlv(struct port *port,
					u32 sub_type,
					boolean_t *success)
{
	struct dcbx1_app_info *app_info;
	struct unpacked_tlv *tlv = create_tlv();
	app_attribs     app_cfg;
	int result;
	u32 i,len;

	*success = FALSE;
	if (!tlv)
		return NULL;

	memset(&app_cfg, 0, sizeof(app_cfg));
	result = get_app(port->ifname, sub_type, &app_cfg);
	if (result == dcb_success) {
		mark_app_sent(port->ifname, sub_type);
		if (!(app_cfg.protocol.Advertise)) {
			free(tlv);
			*success = TRUE;
			return NULL;
		}
	} else {
		free(tlv);
		return NULL;
	}
	len = sizeof(struct  dcbx_tlv_header) + app_cfg.Length;
	app_info = (struct dcbx1_app_info *)malloc(len);
	if (app_info) {
		memset(app_info,0,sizeof(struct  dcbx1_app_info));
		app_info->hdr.oper_version = (u8)app_cfg.protocol.Oper_version;
		app_info->hdr.max_version = (u8)app_cfg.protocol.Max_version;
		/* ewe Enable Willing Error */
		if(app_cfg.protocol.Enable == TRUE)
			app_info->hdr.ewe |= BIT7;
		if(app_cfg.protocol.Willing == TRUE)
			app_info->hdr.ewe |= BIT6;
		if(app_cfg.protocol.Error == TRUE)
			app_info->hdr.ewe |= BIT5;
		app_info->hdr.sub_type = (u8)sub_type;
		for (i = 0; i < (int)app_cfg.Length; i++) {
			app_info->data[i] = app_cfg.AppData[i];
		}
		tlv->length = (u16)len;
	} else {
		printf("bld_dcbx1_app_tlv: Failed to malloc app_info\n");
		free(tlv);
		return NULL;
	}
	tlv->type = DCB_APPLICATION_TLV;
	tlv->info = (u8 *)app_info;
	*success = TRUE;
	return tlv;
}

struct unpacked_tlv *bld_dcbx2_app_tlv(struct port *port,
					u32 sub_type,
					boolean_t *success)
{
	struct dcbx2_app_info *app_info;
	struct unpacked_tlv *tlv = create_tlv();
	u8 oui[DCB_OUI_LEN] = INIT_DCB_OUI;
	app_attribs     app_cfg;
	int result;

	*success = FALSE;
	if (!tlv)
		return NULL;

	memset(&app_cfg, 0, sizeof(app_cfg));
	result = get_app(port->ifname, sub_type, &app_cfg);
	if (result == dcb_success) {
		mark_app_sent(port->ifname, sub_type);
		if (!(app_cfg.protocol.Advertise)) {
			free(tlv);
			*success = TRUE;
			return NULL;
		}
	} else {
		free(tlv);
		return NULL;
	}
	app_info = (struct dcbx2_app_info *)malloc(DCBX2_APP_LEN);
	if (app_info) {
		memset(app_info, 0, DCBX2_APP_LEN);
		app_info->hdr.oper_version = (u8)app_cfg.protocol.Oper_version;
		app_info->hdr.max_version = (u8)app_cfg.protocol.Max_version;
		/* ewe Enable Willing Error */
		if(app_cfg.protocol.Enable == TRUE)
			app_info->hdr.ewe |= BIT7;
		if(app_cfg.protocol.Willing == TRUE)
			app_info->hdr.ewe |= BIT6;
		if(app_cfg.protocol.Error == TRUE)
			app_info->hdr.ewe |= BIT5;
		app_info->hdr.sub_type = (u8)sub_type;

		app_info->data.prot_id = PROTO_ID_FCOE;
		app_info->data.byte1 = (oui[0] & PROTO_ID_OUI_MASK)
			| (PROTO_ID_L2_ETH_TYPE & PROTO_ID_SF_TYPE);
		app_info->data.low_oui = (oui[2]<<8) | oui[1];
		memcpy (&(app_info->data.up_map), &(app_cfg.AppData[0]),
			APP_FCOE_STYPE_LEN);

		tlv->length = DCBX2_APP_LEN;
	} else {
		printf("bld_dcbx2_app_tlv: Failed to malloc app_info\n");
		free(tlv);
		return NULL;
	}
	tlv->type = DCB_APPLICATION_TLV2;
	tlv->info = (u8 *)app_info;
	*success = TRUE;
	return tlv;
}

#ifdef _DCB_LLINK_SUPPORT
struct unpacked_tlv *bld_dcbx_llink_tlv(struct port *port, u32 sub_type,
					boolean_t *success)
{
	struct dcbx_llink_info *llink_info;
	struct unpacked_tlv    *tlv = create_tlv();
	llink_attribs           llk_cfg;
	llink_cfg              *cfg;
	struct dcbx_llink_cfg  *pkt;
	int                     result;


	*success = FALSE;
	if (!tlv) {
		return NULL;
	}
	result = get_llink(port->ifname, sub_type, &llk_cfg);
	if (result == dcb_success) {
		mark_llink_sent(port->ifname, sub_type);
		if (!(llk_cfg.protocol.Advertise)) {
			free(tlv);
			*success = TRUE;
			return NULL;
		}
	} else {
		free(tlv);
		return NULL;
	}

	llink_info = (struct dcbx_llink_info *)malloc(DCBX_LLINK_LEN);
	if (llink_info) {
		memset(llink_info, 0, DCBX_LLINK_LEN);
		llink_info->hdr.oper_version =
			(u8)llk_cfg.protocol.Oper_version;
		llink_info->hdr.max_version = (u8)llk_cfg.protocol.Max_version;
		/* ewe Enable Willing Error */
		if (llk_cfg.protocol.Enable == TRUE)
			llink_info->hdr.ewe |= BIT7;
		if (llk_cfg.protocol.Willing == TRUE)
			llink_info->hdr.ewe |= BIT6;
		if (llk_cfg.protocol.Error == TRUE)
			llink_info->hdr.ewe |= BIT5;
		llink_info->hdr.sub_type = (u8)sub_type;

		cfg = &(llk_cfg.llink);
		pkt = &(llink_info->data);

		if(cfg->llink_status == TRUE)
			pkt->byte1 |= BIT7;
		tlv->length = DCBX_LLINK_LEN;
	} else {
		printf("bld_dcbx_llink_tlv: Failed to malloc llink_info\n");
		free(tlv);
		return NULL;
	}
	tlv->type = DCB_LLINK_TLV;
	tlv->info = (u8 *)llink_info;
	*success = TRUE;
	return tlv;
}
#endif /*_DCB_LLINK_SUPPORT */
