/*******************************************************************************

  Copyright (c) 2001-2009, Intel Corporation
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.

   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   3. Neither the name of the Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived from
      this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.

*******************************************************************************/

#ifndef _LLDP_MOD_H
#define _LLDP_MOD_H

#include <sys/queue.h>
#include <sys/un.h>
#include "lldp_util.h"
#include "ctrl_iface.h"

/* 
 * Ops structure for lldp module callbacks.
 *
 * @lldp_mod_register: return lldp_module struct with tlv or out+subtype 
 * 		       match types
 * @lldp_mod_unregister: cleanup
 * @lldp_mod_gettlv: return packed_tlv for core to append and xmit,
 * 		     module is responsible for sanity checks the core
 * 		     will only verify length.
 * @lldp_mod_rchange: core recv function passing changed tlv to module
 * @lldp_mod_utlv: update tlv called before each xmit
 * @lldp_mod_ifup: notification of rtnetlink LINK_UP event 
 * @lldp_mod_ifdown: notification of rtnetlink LINK_DOWN event 
 * @lldp_mod_recvrt: core passes raw rtnetlink messages 
 * @client_register: any setup required for client interface
 * @client_cmd: process client commands from core lldp
 * @print_tlv: routine for client to pretty print TLVs
 * @lookup_tlv_name: find tlvid given a tlv 'name'
 * @get_arg_handler: return an arg handler list
 */
struct lldp_mod_ops {
	struct lldp_module * 	(* lldp_mod_register)(void);
	void 			(* lldp_mod_unregister)(struct lldp_module *);
	struct packed_tlv * 	(* lldp_mod_gettlv)(struct port *);
	int  			(* lldp_mod_rchange)(struct port *,
						    struct unpacked_tlv *);
	void  			(* lldp_mod_utlv)(struct port *);
	void  			(* lldp_mod_ifup)(char *); 
	void			(* lldp_mod_ifdown)(char *);
	u8 			(* lldp_mod_mibdelete)(struct port *port);
	u32			(* client_register)(void);
	int  			(* client_cmd)(void *data,
					      struct sockaddr_un *from,
					      socklen_t fromlen, char *ibuf,
					      int ilen, char *rbuf);
	int  			(* print_tlv)(u32, u16, char *);
	u32			(* lookup_tlv_name)(char *);
	int			(* print_help)();
	int			(* timer)();
	struct arg_handlers *	(* get_arg_handler)(void);
};

/*
 *	The lldp module structure
 *
 *	lldp module per instance structure.  Used by lldp core to 
 *	track available modules.  Expect lldp core to create link
 *	list of modules types per port.
 *
 */
struct lldp_module {
	int id;		/* match tlv or oui+subtype */
	u8 enable;	/* TX only, RX only, TX+RX, Disabled */
	char *path; 	/* shared library path */
	void *dlhandle; /* dlopen handle for closing */
	void *data;	/* module specific data */
	const struct lldp_mod_ops *ops;
	LIST_ENTRY(lldp_module) lldp;
};

LIST_HEAD(lldp_head, lldp_module);
struct lldp_head lldp_head;


static inline struct lldp_module *find_module_by_id(struct lldp_head *head, int id)
{
 	struct lldp_module *mod;

	LIST_FOREACH(mod, head, lldp) {
		if (mod->id == id)
			return mod;
	}
	return NULL;
}

static inline void *find_module_user_data_by_if(const char *ifname,
						struct lldp_head *head, int id)
{
	struct lldp_module *mod;

	mod = find_module_by_id(head, id);
	if (mod)
		return mod->data;
	return NULL;
}
#endif /* _LLDP_MOD_H */
