#!/bin/bash
#
# Retrospect Client Install script
# 5/2022 
# 
# (C) 2022 Retrospect, Inc. Portions (C) 1989-2010 EMC Corporation. All rights reserved.
# www.retrospect.com
# 
#  Get the system architecture
#
CLIENTDIR=/usr/local/retrospect/client
STATEFILE=/var/log/retroclient.state
EXCLUDEFILE=/etc/retroclient.excludes
PUBKEYS=(./public_key/pubkey.dat ./public_key/pubkey1.dat ./public_key/pubkey2.dat ./public_key/pubkey3.dat ./public_key/pubkey4.dat ./public_key/pubkey5.dat ./public_key/pubkey6.dat ./public_key/pubkey7.dat ./public_key/pubkey8.dat ./public_key/pubkey9.dat)

ISROOT="Yes"

# Check installer dependencies
numErrors=0
if [ "$(id -u)" != "0" ]; then
  echo "Sorry, installation requires that you be logged on as root."
  numErrors=$((numErrors + 1))
fi
if [ ! -d "/usr/local" ]; then
  echo "A /usr/local directory must exist."
  numErrors=$((numErrors + 1))
fi
if  [ ! -f ./RCL.tar ]; then
  echo "The RCL.tar archive should be in the same directory as ${0}."
  numErrors=$((numErrors + 1))
fi
# Check version of GLIBC installed
if ldd --version | sed 's/.* //;q' | awk -F. '{ if ($1 < 2 || ($1 == 2 && $2 <=13)) { exit 0 } else { exit 1} }'; then
  echo "The current system has a GLIBC version older than 2.14"
  echo "You can run ldd --version to see what current GLIBC versions is installed"
  echo "Please go to www.retrospect.com/support/kb/which_versions_of_linux_and_unix_does_the_client_support to see supported Linux operating systems"
  numErrors=$((numErrors + 1))
fi
if [ "$(uname -m)" != "x86_64" ]; then
  echo "This system is not x86_64 and is not supported by this installer"
  numErrors=$((numErrors + 1))
fi 
if [ $numErrors -gt 0 ]; then
  echo "Please correct the above $numErrors issues and try again."
  exit 1
fi

# Set up location for systemd and deprecated init.d and rc.d

# Default location
SYSTEMD=/usr/lib/systemd/system
INITD=/etc/init.d
RCD=rc.d/

# Red Hat and CentOS locations
if [ -r '/etc/redhat-release' ]; then
    INITD=/etc/rc.d/init.d
fi

# Ubuntu systemd location
if [ -d /lib/systemd/system ]; then
    SYSTEMD=/lib/systemd/system
fi

# Check for silent install switch and any switch other than expected exit out
SILENTINSTALL=""

if [ "$1" = "-s" ] || [ "$1" = "--silent" ]; then
    SILENTINSTALL=TRUE
elif [ "$1" ]; then
      echo "Unsupported switch"
      echo "For normal install run ./Install.sh"
      echo "For silent install use the switch -s or --silent"
      exit 1
fi

# Check for pubkey.dat and statefile for silent installs and if they do not exist exit the installer
# At least one of these of files is required for a silent install
if [ $SILENTINSTALL ] && [ ! -f ${PUBKEYS[0]} ] && [ ! -f $STATEFILE ]; then
    echo "ERROR: Silent Install has been selected but file pubkey.dat was not found in folder ./public_key"
    echo "and file retroclient.state was not found in folder /var/log."
    exit 1
fi

UPGRADE=""

if  [ ! -z `which pgrep`  ] && [ "$(pgrep retroclient)" ] || [ "$(ps -e | grep retroclient)" ]; then
    UPGRADE=TRUE
fi

echo
echo -n "Install Retrospect Client? (y/n): "
if [ $SILENTINSTALL ]; then
    X="y"
else
    read X
fi

if [ "$X" != "y" ] && [ "$X" != "Y" ]; then
	echo "Installation canceled."
	exit 1
fi
if [ $UPGRADE ]; then
	$CLIENTDIR/retrocpl -stop > /dev/null
fi
# Remove old install directory, if it exists
if [ -d "/usr/local/dantz" ]; then
   rm -rf /usr/local/dantz
fi
if [ ! -d "/usr/local/retrospect" ]; then
   /bin/mkdir /usr/local/retrospect
fi
if [ ! -d "$CLIENTDIR" ]; then
   /bin/mkdir $CLIENTDIR 
fi

# Extract retroclient files
tar -xf ./RCL.tar
/bin/mv -f ./retroclient ./retrocpl ./Uninstall.sh $CLIENTDIR
# Check if SYSTEMD is used, if not check for INITD
if [ -d  "$SYSTEMD" ]; then
   /bin/mv -f ./retroclient.service $SYSTEMD
   /bin/mv -f ./rcl $CLIENTDIR
# We need to label these files properly so SELinux allows them to execute via retroclient.service
   restorecon -vF $SYSTEMD/retroclient.service 2> /dev/null
   restorecon -vF $CLIENTDIR/retroclient 2> /dev/null
   restorecon -vF $CLIENTDIR/retrocpl 2> /dev/null
   restorecon -vF $CLIENTDIR/rcl 2> /dev/null
   systemctl daemon-reload 2> /dev/null
elif [ -d "$INITD" ]; then
   /bin/mv -f ./rcl $INITD
   ln -sf $INITD/rcl $CLIENTDIR/rcl
fi

# Check if file exists for first element in PUBKEYS, if so attempt to copy pubkey(s) to client dir
if [ -f ${PUBKEYS[0]} ]; then
    for PUBKEY in "${PUBKEYS[@]}"
    do
        if [ -f $PUBKEY ]; then
            /bin/cp -f $PUBKEY $CLIENTDIR
        fi
    done
# Else check if state file exists and if not prompt user to set the password
elif [ ! -f $STATEFILE ]; then
    $CLIENTDIR/retroclient -setpass
fi

if [ ! -f $EXCLUDEFILE ]; then
    /bin/mv -f ./retroclient.excludes /etc
else
    /bin/rm -f ./retroclient.excludes
fi


#
# Install man and info pages
#
if [ ! -d "/usr/local/man/man1" ]; then
   /bin/mkdir -p /usr/local/man/man1
fi
/bin/mv -f ./retroclient.1.gz ./retrocpl.1.gz /usr/local/man/man1
if [ -f ./retroclient.1 ]; then
	/bin/mv -f ./retroclient.1 ./retrocpl.1 /usr/local/man/man1
fi
if [ -d "/usr/man/man1" ]; then
	cp -f /usr/local/man/man1/retroclient.* /usr/man/man1
	cp -f /usr/local/man/man1/retrocpl.* /usr/man/man1
fi

if [ ! -d "/usr/share/info" ]; then
   /bin/mkdir -p /usr/share/info
fi
/bin/mv -f ./retroclient.info.gz /usr/share/info
if  [ -L /usr/share/info/dir ]; then
	INFODIRFILE=/etc/info-dir
else
	INFODIRFILE=/usr/share/info/dir
fi
cat<< EOF >> $INFODIRFILE

Retrospect Client 
* retroclient: (retroclient).					Retrospect Client.
* retrocpl: (retroclient)retrocpl.				Retrospect Client Control Panel.
EOF


#
#  Set up the environment variables
#
#  Bash/sh first
echo ""
echo -n "Adding RETROSPECT_HOME to system profile and login scripts..."

if [ -f /etc/profile ]
then
	ISTHERE=`grep RETROSPECT_HOME /etc/profile`
	if [ ! -z "$ISTHERE" ]
	then
		cp /etc/profile /etc/profile.old
		sed -e "/RETROSPECT_HOME/d" < /etc/profile.old > /etc/profile
	fi
	echo "RETROSPECT_HOME=$CLIENTDIR" >> /etc/profile
	echo "export RETROSPECT_HOME" >> /etc/profile
fi
#
#  csh now
#
if [ -f /etc/.login ]
then
	ISTHERE=`grep RETROSPECT_HOME /etc/.login`
	if [ ! -z "$ISTHERE" ]
	then
		cp /etc/.login /etc/.login.old
		sed -e "/RETROSPECT_HOME/d" < /etc/.login.old > /etc/.login
	fi
	echo "setenv RETROSPECT_HOME $CLIENTDIR" >> /etc/.login
fi
echo "Done!"


echo "Starting client as daemon... "
RETROSPECT_HOME="$CLIENTDIR"
export RETROSPECT_HOME
if [ -d $SYSTEMD ]; then
    systemctl start retroclient.service
else
    $CLIENTDIR/rcl start
fi
echo

if [ ! -z `which pgrep`  ] && [ -z "$(pgrep retroclient)" ] || [ -z "$(ps -e | grep retroclient)" ]; then
    echo "Installation was successful, but Retrospect Client for Linux could not "
    echo "be started. If this is an upgrade, please wait 1 minute, then start "
    echo "the client manually by running "$CLIENTDIR"/rcl start"
else
    echo "Retrospect Client for Linux started successfully!"
fi

# Check if we are upgrading, for systemd systems we need to run
# systemctl enable retroclient.service for users upgrading from
# pre version 19 clients
if [ $UPGRADE ]; then
    if [ -d "$SYSTEMD" ]; then
        systemctl enable retroclient.service
    fi
    echo "Upgrade completed successfully"
else
   if [ -d "$SYSTEMD" ]; then
       echo "Your system is using systemd, would you like "
       echo "to enable the retroclient.service to automatically "
       echo -n "start the Retrospect client for Linux? (y/n): "
       if [ $SILENTINSTALL ]; then
           X="y"
       else
           read X
       fi
       if [ "$X" = "y" ] || [ "$X" = "Y" ]; then
           systemctl enable retroclient.service
       fi
   elif [ -d "/etc/${RCD}rc2.d" ] && [ ! -f "/etc/${RCD}rc2.d/S99retroclient" ]; then
       echo "Your system appears to be compatible with the startup system used  "
       echo "by Red Hat, CentOS, and S.u.S.E. Would you like to create symbolic links "
       echo -n "to automatically start the Retrospect Client for Linux? (y/n): "
       if [ $SILENTINSTALL ]; then
           X="y"
       else
           read X
       fi
       if [ "$X" = "y" ] || [ "$X" = "Y" ]; then
           for Y in 2 3 5; do
              ln -s $CLIENTDIR/rcl /etc/${RCD}rc$Y.d/s99retroclient
           done
           if [ -r '/etc/redhat-release' ]; then
              chkconfig --add rcl
           fi
       fi
   elif [ -f "/etc/rc.local" ]; then
       echo "Your system appears to be compatible with the startup system used "
       echo "by Ubuntu and Debian. Would you like to modify rc.local "
       echo -n "to automatically start Retrospect Client for Linux? (y/n): "
       if [ $SILENTINSTALL ]; then
           X="y"
       else
           read X
       fi
       if [ "$X" = "y" ] || [ "$X" = "Y" ]; then
           sed -i -e '$a /usr/local/retrospect/client/rcl start \n' /etc/rc.local
       fi
   elif [ ! -d "/etc/${RCD}rc2/d" ]; then
       echo "If your system uses an rc.local script, you should be able to start "
       echo "Retrospect Client for Linux automatically by adding the line "
       echo "'$CLIENTDIR/rcl start'. Otherwise, consult your system "
       echo "documentation for the best method."
   fi
fi
# Open firewall ports using firewalld or iptables
if [ ! $UPGRADE ]; then
    echo "Would you like to open the firewall ports needed for "
    echo -n "the Retrospect client to communicate with the Retrospect engine? (y/n): "
    if [ $SILENTINSTALL ]; then
        X="y"
    else  
        read X
    fi
    if [ "$X" = "y" ] || [ "$X" = "Y" ]; then
        if [ -f /usr/bin/firewall-cmd ]; then
	    /usr/bin/firewall-cmd --add-port=497/tcp 2> /dev/null
	    /usr/bin/firewall-cmd --add-port=497/udp 2> /dev/null
	    /usr/bin/firewall-cmd --runtime-to-permanent 2> /dev/null
        elif [ -f /sbin/ufw ]; then
	    /sbin/ufw allow 497/tcp 2> /dev/null
	    /sbin/ufw allow 497/udp 2> /dev/null
        elif [ -f /sbin/iptables ]; then
	    /sbin/iptables -I INPUT -p tcp -m tcp --dport 497 -j ACCEPT 2> /dev/null
            /sbin/iptables -I INPUT -p udp -m udp --dport 497 -j ACCEPT 2> /dev/null
	    /sbin/service iptables save
        else
	    echo "Unable to automatically open port 497 tcp/udp on this system."
	    echo "Please consult your system documenation for help on opening these ports in the firewall."
        fi
    fi
fi
echo
exit 0
