/*
    svg_tools.js
    
    Utility classes for creating svg objects.
    
*/

// svg utility functions
function createCenteredText(textStr, y_val, color_val) {
  // create centered text node with given text
  var textNode = document.createElementNS('http://www.w3.org/2000/svg','text');
  textNode.setAttribute("x", "50%");
  textNode.setAttribute("y", y_val);
  textNode.setAttribute("text-anchor", "middle");
  textNode.setAttribute("stroke", color_val);
  $(textNode).text(textStr);
  return textNode;
}
function polarToCartesian(center, radius, angleInDegrees) {
  var angleInRadians = (angleInDegrees-90) * Math.PI / 180.0;

  return {
    x: center.x + (radius * Math.cos(angleInRadians)),
    y: center.y + (radius * Math.sin(angleInRadians))
  }
}
function originFromSVG(svg) {
    var viewBox = svg.getAttribute("viewBox");
    var origin_x = 100, origin_y = 100; // defaults
    if (viewBox) { // use center of viewBox as origin
      viewBox = viewBox.split(",");
      origin_x = (viewBox[2] - viewBox[0])/2;
      origin_y = (viewBox[3] - viewBox[1])/2;
    } else {
      // try and use width and height
      var w = svg.getAttribute("width");
      var h = svg.getAttribute("height");
      if (w && h) {
        origin_x = w / 2;
        origin_y = h / 2;
      }
    }
    return {
      x: origin_x,
      y: origin_y
    }
}
function drawDisc(svg, options) {
  // creates a circle centered in the svg with a summary text
  var radius = options["radius"] || .25; // default to 1/4 of the size of our svg
  var color_val = options["color"] || "#333";
  var origin = originFromSVG(svg);
  var g = document.createElementNS('http://www.w3.org/2000/svg','g');
  g.setAttribute('class', 'summary');
  var circle = document.createElementNS('http://www.w3.org/2000/svg','circle');
  circle.setAttribute('cx', origin.x);
  circle.setAttribute('cy', origin.y);
  circle.setAttribute('r', origin.x * radius);
  g.appendChild(circle);
  
  // Assume we are getting two words (e.g. "123.4 GB") and we will create two text entries as svg does not wrap
  var title = (options["title"] || "") + "";
  var titles = title.split(" ");
  if (titles[0])
    g.appendChild(createCenteredText(titles[0], "50%", color_val));
  if (titles[1])
    g.appendChild(createCenteredText(titles[1], "57%", color_val)); // empirically, 57% works well

  svg.appendChild(g);
  return g;
}
function createArc(svg, options) {
  // get values from options
  var radius = options["radius"] || .2; // percentage of diameter of svg
  var height = options["height"] || .1; // percentage of diameter of svg
  var start_angle = options["start_angle"] || 0;
  var end_angle = options["end_angle"] || 90;
  var corner_radius = options["corner_radius"] || 16;
  
  var fill = options["fill"];
  var stroke = options["stroke"];
  var stroke_width = options["stroke_width"];
  var class_names = options["class"];
  var svg_origin = originFromSVG(svg); // deduce origin from svg's viewBox/width attribute
  var origin = options["origin"] || svg_origin;
  var diameter = svg_origin.x * .98;
  
  // convert radius and height from percentage of diameter to actual length
  var inner_radius = diameter * radius;
  var outer_radius = inner_radius + diameter * height;
  
  // create svg path description of an arc segment with rounded corners
  // going clockwise, drawing the outer arc, the line to the inner arc, the inner arc and then back
  // we also create an arc for each corner to give a rounded radius
  // c.f. https://developer.mozilla.org/en-US/docs/Web/SVG/Tutorial/Paths#Arcs
  // and https://stackoverflow.com/questions/5736398/how-to-calculate-the-svg-path-for-an-arc-of-a-circle
  // parameters for Arc path:
  // A rx ry rotation large-arc-flag sweep-flag x y
  
  // To create an arc segment with rounded corners, we need to calculate eight points.
  // These will reference the origin, and a translated origin for both the start and end angle
  // using the corner_radius at a 90 degree from the start or end angle
  var start_origin = polarToCartesian(origin, corner_radius, start_angle + 90);
  var end_origin = polarToCartesian(origin, corner_radius, end_angle - 90);
  
  var outer_start_1 = polarToCartesian(origin,        outer_radius - corner_radius, start_angle);
  var outer_start_2 = polarToCartesian(start_origin,  outer_radius,                 start_angle);
  var outer_end_1   = polarToCartesian(end_origin,    outer_radius,                 end_angle);
  var outer_end_2   = polarToCartesian(origin,        outer_radius - corner_radius, end_angle);
  
  // inner goes backwards, from end angle to start angle
  var inner_start_1 = polarToCartesian(origin,        inner_radius + corner_radius, end_angle);
  var inner_start_2 = polarToCartesian(end_origin,    inner_radius,                 end_angle);
  var inner_end_1   = polarToCartesian(start_origin,  inner_radius,                 start_angle);
  var inner_end_2   = polarToCartesian(origin,        inner_radius + corner_radius, start_angle);
    
  var largeArcFlag = end_angle - start_angle <= 180 ? "0" : "1";
  var curveArcFlag = 0;
  var clockwise = "1";
  var counter_clockwise = "0";
  var d = [
    // start at outer start 1
    "M", outer_start_1.x, outer_start_1.y, 
    // corner curve to outer_start_2
    "A", corner_radius, corner_radius, 0, curveArcFlag, clockwise, outer_start_2.x, outer_start_2.y,
    // arc to outer end 1
    "A", outer_radius, outer_radius, 0, largeArcFlag, clockwise, outer_end_1.x, outer_end_1.y,
    // corner curve to outer_end_2
    "A", corner_radius, corner_radius, 0, curveArcFlag, clockwise, outer_end_2.x, outer_end_2.y,
    // line to inner start 1
    "L", inner_start_1.x, inner_start_1.y,
    // corner curve to inner start 2
    "A", corner_radius, corner_radius, 0, curveArcFlag, clockwise, inner_start_2.x, inner_start_2.y,
    // arc back to inner_end_1
    "A", inner_radius, inner_radius, 0, largeArcFlag, counter_clockwise, inner_end_1.x, inner_end_1.y,
    // corner curve to inner_end_2
    "A", corner_radius, corner_radius, 0, curveArcFlag, clockwise, inner_end_2.x, inner_end_2.y,
    // close path
    "Z"
  ].join(" ");

  var path = document.createElementNS('http://www.w3.org/2000/svg','path');
  if (fill)
    path.setAttribute('fill', fill);
  if (stroke)
    path.setAttribute('stroke', stroke);
  if (stroke_width)
    path.setAttribute('stroke-width', stroke_width);
  if (class_names)
    path.setAttribute('class', class_names);
  path.setAttribute('d', d);

  svg.appendChild(path);
  return path;
}
