/*
	dashboard.js
	
		Javascript logic for dashboard.html.
		
		connectSlots() is called once the DOM is fully loaded to connect the
	QT application's signals (updatedActivities, etc.) to the javascript 
	analogs below (updateActivities). This isn't required by the Mac console,
	so connectSlots uses the presence of updatedActivities to decide this.
	
	Note: hiding the 'loading' div is triggered on first update of sources,
	this was chosen as the one container that will always have at least one
	object. c.f. UpdateSources().
*/


$(document).ready(function() {
    // hide all notifications by default
    $("#notifications tr").hide();
    
    // connect close buttons for notices in <tr> tags
    $("a.close_button").unbind().click(function() {
        $(this).parent().parent().hide(); // up one to the <td>, then up again to the <tr> 
    });

	connectSlots();
	
	resizeContentBoxes();
});

var connected = false;
// connectSlots() is a QT specific function to connect the back end to these four functions
// Cocoa can call them without this connectSlots functionality.
// We use this function because connect statements resolve their target once, immediately
// not at signal emission so they must be connected only after the server object has been added to the frame
function connectSlots() {
	if ( !connected ) {
		connected = true;

        if (typeof Fixtures != "undefined") // Fixtures defined
            this.server = new Fixtures(); // use mock data for debugging and screenshots
        
        console.log("connectSlots: attempting to connect to server, server defined: " + (this.server != undefined));
		if (this.server && this.server.updatedActivities) {
			this.server.updatedActivities.connect(this, updateActivities);
			this.server.updatedBackups.connect(this, updateBackups);
			this.server.updatedBackupSets.connect(this, updateSets);
            this.server.updatedSources.connect(this, updateSources);
            this.server.updateServerName.connect(this, updateServerName);
            if (this.server.updatedPastActivities)
                this.server.updatedPastActivities.connect(this, updatePastActivities);
            if (this.server.updatedProactiveActivities)
                this.server.updatedProactiveActivities.connect(this, updateProactive);
        }
        if (this.server && this.server.relaunchRetrospect) {
            console.log("Showing Relaunch header");
            $("#header").removeClass("hidden");
            $("a.relaunch").unbind().click(function() {
                window.server.relaunchRetrospect();
                $(".restarting").show();
            });
        }
        if (this.server && this.server.wasAutoLaunched && this.server.wasAutoLaunched()) {
            window.wasAutoLaunched = true;
            $(".autolaunched").show();
        } else {
            window.wasAutoLaunched = false;
        }
	}
}

/*
		The functions below are both called by, and talk to, the backend, which
    uses RAPI to get the various properities for each object. Because both QT 
    and Cocoa have issues with arrays and other high-level objects, updateXXX()
    is the glue to clear and update a global javascript array of each of these 
    and then call the functions to do the actual logic.
*/
function updateActivities() {
	// console.log("updateActivities() called");
	var numActivities = this.server.numActivities();
	var activities = [];
	for (var dex=0; dex < numActivities; ++dex) {
		var anActivity = this.server.activity(dex);
        if (typeof anActivity == "string")
            eval("anActivity = " + anActivity); // In Cocoa, each object is passed as a string of the hash, not the hash
		if (anActivity["startDate"].valueOf() <= 28800000) { // on Win, default is 0, on Mac, default is 28800000 (1/1/1970)
            // console.log("continuing (invalid startDate)"); // Normal in Win as we report idle activities
            continue;
        }
        var startDate = anActivity["startDate"];
        if (typeof startDate == "string") {
            startDate = Date.parse(startDate);
        }
        if (anActivity["scriptName"] == "")
            continue;
        // Note: this gets remapped again in activities_list.update, so keep these in sync
		activities.push({
						backup_date: startDate,
						source_name:anActivity["sourceName"],
						destination_name:anActivity["destinationName"],
						script_name:anActivity["scriptName"],
						bytes_copied: parseInt(anActivity["bytesCopied"]) || 0,
						bytes_remaining: parseInt(anActivity["bytesRemaining"]) || 0,
						files_copied: parseInt(anActivity["filesCopied"]) || 0,
						files_remaining: parseInt(anActivity["filesRemaining"]) || 0,
						status: anActivity["status"],
                        activity_type: anActivity["activity_type"],
                        pause_or_run: (anActivity["is_paused"] ? "run" : "pause"),
                        activity_id: anActivity["activity_id"],
                        show_status: (anActivity["in_media_request"] ? "hidden" : ""),
                        show_media_request: (anActivity["in_media_request"] ? "" : "hidden"),
                        request_op: anActivity["request_op"],
                        request_member: anActivity["request_member"],
						});
	}
    if (window.wasAutoLaunched && activities.length == 0) {
        // window.server.relaunchRetrospect();
        //$(".restarting").show();
    }
        
    try {
        Updater.refreshActivities(activities);
	}
	catch (err) {
		console.log("updateActivities(): Error: " + err.message);
	}
}

function updatePastActivities() {
    // console.log("updatePastActivities() called");
    try {
        var all_activities = this.server.pastActivities();
        if (typeof all_activities == "string")
            eval("all_activities = " + all_activities); // In Cocoa, each object is passed as a string of the hash, not the hash
        var activities = [];
        all_activities.forEach(function(anActivity) {
            if (typeof anActivity == "string")
                eval("anActivity = " + anActivity); // In Cocoa, each object is passed as a string of the hash, not the hash
            if (anActivity["startDate"].valueOf() <= 28800000) { // on Win, default is 0, on Mac, default is 28800000 (1/1/1970)
                // console.log("continuing (invalid startDate)"); // Normal in Win as we report idle activities
                return;
            }
            var startDate = anActivity["startDate"];
            if (typeof startDate == "string") {
                startDate = Date.parse(startDate);
            }
            if (anActivity["scriptName"] == "")
                return;
            var activity_result = anActivity["activity_result"];
                
            activities.push({
                            backup_date: startDate,
                            source_name:anActivity["sourceName"],
                            destination_name:anActivity["destinationName"],
                            script_name:anActivity["scriptName"],
                            bytes_copied: parseInt(anActivity["bytesCopied"]) || 0,
                            bytes_remaining: parseInt(anActivity["bytesRemaining"]) || 0,
                            files_copied: parseInt(anActivity["filesCopied"]) || 0,
                            files_remaining: parseInt(anActivity["filesRemaining"]) || 0,
                            status: anActivity["status"],
                            activity_type: anActivity["activity_type"],
                            activity_id: anActivity["activity_id"],
                            activity_result: activity_result,
                            });
        });
    
        Updater.refreshPastActivities(activities);
    }
    catch (err) {
        console.log("updatePastActivities(): Error: " + err.message);
    }
}
function updateBackups() {
	// console.log("updateBackups() called");
	var numBackups = this.server.numBackups();
	backups = [];
	for (var dex=0; dex < numBackups; ++dex) {
		var aBackup = this.server.backup(dex);
        if (typeof aBackup == "string")
            eval("aBackup = " + aBackup);       // In Cocoa, each object is passed as a string of the hash, not the hash
		var name = aBackup["volumeName"];
		if (aBackup["clientName"] && aBackup["clientName"].length > 0)
            name += " on " + aBackup["clientName"];
        var backupDate = (typeof aBackup["backupDate"] == "string") ? Date.parse(aBackup["backupDate"]) : aBackup["backupDate"];
        var backupSize = parseInt(aBackup.sizeInBytes);
        var fileCount = parseInt(aBackup.fileCount);
        if (backupSize < 0)
            backupSize = 0;
        backups.push({
						date:backupDate,
						source:name,
						num_files: fileCount,
                        sizeInBytes: backupSize,
                        humanFiles: fileCount.toLocaleString(),
                        humanBytes: formatBytes(backupSize),
						})
	}
	try {
		Updater.refreshBackups(backups);
	}
	catch (err) {
		console.log("Updater.refreshBackups(): Error: " + err.message);
	}
	updateSets();
}
function updateSets() {
	// console.log("updateSets() called");
	var numSets = this.server.numSets();
	storage = [];
	for (var dex = 0; dex < numSets; ++dex) {
		var aSet = this.server.set(dex);
        if (typeof aSet == "string")
            eval("aSet = " + aSet);             // In Cocoa, each object is passed as a string of the hash, not the hash
        var bytesUsed = (typeof aSet["bytesUsed"] == "string") ? parseInt(aSet["bytesUsed"]) : aSet["bytesUsed"];
        var bytesAvailable = (typeof aSet["bytesAvailable"] == "string") ? parseInt(aSet["bytesAvailable"]) : aSet["bytesAvailable"];
        var backupDate = (typeof aSet["modificationDate"] == "string") ? Date.parse(aSet["modificationDate"]) : aSet["modificationDate"];
		storage.push({
						storage_type: aSet["setType"],
						storage_name: aSet["mediaSetName"],
                        can_groom: aSet["canGroom"],
						backup_date: backupDate,
						size_used: bytesUsed,
						total_capacity: bytesUsed + bytesAvailable,
						});
	}
	try {
		Updater.refreshStorage(storage);
	}
	catch (err) {
		console.log("updateSets(): Error: " + err.message);
	}
}
function updateSources() {
	// console.log("updateSources() called");
	try {
		var all_sources = this.server.allSources();		
		if (typeof all_sources == "string") {		
			// In Cocoa, each object is passed as a string of the hash, not the hash		
			try { eval("all_sources = " + all_sources); }		
			catch (err) {		
				console.log("updateSources(): Error parsing allSources: " + err.message);		
				all_sources = [];		
			}
		}
	}	
	catch (err) {		
		console.log("updateSources(): Error calling allSources: " + err.message);		
		all_sources = [];		
	}		
	
	var sources_list = [];
	for (var ndex = 0; ndex < all_sources.length; ndex++) {
		var aSource = all_sources[ndex];
		if (typeof aSource == "string")
			eval("aSource = " + aSource);       // In Cocoa, each object is passed as a string of the hash, not the hash
		var backupDate = (typeof aSource["lastBackupDate"] == "string") ? Date.parse(aSource["lastBackupDate"]) : aSource["lastBackupDate"];
		aSource["backup_date"] = backupDate;
		aSource["source"] = aSource["fullName"];
		sources_list.push(aSource);
    }
    // tell Updater to update all listeners
    try {
        Updater.refreshSources(sources_list);
    }
    catch (err) {
        console.log("updateSources(): Error Updater.refreshSources: " + err.message);
    }
    
	// Always try and hide the 'loading' div
    try {
	    if ($("#loading").is(":visible")) {
			$("#loading").fadeOut('slow');
		}
	}
	catch (err) {
		console.log("updateSources(): Error hiding loading screen: " + err.message);
	}
	// console.log("updateSources() finished");
}
function updateProactive(proactive_status) {
    // console.log("updateProactive(" + proactive_status + ") called");
    var all_proactive = this.server.proactiveActivities();
    if (typeof all_proactive == "string") {
        // In Cocoa, each object is passed as a string of the hash, not the hash
        try { eval("all_proactive = " + all_proactive); }
        catch (err) {
            console.log("updateProactive(): Error parsing all_proactive: " + err.message);
            all_proactive = [];
        }
    }
    var proactive_list = [];
    all_proactive.forEach(function(proActivity) {
        if (typeof proActivity == "string") eval("proActivity = " + proActivity);
        proactive_list.push(proActivity);
    });
    
    // tell Updater to update all listeners
    try {
        Updater.refreshProactive(proactive_list, proactive_status);
    }
    catch (err) {
        console.log("updateProactive(): Error Updater.refreshProactive: " + err.message);
    }
}
function updateServerName(new_server_name) {
    try {
		if (typeof new_server_name != "string" || new_server_name.length == 0)
			new_server_name = this.server.serverName();
        if (new_server_name.length == 0 || new_server_name == Updater.serverName)
            return;                                         // either no name provided or already got it
		console.log("updateServerName: " + new_server_name);
		$(".server_name").text(new_server_name);
        Updater.serverName = new_server_name;
	}
    catch (err) {
        console.log("updateServerName(): Error: " + err.message);
    }
}

/*
	Updater
	
		Generic utility to update html identified with parent_tag using
	the passed in props_list. This will detach (remove from the DOM) the
	first child identified by child_tag and store it in the_rows for future.
	
		This also takes a showEmptyPlaceholder parameter, which will show
	or hide any child element with the class 'empty_placeholder'. This can
	be used to have any styled object as a placeholder if there are no items
	(such as the activities list).
 
        Updater.update_html can be called to automate creating DOM entries for
    each set of properties in the list. The first argument is the CSS selector
    identifying the HTML container to update, the second is the CSS selector
    for the HTML inside the parent HTML to use as a template. The third is the
    properties list. Properties are inserted via two methods. If an element's
    class list contains the property name, then the internal HTML will be replaced
    with the value of the property. Additionally, any class, style, title or id
    attribute with <property_name> will have that tag replaced with the value
    of the property.
        E.g. Given html:
        "<div class='my_list'><p class='template'><span class='<activity_type> icon'></span><span class='activity_name'></span></p></div>"
    then the following code:
 
        my_list = [{activity_type:"backup", activity_name:"Daily Backup"}]
        update('.my_list', '.template', my_list)
    will generate the following HTML:
        <div class='my_list'><p><span class='backup icon'></span><span class='activity_name'>Daily Backup</span></p></div>
 
*/
function UpdaterClass() {
    this.the_rows = {};
    // list of plugins for each type
    this.activity_listeners = [];
    this.past_activity_listeners = [];
    this.backups_listeners = [];
    this.sources_listeners = [];
    this.storage_listeners = [];
    this.proactive_listeners = [];
    // registerForXyz called by plugins
    this.registerForActivities = function(plugin) { this.activity_listeners.push(plugin); }
    this.registerForPastActivities = function(plugin) { this.past_activity_listeners.push(plugin); }
    this.registerForBackups = function(plugin) { this.backups_listeners.push(plugin); }
    this.registerForSources = function(plugin) { this.sources_listeners.push(plugin); }
    this.registerForStorage = function(plugin) { this.storage_listeners.push(plugin); }
    this.registerForProactive = function(plugin) { this.proactive_listeners.push(plugin); }
    
    // updateXyz called by server in response to backend update
    this.refreshActivities = function(new_list) {
        this.activity_listeners.forEach(function(plugin) {
            // call update with type of object and a slice(0) (shallow copy) of the list
            try { plugin.update({activities:new_list.slice(0)}); }
            catch (err) { console.log("Updater.refreshActivities: Error: " + err.message); }
        });
    }
    this.refreshPastActivities = function(new_list) {
        this.past_activity_listeners.forEach(function(plugin) {
            // call update with type of object and a slice(0) (shallow copy) of the list
            try { plugin.update({pastActivities:new_list.slice(0)}); }
            catch (err) { console.log("Updater.refreshPastActivities: Error: " + err.message); }
        });
    }
    this.refreshBackups = function(new_list) {
        this.backups_listeners.forEach(function(plugin) {
            // call update with type of object and a slice(0) (shallow copy) of the list
            try { plugin.update({backups:new_list.slice(0)}); }
            catch (err) { console.log("Updater.refreshBackups: Error: " + err.message); }
        });
    }
    this.refreshSources = function(new_list) {
        this.sources_listeners.forEach(function(plugin) {
            // call update with type of object and a slice(0) (shallow copy) of the list
            try { plugin.update({sources:new_list.slice(0)}); }
            catch (err) { console.log("Updater.refreshSources: Error: " + err.message    ); }
       });
    }
    this.refreshStorage = function(new_list) {
        this.storage_listeners.forEach(function(plugin) {
            // call update with type of object and a slice(0) (shallow copy) of the list
            try { plugin.update({sets:new_list.slice(0)}); }
            catch (err) { console.log("Updater.refreshStorage: Error: " + err.message); }
        });
    }
    this.refreshProactive = function(new_list, proactive_status) {
        this.proactive_listeners.forEach(function(plugin) {
           // call update with type of object and a slice(0) (shallow copy) of the list (along with the new proactive status (polling, etc)
           try { plugin.update({proactive_status: proactive_status, proactive:new_list.slice(0)}); }
           catch (err) { console.log("Updater.proactive_listeners: Error: " + err.message); }
        });
    }

    // utility method for updating HTML dom with a list of properties
	this.update_html = function(parent_tag, child_tag, props_list) {
        // console.log("Updater.update_html parent_tag = '" + parent_tag + "' child_tag = '" + child_tag + "' props_list =" + props_list + " showEmpty = " + showEmptyPlaceholder);
		var parent_body = $(parent_tag).first();
		if (!this.the_rows[parent_tag])
            this.the_rows[parent_tag] = parent_body.children(child_tag).first().detach();
        parent_body.children(child_tag).detach(); // remove any previous updates
        if (props_list.length == 0) {
            parent_body.children(".empty_placeholder").show();
            parent_body.children(".template").hide();
        }
        else
            parent_body.children(".empty_placeholder").hide();
		for (var sdex = 0; sdex < props_list.length; ++sdex) {
			var props = props_list[sdex];
			var new_row = this.replace_in_dom(this.the_rows[parent_tag].clone(), props);
			parent_body.append(new_row);
		}
		// console.log("Updater.update_html() finished");
    }
	this.replace_in_dom = function(new_row, props) {
		for (var key in props) {
			if (props.hasOwnProperty(key)) {
				// console.log ("props[" + sdex + "]: props[" + key +"] = " + props[key])
				new_row.find("." + key).each(function(index) {
					$(this).html(props[key])
                });
				var sub_key = "<" + key + ">";
				var search_for = new RegExp(sub_key, 'gi');
				["class", "style", "title", "id"].forEach(function(attr_name) {
                    // find and replace attrs in new_row
                    var attr_val = new_row.attr(attr_name);
                    if (attr_val) {
                        attr_val = attr_val.replace(search_for, props[key]);
                        new_row.attr(attr_name, attr_val);
                    }
                    // now find in children
                    new_row.find("[" + attr_name + "*='" + sub_key + "']").map(function() {
                        var attr_val = $(this).attr(attr_name)
                        if (attr_val) {
                            attr_val = attr_val.replace(search_for, props[key]);
                            $(this).attr(attr_name, attr_val);
                        }
                    });
                });
			}
		}
		return new_row;
	}
}
Updater = new UpdaterClass();

// Utility functions
function localizedNever() {
	return "-";
}

function shortDateFormat(aDate) {
	if (!aDate || aDate.valueOf() <= 28800000) // on Win, default is 0, on Mac, default is 28800000 (1/1/1970)
        return localizedNever();
	else
        return aDate.toString("yyyy-MM-dd"); // e.g. 2008-04-13
}
function shortDateAndTimeFormat(aDate) {
	if (!aDate || aDate.valueOf() <= 28800000) // on Win, default is 0, on Mac, default is 28800000 (1/1/1970)
        return localizedNever();
	else
        return aDate.toString("yyyy-MM-dd HH:mm"); // e.g. 2008-04-13 11:23
}

Math.rand = function(max) {return Math.floor(Math.random()*(max+1));}

function formatBytes(n, precision) {
	// returns 1-3 digits with appropriate size (e.g. 0B, 123K, 23G, 1P). 12.4M is rounded down, 12.5M is rounded to 13M
    if (typeof n != "number")
        n = 0; // make sure n is a number
	if (typeof precision != "number")
		precision = 0;
	var a = [" B", " KB", " MB", " GB", " TB", " PB", " EB", " ZB", " YB"];
	var dex = 0;
	while( n >= 1024) {++dex; n/=1024;}
    var num = n.toFixed(precision);
    if (precision == 0)
        precision = 4;
    else
        precision += 1;
	return num.substr(0,precision) + a[dex];
}

// Utility for comparing two objects
// Based off of:
// https://stackoverflow.com/questions/201183/how-to-determine-equality-for-two-javascript-objects#answer-16788517
// Note the many ways this can return that two objects are true when they are subtly not. But good enough for our tests
// One change: pass in {keys: []} to restrict which keys to validate.
function objectEquals(x, y, options) {
    'use strict';
    
    if (x === null || x === undefined || y === null || y === undefined) { return x === y; }
    // after this just checking type of one would be enough
    if (x.constructor !== y.constructor) { return false; }
    // if they are functions, they should exactly refer to same one (because of closures)
    if (x instanceof Function) { return x === y; }
    // if they are regexps, they should exactly refer to same one (it is hard to better equality check on current ES)
    if (x instanceof RegExp) { return x === y; }
    if (x === y || x.valueOf() === y.valueOf()) { return true; }
    if (Array.isArray(x)) {
        if (x.length !== y.length) { return false; }
        // skip other tests and compare each element
        return Object.keys(x).every(function (key) { return objectEquals(x[key], y[key], options); });
    }
    
    // if they are dates, they must had equal valueOf
    if (x instanceof Date) { return false; }
    
    // if they are strictly equal, they both need to be object at least
    if (!(x instanceof Object)) { return false; }
    if (!(y instanceof Object)) { return false; }
    
    // recursive object equality check
    var keys = options["keys"];                             // only look at these keys
    if (typeof keys == "undefined") {
        keys = Object.keys(x);
        // quick check to see if y contains all keys present in x
        var y_contains_x_keys = Object.keys(y).every(function (y_key) { return keys.indexOf(y_key) !== -1; });
        if (!y_contains_x_keys)
            return false;
    }
    return keys.every(function (key) { return objectEquals(x[key], y[key], options); });
}


// QT's browser does not support vh/vw units (c.f. )
// so use javascript for setting the height of the box content divs to maintain
// golden ratio (~.681)

// utility function for not calling resize every ms
// c.f. https://stackoverflow.com/questions/2854407/javascript-jquery-window-resize-how-to-fire-after-the-resize-is-completed
var waitForFinalEvent = (function () {
  var timers = {};
  return function (callback, ms, uniqueId) {
    if (!uniqueId) {
      uniqueId = "Don't call this twice without a uniqueId";
    }
    if (timers[uniqueId]) {
      clearTimeout (timers[uniqueId]);
    }
    timers[uniqueId] = setTimeout(callback, ms);
  };
})();

function resizeContentBoxes(){
  var win = $(window); //this = window
  var cols = 3;
  var win_width = win.width();
  if (win_width <= 600) { cols = 1; }
  if (win_width <= 1000) { cols = 2 }
  /* 100vw = 100% of viewport * 0.618 = 61.8vw, 32px for margin and padding of box (8 * 4) */
  var new_height = (win_width * .618 / cols) - 32;
  $(".box .content").height(new_height);
}

$(window).resize(function () {
    waitForFinalEvent(resizeContentBoxes, 100, "resize timer id");
});
