/*
	fixtures.js
	
		Javascript code for creating a mock server for the dashboard.html.
		
*/

// general utility functions
function randomRange(min, max) {
  // var val = (max - min)/2 + min; // used for repeating semi-random like numbers
  if (min > max) return Math.trunc(max);
  var val = (max - min) * Math.random() + min;
  val = Math.round(val);
  return val;
}
function randomDate(start, end = new Date()) {
  return new Date(
    start.getTime() + Math.random() * (end.getTime() - start.getTime())
  );
}
function randomItem(items) {
  if (items && items.length > 0)
    return items[Math.floor(Math.random() * items.length)];
  else return {};
}
function shuffleArray(array) {
  // uses Durstenfeld (Fisher-Yates) shuffle algorithm
  for (var i = array.length - 1; i > 0; i--) {
    var j = Math.floor(Math.random() * (i + 1));
    var temp = array[i];
    array[i] = array[j];
    array[j] = temp;
  }
  return array;
}
function daysAgo(days, fromDate = new Date()) {
  return new Date(fromDate.getTime() - days * 24 * 60 * 60 * 1000);
}

function Fixtures(options = {}) {
  // returns a ServerItem object, each has a name, type, used, free, total and children,
  // each of which are also ServerItems.
  //
  // Pass in a hash of options to override defaults
  // e.g. Fixtures({min_sets: 3});

  // constants for ease-of-use
  const kKB = 1024;
  const kMB = kKB * 1024;
  const kGB = kMB * 1024;
  const kTB = kGB * 1024;

  var m_min_sets = options["min_sets"] || 1;
  var m_max_sets = options["max_sets"] || 10;
  var m_min_free = options["min_free"] || 1 * kMB;
  var m_max_free = options["max_free"] || 8 * kTB;
  var m_min_sources = options["min_sources"] || 1;
  var m_max_sources = options["max_sources"] || 20;
  var m_min_backups = options["min_backups"] || 1; // Cannot have less than 1 backup - source would not show up
  var m_max_backups = options["max_backups"] || 30;
  var m_min_backup_size = options["min_size"] || 1 * kKB;
  var m_max_backup_size = options["max_size"] || 500 * kGB;

  this.loadActivities = function () {
    var script_names = shuffleArray([
      "Daily Backup",
      "Nightly Backup",
      "Server Backup",
    ]);
    var num_items = randomRange(0, script_names.length);
    for (var ndex = 0; ndex < num_items; ndex++) {
      var disk_size = randomRange(1, 1024) * kGB; // 1GB - 1TB, in GB
      var bytes_copied = randomRange(disk_size / 100, disk_size);
      var files_to_copy = Math.floor(
        randomRange(bytes_copied / kKB, bytes_copied / kMB)
      );
      var files_copied = randomRange(0, files_to_copy);
      var source_item = randomItem(this.server.volumes);
      var set_item = randomItem(this.server.sets);
      var status_text =
        randomItem([
          t("app.jobs-scanning-label"),
          t("app.jobs-matching-label"),
          t("app.jobs-copying-label"),
        ]) + source_item.name;
      var activity = {
        type: "activity",
        sourceName: source_item.name,
        parent: this.server,
        startDate: randomDate(daysAgo(1)),
        sourceName: source_item.name,
        destinationName: set_item.name,
        scriptName: script_names[ndex],
        bytesCopied: bytes_copied,
        bytesRemaining: disk_size - bytes_copied,
        filesCopied: files_copied,
        filesRemaining: files_to_copy - files_copied,
        status: status_text,
        activity_type: "backup",
        is_paused: randomItem([true, false]),
        activity_id: randomRange(0, 8),
      };
      this.server.activities.push(activity);
    }
  };
  this.loadBackups = function (set_item) {
    set_item.backups = [];
    set_item.used = 0;
    set_item.total = 0;
    set_item.file_count = 0;
    var num_items = randomRange(m_min_backups, m_max_backups);
    for (var ndex = 0; ndex < num_items > 0; ndex++) {
      var days_ago = randomRange(0, 360) + randomRange(0, 360) - 360; // create a bell curve from -360 - 360
      days_ago = Math.abs(days_ago); // fold back, so more likely to be recent
      var item_date = daysAgo(days_ago);
      var used =
        randomRange(m_min_backup_size, m_max_backup_size) +
        randomRange(m_min_backup_size, m_max_backup_size); // bell curve
      var backup = {
        type: "backup",
        name: item_date.toLocaleString(),
        parent: set_item,
        set: set_item,
        source: randomItem(this.server.volumes),
        free: 0,
        used: used,
        total: used,
        file_count: Math.floor(randomRange(used / kKB, used / kMB)),
        children: [],
        date: item_date,
      };
      set_item.backups.push(backup);
      this.server.backups.push(backup); // master list
      set_item.used += backup.used;
      set_item.total += backup.total;
    }
    set_item.children = set_item.backups; // used by visualizations like Sunburst
    set_item.children.sort(function (a, b) {
      return a.date - b.date;
    });
    set_item.modification_date =
      set_item.children[set_item.children.length - 1].date;
    return set_item;
  };
  this.loadVolumes = function (source_item) {
    source_item.volumes = [];
    var names =
      source_item.os == "Mac"
        ? ["Macintosh HD", "LaCie", "Barracuda"]
        : ["Local Disk (C:)", "WD (E:)", "Samsung (F:)"];
    var num_items = randomRange(1, names.length);
    for (var ndex = 0; ndex < num_items; ndex++) {
      var disk_size = randomRange(1, 1024) * kGB; // 1GB - 1TB, in GB
      var used = randomRange(disk_size / 100, disk_size);
      var volume = {
        type: "volume",
        name: names[ndex] + " on " + source_item.name,
        parent: source_item,
        machine_name: source_item.name,
        volume_name: names[ndex],
        used: used,
        free: disk_size - used,
        last_backup_date: new Date(0),
      };
      source_item.volumes.push(volume);
      this.server.volumes.push(volume); // master list
    }
    return source_item;
  };
  this.loadClients = function () {
    this.server.clients = [];

    var names = shuffleArray([
      "Hannah Abbott",
      "Katie Bell",
      "Cuthbert Binns",
      "Susan Bones",
      "Terry Boot",
      "Lavender Brown",
      "Millicent Bulstrode",
      "Charity Burbage",
      "Cho Chang",
      "Penelope Clearwater",
      "Michael Corner",
      "Vincent Crabbe",
      "Colin Creevey",
      "Dennis Creevey",
      "Roger Davies",
      "Cedric Diggory",
      "Albus Dumbledore",
      "Marietta Edgecombe",
      "Argus Filch",
      "Justin Finch-Fletchley",
      "Seamus Finnigan",
      "Marcus Flint",
      "Filius Flitwick",
      "Anthony Goldstein",
      "Gregory Goyle",
      "Hermione Granger",
      "Rubeus Hagrid",
      "Madam Hooch",
      "Angelina Johnson",
      "Lee Jordan",
      "Neville Longbottom",
      "Luna Lovegood",
      "Draco Malfoy",
      "Ernie Macmillan",
      "Minerva McGonagall",
      "Cormac McLaggen",
      "Graham Montague",
      "Theodore Nott",
      "Pansy Parkinson",
      "Padma Patil",
      "Parvati Patil",
      "Poppy Pomfrey",
      "Harry Potter",
      "Demelza Robins",
      "Aurora Sinistra",
      "Zacharias Smith",
      "Severus Snape",
      "Alicia Spinnet",
      "Pomona Sprout",
      "Dean Thomas",
      "Sybill Trelawney",
      "Romilda Vane",
      "Septima Vector",
      "Fred Weasley",
      "George Weasley",
      "Ginny Weasley",
      "Percy Weasley",
      "Ron Weasley",
      "Oliver Wood",
      "Blaise Zabini",
    ]);
    if (m_max_sources > names.length) m_max_sources = names.length;
    var num_items = randomRange(m_min_sources, m_max_sources);
    for (var ndex = 0; ndex < num_items; ndex++) {
      var os_type = randomItem(["Mac", "PC"]);
      var client = {
        type: "client",
        name: names[ndex] + "'s " + os_type, // names is already shuffled
        machine_name: names[ndex],
        volume_name: "",
        parent: this.server,
        os: os_type,
        free: 0, // Note free and used represent the total free and used for backups. So free is always 0.
        last_backup_date: new Date(0),
      };
      this.loadVolumes(client);
      this.server.clients.push(client);
    }
  };
  this.loadSets = function () {
    this.server.sets = [];
    this.server.used = 0;
    this.server.free = 0;
    this.server.total = 0;
    const letters = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    var num_items = randomRange(m_min_sets, m_max_sets);
    for (var ndex = 0; ndex < num_items; ndex++) {
      var item_name = "Backup Set " + letters[ndex % letters.length];
      var free_space = randomRange(m_min_free, m_max_free);
      var set_item = {
        type: "set",
        name: item_name,
        parent: this.server,
        media_type: randomItem(["disk", "cloud", "tape"]),
        free: free_space,
      };
      this.loadBackups(set_item);
      this.server.sets.push(set_item);
      this.server.used += set_item.used;
      this.server.free += set_item.free;
      this.server.total += set_item.used + set_item.free;
    }
    this.server.children = this.server.sets;
    this.server.children.sort(function (a, b) {
      return b.total - a.total;
    });
  };
  this.loadServer = function (serverName) {
    this.server = {
      type: "server",
      name: serverName,
      parent: {},
      activities: [],
      volumes: [],
      backups: [],
    };
    this.loadClients(); // also loads volumes
    this.loadSets(); // also loads backups
    // now that we have backups, find last backup for each client
    for (var ndex = 0; ndex < this.server.backups.length; ndex++) {
      var backup = this.server.backups[ndex];
      if (backup.date > backup.source.last_backup_date) {
        backup.source.last_backup_date = backup.date;
        if (backup.date > backup.source.parent.last_backup_date)
          backup.source.parent.last_backup_date = backup.date;
      }
    }
    this.loadActivities();
  };

  // shim functions for connecting to dashboard
  this.updatedActivities = {};
  this.updatedActivities.connect = function () {}; // no-op
  this.updatedBackups = {};
  this.updatedBackups.connect = function () {}; // no-op
  this.updatedBackupSets = {};
  this.updatedBackupSets.connect = function () {}; // no-op
  this.updatedSources = {};
  this.updatedSources.connect = function () {}; // no-op
  this.updateServerName = {};
  this.updateServerName.connect = function () {}; // no-op
  this.updatedPastActivities = {};
  this.updatedPastActivities.connect = function () {}; // no-op

  // functions used by the dashboard
  this.numActivities = function () {
    return this.server.activities.length;
  };
  this.activity = function (ndex) {
    var activity_item = this.server.activities[ndex];
    return {
      startDate: activity_item.startDate,
      sourceName: activity_item.sourceName,
      destinationName: activity_item.destinationName,
      scriptName: activity_item.scriptName,
      bytesCopied: activity_item.bytesCopied,
      bytesRemaining: activity_item.bytesRemaining,
      filesCopied: activity_item.filesCopied,
      filesRemaining: activity_item.filesRemaining,
      status: activity_item.status,
      activity_type: activity_item.activity_type,
      is_paused: activity_item.is_paused,
      activity_id: activity_item.activity_id,
    };
  };
  this.pastActivities = function () {
    var results = [];
    this.server.activities.forEach(function (activity_item) {
      results.push({
        startDate: activity_item.startDate,
        sourceName: activity_item.sourceName,
        destinationName: activity_item.destinationName,
        scriptName: activity_item.scriptName,
        bytesCopied: activity_item.bytesCopied,
        bytesRemaining: activity_item.bytesRemaining,
        filesCopied: activity_item.filesCopied,
        filesRemaining: activity_item.filesRemaining,
        status: activity_item.status,
        activity_type: activity_item.activity_type,
        is_paused: activity_item.is_paused,
        activity_id: activity_item.activity_id,
      });
    });
    return results;
  };
  this.proactiveActivities = function () {
    var results = [];
    this.server.activities.forEach(function (activity_item) {
      results.push({
        status: activity_item.status,
        statusKey: "unknown", // ### "Source not available", etc.
        sourceName: activity_item.sourceName,
        destinationName: activity_item.destinationName,
        scriptName: activity_item.scriptName,
      });
    });
    return results;
  };
  this.numBackups = function () {
    return this.server.backups.length;
  };
  this.backup = function (ndex) {
    var backup_item = this.server.backups[ndex];
    return {
      backupDate: backup_item.date,
      volumeName: backup_item.source.name,
      fileCount: backup_item.file_count,
      sizeInBytes: backup_item.used,
    };
  };
  this.numSets = function () {
    return this.server.sets.length;
  };
  this.set = function (ndex) {
    var set_item = this.server.sets[ndex];
    return {
      setType: set_item.media_type,
      mediaSetName: set_item.name,
      modificationDate: set_item.modification_date,
      bytesUsed: set_item.used,
      bytesAvailable: set_item.free,
    };
  };
  this.numSources = function () {
    return this.server.clients.length;
  };
  this.source = function (ndex) {
    var source_item = this.server.clients[ndex];
    return {
      fullName: source_item.name,
      volumeName: source_item.volume_name,
      machineName: source_item.machine_name,
      lastBackupDate: source_item.last_backup_date,
      fullName: source_item.name,
      isClient: "true",
      isEmail: "false",
      isNAS: "false",
      isSubvolume: "false",
    };
  };
  this.allSources = function () {
    var results = [];
    this.server.clients.forEach(function (source_item) {
      results.push({
        fullName: source_item.name,
        volumeName: source_item.volume_name,
        machineName: source_item.machine_name,
        lastBackupDate: source_item.last_backup_date,
        isClient: "true",
        isEmail: "false",
        isNAS: "false",
        isSubvolume: "false",
      });
      source_item.volumes.forEach(function (source_volume) {
        results.push({
          fullName: source_volume.name,
          volumeName: source_volume.volume_name,
          machineName: source_volume.machine_name,
          lastBackupDate: source_volume.last_backup_date,
          isClient: "false",
          isEmail: "false",
          isNAS: "false",
          isSubvolume: "false",
        });
      });
    });
    return results;
  };

  this.loadServer("OBU");
  return this;
}
