/*
 * (C) 2002 Clemson University and The University of Chicago
 *
 * See COPYING in top-level directory.
 */

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <trove.h>
#include <getopt.h>

#include "trove-test.h"

char storage_space[SSPACE_SIZE] = "/tmp/trove-test-space";
char file_system[FS_SIZE] = "fs-foo";

int parse_args(int argc, char **argv);

int main(int argc, char **argv)
{
    int ret, count;
    TROVE_op_id op_id;
    TROVE_coll_id coll_id;
    TROVE_handle root_handle;
    TROVE_ds_state state;
    TROVE_keyval_s key, val;
    TROVE_extent cur_extent;
    TROVE_handle_extent_array extent_array;
    TROVE_context_id trove_context = -1;

    ret = parse_args(argc, argv);
    if (ret < 0) {
	fprintf(stderr, "argument parsing failed.\n");
	return -1;
    }

    /* try to initialize; fails if storage space isn't there? */
    ret = trove_initialize(TROVE_METHOD_DBPF, NULL, storage_space, storage_space, 0);
    if (ret < 0) {
	fprintf(stderr, "warning: initialize failed.  trying to create storage space.\n");

	/* create the storage space */
	/* Q: what good is the op_id here if we have to match on coll_id in test fn? */
	ret = trove_storage_create(TROVE_METHOD_DBPF, storage_space, storage_space, NULL, &op_id);
	if (ret < 0) {
	    fprintf(stderr, "storage create failed.\n");
	    return -1;
	}

	/* second try at initialize, in case it failed first try. */
	ret = trove_initialize(TROVE_METHOD_DBPF, NULL, storage_space, storage_space, 0);
	if (ret < 0) {
	    fprintf(stderr, "initialized failed second time.\n");
	    return -1;
	}
    }

    /* try to look up collection used to store file system */
    ret = trove_collection_lookup(
            TROVE_METHOD_DBPF, file_system, &coll_id, NULL, &op_id);
    if (ret >= 0) {
	fprintf(stderr, "collection lookup succeeded before it should.\n");
	return -1;
    }

    ret = trove_open_context(coll_id, &trove_context);
    if (ret < 0)
    {
        fprintf(stderr, "trove_open_context failed\n");
        return -1;
    }

    /* create the collection for the fs */
    /* Q: why do i get to pick the coll id?  so i can make it the same across nodes? */
    ret = trove_collection_create(file_system, FS_COLL_ID, NULL, &op_id);
    if (ret < 0) {
	fprintf(stderr, "collection create (fs) failed.\n");
	return -1;
    }

    /* lookup collection.  this is redundant because we just gave it a coll. id to use,
     * but it's a good test i guess...
     */
    /* NOTE: can't test on this because we still don't know a coll_id */
    ret = trove_collection_lookup(TROVE_METHOD_DBPF, file_system, &coll_id, NULL, &op_id);
    if (ret < 0) {
	fprintf(stderr, "collection lookup failed.\n");
	return -1;
    }


    /* create a dataspace to hold the root directory */
    /* Q: where are we going to define the dspace types? -- trove-test.h for now. */
#if 0
    printf("creating root directory\n");
#endif
    root_handle = 0;

    cur_extent.first = cur_extent.last = 7;
    extent_array.extent_count = 1;
    extent_array.extent_array = &cur_extent;
    ret = trove_dspace_create(coll_id,
                              &extent_array,
			      &root_handle,
			      TROVE_TEST_DIR,
			      NULL,
			      (TROVE_SYNC | TROVE_FORCE_REQUESTED_HANDLE),
			      NULL,
                              trove_context,
			      &op_id,
                              NULL);
    while (ret == 0) ret = trove_dspace_test(
        coll_id, op_id, trove_context, &count, NULL, NULL, &state,
        TROVE_DEFAULT_TEST_TIMEOUT);
    if (ret < 0) {
	fprintf(stderr, "dspace create (for root dir) failed.\n");
	return -1;
    }

    /* add attribute to collection for root handle */
    /* NOTE: should be using the data_sz field, but it doesn't exist yet. */
    /* NOTE: put ROOT_HANDLE_STRING in trove-test.h; not sure where it should be. */
    key.buffer = ROOT_HANDLE_KEYSTR;
    key.buffer_sz = ROOT_HANDLE_KEYLEN;
    val.buffer = &root_handle;
    val.buffer_sz = sizeof(root_handle);
    ret = trove_collection_seteattr(coll_id, &key, &val,
                                    0, NULL, trove_context, &op_id);
    while (ret == 0) ret = trove_dspace_test(
        coll_id, op_id, trove_context, &count, NULL, NULL, &state,
        TROVE_DEFAULT_TEST_TIMEOUT);
    if (ret < 0) {
	fprintf(stderr, "collection seteattr (for root handle) failed.\n");
	return -1;
    }

    /* add attribute to collection for last used handle ??? */

    printf("fs created (root handle = %d, coll id = %d, root string = %s)\n",
	   (int) root_handle,
	   (int) coll_id,
	   ROOT_HANDLE_KEYSTR);

    trove_close_context(coll_id, trove_context);
    trove_finalize(TROVE_METHOD_DBPF);

    return 0;
}

int parse_args(int argc, char **argv)
{
    int c;

    while ((c = getopt(argc, argv, "s:c:")) != EOF) {
	switch (c) {
	    case 's':
		strncpy(storage_space, optarg, SSPACE_SIZE);
		break;
	    case 'c': /* collection */
		strncpy(file_system, optarg, FS_SIZE);
		break;
	    case '?':
	    default:
		return -1;
	}
    }
    return 0;
}

/*
 * Local variables:
 *  c-indent-level: 4
 *  c-basic-offset: 4
 * End:
 *
 * vim: ts=8 sts=4 sw=4 expandtab
 */
