/*
 * (C) 2001 Clemson University and The University of Chicago
 *
 * See COPYING in top-level directory.
 */

/** \defgroup statecomp statecomp source-to-source translator
 *
 *  statecomp is a source-to-source translator.  It takes state machine
 *  descriptions (.sm extension) as input and creates a corresponding C
 *  source file for compilation.  Both clients and servers rely on these
 *  state machines for concurrent processing.  This executable is built at
 *  compile time and used only during compilation of PVFS2.
 *
 * @{
 */

/** \file
 *
 *  Core of state machine source-to-source translator executable, statecomp,
 *  including processing arguments, calling the parser, and producing
 *  warning and error messages.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef WIN32
#include <unistd.h>
#endif

/* force malloc to not be redefined for statecomp */
#define PVFS_MALLOC_REDEF_OVERRIDE 1
#include "pvfs2-internal.h"
#include "statecomp.h"

#ifdef __GNUC__
#ifdef YYPARSE_PARAM
int yyparse (void *);
#else
int yyparse (void);
#endif
#endif

#ifdef WIN32
#define __func__    __FUNCTION__
#define unlink      _unlink

extern int yyparse();
#endif

/*
 * Global Variables
 */
struct state *states = NULL;
int terminate_path_flag = 0;
int line = 1;
FILE *out_file;
const char *in_file_name;

static const char *progname;
static char *out_file_name;

static void parse_args(int argc, char **argv);
static void finalize(void);

int main(int argc, char **argv)
{
    int retval;
    parse_args(argc, argv);
    retval = yyparse();
    switch (retval)
    {
        case 0:
            /* successful parse */
            break;
        case 1:
            /* syntax error */
            fprintf(stderr,"yyparse returned syntax error\n");
            break;
        case 2:
            /* out of memory error */
            fprintf(stderr,"yyparse returned out of memory error\n");
            break;
        default:
            /* unknown error */
            fprintf(stderr,"yyparse returned unknown error\n");
            break;
    }
    finalize();
    return retval;
}

static void usage(void)
{
    fprintf(stderr, "Usage: %s [-l] input_file.sm [output_file.c]\n", progname);
    exit(1);
}

static void parse_args(int argc, char **argv)
{
    int len;
    const char *cp;

    for (cp=progname=argv[0]; *cp; cp++)
	if (*cp == '/')
	    progname = cp+1;

    if (argc < 2 || argc > 3)
        usage();

    in_file_name = argv[1];
    out_file_name = argc == 3 ? argv[2] : NULL;

    len = strlen(in_file_name);
    if (len <= 3 || strcmp(&in_file_name[len-3], ".sm") != 0)
        usage();

    if (!freopen(in_file_name, "r", stdin))
    {
	perror("open input file");
	exit(1);
    }

    if (!out_file_name) {
	/* construct output file name from input file name */
	out_file_name = estrdup(in_file_name);
	out_file_name[len-2] = 'c';
	out_file_name[len-1] = 0;
    }

    out_file = fopen(out_file_name, "w");
    if (!out_file)
    {
        perror("opening output file");
        exit(1);
    }

    /* dump header comment into out file */
    fprintf(out_file,
    "/* WARNING: THIS FILE IS AUTOMATICALLY GENERATED FROM A .SM FILE.\n");
    fprintf(out_file,
    " * Changes made here will certainly be overwritten.\n");
    fprintf(out_file,
    " */\n\n");
}

static void finalize(void)
{
    if (!terminate_path_flag) {
	fprintf(stderr, "%s: %s: state machine contains no explicit"
                " exit path (terminate or return transition).\n", progname,
                in_file_name);
        unlink(out_file_name);
        exit(1);
    }
    fclose(out_file);
}

void yyerror(char *s)
{
    fprintf(stderr, "%s: %s:%d: %s\n", progname, in_file_name, line, s);
    unlink(out_file_name);
    exit(1);
}

/*
 * Error checking malloc.
 */
void *emalloc(size_t size)
{
    void *p;

    p = malloc(size);
    if (!p) {
	fprintf(stderr, "%s: no more dynamic storage - aborting\n", progname);
	exit(1);
    }
    return p;
}

char *estrdup(const char *oldstring)
{
    char *s;

    s = emalloc(strlen(oldstring)+1);
    strcpy(s, oldstring);
    return s;
}

struct state *new_state(char *name)
{
    struct state *s, **sprev = &states;

    for (s=states; s; s=s->next) {
        if (!strcmp(s->name, name)) {
            fprintf(stderr, "%s: state %s already exists\n", __func__, name);
            exit(1);
        }
        sprev = &s->next;
    }
    s = emalloc(sizeof(*s));
    s->name = name;
    s->transition = NULL;
    s->task = NULL;
    s->next = NULL;
    *sprev = s;
    return s;
}

struct transition *new_transition(struct state *s, char *return_code)
{
    struct transition *t, **tprev = &s->transition;

    for (t=s->transition; t; t=t->next) {
        if (!strcmp(t->return_code, return_code)) {
            fprintf(stderr, "%s: state %s: return code %s already exists\n",
                    __func__, s->name, return_code);
            exit(1);
        }
        tprev = &t->next;
    }
    t = emalloc(sizeof(*t));
    t->return_code = return_code;
    t->next = NULL;
    *tprev = t;
    return t;
}

struct task *new_task(
    struct state *s, char *return_code, char *task_name)
{
    struct task *t, **tprev;

    /* allocate for the new task */
    t = emalloc(sizeof(*t));
    t->return_code = return_code;
    t->task_name = task_name;
    t->next = NULL;

    /* move to the end of the task list */
    tprev = &s->task;
    while(*tprev)
    {
        tprev = &(*tprev)->next;
    }

    /* add the new task to the end of the list */
    *tprev = t;
    return t;
}

/* @} */

/*
 * Local variables:
 *  c-indent-level: 4
 *  c-basic-offset: 4
 * End:
 *
 * vim: ts=8 sts=4 sw=4 expandtab
 */
