/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/err.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/of.h>
#include <video/of_display_timing.h>
#include <linux/fb.h>
#include <linux/lcd.h>
#include <linux/dma-mapping.h>
#include <linux/pinctrl/pinctrl.h>
#include <linux/platform_data/video-mx3fb.h>

#define NR_IRQS_LEGACY	16
#define MX35_INT_IPU_ERR	(NR_IRQS_LEGACY + 41)
#define MX35_INT_IPU_SYN	(NR_IRQS_LEGACY + 42)

#define MX35_AIPS2_BASE_ADDR		0x53f00000
#define MX35_AIPS2_SIZE			SZ_1M
#define MX35_IPU_CTRL_BASE_ADDR	(MX35_AIPS2_BASE_ADDR + 0xc0000)

extern int cicada_hw_type(void);

struct cicada_lcd_data {
	struct device *dev;
	struct platform_device *ipu;
	struct platform_device *fb;
	struct platform_device_info fbdev_info;
	struct mx3fb_platform_data mx3fb_pdata;
	struct lcd_device *lcd_dev;
};

static int cicada_lcd_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct cicada_lcd_data *data;
	struct device *ipu_dev;
	struct platform_device *ipu_pdev;
	struct resource *ipu_res;
	struct resource fb_res;
	int retcode;

	dev_dbg(dev, "%s\n", __func__);

	retcode = cicada_hw_type();
	if (retcode < 0)
		return retcode;

	if (retcode != 0)
		/* remote unit with no lcd display */
		return -ENODEV;

	if (!np)
		return -EINVAL;

	if (!of_device_is_available(np))
		return -ENODEV;

	data = devm_kzalloc(dev, sizeof(*data), GFP_KERNEL);
	if (!data) {
		dev_err(dev, "could not allocate object\n");
		return -ENOMEM;
	}

	data->dev = dev;
	dev_set_drvdata(dev, data);

	ipu_dev = bus_find_device_by_name(&platform_bus_type, NULL, "ipu-core");
	if (IS_ERR_OR_NULL(ipu_dev)) {
		dev_err(dev, "failed to find ipu-core device\n");
		return PTR_ERR(ipu_dev);
	}

	ipu_pdev = to_platform_device(ipu_dev);
	ipu_res = platform_get_resource(ipu_pdev, IORESOURCE_MEM, 0);
	if (IS_ERR_OR_NULL(ipu_res)) {
		dev_err(dev, "failed to get ipu resource\n");
		return -ENODEV;
	}

	/* ipu registers
	 *	SDC (synchronous display controller) registers
	 *	ADC (asynchronous display controller) registers
	 *	DI (Display Interface) registers */
	fb_res.start = ipu_res->start + 0xb4;
	fb_res.end =  ipu_res->start + 0x1bf;
	fb_res.flags = IORESOURCE_MEM;

	data->mx3fb_pdata.dma_dev = ipu_dev;

	/* using only the native mode */
	data->mx3fb_pdata.num_modes = 1;

	data->mx3fb_pdata.mode = devm_kzalloc(dev,
			sizeof(struct fb_videomode), GFP_KERNEL);
	if (!data->mx3fb_pdata.mode) {
		return -ENOMEM;
	}

	retcode = of_property_read_string(np, "model", &data->mx3fb_pdata.name);
	if (retcode)
		data->mx3fb_pdata.name = NULL;

	retcode = of_get_fb_videomode(np, (struct fb_videomode *)data->mx3fb_pdata.mode, OF_USE_NATIVE_MODE);
	if (retcode) {
		dev_err(dev, "Failed to get videomode from DT\n");
		return retcode;
	}

	data->fbdev_info.name = "mx3_sdc_fb",
	data->fbdev_info.id = -1,
	data->fbdev_info.res = &fb_res,
	data->fbdev_info.num_res = 1,
	data->fbdev_info.data = &data->mx3fb_pdata,
	data->fbdev_info.size_data = sizeof(data->mx3fb_pdata),
	data->fbdev_info.dma_mask = DMA_BIT_MASK(32),

#if 0
	data->lcd_dev = devm_lcd_device_register(dev, "mx3_fb", NULL, &data->fbdev_info, NULL);
	if (IS_ERR_OR_NULL(data->lcd_dev)) {
		dev_err(dev, "failed to register pdev mx3_fb\n");
		return -ENODEV;
	}
#else
	data->fb = platform_device_register_full(&data->fbdev_info);
	if (IS_ERR_OR_NULL(data->fb)) {
		dev_err(dev, "failed to register pdev mx3_sdc_fb\n");
		return -ENODEV;
	}
#endif

	return 0;
}

static int cicada_lcd_remove(struct platform_device *pdev)
{
	dev_dbg(&pdev->dev, "%s\n", __func__);

	return 0;
}

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-lcd", .data = NULL },
	{}
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

static struct platform_driver cicada_lcd_driver = {
	.driver = {
		.name = "cicada-lcd",
		.of_match_table = cicada_device_ids,
	},
	.probe	  = cicada_lcd_probe,
	.remove   = cicada_lcd_remove,
};
module_platform_driver(cicada_lcd_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("Cicada Lcd");


