/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/gpio/consumer.h>
#include <linux/of.h>
#include <linux/power_supply.h>
#include <linux/types.h>

struct lt3650_charger {
	struct device *dev;

	const char *name;
	struct power_supply *psy;
	struct power_supply_desc psy_desc;
	enum power_supply_type type;

	char **supplied_to;
	size_t num_supplicants;

	struct gpio_desc *enable_gpio;
	struct gpio_desc *online_gpio;
	struct gpio_desc *charging_gpio;
	struct gpio_desc *fault_gpio;
};

static ssize_t enable_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct lt3650_charger *data = power_supply_get_drvdata(psy);

	return sprintf(buf, "%u\n", gpiod_get_value(data->enable_gpio));
}

static ssize_t enable_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t n)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct lt3650_charger *data = power_supply_get_drvdata(psy);
	unsigned int value;
	if (kstrtouint(buf, 2, &value) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	gpiod_set_value(data->enable_gpio, value);

	return n;
}

static DEVICE_ATTR_RW(enable);

static struct attribute *dev_attrs[] = {
	&dev_attr_enable.attr,
	NULL
};

static const struct attribute_group dev_attr_group = {
	.attrs = dev_attrs,
};

static int lt3650_charger_get_property(struct power_supply *psy,
		enum power_supply_property psp, union power_supply_propval *propval)
{
	struct lt3650_charger *data = power_supply_get_drvdata(psy);
	int val;

	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		val = (gpiod_get_value(data->charging_gpio) << 1) |
		      gpiod_get_value(data->fault_gpio);
		switch (val) {
		case 0:
			/* not charging */
			propval->intval = POWER_SUPPLY_STATUS_NOT_CHARGING;
			break;
		case 1:
			/* bad battery fault */
			propval->intval = POWER_SUPPLY_STATUS_NOT_CHARGING;
			break;
		case 2:
			/* charging */
			propval->intval = POWER_SUPPLY_STATUS_CHARGING;
			break;
		case 3:
			/* temp fault */
			propval->intval = POWER_SUPPLY_STATUS_NOT_CHARGING;
			break;
		default:
			propval->intval = POWER_SUPPLY_STATUS_UNKNOWN;
			break;
		}
		break;
	case POWER_SUPPLY_PROP_ONLINE:
		propval->intval = gpiod_get_value(data->online_gpio);
		break;
	case POWER_SUPPLY_PROP_HEALTH:
		val = (gpiod_get_value(data->charging_gpio) << 1) |
		      gpiod_get_value(data->fault_gpio);
		switch (val) {
		case 0:
			/* not charging */
			propval->intval = POWER_SUPPLY_HEALTH_GOOD;
			break;
		case 1:
			/* bad battery fault */
			propval->intval = POWER_SUPPLY_HEALTH_UNSPEC_FAILURE;
			break;
		case 2:
			/* charging */
			propval->intval = POWER_SUPPLY_HEALTH_GOOD;
			break;
		case 3:
			/* temp fault */
			propval->intval = POWER_SUPPLY_HEALTH_OVERHEAT;
			break;
		default:
			propval->intval = POWER_SUPPLY_HEALTH_UNKNOWN;
			break;
		}
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

static enum power_supply_property lt3650_charger_properties[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_ONLINE,
	POWER_SUPPLY_PROP_HEALTH,
};

static int lt3650_charger_parse_dt(struct device *dev, struct lt3650_charger *data)
{
	struct device_node *np = dev->of_node;

	if (!np)
		return -ENOENT;

	if (!of_device_is_available(np))
		return -ENODEV;

	data->name = np->name;
	data->type = POWER_SUPPLY_TYPE_MAINS;

	data->enable_gpio = devm_gpiod_get(dev, "enable", GPIOD_OUT_HIGH);
	if (IS_ERR(data->enable_gpio)) {
		dev_err(dev, "could not get enable gpio\n");
		return PTR_ERR(data->enable_gpio);
	}

	data->fault_gpio = devm_gpiod_get(dev, "fault", GPIOD_IN);
	if (IS_ERR(data->fault_gpio)) {
		dev_err(dev, "could not get fault gpio\n");
		return PTR_ERR(data->fault_gpio);
	}

	data->charging_gpio = devm_gpiod_get(dev, "charging", GPIOD_IN);
	if (IS_ERR(data->charging_gpio)) {
		dev_err(dev, "could not get charging gpio\n");
		return PTR_ERR(data->charging_gpio);
	}

	data->online_gpio = devm_gpiod_get(dev, "online", GPIOD_IN);
	if (IS_ERR(data->online_gpio)) {
		dev_err(dev, "could not get online gpio\n");
		return PTR_ERR(data->online_gpio);
	}

	return 0;
}

static char *supplied_to[] = {
	"main_battery",
};

static int lt3650_charger_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct power_supply_config psy_cfg = {};
	struct lt3650_charger *data;
	struct power_supply_desc *psy_desc;
	int ret;

	dev_info(&pdev->dev, "%s:\n", __func__);

	data = devm_kzalloc(&pdev->dev, sizeof(*data), GFP_KERNEL);
	if (IS_ERR(data)) {
		dev_err(&pdev->dev, "no memory\n");
		return PTR_ERR(data);
	}

	ret = lt3650_charger_parse_dt(&pdev->dev, data);
	if (ret) {
		dev_err(&pdev->dev, "probe failed\n");
		return ret;
	}

	data->dev = dev;
	dev_set_drvdata(dev, data);

	psy_desc = &data->psy_desc;
	psy_desc->name = data->name ? data->name : "lt3650-charger";
	psy_desc->type = POWER_SUPPLY_TYPE_MAINS;
	psy_desc->properties = lt3650_charger_properties;
	psy_desc->num_properties = ARRAY_SIZE(lt3650_charger_properties);
	psy_desc->get_property = lt3650_charger_get_property;

	psy_cfg.supplied_to = supplied_to;
	psy_cfg.num_supplicants = ARRAY_SIZE(supplied_to);
	psy_cfg.of_node = pdev->dev.of_node;
	psy_cfg.drv_data = data;

	data->psy = devm_power_supply_register(&pdev->dev, psy_desc, &psy_cfg);
	if (IS_ERR(data->psy)) {
		ret = PTR_ERR(data->psy);
		dev_err(&pdev->dev, "Failed to register power supply: %d\n",
			ret);
		return PTR_ERR(data->psy);
	}

	ret = sysfs_create_group(&data->psy->dev.kobj, &dev_attr_group);
	if (ret) {
		dev_err(&pdev->dev, "%s, sysfs_create_group failed\n", __func__);
		return ret;
	}

	dev_info(&pdev->dev, "probe succeded\n");

	return 0;
}

static int lt3650_charger_remove(struct platform_device *pdev)
{
	struct lt3650_charger *data = platform_get_drvdata(pdev);

	dev_dbg(&pdev->dev, "%s:\n", __func__);

	sysfs_remove_group(&data->psy->dev.kobj, &dev_attr_group);

	return 0;
}

static const struct of_device_id lt3650_device_ids[] = {
	{ .compatible = "fnet,lt3650_charger", },
	{}
};
MODULE_DEVICE_TABLE(of, lt3650_device_ids);

static struct platform_driver lt3650_charger_driver = {
	.driver = {
		.name = "lt3650-charger",
		.of_match_table = lt3650_device_ids,
	},
	.probe	  = lt3650_charger_probe,
	.remove   = lt3650_charger_remove,
};

module_platform_driver(lt3650_charger_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("lt3650 battery charger");

