/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <linux/module.h>
#include <linux/err.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/platform_device.h>

extern int cicada_hw_type(void);

enum {
	RMT_LED_PASS	= 0x01,
	RMT_LED_TEST	= 0x02,
	RMT_LED_FAIL	= 0x04,
	RMT_LED_TALK	= 0x08,
	RMT_LED_TONE	= 0x10,
	RMT_LED_LOWBATT	= 0x20,
	RMT_LED_ALL	= 0x3F,
} rmt_led_t;

struct cicada_rmt_led_data {
	struct device *dev;

	struct gpio_desc *gpio_led_tone;
	struct gpio_desc *gpio_led_lowbat;
	struct gpio_desc *gpio_led_pass;
	struct gpio_desc *gpio_led_test;
	struct gpio_desc *gpio_led_fail;
	struct gpio_desc *gpio_led_talk;
};

static ssize_t pass_show(struct device *dev,
			 struct device_attribute *attr,
			 char *buf)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", gpiod_get_value(data->gpio_led_pass));
}

static ssize_t pass_store(struct device *dev,
			  struct device_attribute *attr,
			  const char *buf, size_t n)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 2, &value) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	gpiod_set_value(data->gpio_led_pass, value);

	return n;
}

DEVICE_ATTR_RW(pass);

static ssize_t test_show(struct device *dev,
			 struct device_attribute *attr,
			 char *buf)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", gpiod_get_value(data->gpio_led_test));
}

static ssize_t test_store(struct device *dev,
			  struct device_attribute *attr,
			  const char *buf, size_t n)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 2, &value) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	gpiod_set_value(data->gpio_led_test, value);

	return n;
}

DEVICE_ATTR_RW(test);

static ssize_t fail_show(struct device *dev,
			struct device_attribute *attr,
			char *buf)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", gpiod_get_value(data->gpio_led_fail));
}

static ssize_t fail_store(struct device *dev,
			struct device_attribute *attr,
			const char *buf, size_t n)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 2, &value) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	gpiod_set_value(data->gpio_led_fail, value);

	return n;
}

DEVICE_ATTR_RW(fail);

static ssize_t talk_show(struct device *dev,
			struct device_attribute *attr,
			char *buf)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", gpiod_get_value(data->gpio_led_talk));
}

static ssize_t talk_store(struct device *dev,
			struct device_attribute *attr,
			const char *buf, size_t n)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 2, &value) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	gpiod_set_value(data->gpio_led_talk, value);

	return n;
}

DEVICE_ATTR_RW(talk);

static ssize_t tone_show(struct device *dev,
			struct device_attribute *attr,
			char *buf)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", gpiod_get_value(data->gpio_led_tone));
}

static ssize_t tone_store(struct device *dev,
			struct device_attribute *attr,
			const char *buf, size_t n)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 2, &value) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	gpiod_set_value(data->gpio_led_tone, value);

	return n;
}

DEVICE_ATTR_RW(tone);

static ssize_t lowbatt_show(struct device *dev,
			struct device_attribute *attr,
			char *buf)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", gpiod_get_value(data->gpio_led_lowbat));
}

static ssize_t lowbatt_store(struct device *dev,
			     struct device_attribute *attr,
			     const char *buf, size_t n)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 2, &value) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	gpiod_set_value(data->gpio_led_lowbat, value);

	return n;
}

DEVICE_ATTR_RW(lowbatt);

static ssize_t all_show(struct device *dev,
			struct device_attribute *attr,
			char *buf)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);
	static char *led_text[] = {
		"PASS", "TEST", "FAIL", "TALK", "TONE", "LOWBATT"
	};
	int val = 0;
	int ccount = 0;
	int i;

	val |= gpiod_get_value(data->gpio_led_pass) ? RMT_LED_PASS : 0;
	val |= gpiod_get_value(data->gpio_led_test) ? RMT_LED_TEST : 0;
	val |= gpiod_get_value(data->gpio_led_fail) ? RMT_LED_FAIL : 0;
	val |= gpiod_get_value(data->gpio_led_talk) ? RMT_LED_TALK : 0;
	val |= gpiod_get_value(data->gpio_led_tone) ? RMT_LED_TONE : 0;
	val |= gpiod_get_value(data->gpio_led_lowbat) ? RMT_LED_LOWBATT : 0;

	ccount += sprintf(buf + ccount, "0x%02x ", val);
	for (i = sizeof(led_text); i >= 0; i--) {
		if (val & (1 << i))
			ccount += sprintf(buf + ccount, "%s ", led_text[i]);
	}

	ccount += sprintf(buf + ccount, "\n");

	return ccount;
}

static ssize_t all_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t n)
{
	struct cicada_rmt_led_data *data = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 0, &value) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	gpiod_set_value(data->gpio_led_tone, !!(value & RMT_LED_TONE));
	gpiod_set_value(data->gpio_led_lowbat, !!(value & RMT_LED_LOWBATT));
	gpiod_set_value(data->gpio_led_pass, !!(value & RMT_LED_PASS));
	gpiod_set_value(data->gpio_led_test, !!(value & RMT_LED_TEST));
	gpiod_set_value(data->gpio_led_fail, !!(value & RMT_LED_FAIL));
	gpiod_set_value(data->gpio_led_talk, !!(value & RMT_LED_TALK));

	return n;
}

DEVICE_ATTR_RW(all);


static struct attribute *dev_attrs[] = {
	&dev_attr_pass.attr,
	&dev_attr_test.attr,
	&dev_attr_fail.attr,
	&dev_attr_talk.attr,
	&dev_attr_tone.attr,
	&dev_attr_lowbatt.attr,
	&dev_attr_all.attr,
	NULL
};

static const struct attribute_group dev_attr_group = {
	.attrs = dev_attrs,
};

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-rmt-led" },
	{}
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

static int cicada_rmt_led_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct cicada_rmt_led_data *data;
	int retval = 0;

	pr_info("Cicada Platform Hardware Id: remote led display\n");

	if (!np)
		return -ENOENT;

	if (!of_device_is_available(np))
		return -ENODEV;

	data = devm_kzalloc(dev, sizeof(*data), GFP_KERNEL);
	if (!data) {
		dev_err(dev, "no memory\n");
		return -ENOMEM;
	}

	dev_set_drvdata(dev, data);

	data->gpio_led_tone = devm_gpiod_get(dev, "led-tone", GPIOD_OUT_HIGH);
	if (IS_ERR(data->gpio_led_tone)) {
		dev_err(dev, "could not get led-tone gpio\n");
		return PTR_ERR(data->gpio_led_tone);
	}

	data->gpio_led_lowbat = devm_gpiod_get(dev, "led-lowbat",
					       GPIOD_OUT_HIGH);
	if (IS_ERR(data->gpio_led_lowbat)) {
		dev_err(dev, "could not get led-lowbat gpio\n");
		return PTR_ERR(data->gpio_led_lowbat);
	}

	data->gpio_led_pass = devm_gpiod_get(dev, "led-pass", GPIOD_OUT_HIGH);
	if (IS_ERR(data->gpio_led_pass)) {
		dev_err(dev, "could not get led-pass gpio\n");
		return PTR_ERR(data->gpio_led_pass);
	}

	data->gpio_led_test = devm_gpiod_get(dev, "led-test", GPIOD_OUT_HIGH);
	if (IS_ERR(data->gpio_led_test)) {
		dev_err(dev, "could not get led-test gpio\n");
		return PTR_ERR(data->gpio_led_test);
	}

	data->gpio_led_fail = devm_gpiod_get(dev, "led-fail", GPIOD_OUT_HIGH);
	if (IS_ERR(data->gpio_led_fail)) {
		dev_err(dev, "could not get led-fail gpio\n");
		return PTR_ERR(data->gpio_led_fail);
	}

	data->gpio_led_talk = devm_gpiod_get(dev, "led-talk", GPIOD_OUT_HIGH);
	if (IS_ERR(data->gpio_led_talk)) {
		dev_err(dev, "could not get led-talk gpio\n");
		return PTR_ERR(data->gpio_led_talk);
	}

	retval = sysfs_create_group(&dev->kobj, &dev_attr_group);
	if (retval) {
		dev_err(dev, "%s, sysfs_create_group failed\n", __func__);
		goto error;
	}

	dev_info(dev, "success\n");

	return 0;

error:
	return retval;
}

static int cicada_rmt_led_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;

	dev_dbg(dev, "%s\n", __func__);

	sysfs_remove_group(&dev->kobj, &dev_attr_group);

	return 0;
}

static struct platform_driver cicada_rmt_led_driver = {
	.driver = {
		.name = "cicada-rmt-led",
		.of_match_table = cicada_device_ids,
	},
	.probe	  = cicada_rmt_led_probe,
	.remove   = cicada_rmt_led_remove,
};

module_platform_driver(cicada_rmt_led_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("Cicada Remote LED");

