/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define DEBUG
#include <linux/printk.h>
#include <linux/module.h>
#include <linux/pm.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>

struct cicada_module_if_data {
	struct device *dev;

	struct gpio_desc *special_enable_gpios;
	struct gpio_desc *special_data_gpios;
	struct gpio_desc *vbat_enable_gpios;
	struct gpio_desc *vbat_fault_gpios;
};

struct cicada_module_if_data *cicada_module_if_data = NULL;

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-module-if", .data = NULL },
	{}
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

/*=================================================*/
void cicada_set_special_enable(int enable)
{
	if (!cicada_module_if_data) {
		pr_err("%s, no device\n", __func__);
		return;
	}

	pr_debug("%s, %s\n", __func__, (enable ? "enabled" : "disabled"));

	gpiod_set_value(cicada_module_if_data->special_enable_gpios, !!enable);
}
EXPORT_SYMBOL(cicada_set_special_enable);

void cicada_set_special(int val)
{
	if (!cicada_module_if_data) {
		pr_err("%s, no device\n", __func__);
		return;
	}

	pr_debug("%s, %s\n", __func__, (val ? "high" : "low"));

	gpiod_set_value(cicada_module_if_data->special_data_gpios, !!val);
}
EXPORT_SYMBOL(cicada_set_special);

int cicada_get_special(void)
{
	int val;

	if (!cicada_module_if_data) {
		pr_err("%s, no device\n", __func__);
		return -ENODEV;
	}

	val = gpiod_get_value(cicada_module_if_data->special_data_gpios);
	pr_debug("%s, %s\n", __func__, (val ? "high" : "low"));

	return val;
}
EXPORT_SYMBOL(cicada_get_special);

void cicada_set_module_vbat(int enable)
{
	if (!cicada_module_if_data) {
		pr_err("%s, no device\n", __func__);
		return;
	}

	pr_debug("%s, %s\n", __func__, (enable ? "enabled" : "disabled"));

	gpiod_set_value(cicada_module_if_data->vbat_enable_gpios, !!enable);
}
EXPORT_SYMBOL(cicada_set_module_vbat);

#define CICADA_VBAT_ENABLE	0
#define CICADA_VBAT_FAULT	1
int cicada_get_module_vbat(void)
{
	int val;

	if (!cicada_module_if_data) {
		pr_err("%s, no device\n", __func__);
		return -ENODEV;
	}

	val = gpiod_get_value(cicada_module_if_data->vbat_enable_gpios) << CICADA_VBAT_ENABLE;
	val |= !!gpiod_get_value(cicada_module_if_data->vbat_fault_gpios) << CICADA_VBAT_FAULT;

	pr_debug("%s, %s %s\n", __func__,
                 ((val & (1 << CICADA_VBAT_ENABLE)) ? "enabled" : "disabled"),
                 ((val & (1 << CICADA_VBAT_FAULT)) ? "fault" : ""));

	return val;
}
EXPORT_SYMBOL(cicada_get_module_vbat);

/*=================================================*/
static ssize_t special_enable_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	struct cicada_module_if_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", gpiod_get_value(data->special_enable_gpios));
}

static ssize_t special_enable_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t n)
{
	struct cicada_module_if_data *data = dev_get_drvdata(dev);
	unsigned int val;
	if (kstrtouint(buf, 2, &val) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	gpiod_set_value(data->special_enable_gpios, val);

	return n;
}

static DEVICE_ATTR_RW(special_enable);

static ssize_t special_data_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	struct cicada_module_if_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", gpiod_get_value(data->special_data_gpios));
}

static ssize_t special_data_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t n)
{
	struct cicada_module_if_data *data = dev_get_drvdata(dev);
	unsigned int val;
	if (kstrtouint(buf, 2, &val) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	gpiod_set_value(data->special_data_gpios, val);

	return n;
}

static DEVICE_ATTR_RW(special_data);

static ssize_t vbat_enable_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	struct cicada_module_if_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", gpiod_get_value(data->vbat_enable_gpios));
}

static ssize_t vbat_enable_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t n)
{
	struct cicada_module_if_data *data = dev_get_drvdata(dev);
	unsigned int val;
	if (kstrtouint(buf, 2, &val) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	gpiod_set_value(data->vbat_enable_gpios, val);

	return n;
}

static DEVICE_ATTR_RW(vbat_enable);

static ssize_t vbat_fault_show(struct device *dev,
			   struct device_attribute *attr,
			   char *buf)
{
	struct cicada_module_if_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", gpiod_get_value(data->vbat_fault_gpios));
}

static DEVICE_ATTR_RO(vbat_fault);

static struct attribute *dev_attrs[] = {
	&dev_attr_special_enable.attr,
	&dev_attr_special_data.attr,
	&dev_attr_vbat_enable.attr,
	&dev_attr_vbat_fault.attr,
	NULL,
};

static const struct attribute_group dev_attr_group = {
	.attrs = dev_attrs,
};

/*=================================================*/
static int cicada_module_if_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct cicada_module_if_data *data;
	int ret = 0;

	dev_dbg(dev, "%s\n", __func__);

	if (cicada_module_if_data) {
		dev_err(dev, "already allocated\n");
		return -ENODEV;
	}

	if (!np)
		return -ENOENT;

	if (!of_device_is_available(np))
		return -ENODEV;

	data = devm_kzalloc(dev, sizeof(*data), GFP_KERNEL);
	if (!data) {
		dev_err(dev, "no memory\n");
		return -ENOMEM;
	}

	/* enable special, output */
	data->special_enable_gpios = devm_gpiod_get(dev, "special-enable", GPIOD_OUT_LOW);
	if (IS_ERR(data->special_enable_gpios)) {
		dev_err(dev, "could not get special-enable gpio\n");
		return PTR_ERR(data->special_enable_gpios);
	}

	/* special data, output */
	data->special_data_gpios = devm_gpiod_get(dev, "special-data", GPIOD_OUT_HIGH);
	if (IS_ERR(data->special_data_gpios)) {
		dev_err(dev, "could not get special-data gpio\n");
		return PTR_ERR(data->special_data_gpios);
	}

	/* v bat enable, output */
	data->vbat_enable_gpios = devm_gpiod_get(dev, "vbat-enable", GPIOD_OUT_LOW);
	if (IS_ERR(data->vbat_enable_gpios)) {
		dev_err(dev, "could not get vbat-enable gpio\n");
		return PTR_ERR(data->vbat_enable_gpios);
	}

	/* v bat over current, input */
	data->vbat_fault_gpios = devm_gpiod_get(dev, "vbat-fault", GPIOD_IN);
	if (IS_ERR(data->vbat_fault_gpios)) {
		dev_err(dev, "could not get vbat-fault gpio\n");
		return PTR_ERR(data->vbat_fault_gpios);
	}

	data->dev = dev;

	dev_set_drvdata(dev, data);

	cicada_module_if_data = data;

	ret = sysfs_create_group(&dev->kobj, &dev_attr_group);
	if (ret) {
		dev_err(dev, "%s, sysfs_create_group failed\n", __func__);
		return -ENODEV;
	}

	return 0;
}

static int cicada_module_if_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;

	sysfs_remove_group(&dev->kobj, &dev_attr_group);

	return 0;
}

static struct platform_driver cicada_module_if_driver = {
	.driver = {
		.name = "cicada-module-if",
		.of_match_table = cicada_device_ids,
	},
	.probe	  = cicada_module_if_probe,
	.remove   = cicada_module_if_remove,
};

module_platform_driver(cicada_module_if_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("Cicada Module Interface");




