/*
 *   fb.c
 *
 *   (C) Richard Drummond 2001-2007
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 */

#include "defs.h"
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <string.h>

#include "fb.h"
#include "surface.h"
#include "cmap.h"

#ifdef SUPPORT_VGA16FB
#include "vga.h"
#endif


/*
 * Type used to map the framebuffer's bitfield signature
 * to a Surface pixel format
 */
struct pixel_type_mapping
{
  unsigned short     pixel_type;
  unsigned short     bits_per_pixel;
  struct fb_bitfield red;
  struct fb_bitfield green;
  struct fb_bitfield blue;
};

/*
 * Table mapping truecolor/directcolor packed-pixel bitfield signatures
 * to pixel formats (we're ignoring any alpha channel here).
 */
static const struct pixel_type_mapping packed_type_map[] =
{
   { PIXEL_TYPE_x1r5g5b5, 16, { 10, 5, 0 }, { 5, 5, 0 }, { 0, 5, 0 } },
   { PIXEL_TYPE_r5g6b5,   16, { 11, 5, 0 }, { 5, 6, 0 }, { 0, 5, 0 } },
   { PIXEL_TYPE_r8g8b8,   24, { 16, 8, 0 }, { 8, 8, 0 }, { 0, 8, 0 } },
   { PIXEL_TYPE_x8r8g8b8, 32, { 16, 8, 0 }, { 8, 8, 0 }, { 0, 8, 0 } },
   { PIXEL_TYPE_UNKNOWN,  0,  {  0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 } }
};

/*
 * Return the pixel format type of a packed-pixel framebuffer
 */
static int get_packed_pixel_type( FBHandle *fb )
{
  int i = 0;

  if( fb->fix.visual == FB_VISUAL_PSEUDOCOLOR || fb->fix.visual == FB_VISUAL_STATIC_PSEUDOCOLOR )
  {
     if( fb->var.bits_per_pixel == 8 )
       return PIXEL_TYPE_CLUT8;
  }
  else
    while( packed_type_map[i].pixel_type != PIXEL_TYPE_UNKNOWN )
    {
      if( fb->var.bits_per_pixel == packed_type_map[i].bits_per_pixel )
      {
        if( fb->var.red.length   == packed_type_map[i].red.length   &&
            fb->var.red.offset   == packed_type_map[i].red.offset   &&
            fb->var.green.length == packed_type_map[i].green.length &&
            fb->var.green.offset == packed_type_map[i].green.offset &&
            fb->var.blue.length  == packed_type_map[i].blue.length  &&
            fb->var.blue.offset  == packed_type_map[i].blue.offset )
        return packed_type_map[i].pixel_type;
      }
      i++;
    } // while

  return PIXEL_TYPE_UNKNOWN;
}

/*
 * Get the pixel format of a framebuffer
 *
 * Returns PIXEL_TYPE_UNKNOWN for unsupported framebuffer types
 */
static int get_fb_pixel_type( FBHandle *fb )
{
  int type = PIXEL_TYPE_UNKNOWN;

  switch( fb->fix.type )
  {
    case FB_TYPE_VGA_PLANES:
      if( fb->var.bits_per_pixel == 4 )
        type = PIXEL_TYPE_VGA4;
      break;

    case FB_TYPE_PACKED_PIXELS:
      type = get_packed_pixel_type( fb );
      break;
  }
  return type;
}


/*
 * Open an initalize a framebuffer device
 *
 * Returns 0 on success, -1 on error.
 */
int FB_open( const char *device, FBHandle *fb )
{
  int fd;

  memset( fb, 0, sizeof(FBHandle) );

  /*
   * Try to open the specified fb device
   */
  if ( (fd = open( device, O_RDWR )) != -1 )
  {
    fb->fd = fd;
    /*
     * Okay, now retrieve the fb device's fixed properties
     */
    if( (ioctl( fd, FBIOGET_FSCREENINFO, &fb->fix )) != -1 )
    {
      /*
       * Find the framebuffer size and map it into our memory space
       */
      fb->len = fb->fix.smem_len; // fb->fix.line_length * fb->var.yres_virtual;

      if( (fb->mem = mmap( 0, fb->len, PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0
                                                              )) != MAP_FAILED )
      {
#ifdef SUPPORT_VGA16FB
        /* Initalize VGA access for VGA fb */
        if( fb->fix.type == FB_TYPE_VGA_PLANES )
          return VGA_init();
#endif
        /* success */
        return 0;
      }
    }
    close( fd );
  }
  return -1;
}


/*
 * Return a copy of the framebuffer's colormap
 */
ColorMap *FB_getColorMap( FBHandle *fb )
{
  ColorMap *cmap;

  // Allocate a ColorMap instance
  if( (cmap = ColorMap_alloc( 256 )) != 0 )
  {
    // And try to get a copy of the current palette
    if( ( ioctl( fb->fd, FBIOGETCMAP, cmap)) != -1 )
    {
      if( fb->fix.visual == FB_VISUAL_STATIC_PSEUDOCOLOR )
      {
        // If we're running on a static pseudocolor fb then
        // FBIOGETCMAP returns an empty Colormap. We assume
        // that, since the palette is static, getting a copy
        // of the default 16 pens will do.
        ColorMap_copyDefault16( cmap );
      }
    }
    else
    {
      // FBIOGETCMAP failed - clean up
      ColorMap_free( cmap );
      cmap = 0;
    }
  }
#ifdef DEBUG
  ColorMap_dump( cmap, stderr );
#endif
  return cmap;
}


/*
 * Get a handle on the visible region displayed
 * by the framebuffer
 */
int FB_getVisibleSurface( FBHandle *fb, Surface *s )
{
  /*
   * Get the framebuffer's current variable properties
   */
  if( (ioctl( fb->fd, FBIOGET_VSCREENINFO, &fb->var )) != -1 )
  {
    /*
     * Get a copy of the fb's colormap
     */
    if( (fb->cmap = FB_getColorMap( fb )) )
    {
      /* Now construct the Surface instance */
      s->left     = fb->var.xoffset;
      s->top      = fb->var.yoffset;
      s->width    = fb->var.xres;
      s->height   = fb->var.yres;
      s->bpp      = fb->var.bits_per_pixel;
      s->type     = get_fb_pixel_type( fb );
      s->clrmodel = fb->fix.visual;
      s->modulo   = fb->fix.line_length;
      s->buffer   = fb->mem;
      s->cmap     = fb->cmap;
    }
#ifdef DEBUG
   fprintf( stderr, "Pixel type:%d\n", s->type );
#endif
    return 0;
  }
  return -1;
}


/*
 * Close framebuffer
 */
void FB_close( FBHandle *fb )
{
  ColorMap_free( fb->cmap );
  munmap( fb->mem, fb->len );
  close( fb->fd );
}
