/*
 * Fluke Networks Cicada Project Meas Driver Debug Tracer
 *
 * Copyright (C) 2011 Fluke Networks, Premises Infrastructure Test
 * Software Team, Cicada/Cicada Project.
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 * This tracer can be used to record up to 32 channels of trace info
 * using the function cicada_trace_record() which is typically wrapped
 * in an an application specific macro.  This macro should check the
 * channel bit before calling the record function to minimize the
 * overhead for disabled channels.
 *
 * It is used in two ways:
 *
 *   - normally, this file is included in cicada_meas.c to provide
 *     basic runtime tracing of the drivers operations.
 *
 *   - it can be compiled into a special instrumented kernel to provide
 *     visibility into the USB core functions.  This kernel is for learning
 *     and troubleshooting and is never included in any product releases.
 *
 *   - it can also be excluded.
 *
 */

#include <linux/module.h>
#include <linux/device.h>
#include <linux/spinlock.h>
#include <linux/uaccess.h>    /* for copy_to_user */
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/mutex.h>

#include "cicada_trace.h"

#define CICADA_TRACE_SYS_CLASS		"cicada_trace"
#define CICADA_TRACE_PROC_DEV_NAME	"cicada_trace"
#define CICADA_TRACE_FIRST_MINOR	0
#define CICADA_TRACE_NUM_MINORS		1

struct cicada_trace_cfg
{
    struct cdev    cdev;
    struct class  *class;
    struct device *dev_node;
    dev_t	   dinfo;
};

static const int TRACE_DUMP_PRINTED_LAST = -2;
static const int TRACE_DUMP_IDLE         = -1;


static struct cicada_trace_cfg trace_cfg;

static struct device_attribute dev_attr_trace_enable;
static struct device_attribute dev_attr_trace_enabled_chans;

/* these are global so prefix name with "cicada".
 *
 * We cheat just a little bit with cicada_trace_enabled_chans.  It is
 * rarely set, so we lock around setting the value, but allow reads
 * without any locks because it is done all of the time.  The worst
 * thing that should happen is that you might miss one or get an extra
 * trace entry.
*/
uint32_t             cicada_trace_enabled_chans = 0;
struct cicada_trace *cicada_trace_struct = 0;

int cicada_trace_start(void);
static int cicada_trace_init(int wrap_mode);

static const struct file_operations trace_dump_fops_table;
static int     trace_dump_open(struct inode *inode, struct file *file);
static ssize_t trace_dump_read(struct file *file, char *buffer, size_t count,
			loff_t *ppos);
static int     trace_dump_release(struct inode *inode, struct file *file);
static int     trace_dump_flush(struct file *file, fl_owner_t id);

/*#define DEBUG_LOG*/
#ifdef  DEBUG_LOG
  #define LOG(fmt, args...) printk(fmt, ## args)
#else
  #define LOG(fmt, args...)
#endif


static unsigned int get_us(void)
{
    struct timeval tv;
    do_gettimeofday(&tv);
    return ((tv.tv_sec % 1000) * 1000000) + tv.tv_usec;
}


static uint32_t delta_us(uint32_t prev, uint32_t curr)
{
	int32_t delta = (int32_t)curr - (int32_t)prev;
	if (delta < 0)
		delta += 1000000;
	return (uint32_t) delta;
}


int cicada_trace_create(int max_entries, int wrap_mode)
{
	int buf_bytes = 0;
	int ret = 0;

	if (max_entries > CICADA_TRACE_ENTRIES_MAX)
		return -1;

	buf_bytes = (max_entries * sizeof(struct cicada_trace_entry)) + sizeof(struct cicada_trace);

	cicada_trace_struct = (struct cicada_trace *) kzalloc(buf_bytes, GFP_KERNEL);
	spin_lock_init(&cicada_trace_struct->lock);
	cicada_trace_struct->max_entries = max_entries;
	cicada_trace_init(wrap_mode);

	trace_cfg.class = class_create(THIS_MODULE, CICADA_TRACE_SYS_CLASS);
	if (!trace_cfg.class) {
		pr_err("class creation failed");
		goto cleanup_buf;
	}

	/* get our major/minor numbers */
	ret = alloc_chrdev_region(&trace_cfg.dinfo, 
				  CICADA_TRACE_FIRST_MINOR,
				  CICADA_TRACE_NUM_MINORS,
				  CICADA_TRACE_PROC_DEV_NAME);
	if (ret < 0) {
		pr_err("cicada_trace_create, alloc_chrdev_region failed\n");
		goto cleanup_class;
	}

	/* associate fops table with our char device and register it */
	cdev_init(&trace_cfg.cdev, &trace_dump_fops_table);
	trace_cfg.cdev.owner = THIS_MODULE;

	if (cdev_add(&trace_cfg.cdev, trace_cfg.dinfo, 1) < 0) {
		pr_err("cicada_trace_create: cdev_add failed\n");
		goto cleanup_chrdev_region;
	}

	/* create the actual dev node */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 27)
	trace_cfg.dev_node = device_create(trace_cfg.class,
					NULL, trace_cfg.dinfo, NULL,
					"%s", "cicada_trace");
#else
	trace_cfg.dev_node = device_create(trace_cfg.class,
					NULL, trace_cfg.dinfo,
					"%s", "cicada_trace");
#endif
	if (!trace_cfg.dev_node) {
		pr_err("cicada_trace_create: can't create dev");
		goto cleanup_cdev;
	}

	/* read and write this sys node to enable/disable trace chans */
	if (device_create_file(trace_cfg.dev_node, &dev_attr_trace_enabled_chans)) {
		pr_err("cicada_trace_create: device_create_file for trace_enabled_chans failed\n");
		goto cleanup_dev;
	}

	/* read and write this sys node to enable/disable trace chans */
	if (device_create_file(trace_cfg.dev_node, &dev_attr_trace_enable)) {
		pr_err("cicada_trace_create: device_create_file for trace_enable failed\n");
		goto cleanup_attr_trace_enabled_chans;
	}
	cicada_trace_start();
	return 0;

cleanup_attr_trace_enabled_chans:
	device_remove_file(trace_cfg.dev_node, &dev_attr_trace_enabled_chans);

cleanup_dev:
	device_destroy(trace_cfg.class, trace_cfg.dinfo);

cleanup_cdev:
	cdev_del(&trace_cfg.cdev);

cleanup_chrdev_region:
	unregister_chrdev_region(trace_cfg.dinfo, CICADA_TRACE_NUM_MINORS);

cleanup_class:
	class_destroy(trace_cfg.class);

cleanup_buf:
	kfree(cicada_trace_struct);

	return -1; /* TBD return something better? */
}
EXPORT_SYMBOL_GPL(cicada_trace_create);



int cicada_trace_destroy(void)
{
	device_remove_file(trace_cfg.dev_node, &dev_attr_trace_enable);
	device_remove_file(trace_cfg.dev_node, &dev_attr_trace_enabled_chans);

	device_destroy(trace_cfg.class, trace_cfg.dinfo);
	cdev_del(&trace_cfg.cdev);
	unregister_chrdev_region(trace_cfg.dinfo, CICADA_TRACE_NUM_MINORS);
	class_destroy(trace_cfg.class);
	kfree(cicada_trace_struct);

	return 0;
}
EXPORT_SYMBOL_GPL(cicada_trace_destroy);



static int cicada_trace_init(int wrap_mode)
{
	if (wrap_mode != -1)
		cicada_trace_struct->wrap_mode = wrap_mode;

	cicada_trace_struct->enabled = 0;
	cicada_trace_struct->save_enabled = 0;
	cicada_trace_struct->put_index = 0;
	cicada_trace_struct->get_index = TRACE_DUMP_IDLE;  /* haven't dumped anything */
	cicada_trace_struct->has_wrapped = 0;
	return 0;
}


int cicada_trace_start(void)
{
	int status = -1;
	unsigned long lock_flags;
	spin_lock_irqsave(&cicada_trace_struct->lock, lock_flags);
	if (cicada_trace_struct->get_index == TRACE_DUMP_IDLE) {
		/* start - we're not dumping */
		cicada_trace_struct->enabled = 1;
		cicada_trace_struct->save_enabled = 0;
		cicada_trace_struct->put_index = 0;
		cicada_trace_struct->get_index = TRACE_DUMP_IDLE;  /* haven't dumped anything */
		cicada_trace_struct->has_wrapped = 0;
		status = 0;
	}
	spin_unlock_irqrestore(&cicada_trace_struct->lock, lock_flags);
	return status;
}
EXPORT_SYMBOL_GPL(cicada_trace_start);


int cicada_trace_stop(void)
{
	unsigned long lock_flags;
	spin_lock_irqsave(&cicada_trace_struct->lock, lock_flags);
	cicada_trace_struct->save_enabled = cicada_trace_struct->enabled;
	cicada_trace_struct->enabled = 0;
	spin_unlock_irqrestore(&cicada_trace_struct->lock, lock_flags);
	return 0;
}


int cicada_trace_restart(void)
{
	int save_enabled = 0;
	unsigned long lock_flags;
	spin_lock_irqsave(&cicada_trace_struct->lock, lock_flags);
	save_enabled = cicada_trace_struct->save_enabled;
	spin_unlock_irqrestore(&cicada_trace_struct->lock, lock_flags);

	if (save_enabled)
		return cicada_trace_start();

	return 0;
}

int cicada_trace_get_enabled(void)
{
	int enabled;
    	unsigned long lock_flags;
	spin_lock_irqsave(&cicada_trace_struct->lock, lock_flags);
	enabled = cicada_trace_struct->enabled;
	spin_unlock_irqrestore(&cicada_trace_struct->lock, lock_flags);
	return enabled;
}

/* */
int cicada_trace_set_enabled_chans(uint32_t enabled_chans)
{
    	unsigned long lock_flags;
	spin_lock_irqsave(&cicada_trace_struct->lock, lock_flags);
	cicada_trace_enabled_chans = enabled_chans;
	spin_unlock_irqrestore(&cicada_trace_struct->lock, lock_flags);
	return 0;
}
EXPORT_SYMBOL_GPL(cicada_trace_set_enabled_chans);

/* */
int cicada_trace_add_enabled_chans(uint32_t enabled_chans)
{
    	unsigned long lock_flags;
	spin_lock_irqsave(&cicada_trace_struct->lock, lock_flags);
	cicada_trace_enabled_chans |= enabled_chans;
	spin_unlock_irqrestore(&cicada_trace_struct->lock, lock_flags);
	return 0;
}
EXPORT_SYMBOL_GPL(cicada_trace_add_enabled_chans);

/* */
int cicada_trace_remove_enabled_chans(uint32_t enabled_chans)
{
    	unsigned long lock_flags;
	spin_lock_irqsave(&cicada_trace_struct->lock, lock_flags);
	cicada_trace_enabled_chans &= ~enabled_chans;
	spin_unlock_irqrestore(&cicada_trace_struct->lock, lock_flags);
	return 0;
}
EXPORT_SYMBOL_GPL(cicada_trace_remove_enabled_chans);

int cicada_trace_get_index(void)
{
	int get_index;
    	unsigned long lock_flags;
	spin_lock_irqsave(&cicada_trace_struct->lock, lock_flags);
	get_index = cicada_trace_struct->get_index;
	spin_unlock_irqrestore(&cicada_trace_struct->lock, lock_flags);
	return get_index;
}


void cicada_trace_record(uint32_t chan, const char* fmt, ...)
{
	struct cicada_trace_entry* p;
	unsigned long lock_flags;
	va_list ap;
	va_start(ap, fmt);

        if (!cicada_trace_struct) {
            return;
        }
	spin_lock_irqsave(&cicada_trace_struct->lock, lock_flags);

	if (cicada_trace_struct->enabled) {

		p = &cicada_trace_struct->entries[cicada_trace_struct->put_index];
		p->time = get_us();
		p->chan = chan;
		p->msg  = fmt;
		p->arg1 = va_arg(ap, unsigned long);
		p->arg2 = va_arg(ap, unsigned long);

		cicada_trace_struct->put_index++;

		/* is the trace buffer full? */
		if (cicada_trace_struct->put_index >= cicada_trace_struct->max_entries) {
			if (cicada_trace_struct->wrap_mode) {
				cicada_trace_struct->put_index = 0;
				cicada_trace_struct->has_wrapped = 1;
			}
			else {
				cicada_trace_struct->enabled = 0;
			}
		}
	}
	spin_unlock_irqrestore(&cicada_trace_struct->lock, lock_flags);
	va_end(ap);
}
EXPORT_SYMBOL_GPL(cicada_trace_record);


/* Print one trace buffer entry to the given buffer and return the
 * number of bytes.  Assume that tracing has already been disabled. */
static int cicada_trace_print(struct cicada_trace_entry *p, uint32_t initial_time,
			      uint32_t* pPrev_time, char* pbuf, int bufbytes)
{
	long curr_rel_time;
	int nsnprintf = 0;
	int ntotal = 0;

	curr_rel_time = delta_us(initial_time, p->time);

	/* print timestamps and channel id */
	nsnprintf = snprintf(pbuf, bufbytes, "%10lu us    %10u us    %08x    ",
			  curr_rel_time,
			  delta_us(*pPrev_time, curr_rel_time),
			  p->chan);
	pbuf     += nsnprintf;
	ntotal   += nsnprintf;
	bufbytes -= nsnprintf;

	/* print traced message and its args - extra zeros are in case
	 * too many args in fmt */
	nsnprintf = snprintf(pbuf, bufbytes, p->msg, p->arg1, p->arg2, 0,0,0,0,0,0);
	pbuf     += nsnprintf;
	ntotal   += nsnprintf;
	bufbytes -= nsnprintf;

	/* finish with a newline */
	nsnprintf = snprintf(pbuf, bufbytes, "\n");
	pbuf     += nsnprintf;
	ntotal   += nsnprintf;
	bufbytes -= nsnprintf;

	/* return updated previous time */
	*pPrev_time = curr_rel_time;

	return ntotal;
}


static const struct file_operations trace_dump_fops_table = {
	.owner =	THIS_MODULE,
	.read =		trace_dump_read,
	.open =		trace_dump_open,
	.release =	trace_dump_release,
	.flush =	trace_dump_flush,
};

static int trace_dump_open(struct inode *inode, struct file *file)
{
	return 0;
}

/* Read the trace buffer output.  You can use "cat /dev/cicada_trace".
 *
 * It performs the following actions:
 *
 *    - stop tracing
 *    - convert trace output to ASCII
 *         - until all has been converted or
 *         - until you get more than 3k bytes of ascii (leaving 1k of margin)
 *    - copy the buffer to user space
 *    - return the number of ASCII bytes in the buffer
 *    - repeat until the entire trace buffer has been converted
 *    - return EOF (zero) on the read after the last line has been output.
 *
 * The program "cat" will continue to do reads until it receives EOF.  So,
 * it will do one additional call to trace_dump_read() after it has received
 * the last buffer of ASCII data.  In that case, we respond with an EOF (zero).
 *
 * Tracing remains off until explicitly reenabled.
 */
static ssize_t trace_dump_read(struct file *file, char *buffer, size_t count,
			       loff_t *ppos)
{
	char* pbuf_orig = 0;
	char* pbuf = 0;
	int retval = 0;
	int nbytes = 0;
	struct cicada_trace_entry *pEntry = 0;
	static const int PAGE_MARGIN = 1 * 1024;

	LOG("trace_dump_read: entry\n");

	pbuf_orig = kzalloc(PAGE_SIZE, GFP_KERNEL);
	pbuf = pbuf_orig;

	while(1) {
		LOG("trace_dump_read: get_index = %d, put_index = %d\n",
		    cicada_trace_struct->get_index,
		    cicada_trace_struct->put_index);

		LOG("trace_dump_read: wrap_mode = %d, has_wrapped = %d\n",
		    cicada_trace_struct->wrap_mode,
		    cicada_trace_struct->has_wrapped);

		if (cicada_trace_get_index() == TRACE_DUMP_IDLE) {

			LOG("trace_dump_read: state = TRACE_DUMP_IDLE\n");

			/* disable tracing - once tracing has been stopped, you
			   can access cicada_trace_struct without locks */
			cicada_trace_stop();
			LOG("trace_dump_read: save_enabled = %d\n", cicada_trace_struct->save_enabled);

			/* first dump */
			if (cicada_trace_struct->wrap_mode && cicada_trace_struct->has_wrapped)
				cicada_trace_struct->get_index = cicada_trace_struct->put_index;
			else {
				if (cicada_trace_struct->put_index == 0) {
					LOG("trace_dump_read: buffer is empty\n");
					cicada_trace_restart();
					retval = 0;	/* trace buffer is empty */
					break;
				}
				cicada_trace_struct->get_index = 0;
			}

			/* all times are shown relative to the first entry */
			pEntry = &cicada_trace_struct->entries[cicada_trace_struct->get_index];
			cicada_trace_struct->initial_time = pEntry->time;
			cicada_trace_struct->prev_time = 0;
		}
		else if (cicada_trace_struct->get_index == TRACE_DUMP_PRINTED_LAST) {
			LOG("trace_dump_read: state = TRACE_DUMP_PRINTED_LAST\n");

			/* idle the outputter and return EOF (zero) */
			cicada_trace_struct->get_index = TRACE_DUMP_IDLE;
			retval = 0;

			/* restore tracer enabled state */
			LOG("trace_dump_read: save_enabled = %d\n", cicada_trace_struct->save_enabled);
			cicada_trace_restart();
			LOG("trace_dump_read: enabled = %d\n", cicada_trace_struct->enabled);
			break;
		}
		else {
			pEntry = &cicada_trace_struct->entries[cicada_trace_struct->get_index];
		}

		LOG("trace_dump_read: before call: get_index = %d\n", cicada_trace_struct->get_index);

		/* print one line */
		retval = cicada_trace_print(pEntry,
					    cicada_trace_struct->initial_time,
					    &cicada_trace_struct->prev_time, /* call modifies */
					    pbuf, PAGE_SIZE);
		if (retval < 0) {
			pr_err("trace_dump_read: cicada_trace_dump returned an error\n");
			break;
		}

		nbytes += retval;
		pbuf   += retval;

		/* increment and wrap the trace buffer index */
		cicada_trace_struct->get_index++;

		if (cicada_trace_struct->wrap_mode)
			cicada_trace_struct->get_index %= cicada_trace_struct->max_entries;

		LOG("trace_dump_read: stop check: get_index = %d, put_index = %d\n",
		    cicada_trace_struct->get_index, cicada_trace_struct->put_index);

		if (cicada_trace_struct->get_index == cicada_trace_struct->put_index) {
		     /* we've printed the last of the trace data, but we can't return
			EOF yet because we've got data this time.  Setting get_index
			to the special value tells it to return EOF next time */
			cicada_trace_struct->get_index = TRACE_DUMP_PRINTED_LAST;
			break;
		}

		LOG("trace_dump_read: nbytes = %d\n", nbytes);

		/* stop when the page is mostly full */
		if (nbytes > (PAGE_SIZE - PAGE_MARGIN))
			break;

	}

	if (retval > 0) {
		if (copy_to_user(buffer, pbuf_orig, nbytes)) {
			pr_err("trace_dump_read: copy_to_user failed");
			retval = -EFAULT;
		}
	}
	else {
		nbytes = retval;
	}

	kfree(pbuf_orig);

	LOG("trace_dump_read: ALL DONE, nbytes = %d\n", nbytes);

	return nbytes;
}

static int trace_dump_release(struct inode *inode, struct file *file)
{
	return 0;
}

static int trace_dump_flush(struct file *file, fl_owner_t id)
{
	return 0;
}




/* trace enable */
static ssize_t cicada_show_trace_enable(struct device *dev,
				  struct device_attribute *attr,
				  char *buf)
{
	return sprintf(buf, "%d\n", cicada_trace_get_enabled());
}

static ssize_t cicada_set_trace_enable(struct device *dev,
				     struct device_attribute *attr,
				     const char *buf, size_t count)
{
	int value;
	if (sscanf(buf, "%d", &value) != 1) {
		pr_err("set_trace_enabled_chans: error: sscanf couldn't parse\n");
		return -EINVAL;
	}
	if (value < 0 || value > 1) {
		pr_err("set_trace_enabled_chans: error: value must be 0 or 1\n");
		return -EINVAL;
	}
	if (value)
		cicada_trace_start(); /* enable + start */
	else
		cicada_trace_stop(); /* disable */
	return count;
}


static DEVICE_ATTR(trace_enable, 0664,
		   cicada_show_trace_enable, cicada_set_trace_enable);




/* trace enable */
static ssize_t set_trace_enabled_chans(struct device *dev,
			     struct device_attribute *attr,
			     const char *buf, size_t count)
{
	int value;
	if (sscanf(buf, "%i", &value) != 1) {
		pr_err("set_trace_enabled_chans: error: sscanf couldn't parse\n");
		return -EINVAL;
	}
	cicada_trace_set_enabled_chans(value);
	return count;
}

static ssize_t show_trace_enabled_chans(struct device *dev,
				  struct device_attribute *attr,
				  char *buf)
{
	return sprintf(buf, "%08x\n", cicada_trace_enabled_chans);
}

static DEVICE_ATTR(trace_enabled_chans, 0664, show_trace_enabled_chans,
		   set_trace_enabled_chans);


