/*
 * This file defines the common USB messages and identifiers that are
 * used by cicada_meas and cicada_rcom.
 *
 * All multi-byte fields are stored in little-endian order.
 */

#ifndef CICADA_MEAS_DRVDEFS_H
#define CICADA_MEAS_DRVDEFS_H

/*
 * Common header for Cicada USB commands, responses, and interrupt
 * endpoint messages.
 *
 * The transfer size specifies the number of bytes in the total
 * message including this header.
 */

struct USB_MSG_HEADER
{
	uint8_t		MagicNum;	/* fixed value for a standard header */
	uint8_t		SeqNum;		/* Incrementing sequence number */
	uint8_t		SeqInv;		/* 1's complement of SeqNum */
	uint8_t		RouteID;	/* Routing code for int endpoint packets */
	uint32_t	TransferSize;	/* Total bytes including this header */
	uint16_t	Reserved;	/* reserved */
	uint16_t	MsgID;		/* message ID (cmd, response or int msg) */
} __attribute__ ((packed));
typedef struct USB_MSG_HEADER UsbMsgHead;



/********************************************************************/
/**** Commands ******************************************************/
/********************************************************************/

/*
 * Configure Communications Command
 */
struct USB_CMD_CONFIGMESG
{
	UsbMsgHead	Head;		/* Standard message header */
	uint16_t	CommFlags;	/* Communication configuration bits */
	uint8_t		ChanPriA;	/* Highest Priority Channel # (1-5), LL: Tx port to use   */
	uint8_t		ChanPriB;	/* Second highest priority chan #,   LL: Wl source to use */
	uint8_t		ChanPriC;	/* Third  highest priority chan #,   LL: Rx port to use   */
	uint8_t		ChanPriD;	/* Fourth highest priority chan # */
	uint8_t		ChanPriE;	/* Lowest priority chan #         */
	uint8_t		Unused;		/* unused */
	uint16_t	Baud;		/* LL: rate in Kbps for comm channel (3 - 500, 0 defaults to 200) */
	uint16_t	Spare1;		/* unused */
	uint16_t	Spare2;		/* unused */
	uint16_t	Spare3;		/* unused */
} __attribute__ ((packed)) ;
typedef struct USB_CMD_CONFIGMESG UsbCmdConfigMesg;


/*
 *  Send Comm Message Command
 */
struct USB_CMD_SENDMESG
{
	UsbMsgHead	Head;		/* Standard message header */
	uint32_t	Spare1;		/* Available for future use */
	uint32_t	Length;		/* # bytes of message data */
	uint8_t		Data[0];	/* Message section, up to 1520 bytes total */
	/* TBD changed length from 4 to 0, make sure */
	/* OK with DSP code */
} __attribute__ ((packed)) ;	/*  in up to 2 additional packets */
typedef struct USB_CMD_SENDMESG UsbCmdSendMesg;


/*
 *  Read Comm Message Command
 */
struct USB_CMD_READMESG
{
	UsbMsgHead	Head;		/* Standard message header */
	uint32_t	Spare1;		/* Available for future use */
	uint32_t	MaxLength;	/* Maximum # bytes of message data to xfer */
} __attribute__	((packed)) ;		/*  in up to 2 additional packets */
typedef struct USB_CMD_READMESG UsbCmdReadMesg;

/*
 *  Message format for Cicada command response without any data
 */
struct USB_RESP_STD
{
	UsbMsgHead	Head;		/* Standard message header */
	uint16_t	CCStat;		/* Command Completion Status Value */
	uint16_t	OptionalDebug;	/* Optional module specific status word */
} __attribute__ ((packed)) ;
typedef struct USB_RESP_STD UsbRespStd;




/********************************************************************/
/**** Responses *****************************************************/
/********************************************************************/

/*
 * Read Comm Message Response
 */
struct USB_RESP_READMESG
{
	UsbMsgHead	Head;		/* Standard message header */
	uint16_t	CCStat;		/* Command Completion Status Value */
	uint16_t	OptionalDebug;	/* Optional module specific status word */
	uint16_t	CommStat;	/* Communication Status word */
	uint16_t	Spare1;		/* Pad to multiple of 4 bytes */
	uint32_t	Length;		/* # bytes of message data (not incl. padding) */
	uint8_t		MesgData[0];	/* Start of message data */
} __attribute__ ((packed)) ;
typedef struct USB_RESP_READMESG UsbRespReadMesg;



/********************************************************************/
/**** Interrupt Messsages *******************************************/
/********************************************************************/

struct USB_INT_MESGRCV
{
	UsbMsgHead	Head;		/* Standard message header */
	uint16_t	MesgStat;	/* Message Status */
	uint16_t	Spare1;		/* Unused */
	uint16_t	MesgLen;	/* # bytes in the message */
	uint16_t	Spare2;		/* Unused */
} __attribute__ ((packed)) ;
typedef struct USB_INT_MESGRCV UsbIntMesgRcv;

/// Comm Info Interrupt
struct USB_INT_COMMINFO
{
    UsbMsgHead  Head;        ///< Standard message header
    uint8_t     OldRmanVer;  ///< Found remote with old rman version
    uint8_t     Role;        ///< Our role (initiator or responder)
    uint8_t     Speed;       ///< Comm Speed
    uint8_t     Spare1;      ///< Unused
} ;
typedef struct USB_INT_COMMINFO UsbIntCommInfo;



/********************************************************************/
/**** Message Identifiers *******************************************/
/********************************************************************/

/* These messages are shared by all Cicada modules */
#define	CICADA_MSG_BASE_DRVDEFS	0x0

/* Command Message Identifiers */
#define CMD_CODE_INVALID	(CICADA_MSG_BASE_DRVDEFS + 0x00)  /* never use this one */
#define CMD_CODE_CONFIGCOMM	(CICADA_MSG_BASE_DRVDEFS + 0x01)
#define CMD_CODE_SENDMESG	(CICADA_MSG_BASE_DRVDEFS + 0x02)
#define CMD_CODE_READMESG	(CICADA_MSG_BASE_DRVDEFS + 0x03)
#define CMD_CODE_START_COMM	(CICADA_MSG_BASE_DRVDEFS + 0x04)
#define CMD_CODE_STOP_COMM	(CICADA_MSG_BASE_DRVDEFS + 0x05)

/* Response Message Identifiers */
#define DEV_RESP_STD		(CICADA_MSG_BASE_DRVDEFS + 0x80)
#define DEV_RESP_READMESG	(CICADA_MSG_BASE_DRVDEFS + 0x81)

/* Interrupt Endpoint Message Identifiers */
#define INT_CODE_MESGRCV	(CICADA_MSG_BASE_DRVDEFS + 0xe0)
#define INT_CODE_COMMINFO	(CICADA_MSG_BASE_DRVDEFS + 0xe4)



/********************************************************************/
/**** Other Identifiers *********************************************/
/********************************************************************/

/* Device Comm Flag Definitions for Configure Communications Command */
#define CMD_FLAG_MESGENA	(0x0001)
#define CMD_FLAG_LONGREACH	(0x0010)
#define CMD_FLAG_LL_TX_EN   (0x0001)
#define CMD_FLAG_LL_RX_EN   (0x0002)

/* Custom IOCTL commands - All will enable comm unless otherwise specified. */
/* ifreq.ifr_data: Pointer to uint8_t[NUM_CHAN_PRI], where ChanPriA = idx 0; B = idx 1 ... */
#define IOCTL_SET_CHAN_PRI      SIOCDEVPRIVATE+0

/* ifreq.ifr_data: Pointer to uint32_t, where Baud is in BPS */
#define IOCTL_SET_BAUD          SIOCDEVPRIVATE+1

/* ifreq.ifr_data: void */
#define IOCTL_DISABLE_COMM      SIOCDEVPRIVATE+2

/* START_COMM and STOP_COMM are for starting/stopping comm via Mantis's new
 * module-to-module method. They do not interact with the configcomm cmd. */
/* ifreq.ifr_data: void */
#define IOCTL_START_COMM      SIOCDEVPRIVATE+3
#define IOCTL_STOP_COMM      SIOCDEVPRIVATE+4

/* Valid value for MagicNum (in USB_MSG_HEADER) */
#define CICADA_CMD_MAGIC 0xa5

/* routing codes for interrupt endpoint messages (in USB_MSG_HEADER) */
#define ROUTE_INVALID	(0x00)	  /* */
#define ROUTE_EVENT	(0x01)	  /* Event Interrupt */
#define ROUTE_NET	(0x02)	  /* Communications related */
#define ROUTE_CNTL	(0x03)	  /* Measurement related */
#define ROUTE_N         (0x04)

/* Command Status (ccstat) Definitions (in all responses) */
#define CMD_STAT_OK		(0)	/* Command completed ok */
#define CMD_STAT_BADCMD		(1)	/* Illegal command */
#define CMD_STAT_BADPARAM	(2)	/* Command parameter error */
#define CMD_STAT_ABORT		(3)	/* Command was aborted */
#define CMD_STAT_EXEC		(4)	/* General execution error */

#define CMD_STAT_QUEUE_BUSY    (11) ///< SLOW command was not queued to task2


/* Communication Status Bit Definitions
 *
 * These bits are used in USB_RESP_READMESG
 */
#define COMM_STAT_MESGENA	(0X0001)	/* Message reception enabled */
#define COMM_STAT_MESGRCVD	(0X0002)	/* Message waiting */
#define COMM_STAT_MESGERR	(0X0004)	/* Message waiting w/error */
#define COMM_STAT_MESGMODE	(0X0008)	/* 0 = HS, 1 = Long Reach */
#define COMM_STAT_TYPE_ERR	(0X0010)	/* Message not host-host type */
#define COMM_STAT_MESGCHAN	(0X00E0)	/* Last channel used (1-4) */



/* The maximum size of the entire network packet, i.e. the data payload of the
 * SendMesg or ReadMesg commands. */
#define MAX_PACKET_DATA 1512

/* OTDR wants to read 64K + a little bit so make the buffer accomodate them */
#define	CICADA_MEAS_URB_BUFBYTES	(66*1024)


/********************************************************************/
/**** Shared Driver Implementation **********************************/
/********************************************************************/

struct brb;

/* Bulk struct which provides the info to generate a Bulk Endpoing Urb */
typedef void (*brb_callback)(struct brb *brb);

struct brb
{
        int             status;                 /* returned to submitter (same as urb) */
        void           *mmstate;                /* cicada_meas driver state */
        uint8_t        *transfer_buffer;        /* data buffer */
        int             transfer_buffer_length; /* same as urb */
        int             actual_length;          /* same as urb */
        brb_callback    callback;
        void           *context;                /* owned by submitter */
        int             routing_code;           /* bulk driver uses to route response */
};

/* use to initialize brb struct */
inline void brb_init(struct brb *brb, uint8_t* transfer_buffer, int transfer_buffer_length,
                     brb_callback callback, void* mmstate, void* context,
                     int routing_code)
{
        brb->status = 0;
        brb->transfer_buffer = transfer_buffer;
        brb->transfer_buffer_length = transfer_buffer_length;
        brb->actual_length = 0;
        brb->callback = callback;
        brb->mmstate = mmstate;
        brb->context = context;
        brb->routing_code = routing_code;
}

inline void brb_free(struct brb *brb)
{
    if (brb) {
	kfree(brb->transfer_buffer);
	kfree(brb);
    }
}


#endif /* CICADA_MEAS_DRVDEFS_H */
