/*
 * USB Skeleton driver - 2.2
 *
 * Copyright (C) 2001-2004 Greg Kroah-Hartman (greg@kroah.com)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 * This driver is based on the 2.6.3 version of drivers/usb/usb-skeleton.c
 * but has been rewritten to be easier to read and use.
 *
 *----------------------------------------------------------------
 *
 * Fluke Networks USB Measurement Module Driver
 *
 * Copyright (C) 2012 Fluke Networks, Premises Infrastructure Test
 * Software Team, Cicada/Mantis Project.
 *
 * While it is much different, this driver is based on the USB
 * Skeleton Driver as acknowledged by the above copyright.
 *
 *----------------------------------------------------------------
 *
 * There are two Cicada USB measurement module drivers:
 *
 *    - cicada_meas
 *    - cicada_rcom
 *
 * cicada_meas supports a command/response protocol between the
 * devnode cicada_meas_cmd0 and the module's bulk in/out endpoints.
 * The userspace program reads devnode cicada_meas_evt0 to get
 * interrupt endpoint data.
 *
 * cicada_rcom supports network communications over the module's media
 * interface for tests that require a remote unit.  Its interface to
 * the higher levels is through its Linux network interface.
 *
 * cicada_meas is the primary driver.  Its USB probe function is
 * called when the device is plugged in.  It then calls probe
 * functions for its subdrivers and then for cicada_rcom.  Likewise,
 * its disconnect function is called first and it then calls the
 * disconnect function of the subdrivers and cicada_rcom.
 *
 * cicada_meas contains the following subdrivers:
 *
 *    - cmd (command response over bulk endpoint)
 *    - evt (reporting events to userspace from interrupt endpoint)
 *    - bulk (bulk resp routing to cmd or net)
 *    - int (interrupt endpoint packet handling and routing to cmd or net)
 *    - modpower (control of module battery supply on/off switch)
 *    - hubpower (control of hub port power to module)
 *
 * The interrupt subdriver exists to route interrupt endpoint packets
 * to either cicada_rcom or the evt subdriver.
 *
 * The bulk driver is messy because the cicada_rcom an cicada_meas::cmd
 * drivers share the bulk endpoints and can steal each other's response
 * packets.
 *
 * Note: for a long time, you could send one command to
 * /dev/cicada_meas_cmd0 and do repeated reads to get a large
 * response.  Adding the bulk driver to address the packet stealing
 * problem imposes a new requirement that each response urb must have
 * a header so we can use the routing code if there is any confusion.
 * When using multiple reads to get a large response, only the first
 * urb has a header so this approach won't work anymore.
 *
 * NOTE: there is a significant amount of application specific
 *	 checking and tracing in the driver.  The original intent was
 *	 to have it be largely ignorant of the packet contents.
 *	 However, we often needed to get at the routing code, so I
 *	 found it much easier to see what was going on to add a bit
 *	 more visibility.
 *
 *----------------------------------------------------------------
 */

#include <linux/errno.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kref.h>
#include <linux/uaccess.h>
#include <linux/usb.h>
#include <linux/mutex.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/workqueue.h>
#include <linux/version.h>
#include <linux/spinlock.h>
#include <linux/usb/ch9.h>
#include <linux/slab.h>

/* at one time, you could exclude all of cicada_trace with a define
 * so it was conditionally included as a .c.  */
#include "cicada_trace.c"

#include "cicada_meas.h"
#include "cicada_meas_drvdefs.h"

/* Version of this driver - increment it for all significant changes */
#define CICADA_MEAS_VERSION	40

/* Cicada module usb enumeration constants */
#define CICADA_MEAS_VENDOR_ID	0x0f7e
#define CICADA_MEAS_PRODUCT_ID	0x0050

/* table of devices that work with this driver */
static struct usb_device_id cicada_meas_table[] = {
	{ USB_DEVICE(CICADA_MEAS_VENDOR_ID, CICADA_MEAS_PRODUCT_ID) },
	{ }					/* Terminating entry */
};
MODULE_DEVICE_TABLE(usb, cicada_meas_table);

/* class and dev-node constants */
#define CICADA_MEAS_PROC_DEV_NAME	"cicada_meas"
#define CICADA_MEAS_SYS_CLASS		"cicada_meas"
#define CICADA_MEAS_FIRST_MINOR		0
#define CICADA_MEAS_NUM_MINORS		2	     /* for cmd and evt dev nodes */

/* This driver supports this number of modules at the same time.  This
 * number could be increased, but we only use two at most, and only
 * for development purposes. */
#define MAX_MODULE_INSTANCES 2


#define to_cicada_meas_state(d) container_of(d, struct cicada_meas_state, kref)


/* cmd subdriver: user space sends and receives commmand and response
 *		  packets to and from /dev/cicada_meas_cmd dev node.
 */

/* cmd subdriver: state info for cmd specific part of driver */
struct cmd_state {
	struct cicada_meas_state *mmstate;	/* parent struct */

	struct cdev	       *cdev;		/* char device struct */
	struct device	       *char_dev_node;
	int			major;
	int			minor;

	int			open_count;	 /* control number of opens */

	int			errors;		 /* the last request error code */
	spinlock_t		lock;		 /* lock for errors and stuff */

	int			brb_is_queued;	 /* don't queue more than 1 urb at one time */

	wait_queue_head_t	read_wait_queue; /* for blocking read */
	int			read_done;	 /* */
	struct brb	       *read_brb;	 /* */
	int			read_byte_index; /* index for partial reads */
};


/* cmd subdriver: constant usb interface info for cmd driver endpoints */
struct cmd_usb_desc {
	size_t	bulk_in_size;		/* size of the receive buffer */
	uint8_t	bulk_in_endpoint_addr;	/* address of the bulk in endpoint */
	uint8_t	bulk_out_endpoint_addr;	/* address of the bulk out endpoint */
};




/* bulk transaction subdriver: routes bulk endpoint receive packets back
 *	to the cmd or net drivers.
 */

struct bulk_transaction
{
	/* this struct isn't necessary at this time, but could be handy if we
	* decide later that there is one more bit of information that we'd
	* like to remember. */
	struct brb	       *brb; /* bulk request block */
};

struct bulk_state
{
	spinlock_t		snd_lock;
	spinlock_t		rcv_lock;

	/* The transaction array has pointer to one brb for each routeId.  The
	 * pointer is non-null if the brb is outstanding. */
	struct bulk_transaction transactions[ROUTE_N];

	/* rcv_urb_in_limbo is a pointer to a urb that temporarily has no
	 * brb registered in "transactions" above. */
	struct urb	       *rcv_urb_in_limbo;

	/* count of outstanding urbs */
	int			rcv_urbs_active;
};



/* event subdriver: data from interrupt endpoint urb is copied into
 * one of these so the urb can be requeued.  Initially, we allocate
 * EVT_DATA_BUF_N of them.  This should typically be enough, but if
 * under high userspace load, it gets behind, we dynamically allocate
 * new temporary buffers/descriptors.  The temporary ones are deleted
 * after the data is read out of them.
 */
struct evt_data_buf_desc {
	struct list_head	list;
	int			nbytes;
	u8		       *pbuf;	   /* data from urb */
	int			status;	   /* status from urb */
	int			temporary; /* true when dynamically added */
};

struct work_state  {
	struct work_struct	  work_struct;
	struct cicada_meas_state *mmstate;
};



/* interrupt endpoint subdriver uses this struct to keep track of
 * registered callbacks.
 */

typedef void (*int_callback)(struct urb *urb, void *context);

struct int_cb_info {
	struct list_head list;
	int		 routing_code; /* route to net or cmd */
	int_callback	 func;	       /* callback func ptr */
	void		*context;      /* passed as-is to callback */
};



/* Maximum number of urbs queued at once */
#define	INT_URBS		20
#define	INT_URB_BUFBYTES	1024
#define	INT_URBS_MAX		1000   /* limit on module parameter */

/* interrupt subdriver: state info for interrupt endpoint specific
 * part of driver */
struct int_state {
	struct workqueue_struct *workqueue;
	struct work_state	work_state;

	struct list_head	free_work_urb_list;   /* list of free work_urbs */
	struct list_head	active_work_urb_list; /* list of work_urbs in use */

	struct list_head	cb_list;	      /* client callback list */
	struct list_head	urbs_allocated_list;  /* list of allocated urbs */

	struct usb_anchor	urb_anchor;	/* use usb anchors */

	int			urb_count;	/* for debugging */
	int			urb_count_max;	/* for debugging */

	spinlock_t		lock;		/* lock for errors and stuff */
	int			nest_count;
	int			shutting_down;
};

/* constant usb interface info for interrupt driver endpoint */
struct int_usb_desc {
	size_t	int_in_size;		/* the size of the receive buffer */
	uint8_t	int_in_endpoint_addr;	/* the address of the int in endpoint */
	int	interval;
};

/* interrupt/event drivers: we allow multiple interrupt urbs to be in
 * process at the same time.  Therefore, we need to keep a list to
 * hold them until the work callback can process them.
 */
struct urb_carrier {
	struct list_head list;
	struct urb *urb;		/* arg to work callback */
};

/* Minimum number of read buffers held between evt_callback() and
   evt_read().	*/
#define	EVT_DATA_BUF_N		10
#define	EVT_DATA_BUF_BYTES	INT_URB_BUFBYTES

/* Start dropping data if you get more than this many queued up */
#define	EVT_DATA_BUF_MAX	500



/* event subdriver: state info for evt specific part of driver */
struct evt_state {
	struct cicada_meas_state *mmstate;   /* parent struct */

	struct cdev	       *cdev;
	struct device	       *char_dev_node;
	int			major;
	int			minor;

	int			open_count;  /* limit number of opens */

	int			error;	     /* report callback errors */
	struct mutex		buf_mutex;   /* protect list and shared data */

	wait_queue_head_t	wait_queue;  /* for blocking read */

	struct list_head	free_list;   /* free buffers for devnode read */
	struct list_head	rcv_list;    /* rcv buffers for devnode read */

	int			read_byte_index;  /* index for partial reads */

	int			buf_count;	/* for debugging */
	int			buf_count_max;	/* for debugging */

	int			exiting;

#ifdef TIMING_HISTOGRAM_ENABLE
	uint32_t		  us_prev;
	struct timing_histogram	 *histogram;
#endif
};

#define	USB_BUS_ID_MAX	128




/* Overall driver: Structure to hold our device specific stuff */
struct cicada_meas_state {
	/* protect access to shared elements of this struct.
	 * Contained structs have their own locks when needed so
	 * they can operate mostly independently.
	 */
	struct mutex		  state_struct_mutex;

	/* kref keeps this struct from going out of scope too soon */
	struct kref		  kref;

	/* The "active_list" keeps track of active mmstate structs.  This
	 * embedded list pointer allows this struct to be put into that list.
	 */
	struct list_head	  active_list;

	struct usb_device	 *udev;		/* from usb.h */
	struct usb_interface	 *interface;	/* from usb.h */

	dev_t			  dev_info;

	int			  suffix;    /* dev node suffix same for cmd
						& evt on the same interface */

	/* command subdriver */
	struct cmd_usb_desc	  cmd_config;
	struct cmd_state	  cmds;

	/* bulk transaction subdriver state */
	struct bulk_state	  bulks;

	/* event subdriver */
	int			  evt_enabled;
	struct evt_state	  evts;

	/* interrupt endpoint subdriver */
	struct int_usb_desc	  int_config;
	int			  int_enabled;
	struct int_state	  ints;

	/* Misc stuff */
	int			  rcom_enabled;
	int			  net_routing_code;
	int_callback		  net_callback;
	void			 *net_context;
	int			  matlab_int_cb_override;

	char			  usb_bus_id[USB_BUS_ID_MAX];
};


#define CHAN_DRIVER_FLOW	0x00000001
#define FLOW(fmt, args...) if (cicada_trace_enabled_chans & CHAN_DRIVER_FLOW) \
				 cicada_trace_record(CHAN_DRIVER_FLOW, fmt, ## args);

#define CHAN_DRIVER_FLOW2	0x00000002
#define FLOW2(fmt, args...) if (cicada_trace_enabled_chans & CHAN_DRIVER_FLOW2) \
				  cicada_trace_record(CHAN_DRIVER_FLOW2, fmt, ## args)

#define CHAN_DRIVER_DEBUG	0x00000004
#define DBG(fmt, args...) if (cicada_trace_enabled_chans & CHAN_DRIVER_DEBUG) \
				cicada_trace_record(CHAN_DRIVER_DEBUG, fmt, ## args)

#define CHAN_DRIVER_LOW		0x00000008
#define LOW(fmt, args...) if (cicada_trace_enabled_chans & CHAN_DRIVER_LOW) \
				cicada_trace_record(CHAN_DRIVER_LOW, fmt, ## args)

#define CHAN_DRIVER_LOW2	0x00000010
#define LOW2(fmt, args...) if (cicada_trace_enabled_chans & CHAN_DRIVER_LOW2) \
				cicada_trace_record(CHAN_DRIVER_LOW2, fmt, ## args)

#define CHAN_DRIVER_RCRX	0x00000020
#define RCRX(fmt, args...) if (cicada_trace_enabled_chans & CHAN_DRIVER_RCRX) \
				 cicada_trace_record(CHAN_DRIVER_RCRX, fmt, ## args)

/* for development, shouldn't release with any of these in place */
#define CHAN_DRIVER_TEMP	0x00000080
#define TEMP(fmt, args...) if (cicada_trace_enabled_chans & CHAN_DRIVER_TEMP) \
				 cicada_trace_record(CHAN_DRIVER_TEMP, fmt, ## args)

static const uint32_t DEFAULT_TRACE_CHANS = 0xffff;


/* error is always enabled, but records to FLOW channel, to conserve channels */
/* Its output also goes to klogger */
#define ERR(fmt, args...) { cicada_trace_record(CHAN_DRIVER_FLOW, fmt, ## args); pr_err(fmt, ## args); }

/* always trace these error cases, but conditionally send them to klog too */
//#define ERRDBG_ENABLE
#ifdef	ERRDBG_ENABLE
#warning "ERRDBG_ENABLE enabled"
#define ERRDBG(fmt, args...) { cicada_trace_record(CHAN_DRIVER_FLOW, fmt, ## args); pr_err(fmt, ## args); }
#else
#define ERRDBG(fmt, args...) { cicada_trace_record(CHAN_DRIVER_FLOW, fmt, ## args); }
#endif

/* very useful for debugging probe/disconnect, etc...
   Module parameter enable_config_logging enables printk too.
 */
#define	CONFIG(fmt, args...) { \
		if (enable_config_logging) { printk(fmt "\n", ## args); } \
		if (cicada_trace_enabled_chans & CHAN_DRIVER_FLOW) \
			cicada_trace_record(CHAN_DRIVER_FLOW, fmt, ## args); \
	}

extern uint32_t cicada_trace_enabled_chans;
extern int  cicada_trace_create(int max_entries, int wrap_mode);
extern int  cicada_trace_destroy(void);
extern void cicada_trace_record(uint32_t chan, const char* fmt, ...);
extern int  cicada_trace_add_enabled_chans(uint32_t enabled_chans);
extern int  cicada_trace_remove_enabled_chans(uint32_t enabled_chans);

struct usb_driver cicada_meas_driver;

#include "cicada_meas_drvdefs.h"

int cicada_rcom_probe(struct usb_device *usbdev, struct usb_interface *usbif,
		unsigned int bulk_in_endpointAddr,
		unsigned int bulk_out_endpointAddr,
		int *route,
		void (**net_int_cb)(struct urb *, void *),
		void **pnetdev,
		int  (*bulk_submit_snd_urb)(void *mmstate, struct urb *urb,
					    gfp_t mem_flags, int routing_code),
		int  (*bulk_submit_rcv_brb)(struct brb* brb),
		void* mmstate,
		void (*cicada_meas_state_get)(void* mmstate),
		void (*cicada_meas_state_put)(void* mmstate),
		uint32_t* ptrace_enabled_chans,
		void (*trace_record)(uint32_t chan, const char* fmt, ...),
		int enable_config_logging_arg);
void cicada_rcom_disconnect(void* context);


static void cicada_meas_delete(struct kref *kref);
static int cmd_draw_down(struct cicada_meas_state *mmstate);
static const struct file_operations cmd_fops_table;

static void int_register_callback(struct cicada_meas_state *dev, int
			routing_code, int_callback func, void *context);
static void int_urb_callback(struct urb *urb);
static void int_work_handler(struct work_struct *work);

static void evt_callback(struct urb *urb, void *context);
static const struct file_operations evt_fops_table;

static int bulk_submit_snd_urb(void *mmstate_vp,
			       struct urb *client_snd_urb,
			       gfp_t mem_flags,
			       int routing_code);
static int bulk_submit_rcv_brb(struct brb *brb);

static void bulk_urb_call_cb(struct cicada_meas_state *mmstate,
			     struct brb *brb,
			     struct urb *urb,
			     bool immediate);

/* there's only one class for all instances */
static struct class *cicada_meas_class;

/* remember major and base of minor numbers so we can use the same
   major if we've got multiple modules installed. */
static int driver_major;
static int minor_base;


/*----------------------------------------------------------------------------
 * Module Parameters - can be set when loading the module to configure.
 */

/* can modify the number of interrupt endpoint urbs for testing */
static int int_urbs = INT_URBS;
module_param(int_urbs, int, S_IRUGO|S_IWUSR);

/* enables logging facility of CONFIG macros (normally trace, but no logging) */
static int enable_config_logging = 0;
module_param(enable_config_logging, int, S_IRUGO|S_IWUSR);


/*----------------------------------------------------------------------------
 * cicada_rcom calls these functions to ensure that mmstate stays alive.
 */

void cicada_meas_state_get(void* v_mmstate)
{
	struct cicada_meas_state *mmstate = (struct cicada_meas_state*) v_mmstate;
	CONFIG("cicada_meas_state_get() - called from cicada_rcom - mmstate = %p", mmstate);
	kref_get(&mmstate->kref);
}

void cicada_meas_state_put(void* v_mmstate)
{
	struct cicada_meas_state *mmstate = (struct cicada_meas_state*) v_mmstate;
	CONFIG("cicada_meas_state_put() - called from cicada_rcom - mmstate = %p", mmstate);
	kref_put(&mmstate->kref, cicada_meas_delete);
}

/*----------------------------------------------------------------------------
 *---- keep track of active mmstate struct -----------------------------------
 *----------------------------------------------------------------------------
 */

struct active_mmstate_info {
	struct list_head	active_list;
	struct mutex		mutex;   /* protect list and shared data */
};

struct active_mmstate_info active_mmstate;

static void init_mmstate_active(void)
{
	INIT_LIST_HEAD(&active_mmstate.active_list);
	mutex_init(&active_mmstate.mutex);
}

static void add_mmstate_active(struct cicada_meas_state *mmstate)
{
	CONFIG("add_mmstate_active() - mmstate = %p", mmstate);
	mutex_lock(&active_mmstate.mutex);
	list_add_tail(&mmstate->active_list, &active_mmstate.active_list);
	mutex_unlock(&active_mmstate.mutex);
}

static void remove_mmstate_active(struct cicada_meas_state *mmstate)
{
	struct cicada_meas_state *list_mmstate = NULL;
	struct list_head *ptr = NULL;
	struct list_head *next = NULL;

	mutex_lock(&active_mmstate.mutex);
	list_for_each_safe(ptr, next, &active_mmstate.active_list) {
		list_mmstate = list_entry(ptr, struct cicada_meas_state, active_list);
		if (list_mmstate && list_mmstate == mmstate) {
			CONFIG("remove_mmstate_active: mmstate = %p", mmstate);
			list_del(&(mmstate->active_list));
			break;
		}
	}
	mutex_unlock(&active_mmstate.mutex);
}

static struct cdev* cmd_cdev_from_mmstate(struct cicada_meas_state *mmstate)
{
	return mmstate ? mmstate->cmds.cdev : NULL;
}

static struct cdev* evt_cdev_from_mmstate(struct cicada_meas_state *mmstate)
{
	return mmstate ? mmstate->evts.cdev : NULL;
}

typedef struct cdev* (*get_cdev_from_mmstate)(struct cicada_meas_state *mmstate);


static struct cicada_meas_state* get_mmstate_active(get_cdev_from_mmstate get_func,
						    struct cdev* cdev)
{
	struct cicada_meas_state *list_mmstate = NULL;
	struct cdev *list_cdev = NULL;
	struct list_head *ptr = NULL;

	mutex_lock(&active_mmstate.mutex);
	list_for_each(ptr, &active_mmstate.active_list) {
		list_mmstate = list_entry(ptr, struct cicada_meas_state, active_list);

		if (get_func) {
			list_cdev = get_func(list_mmstate);
			if (list_cdev && list_cdev == cdev) {
				CONFIG("get_mmstate_active(): cdev = %p, mmstate = %p",
					cdev, list_mmstate);
				mutex_unlock(&active_mmstate.mutex);
				return list_mmstate;
			}
		}
	}
	mutex_unlock(&active_mmstate.mutex);
	return NULL;
}


/*----------------------------------------------------------------------------
 *---- cmd subdriver ---------------------------------------------------------
 *----------------------------------------------------------------------------
 *
 * The cmd subdriver supports a basic command/response protocol
 * between the devnode cicada_meas_cmd0 and the module's bulk in/out
 * endpoints.
 *
 * The driver reads data from the module in chunks up to
 * CICADA_MEAS_URB_BUFBYTES.  However, the read() method provides a
 * fifo-like interface.	 So, if you call read() with a byte count less
 * than the number of bytes received, it will return the number of
 * bytes you requested and keep the buffer around so you can read the
 * rest of it with subsequent reads.
 */

static struct device_attribute dev_attr_ctlmsg;

int cmd_probe(struct cicada_meas_state *mmstate, int major, int minor,
	      int suffix)
{
	dev_t dinfo;

	CONFIG("cmd_probe: entry major = %d, minor = %d", major, minor);
	FLOW("cmd_probe: urb buffer size = %d", CICADA_MEAS_URB_BUFBYTES);

	kref_get(&mmstate->kref);

	init_waitqueue_head(&mmstate->cmds.read_wait_queue);
	spin_lock_init(&mmstate->cmds.lock);

	mmstate->cmds.major = major;
	mmstate->cmds.minor = minor;
	dinfo = MKDEV(major, minor);

	/* Create cmd char device and associate its fops table.
	 * mmstate must have pointer to cdev (cdev must not be embedded in mmstate).
	 */
	mmstate->cmds.cdev = cdev_alloc();
	if (!mmstate->cmds.cdev) {
		ERR("cmd_probe: cdev_alloc failed\n");
		goto cmd_probe_error;
	}

	mmstate->cmds.cdev->owner = THIS_MODULE;
	mmstate->cmds.cdev->ops = &cmd_fops_table;
	mmstate->cmds.mmstate = mmstate;

	if (cdev_add(mmstate->cmds.cdev, dinfo, 1) < 0) {
		ERR("cmd_probe: cdev_add failed\n");
		goto cmd_delete_cdev;
	}
	mmstate->cmds.char_dev_node =
				device_create(cicada_meas_class,
					&mmstate->interface->dev,
					dinfo, NULL,
					"%s%d", "cicada_meas_cmd", suffix);

	if (IS_ERR(mmstate->cmds.char_dev_node)) {
		ERR("cmd_probe: device_create failed\n");
		goto cmd_delete_cdev;
	}

	if (device_create_file(&mmstate->interface->dev, &dev_attr_ctlmsg)) {
		ERR("cmd_probe: device_create_file for ctlmsg failed\n");
		goto cmd_destroy_dev_node;
	}
	CONFIG("cmd_probe: exit");

	return 0;

cmd_destroy_dev_node:
	device_destroy(cicada_meas_class, dinfo);

cmd_delete_cdev:
	cdev_del(mmstate->cmds.cdev);

cmd_probe_error:
	kref_put(&mmstate->kref, &cicada_meas_delete);

	return -ENODEV;
}


/* Note: this is run within a lock of mmstate->state_struct_mutex */
int cmd_disconnect(struct cicada_meas_state *mmstate)
{
	dev_t dinfo;

	CONFIG("cmd_disconnect: entry");
	device_remove_file(&mmstate->interface->dev, &dev_attr_ctlmsg);

	dinfo = MKDEV(mmstate->cmds.major, mmstate->cmds.minor);

	if (mmstate->cmds.char_dev_node)
		device_destroy(cicada_meas_class, dinfo);

	/* cdev ops may still be used by file system after cdev_del() is called.
	 * The memory pointed to by mmstate->cmds.cdev will be freed after all users
	 * have finished. */
	cdev_del(mmstate->cmds.cdev);
	mmstate->cmds.cdev = NULL;

	kref_put(&mmstate->kref, &cicada_meas_delete);

	CONFIG("cmd_disconnect:exit");

	return 0;
}


static int cmd_open(struct inode *inode, struct file *file)
{
	int retval = 0;
	struct cicada_meas_state *mmstate;

	mmstate = get_mmstate_active(cmd_cdev_from_mmstate, inode->i_cdev);
	CONFIG("cmd_open: entry: mmstate = %p", mmstate);
	if (!mmstate) return -ENODEV;

	/* we haven't scheduled a urb yet */
	mmstate->cmds.brb_is_queued = 0;

	/* we're not holding a received urb yet */
	mmstate->cmds.read_brb = 0;
	mmstate->cmds.read_byte_index = 0;

	/* increment our usage count for the device while open */
	kref_get(&mmstate->kref);

	/* open the device */
	mutex_lock(&mmstate->state_struct_mutex);
	{
		/* make sure interface is still there */
		if (!mmstate->interface) {
			retval = -ENODEV;
			goto error_exit;
		}

		if (!mmstate->cmds.open_count) {
			mmstate->cmds.open_count++;
		}
		else { /* uncomment this block if you want exclusive open */
			retval = -EBUSY;
			ERR("cmd_open: error: open_count = %d",
			    mmstate->cmds.open_count);
			goto error_exit;
		}

		/* save our object in the file's private structure */
		file->private_data = mmstate;
	}
	mutex_unlock(&mmstate->state_struct_mutex);

	CONFIG("cmd_open: exit: %d", retval);
	return retval;

error_exit:
	mutex_unlock(&mmstate->state_struct_mutex);
	kref_put(&mmstate->kref, cicada_meas_delete);
	CONFIG("cmd_open: error exit: %d", retval);
	return retval;
}


static int cmd_release(struct inode *inode, struct file *file)
{
	struct cicada_meas_state *mmstate;

	CONFIG("cmd_release: entry");
	mmstate = (struct cicada_meas_state *)file->private_data;
	if (!mmstate) return -ENODEV;

	mutex_lock(&mmstate->state_struct_mutex);
	{
		mmstate->cmds.open_count--;
	}
	mutex_unlock(&mmstate->state_struct_mutex);

	/* decrement the count on our device */
	kref_put(&mmstate->kref, cicada_meas_delete);

	CONFIG("cmd_release: exit");
	return 0;
}


static int cmd_flush(struct file *file, fl_owner_t id)
{
	struct cicada_meas_state *mmstate;
	int retval;
	unsigned long lock_flags;

	CONFIG("cmd_flush: entry");
	mmstate = (struct cicada_meas_state *)file->private_data;
	if (!mmstate) return -ENODEV;

	/* wait for io to stop */
	mutex_lock(&mmstate->state_struct_mutex);
	{
		if (mmstate->cmds.read_brb)
			brb_free(mmstate->cmds.read_brb);
		mmstate->cmds.read_brb = NULL;
		mmstate->cmds.read_byte_index = 0;

		cmd_draw_down(mmstate);
		mmstate->cmds.brb_is_queued = 0;

		/* read out errors, leave subsequent opens a clean slate */
		spin_lock_irqsave(&mmstate->cmds.lock, lock_flags);
		{
			retval = mmstate->cmds.errors ?
				(mmstate->cmds.errors == -EPIPE
				 ? -EPIPE : -EIO) : 0;
			mmstate->cmds.errors = 0;
		}
		spin_unlock_irqrestore(&mmstate->cmds.lock,
				       lock_flags);
	}
	mutex_unlock(&mmstate->state_struct_mutex);

	CONFIG("cmd_flush: exit: %d", retval);

	return retval;
}


/*
 * read callback - callback for received brb
 * 
 * Don't forget that this normally runs in interrupt context (except
 * when called with an out-of-sync brb).  This function will set but
 * not clear cmd_state.errors.
 */

static void cmd_read_callback(struct brb *brb)
{
	struct cicada_meas_state *mmstate;
	unsigned long lock_flags;

	mmstate = (struct cicada_meas_state *)brb->mmstate;
	if (!mmstate) return;

	LOW("cmd_read_callback: brb->status = %d, brb->actual_length = %d",
		       brb->status, brb->actual_length);

	/* sync/async unlink faults aren't errors */
	if (brb->status) {

		if (!(brb->status == -ENOENT ||
		      brb->status == -ECONNRESET ||
		      brb->status == -ESHUTDOWN ||
		      brb->status == -EPROTO))
			ERR("%s - nonzero read bulk status received: %d",
			    __func__, brb->status);

		/* only overwrite error flag if there was an error */
		spin_lock_irqsave(&mmstate->cmds.lock, lock_flags);
		{
			mmstate->cmds.errors = brb->status;
		}
		spin_unlock_irqrestore(&mmstate->cmds.lock, lock_flags);
	}
	else {
		if (brb->actual_length == 0) {
			ERR("%s - should never get ZLP anymore\n", __func__);
		}
	}

	spin_lock_irqsave(&mmstate->cmds.lock, lock_flags);
	{
		mmstate->cmds.read_done = 1;
		mmstate->cmds.read_brb = brb;
	}
	spin_unlock_irqrestore(&mmstate->cmds.lock, lock_flags);

	wake_up_interruptible(&mmstate->cmds.read_wait_queue);
	
	LOW2("cmd_read_callback: exit");
}


static ssize_t cmd_read(struct file *file, char *buffer, size_t count,
			loff_t *ppos)
{
	struct cicada_meas_state *mmstate;
	int retval;
	unsigned char *buf = NULL;
	struct brb *brb = NULL;
	int uncopied_bytes = 0;
	int bytes_returned = 0;
	unsigned long lock_flags;
	struct USB_MSG_HEADER *phdr = 0;
	uint16_t ccstat = 0;

	mmstate = (struct cicada_meas_state *)file->private_data;
	if (!mmstate) return -ENODEV;

	FLOW2("cmd_read: entry: count = %d", count);

	mutex_lock(&mmstate->state_struct_mutex);

	if (mmstate->cmds.read_brb) {
		uncopied_bytes = mmstate->cmds.read_brb->actual_length - mmstate->cmds.read_byte_index;
		mutex_unlock(&mmstate->state_struct_mutex);
		FLOW2("cmd_read: have an urb: uncopied bytes = %d", uncopied_bytes);
	}
	else {
		if (mmstate->cmds.brb_is_queued) {
			FLOW2("cmd_read: an urb is already queued");
			mutex_unlock(&mmstate->state_struct_mutex);
		}
		else {
			FLOW2("cmd_read: schedule a fresh urb, uncopied_bytes = %d", uncopied_bytes);

			if (!mmstate->interface) {	/* disconnect was called */
				mutex_unlock(&mmstate->state_struct_mutex);
				retval = -ENODEV;
				goto cmd_read_exit;
			}

			/* create a urb and allocate a buffer to hold the rcv data */
			brb = kmalloc(sizeof(struct brb), GFP_KERNEL);
			if (!brb) {
				ERR("cmd_read: can't allocate brb");
				mutex_unlock(&mmstate->state_struct_mutex);
				retval = -ENOMEM;
				goto cmd_read_exit;
			}

			buf = kmalloc(CICADA_MEAS_URB_BUFBYTES, GFP_KERNEL);
			if (!buf) {
				ERR("cmd_read: can't allocate brb buffer");
				kfree(brb);
				mutex_unlock(&mmstate->state_struct_mutex);
				retval = -ENOMEM;
				goto cmd_read_exit;
			}

			/* initialize the brb properly */
			brb_init(brb, buf, CICADA_MEAS_URB_BUFBYTES,
				 cmd_read_callback, mmstate, NULL, ROUTE_CNTL);

			/* clear these state variables before submitting the brb who's
			 * callback will change their state  */
			mmstate->cmds.read_done = 0;
			mmstate->cmds.read_brb = NULL;

			retval = bulk_submit_rcv_brb(brb);

			mutex_unlock(&mmstate->state_struct_mutex);

			if (retval) {
				ERR("%s - failed submitting read brb, error %d",
				    __func__, retval);
				goto cmd_read_error_free_brb;
			}
			mmstate->cmds.brb_is_queued = 1;
		}


		if (file->f_flags & O_NONBLOCK) {
			FLOW2("cmd_read: non-blocking read, no data");
			/* EAGAIN and EWOULDBLOCK have the same value */
			return -EAGAIN;
		}

		/* wait for callback to tell us that it has something */
		if (wait_event_interruptible(mmstate->cmds.read_wait_queue,
					     mmstate->cmds.read_done == 1)) {
			FLOW("cmd_read: wait_event_interruptible exited, -ERESTARTSYS");
			retval = -ERESTARTSYS;
			goto cmd_read_error_free_brb;
		}

		/* exit with error if read callback detected an error */
		spin_lock_irqsave(&mmstate->cmds.lock, lock_flags);
		{
			if (mmstate->cmds.errors) {
				retval = mmstate->cmds.errors;
				mmstate->cmds.errors = 0;
			}
		}
		spin_unlock_irqrestore(&mmstate->cmds.lock, lock_flags);
		if (retval) {
			ERRDBG("cmd_read: mmstate->cmds.errors = %d", retval);
			goto cmd_read_exit;
		}

		/* verify that we got back the one we submitted */
		if (brb != mmstate->cmds.read_brb) {
			ERR("cmd_read: not same URB as we submitted");
			retval = -EFAULT;
			goto cmd_read_error_free_brb;
		}
		uncopied_bytes = mmstate->cmds.read_brb->actual_length;
	}
	bytes_returned = count < uncopied_bytes ? count : uncopied_bytes;

	if (mmstate->cmds.read_byte_index == 0) {
		/* Only output response id when we're at the start of a buffer */
		/* This will be ok for single packet responses.	 However, it will */
		/* not be correct for intermediate packets in a multipacket transfer. */
		phdr = (struct USB_MSG_HEADER*) mmstate->cmds.read_brb->transfer_buffer;
		FLOW2("cmd_read: respId = 0x%03x, bytes = %d", phdr->MsgID, phdr->TransferSize);
		if (phdr->TransferSize >= (sizeof(struct USB_MSG_HEADER)+sizeof(uint16_t))) {
		    FLOW2("cmd_read: rc = %d, SeqNum = 0x%x", phdr->RouteID, phdr->SeqNum);
		    ccstat = *(((uint16_t*)&phdr->MsgID)+1);
		    /* elevate level if ccstat non-zero */
		    if (ccstat) {  FLOW("cmd_read: ccstat = 0x%x", ccstat); }
		    else	{ FLOW2("cmd_read: ccstat = 0x%x", ccstat); }
		}
	}

	/* copy only the requested number of bytes to output device */
	if (copy_to_user(buffer,
			 mmstate->cmds.read_brb->transfer_buffer + mmstate->cmds.read_byte_index,
			 bytes_returned)) {
		ERR("cmd_read: copy_to_user failed");
		retval = -EFAULT;
		goto cmd_read_error_free_brb;
	}

	mmstate->cmds.read_byte_index += bytes_returned;
	FLOW2("cmd_read: read_byte_index = %d, urb->actual_length = %d",
	     mmstate->cmds.read_byte_index, mmstate->cmds.read_brb->actual_length);

	if (mmstate->cmds.read_byte_index >= mmstate->cmds.read_brb->actual_length) {
		FLOW2("cmd_read: done with urb");
		brb_free(mmstate->cmds.read_brb);
		mmstate->cmds.read_brb = 0;
		mmstate->cmds.read_byte_index = 0;
		mmstate->cmds.brb_is_queued = 0;
	}
	FLOW2("cmd_read: exit: bytes_returned = %d, remaining = %d",
	     bytes_returned, uncopied_bytes - bytes_returned);

	return bytes_returned;

cmd_read_error_free_brb:
	brb_free(brb);

cmd_read_exit:
	mmstate->cmds.brb_is_queued = 0;
	FLOW("cmd_read: error exit = %d", retval);
	return retval;
}


/*
  write callback - don't forget that this runs in interrupt context
*/
static void cmd_write_callback(struct urb *urb)
{
	struct cicada_meas_state *mmstate;
	unsigned long lock_flags;

	mmstate = (struct cicada_meas_state *)urb->context;
	if (!mmstate) return;

	LOW("cmd_write_callback: entry: urb = %p, urb->status = %d",
	    urb, urb->status);

	/* sync/async unlink faults aren't errors */
	if (urb->status) {

		if (!(urb->status == -ENOENT ||
		      urb->status == -ECONNRESET ||
		      urb->status == -ESHUTDOWN ||
		      urb->status == -EPROTO))
			ERR("%s - nonzero write bulk status received: %d",
			    __func__, urb->status);

		spin_lock_irqsave(&mmstate->cmds.lock,
				  lock_flags);
		{
			LOW("cmd_write_callback: urb->status = %d", urb->status);
			mmstate->cmds.errors = urb->status;
		}
		spin_unlock_irqrestore(&mmstate->cmds.lock,
				       lock_flags);
	}
	LOW2("cmd_write_callback: exit");
}


static ssize_t cmd_write(struct file *file, const char *user_buffer,
			 size_t count, loff_t *ppos)
{
	struct cicada_meas_state *mmstate;
	int retval = 0;
	struct urb *urb = NULL;
	char *buf = NULL;
	unsigned long lock_flags;
	size_t writesize = 0;
	struct USB_MSG_HEADER *phdr;

	mmstate = (struct cicada_meas_state *)file->private_data;
	if (!mmstate) return -ENODEV;

	FLOW2("cmd_write: count = %d", count);

	/* verify that we actually have some data to write */
	if (count == 0)
		goto cmd_write_exit;

	spin_lock_irqsave(&mmstate->cmds.lock, lock_flags);
	{
		retval = mmstate->cmds.errors;
		if (retval < 0) {
			/* any error is reported once to preserve
			 * notifications about reset */
			mmstate->cmds.errors = 0;
			retval = (retval == -EPIPE) ? retval : -EIO;
		}
	}
	spin_unlock_irqrestore(&mmstate->cmds.lock, lock_flags);
	if (retval < 0)
		goto cmd_write_error;

	/* Create a urb, and a buffer for it, and copy the data to the urb.
	   urb ref count is set to 1.  */
	urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!urb) {
		retval = -ENOMEM;
		goto cmd_write_error;
	}

	buf = kmalloc(CICADA_MEAS_URB_BUFBYTES, GFP_KERNEL);
	if (!buf) {
		retval = -ENOMEM;
		goto cmd_write_error;
	}

	/* Limit write to max urb buffer size.	Actual writesize is
	   returned so the caller can write the rest in subsequent calls. */
	writesize = min(count, (size_t)CICADA_MEAS_URB_BUFBYTES);
	if (copy_from_user(buf, user_buffer, writesize)) {
		retval = -EFAULT;
		goto cmd_write_error;
	}

	FLOW2("cmd_write: urb = %p, buf = %p", urb, buf);
	phdr = (struct USB_MSG_HEADER *) buf;
	FLOW2("cmd_write: cmdId = 0x%03x, bytes = %d", phdr->MsgID, phdr->TransferSize);
	FLOW2("cmd_write: rc = %d, SeqNum = 0x%x", phdr->RouteID, phdr->SeqNum);

	/* this lock makes sure we don't submit URBs to gone devices */
	mutex_lock(&mmstate->state_struct_mutex);
	{
		if (!mmstate->interface) {	/* disconnect was called */
			mutex_unlock(&mmstate->state_struct_mutex);
			retval = -ENODEV;
			goto cmd_write_error;
		}

		/* initialize the urb properly */
		usb_fill_bulk_urb(urb, mmstate->udev,
			usb_sndbulkpipe(mmstate->udev,
				mmstate->cmd_config.bulk_out_endpoint_addr),
			buf, writesize, cmd_write_callback, mmstate);
		urb->transfer_flags |= URB_FREE_BUFFER | URB_ZERO_PACKET;

		/* send the data out the bulk port - sets refcount to 3 */
		retval = bulk_submit_snd_urb(mmstate, urb, GFP_KERNEL, ROUTE_CNTL);
	}
	mutex_unlock(&mmstate->state_struct_mutex);
	if (retval) {
		ERR("%s - failed submitting write urb, error %d",
			__func__, retval);
		goto cmd_write_error;
	}

	/* release our reference to this urb - sets refcount to 2 */
	usb_free_urb(urb);

	FLOW2("cmd_write: exit - writesize = %d", writesize);

	return writesize;

cmd_write_error:
	usb_free_urb(urb);

cmd_write_exit:
	FLOW("cmd_write: exit = %d", retval);

	return retval;
}


static uint32_t cmd_poll(struct file *filp, poll_table *wait)
{
	uint32_t mask = 0;
	struct cicada_meas_state *mmstate;
	mmstate = (struct cicada_meas_state *)filp->private_data;
	if (!mmstate) return -ENODEV;

	FLOW2("cmd_poll: entry");

	mutex_lock(&mmstate->state_struct_mutex);
	poll_wait(filp, &mmstate->cmds.read_wait_queue, wait);
	if (mmstate->cmds.read_brb)
		mask |= POLLIN | POLLRDNORM;
	mutex_unlock(&mmstate->state_struct_mutex);

	FLOW2("cmd_poll: exit: mask = 0x%x", mask);
	return mask;
}


int cmd_draw_down(struct cicada_meas_state *mmstate)
{
	CONFIG("cmd_draw_down: mmstate = %p", mmstate);
	if (!mmstate) return -ENODEV;
	return 0;
}


void cmd_delete(struct kref *kref)
{
	CONFIG("cmd_delete: entry/exit");
}


/* write to ctlmsg sysnode to pass the byte value on to the control endpoint */
static ssize_t cmd_set_ctlmsg(struct device *dev,
			     struct device_attribute *attr,
			     const char *buf, size_t count)
{
	int value;
	int ret;
	struct usb_interface *intf = to_usb_interface(dev);
	struct cicada_meas_state *mmstate = usb_get_intfdata(intf);

	if (!mmstate) return -ENODEV;

	if (sscanf(buf, "%u", &value) != 1 || value < 1 || value > 255) {
		ERR("cmd_set_ctlmsg: error: argument must be between 1 and 255\n");
		return -EINVAL;
	}
	FLOW("cmd_set_ctlmsg: val = %u", value);

	/* send out the control message */
	ret = usb_control_msg(mmstate->udev,
		      usb_sndctrlpipe(mmstate->udev, 0),
		      value,			 /* bRequest */
		      USB_DIR_OUT | USB_TYPE_VENDOR |
		      USB_RECIP_DEVICE,		/* bmRequestType */
		      0,			/* wValue */
		      0,			/* wIndex */
		      0,			/* *data */
		      0,			/* size */
		      5000);			/* timeout */

	if (ret != 0)
		printk(KERN_NOTICE "cmd_set_ctlmsg: message error = %d.\n", ret);

	FLOW2("cmd_set_ctlmsg: done\n");
	return count;
}
static DEVICE_ATTR(ctlmsg, 0220, NULL, cmd_set_ctlmsg);

static const struct file_operations cmd_fops_table = {
	.owner =	THIS_MODULE,
	.read =		cmd_read,
	.write =	cmd_write,
	.open =		cmd_open,
	.release =	cmd_release,
	.flush =	cmd_flush,
	.poll =		cmd_poll,
};




/*----------------------------------------------------------------------------
 *---- bulk subdriver --------------------------------------------------------
 *----------------------------------------------------------------------------
 *
 */
static void bulk_rcv_urb_cb(struct urb *urb);

/* called by probe to perform bulk subdriver specific actions */
int bulk_probe(struct cicada_meas_state *mmstate)
{
	int routeId;
	int retval = 0;
	CONFIG("bulk_probe: entry: mmstate = %p", mmstate);

	kref_get(&mmstate->kref);

	spin_lock_init(&mmstate->bulks.snd_lock);
	spin_lock_init(&mmstate->bulks.rcv_lock);

	/* Initialize transaction array, including ones never used. */
	for (routeId = 0; routeId < ROUTE_N; routeId++) {
		mmstate->bulks.transactions[routeId].brb = NULL;
	}

	mmstate->bulks.rcv_urb_in_limbo = NULL;
	mmstate->bulks.rcv_urbs_active = 0;

	CONFIG("bulk_probe: exit retval = %d", retval);

	return retval;
}


/* called by disconnect to perform int subdriver specific actions
 * Note: this is run within a lock of mmstate->state_struct_mutex Also
 * note: the USB subsystem should have called all urb callbacks before
 * we get here.
 */

int bulk_disconnect(struct cicada_meas_state *mmstate)
{
	/* Note: all urb's have been killed already */
	CONFIG("bulk_disconnect: entry: mmstate = %p", mmstate);

	/* debug messages */
	if (mmstate->bulks.rcv_urb_in_limbo) {
		FLOW("bulk_disconnect: rcv_urb_in_limbo = %p", mmstate->bulks.rcv_urb_in_limbo);
	}
	if (mmstate->bulks.rcv_urbs_active) {
		FLOW("bulk_disconnect: rcv_urbs_active = %d", mmstate->bulks.rcv_urbs_active);
	}
	if (mmstate->bulks.transactions[ROUTE_CNTL].brb) {
		FLOW("bulk_disconnect: transactions[ROUTE_CNTL].brb = %p", mmstate->bulks.transactions[ROUTE_CNTL].brb);
	}

	if (mmstate->bulks.transactions[ROUTE_NET].brb) {
		FLOW("bulk_disconnect: transactions[ROUTE_NET].brb = %p", mmstate->bulks.transactions[ROUTE_NET].brb);
	}

	kref_put(&mmstate->kref, &cicada_meas_delete);

	CONFIG("bulk_disconnect: exit");

	return 0;
}

void bulk_delete(struct cicada_meas_state *mmstate)
{
	CONFIG("bulk_delete: entry");
	kfree(mmstate->bulks.rcv_urb_in_limbo);
	brb_free(mmstate->bulks.transactions[ROUTE_CNTL].brb);
	brb_free(mmstate->bulks.transactions[ROUTE_NET].brb);
}


/* Submit transmit urb.	 This only exists so we can see the order and
 * timing of all bulk transactions.  This is invaluable when
 * troubleshooting interactions between cicada_rcom and cicada_meas.
 */
static int bulk_submit_snd_urb(void *mmstate_vp,
			       struct urb *client_snd_urb,
			       gfp_t mem_flags,
			       int routing_code)
{
	unsigned long lock_flags;
	int retval = 0;
	struct USB_MSG_HEADER *phdr;
	struct cicada_meas_state *mmstate = (struct cicada_meas_state*) mmstate_vp;

	if (!mmstate) return -ENODEV;

	FLOW2("bulk_submit_snd_urb: entry: client_snd_urb = %p, mem_flags = %d", client_snd_urb, mem_flags);
	phdr = (struct USB_MSG_HEADER *) client_snd_urb->transfer_buffer;
	FLOW("bulk_submit_snd_urb: cmdId  = 0x%03x, bytes = %4d", phdr->MsgID, phdr->TransferSize);
	FLOW2("bulk_submit_snd_urb: rc = %d, SeqNum = 0x%x", routing_code, phdr->SeqNum);

	// Sorry, this is a bad break in encapsulation, but seeing
	// a few parts of RxCmd make connecting the layers possible.
	// This should be installable, but I'm in kind of a hurry.
	//
	// Size of IP/UDP/USB overhead - maybe get this from a better
	// source
	if (phdr->MsgID == CMD_CODE_SENDMESG) {
		static const int OVERHEAD_BYTES   = 50;
		static const int RXCMD_ID_OFFSET  =	 0;
		static const int RXCMD_SEQ_OFFSET =	 8;
		static const int rxCmdIdOffset = sizeof(struct USB_MSG_HEADER) +
						 OVERHEAD_BYTES +
						 RXCMD_ID_OFFSET;
		static const int rxCmdSeqOffset = rxCmdIdOffset + RXCMD_SEQ_OFFSET;

		if (phdr->TransferSize >= (rxCmdSeqOffset+sizeof(uint32_t))) {
			uint8_t* buf = client_snd_urb->transfer_buffer;
			uint32_t rxCmdId = 0;
			rxCmdId	 = *(buf + rxCmdIdOffset);
			rxCmdId |= *(buf + rxCmdIdOffset+1) <<	8;
			rxCmdId |= *(buf + rxCmdIdOffset+2) << 16;
			rxCmdId |= *(buf + rxCmdIdOffset+3) << 24;

			if ((rxCmdId >> 16) == 0x0081) {
				uint32_t rxCmdSeq = 0;
				rxCmdSeq  = *(buf + rxCmdSeqOffset);
				rxCmdSeq |= *(buf + rxCmdSeqOffset+1) <<  8;
				rxCmdSeq |= *(buf + rxCmdSeqOffset+2) << 16;
				rxCmdSeq |= *(buf + rxCmdSeqOffset+3) << 24;
				RCRX("bulk_submit_snd_urb: sendMsg RxCmd:  id = 0x%08x, seq = %10u", rxCmdId, rxCmdSeq);
			}
		}
	}

	spin_lock_irqsave(&mmstate->bulks.snd_lock, lock_flags);
	{
		retval = usb_submit_urb(client_snd_urb, mem_flags);
	}
	spin_unlock_irqrestore(&mmstate->bulks.snd_lock, lock_flags);

	FLOW2("bulk_submit_snd_urb: exit");

	return retval;
}

/* helper function for copying the relevant urb to the brb and calling
 * the brb callback.
 */
static void bulk_urb_call_cb(struct cicada_meas_state *mmstate,
			     struct brb *brb,
			     struct urb *urb,
			     bool immediate)
{
	(void) immediate;

	if (urb) {
		/* copy data from urb to brb */
		brb->status = urb->status;
	}
	else {
		brb->status = -EPROTO;
	}

	if (brb->status) {
		/* there was an urb error */
		brb->actual_length = 0;
	}
	else if (urb) {
		brb->actual_length = urb->actual_length;
		memcpy(brb->transfer_buffer, urb->transfer_buffer, urb->actual_length);
	}
	else {
	    pr_err("should not be possible to get here");
	}

	/* call completion callback */
	brb->callback(brb);

	mmstate->bulks.rcv_urbs_active--;
}


/* submit receive brb.
 *
 * 
 */
static int bulk_submit_rcv_brb(struct brb *brb)
{
	unsigned long lock_flags;
	int retval = -EINVAL;
	unsigned char *buf = NULL;
	struct urb *urb = 0;
	struct bulk_transaction *transaction = 0;
	struct cicada_meas_state *mmstate = (struct cicada_meas_state*) brb->mmstate;
	int    callback_immediately = 0;
	int rc = brb->routing_code;

	if (!mmstate) return -ENODEV;

	spin_lock_irqsave(&mmstate->bulks.rcv_lock, lock_flags);
	FLOW2("bulk_submit_rcv_brb: brb = %p, rc = %d: entry", brb, rc);
	transaction = &mmstate->bulks.transactions[rc];

	do
	{
		if (transaction->brb != NULL) {
			ERRDBG("bulk_submit_rcv_brb: ERROR: brb for rc = %d still pending", rc);
			retval = -EINVAL;
			break;
		}


		if (!mmstate->interface) {	/* disconnect was called */
			retval = -ENODEV;
			break;
		}

		if (mmstate->bulks.rcv_urb_in_limbo) {
			struct USB_MSG_HEADER *phdr = (struct USB_MSG_HEADER *)
						    mmstate->bulks.rcv_urb_in_limbo->transfer_buffer;
			if (phdr->RouteID == rc) {
				/* The response to this command has already been received in the
				 * receive urb for a command from the other driver.  Call the
				 * callback for this brb later in this function.
				 *
				 * Note: We still need schedule a new urb so the other command's
				 * response will have somewhere to go.
				 */
				callback_immediately = 1;
			}
		}

		/* create a urb and allocate a buffer to hold the rcv data */
		urb = usb_alloc_urb(0, GFP_ATOMIC);
		if (!urb) {
			ERR("bulk_submit_rcv_brb: can't allocate urb");
			retval = -ENOMEM;
			break;
		}

		buf = kmalloc(CICADA_MEAS_URB_BUFBYTES, GFP_ATOMIC);
		if (!buf) {
			ERR("bulk_submit_rcv_brb: can't allocate urb buffer");
			usb_free_urb(urb);
			retval = -ENOMEM;
			break;
		}

		/* initialize the urb properly */
		usb_fill_bulk_urb(urb, mmstate->udev,
				  usb_rcvbulkpipe(mmstate->udev,
					mmstate->cmd_config.bulk_in_endpoint_addr),
				  buf, CICADA_MEAS_URB_BUFBYTES,
				  bulk_rcv_urb_cb, mmstate);
		urb->transfer_flags |= URB_FREE_BUFFER;

		retval = usb_submit_urb(urb, GFP_ATOMIC);
		if (retval) {
			usb_free_urb(urb);
		}
		else {
			transaction->brb = brb;
			FLOW2("bulk_submit_rcv_brb: brb = %p, rcv_urbs_active = %d",
				     brb, mmstate->bulks.rcv_urbs_active);
			mmstate->bulks.rcv_urbs_active++;

			/* release our reference to this urb - sets refcount to 2 */
			usb_free_urb(urb);
		}
	}
	while (0);

	if (callback_immediately) {
		FLOW("**** bulk_submit_rcv_brb: call bulk callback immediately: brb = %p, urb_in_limbo = %p\n",
				    brb, mmstate->bulks.rcv_urb_in_limbo);

		/* call the callback with the rcv_urb_in_limbo and free it */
		bulk_urb_call_cb(mmstate, brb, mmstate->bulks.rcv_urb_in_limbo, true);
		usb_free_urb(mmstate->bulks.rcv_urb_in_limbo);
		mmstate->bulks.rcv_urb_in_limbo = NULL;
		mmstate->bulks.rcv_urbs_active--;

		/* This brb has been handled */
		transaction->brb = NULL;
		retval = 0;
	}

	spin_unlock_irqrestore(&mmstate->bulks.rcv_lock, lock_flags);

	FLOW2("bulk_submit_rcv_brb: exit: retval = %d", retval);

	return retval;
}


/* Callback for bulk receive urbs.
 *
 * It will normally get called for each receive urb, copy the urb
 * contents to the associated brb and call the brb callback.
 *
 * Because the cicada_rcom and cicada_meas both use the bulk
 * endpoints, it needs to handle the case where one driver steals the
 * other's response.
 *
 * For example, the sequence may go as follows:
 *    - meas driver sends command such as "writeperiph"
 *    - rcom driver sends a ReadMesg command
 *    - rcom driver submits a recv urb
 *    - response to writeperiph command comes back and is put
 *	    into the recv usb for the ReadMesg command.
 *
 * This function detects this situation and remembers the received
 * urb using the "limbo" pointer.  This urb is then connected to
 * its brb at a later time.
 */
static void bulk_rcv_urb_cb(struct urb *urb)
{
	struct bulk_transaction *transaction;
	struct cicada_meas_state *mmstate =
			(struct cicada_meas_state *)urb->context;
	unsigned long lock_flags = 0;
	struct USB_MSG_HEADER *phdr = 0;
	int rc = ROUTE_INVALID;
	int retval = 0;
	uint16_t ccstat = 0;

	/* We normally complete the transaction even if there was an error
	 * returned by the urb (so the submitter knows) */
	int process_urb = 1;

	if (!mmstate) return;

	spin_lock_irqsave(&mmstate->bulks.rcv_lock, lock_flags);
	FLOW2("bulk_rcv_urb_cb:	    entry: urb = %p, urb->status = %d", urb, urb->status);
	FLOW2("bulk_rcv_urb_cb:	    entry: rcv_urbs_active = %d", mmstate->bulks.rcv_urbs_active);

	do {
		if (urb->status) {
			/* this urb had an error so we can't look at the contents */
			FLOW("bulk_rcv_urb_cb:	   urb->status = %d", urb->status);
			break;
		}

		if (urb->actual_length == 0) {
			/* resubmit urb because we just received a zero-length
			 * packet.  Go get the one we really want */
			retval = usb_submit_urb(urb, GFP_ATOMIC);
			if (retval) {
				ERR("%s - failed resubmitting read urb, error %d",
				    __func__, retval);
			}
			process_urb = 0;    /* zlp is special since there's nothing to do */
			FLOW("bulk_rcv_urb_cb: exit resubmit after ZLP");
			break;
		}

		if (urb->actual_length < sizeof(struct USB_MSG_HEADER)) {
			/* this shouldn't fail, just in case */
			ERR("bulk_rcv_urb_cb:	  usb message too small = %d\n",
						urb->actual_length);
			break;
		}

		/* check header and get routing code from data packet */
		phdr = (struct USB_MSG_HEADER *) urb->transfer_buffer;

		if (phdr->MagicNum != CICADA_CMD_MAGIC) {
			ERRDBG("bulk_rcv_urb_cb:     magic number = %d, expected = %d\n",
			    phdr->MagicNum, CICADA_CMD_MAGIC);
			break;
		}

		if ((uint8_t)phdr->SeqNum != (uint8_t)~phdr->SeqInv) {
			ERRDBG("bulk_rcv_urb_cb:     SeqNum = 0x%x, seqInv = 0x%x\n",
			    phdr->SeqNum, phdr->SeqInv);
			break;
		}

		FLOW("bulk_rcv_urb_cb:	   respId = 0x%03x, bytes = %4d\n", phdr->MsgID, phdr->TransferSize);
		FLOW2("bulk_rcv_urb_cb:	    rc=%d, SeqNum = 0x%x", phdr->RouteID, phdr->SeqNum);

		/* assign routing code to "rc" if valid */
		if (phdr->RouteID == ROUTE_CNTL || phdr->RouteID == ROUTE_NET) {
			rc = phdr->RouteID;
			if (phdr->TransferSize >= (sizeof(struct USB_MSG_HEADER)+sizeof(uint16_t))) {
				ccstat = *(((uint16_t*)&phdr->MsgID)+1);
				/* elevate level if ccstat non-zero */
				if (ccstat) {  FLOW("bulk_rcv_urb_cb:	  ccstat = 0x%x", ccstat); }
				else	    { FLOW2("bulk_rcv_urb_cb:	  ccstat = 0x%x", ccstat); }
			}

			// Sorry, this is a bad break in encapsulation, but seeing
			// a few parts of RxCmd make connecting the layers possible.
			// This should be installable, but I'm in kind of a hurry.
			//
			// Size of IP/UDP/USB overhead - maybe get this from a better
			// source
			if (phdr->MsgID == DEV_RESP_READMESG) {
				static const int OVERHEAD_BYTES    = 54;
				static const int RXRESP_ID_OFFSET  =  0;
				static const int RXRESP_SEQ_OFFSET =  8;
				static const int rxRespIdOffset = sizeof(struct USB_MSG_HEADER) +
									OVERHEAD_BYTES +
									RXRESP_ID_OFFSET;
				static const int rxRespSeqOffset = rxRespIdOffset + RXRESP_SEQ_OFFSET;

				if (phdr->TransferSize >= (rxRespSeqOffset+sizeof(uint32_t))) {
					uint8_t* buf = (uint8_t*)urb->transfer_buffer;
					uint32_t rxRespId = 0;
					rxRespId  = *(buf + rxRespIdOffset);
					rxRespId |= *(buf + rxRespIdOffset+1) <<  8;
					rxRespId |= *(buf + rxRespIdOffset+2) << 16;
					rxRespId |= *(buf + rxRespIdOffset+3) << 24;

					if ((rxRespId >> 16) == 0x0081) {
						uint32_t rxRespSeq = 0;
						rxRespSeq  = *(buf + rxRespSeqOffset);
						rxRespSeq |= *(buf + rxRespSeqOffset+1) <<	8;
						rxRespSeq |= *(buf + rxRespSeqOffset+2) << 16;
						rxRespSeq |= *(buf + rxRespSeqOffset+3) << 24;
						RCRX("bulk_rcv_urb_cb:     recvMsg RxResp: id = 0x%08x, seq = %10u", rxRespId, rxRespSeq);
					}
				}
			}
		}
	}
	while (0);

	if (!process_urb) {
		spin_unlock_irqrestore(&mmstate->bulks.rcv_lock, lock_flags);
		return;
	}

	if (!mmstate->bulks.rcv_urbs_active) {
		ERRDBG("**** bulk_rcv_urb_cb: should always have a transaction at this point");
		spin_unlock_irqrestore(&mmstate->bulks.rcv_lock, lock_flags);
		LOW2("bulk_rcv_urb_cb:	   exit");
		return;
	}

	if (rc == ROUTE_INVALID) {
		/* USB error or some other error with the packet.
		 *  Notify all registered brbs. */
		if (urb->status == 0) {
			/* make sure URB status reports an error to brb */
			urb->status = -EINVAL;
		}

		/* check for a pending cmd/cntl transaction */
		transaction = &mmstate->bulks.transactions[ROUTE_CNTL];
		if (transaction->brb) {
			FLOW("**** bulk_rcv_urb_cb: urb = %p, rc = INVALID, Error callback for ROUTE_CNTL, urb->status = %d", urb, urb->status);
			bulk_urb_call_cb(mmstate, transaction->brb, urb, false);
			transaction->brb = NULL;
		}

		/* check for a pending net transaction */
		transaction = &mmstate->bulks.transactions[ROUTE_NET];
		if (transaction->brb) {
			FLOW("**** bulk_rcv_urb_cb: urb = %p, rc = INVALID, Error callback for ROUTE_NET, urb->status = %d", urb, urb->status);
			bulk_urb_call_cb(mmstate, transaction->brb, urb, false);
			transaction->brb = NULL;
		}


		if (mmstate->bulks.rcv_urb_in_limbo) {
			pr_err("**** bulk_rcv_urb_cb: urb = %p, rc = INVALID, Error callback, there was a limbo", urb);
			usb_free_urb(mmstate->bulks.rcv_urb_in_limbo);
			mmstate->bulks.rcv_urb_in_limbo = NULL;
		}
		mmstate->bulks.rcv_urbs_active = 0;
	}
	else {
		/* get pointer to original brb */
		transaction = &mmstate->bulks.transactions[rc];
		if (transaction->brb) {
			/* call callback for urb */
			FLOW2("bulk_rcv_urb_cb:	    urb = %p, rc = %d: calling callback func", urb, rc);
			FLOW2("bulk_rcv_urb_cb:	    brb = %p", transaction->brb);
			bulk_urb_call_cb(mmstate, transaction->brb, urb, false);
			transaction->brb = NULL;

			if (mmstate->bulks.rcv_urb_in_limbo) {
				/* we also have a URB in limbo so send it on its way */

				if (rc == ROUTE_CNTL) {
					transaction = &mmstate->bulks.transactions[ROUTE_NET];
					FLOW("**** bulk_rcv_urb_cb: brb = %p, limbo_urb = %p, ROUTE_NET",
					       transaction->brb,
					       mmstate->bulks.rcv_urb_in_limbo);
					bulk_urb_call_cb(mmstate, transaction->brb,
							 mmstate->bulks.rcv_urb_in_limbo, false);
					mmstate->bulks.transactions[ROUTE_NET].brb = NULL;
				}
				else {
					transaction = &mmstate->bulks.transactions[ROUTE_CNTL];
					FLOW("**** bulk_rcv_urb_cb: brb = %p, limbo_urb = %p, ROUTE_CNTL",
					       transaction->brb,
					       mmstate->bulks.rcv_urb_in_limbo);
					bulk_urb_call_cb(mmstate, transaction->brb,
							 mmstate->bulks.rcv_urb_in_limbo, false);
					mmstate->bulks.transactions[ROUTE_CNTL].brb = NULL;
				}
				mmstate->bulks.rcv_urb_in_limbo = NULL;
				mmstate->bulks.rcv_urbs_active--;
			}
		}
		else {
			if (mmstate->bulks.rcv_urb_in_limbo) {
				pr_err("**** bulk_rcv_urb_cb: no brb for rc = %d, urb = %p LIMBO FULL ************\n", rc, urb);
				FLOW("**** bulk_rcv_urb_cb: no brb for rc = %d, urb = %p LIMBO FULL ************\n", rc, urb);
			}
			else {
				/* don't know which one to send this to yet, so save it */
				FLOW("**** bulk_rcv_urb_cb: no brb for rc = %d, urb = %p to limbo\n", rc, urb);
				usb_get_urb(urb);
				mmstate->bulks.rcv_urb_in_limbo = urb;
				mmstate->bulks.rcv_urbs_active++;
			}
		}
	}

	spin_unlock_irqrestore(&mmstate->bulks.rcv_lock, lock_flags);

	LOW2("bulk_rcv_urb_cb:	   exit");
}



/*----------------------------------------------------------------------------
 *---- event subdriver -------------------------------------------------------
 *----------------------------------------------------------------------------
 *
 * The Event Driver provides the conduit to user space for interrupt
 * endpoint packets.  Separating handling of the interrupt endpoint
 * packets into the event and interrupt subdrivers was necessary
 * because some interrupt packets are routed to cicada_rcom.  Event
 * data destined for userspace can be read by reading
 * /dev/cicada_meas_evt0.
 *
 * As with the cmd subdriver, read() provides a fifo-like interface.
 *
 * The event subdriver supports non-blocking reads.
 */

static struct device_attribute dev_attr_evt_buf_count_max;
static struct device_attribute dev_attr_evt_matlab_comm;

int evt_probe(struct cicada_meas_state *mmstate, int major, int minor,
	      int suffix)
{
	dev_t dinfo;

	CONFIG("evt_probe: entry: major = %d, minor = %d", major, minor);

	if (!mmstate) return -ENODEV;

	kref_get(&mmstate->kref);

	/* create char device and associate its fops table */
	mmstate->evts.major = major;
	mmstate->evts.minor = minor;
	dinfo = MKDEV(major, minor);

	init_waitqueue_head(&mmstate->evts.wait_queue);
	mutex_init(&mmstate->evts.buf_mutex);

	mmstate->evts.buf_count = 0;
	mmstate->evts.buf_count_max = 0;

	if (device_create_file(&mmstate->interface->dev, &dev_attr_evt_buf_count_max)) {
		ERR("evt_probe: device_create_file for buf count failed\n");
		goto evt_probe_error;
	}

	if (device_create_file(&mmstate->interface->dev, &dev_attr_evt_matlab_comm)) {
		ERR("evt_probe: device_create_file for matlab comm failed\n");
		goto evt_attr_buf_count_max_error;
	}

	/* Create evt char device and associate its fops table.
	 * mmstate must have pointer to cdev (cdev must not be embedded in mmstate).
	 */
	mmstate->evts.cdev = cdev_alloc();
	if (!mmstate->evts.cdev) {
		ERR("evt_probe: cdev_alloc failed\n");
		goto evt_attr_matlab_comm_error;
	}
	mmstate->evts.cdev->owner = THIS_MODULE;
	mmstate->evts.cdev->ops = &evt_fops_table;
	mmstate->evts.mmstate = mmstate;

	if (cdev_add(mmstate->evts.cdev, dinfo, 1) < 0) {
		ERR("evt_probe: cdev_add failed\n");
		goto evt_delete_cdev;
	}

	mmstate->evts.char_dev_node = device_create(
						cicada_meas_class,
						&mmstate->interface->dev,
						dinfo, NULL,
						"%s%d", "cicada_meas_evt", suffix);

	if (IS_ERR(mmstate->evts.char_dev_node)) {
		ERR("evt_probe: device_create failed\n");
		goto evt_delete_cdev;
	}

	mmstate->evts.exiting = 0;

	CONFIG("evt_probe: exit");

	return 0;

evt_delete_cdev:
	cdev_del(mmstate->evts.cdev);

evt_attr_matlab_comm_error:
	device_remove_file(&mmstate->interface->dev, &dev_attr_evt_matlab_comm);

evt_attr_buf_count_max_error:
	device_remove_file(&mmstate->interface->dev, &dev_attr_evt_buf_count_max);

evt_probe_error:
	kref_put(&mmstate->kref, &cicada_meas_delete);

	return -ENODEV;
}


/* Note: this is run within a lock of mmstate->state_struct_mutex */
int evt_disconnect(struct cicada_meas_state *mmstate)
{
	dev_t  dinfo;

	CONFIG("evt_disconnect: entry: major = %d, minor = %d",
	       mmstate->evts.major, mmstate->evts.minor);

	/* wake up evt_read() or poll if blocked so we can return a
	 * read error */
	mmstate->evts.buf_count = 1;
	mmstate->evts.exiting = 1;
	wake_up_interruptible(&mmstate->evts.wait_queue);

	device_remove_file(&mmstate->interface->dev, &dev_attr_evt_buf_count_max);
	device_remove_file(&mmstate->interface->dev, &dev_attr_evt_matlab_comm);
	dinfo = MKDEV(mmstate->evts.major, mmstate->evts.minor);

	if (mmstate->evts.char_dev_node)
		device_destroy(cicada_meas_class, dinfo);

	/* cdev ops may still be used by file system after cdev_del() is called.
	 * The memory pointed to by mmstate->evts.cdev will be freed after all users
	 * have finished. */
	cdev_del(mmstate->evts.cdev);
	mmstate->evts.cdev = NULL;

	kref_put(&mmstate->kref, &cicada_meas_delete);

	CONFIG("evt_disconnect: exit");

	return 0;
}


static int evt_open(struct inode *inode, struct file *file)
{
	int i;	  
	int retval = 0;
	struct cicada_meas_state *mmstate;
	struct evt_data_buf_desc *dbdesc;

	CONFIG("evt_open: entry - inode = %p, file = %p", inode, file);
	mmstate = get_mmstate_active(evt_cdev_from_mmstate, inode->i_cdev);
	if (!mmstate) {
		CONFIG("evt_open: mmstate is null");
		return -ENODEV;
	}

	/* increment our usage count for the device */
	kref_get(&mmstate->kref);

	/* lock the device to allow correctly handling errors
	 * in resumption */
	mutex_lock(&mmstate->state_struct_mutex);
	{
		/* make sure interface is still there */
		if (!mmstate->interface) {
			retval = -ENODEV;
			goto open_error;
		}

		if (!mmstate->evts.open_count) {
			mmstate->evts.open_count++;
		}
		else {
			retval = -EBUSY;
			ERR("evt_open: error: open_count = %d",
			    mmstate->evts.open_count);
			goto open_error;
		}
	}
	mutex_unlock(&mmstate->state_struct_mutex);

	mutex_lock(&mmstate->evts.buf_mutex);
	{
		/* Initialize lists */
		INIT_LIST_HEAD(&mmstate->evts.rcv_list);
		INIT_LIST_HEAD(&mmstate->evts.free_list);

		/* populate the free list (create empty buffer descs and buffers) */
		for (i = 0; i < EVT_DATA_BUF_N; i++) {
			dbdesc = kzalloc(sizeof(*dbdesc), GFP_KERNEL);
			dbdesc->pbuf = kzalloc(EVT_DATA_BUF_BYTES, GFP_KERNEL);
			list_add_tail(&dbdesc->list,
				      &mmstate->evts.free_list);
			DBG("evt_open: %d:", i);
			DBG("evt_open: data_buf_desc = %p, data_buf = %p",
			       dbdesc, dbdesc->pbuf);
		}

		/* index into urb to allow partial reads*/
		mmstate->evts.read_byte_index = 0;
		mmstate->evts.buf_count = 0;

		/* save our object in the file's private structure */
		file->private_data = mmstate;

		/* register to get our events from the interrupt endpoint */
		int_register_callback(mmstate, ROUTE_EVENT, &evt_callback, 0);
	}
	mutex_unlock(&mmstate->evts.buf_mutex);

	CONFIG("evt_open: exit: retval = %d", retval);
	return retval;

open_error:
	mutex_unlock(&mmstate->state_struct_mutex);
	kref_put(&mmstate->kref, cicada_meas_delete);

	CONFIG("evt_open: error exit: %d", retval);
	return retval;
}


static int evt_release(struct inode *inode, struct file *file)
{
	struct evt_data_buf_desc *dbdesc;
	struct list_head *ptr;
	struct list_head *next;

	struct cicada_meas_state *mmstate =
		(struct cicada_meas_state *)file->private_data;
	if (!mmstate) return -ENODEV;

	CONFIG("evt_release: entry");

	mutex_lock(&mmstate->state_struct_mutex);
	{
		mmstate->evts.open_count--;
	}
	mutex_unlock(&mmstate->state_struct_mutex);

	mutex_lock(&mmstate->evts.buf_mutex);
	{
		/* unregister for notifications from interrupt endpoint */
		int_register_callback(mmstate, ROUTE_EVENT, 0, 0);

		/* delete all received buffers (including temporaries) */
		list_for_each_safe(ptr, next, &mmstate->evts.rcv_list) {
			dbdesc = list_entry(ptr, struct evt_data_buf_desc, list);
			DBG("evt_release: delete rcv buf desc %p, buf = %p",
			       dbdesc, dbdesc->pbuf);
			list_del(&(dbdesc->list));
			kfree(dbdesc->pbuf);
			kfree(dbdesc);
		}

		/* delete all free buffers (if any) */
		list_for_each_safe(ptr, next, &mmstate->evts.free_list) {
			dbdesc = list_entry(ptr, struct evt_data_buf_desc, list);
			DBG("evt_release: delete free buf desc %p, buf = %p",
			       dbdesc, dbdesc->pbuf);
			list_del(&(dbdesc->list));
			kfree(dbdesc->pbuf);
			kfree(dbdesc);
		}
	}
	mutex_unlock(&mmstate->evts.buf_mutex);

	/* decrement the count on our device */
	kref_put(&mmstate->kref, cicada_meas_delete);

	CONFIG("evt_release: exit");
	return 0;
}



/* Callback from interrupt endpoint manager with our data.
 *
 * Note, this runs in workqueue context so we can use a mutex.
 */
void evt_callback(struct urb *urb, void *context)
{
	struct evt_data_buf_desc *dbdesc = 0;
	struct cicada_meas_state *mmstate = urb->context;

	if (!mmstate) return;

	LOW("evt_callback: entry: urb = %p, status = %d",
		 urb, urb->status);

	/* must lock around list manipulations and access to shared data */
	mutex_lock(&mmstate->evts.buf_mutex);
	{
		switch (urb->status) {
		case 0: /* success */

			if (urb->actual_length > EVT_DATA_BUF_BYTES) {
				ERR("evt_callback: data too big = %d\n",
						urb->actual_length);
				mmstate->evts.error = -EFAULT;
				break;
			}

			if (list_empty(&mmstate->evts.free_list)) {
				if (mmstate->evts.buf_count > EVT_DATA_BUF_MAX) {
					FLOW("evt_callback: drop buffer (buf_count = %d)\n",
						mmstate->evts.buf_count);
					break;
				}
				else {
					/* allocate a new one*/
					dbdesc = kzalloc(sizeof(*dbdesc), GFP_KERNEL);
					dbdesc->pbuf = kzalloc(EVT_DATA_BUF_BYTES,
									GFP_KERNEL);
					dbdesc->temporary = 1;
					FLOW2("evt_callback: allocate a TEMP buffer (pbuf = %p, buf_count = %d)\n",
					     dbdesc->pbuf, mmstate->evts.buf_count);
				}
			} else {
				/* get buffer descriptor from the free list */
				dbdesc = list_first_entry(
						&mmstate->evts.free_list,
						struct evt_data_buf_desc, list);

				list_del_init(&dbdesc->list);
			}

			if (++mmstate->evts.buf_count >
					mmstate->evts.buf_count_max)
				mmstate->evts.buf_count_max =
						mmstate->evts.buf_count;

			FLOW2("evt_callback: buf_count = %d, buf_count_max = %d",
			    mmstate->evts.buf_count, mmstate->evts.buf_count_max);

			memcpy(dbdesc->pbuf, urb->transfer_buffer,
					     urb->actual_length);
			dbdesc->nbytes = urb->actual_length;

			/* put the buffer descriptor on the received
			   list and updated evt status */
			list_add_tail(&dbdesc->list, &mmstate->evts.rcv_list);

			wake_up_interruptible(&mmstate->evts.wait_queue);
			break;

		default:		/* error */
			ERRDBG("evt_callback: urb error = %d\n", urb->status);
		}

		if (mmstate->evts.error == 0)
			mmstate->evts.error = urb->status;
	}
	mutex_unlock(&mmstate->evts.buf_mutex);

	LOW2("evt_callback: exit: status = %d", urb->status);
}


/* read count bytes from evt devnode.
 *
 * supports O_NONBLOCK.
*/
static ssize_t evt_read(struct file *file, char *buffer, size_t count,
			loff_t *ppos)
{
	struct cicada_meas_state *mmstate;
	struct evt_data_buf_desc *dbdesc = 0;
	int	retval = 0;
	int	already_have_a_buf = 0;
	int	bytes_available = 0;
	int	bytes_this_time;
	struct USB_MSG_HEADER *phdr = 0;
	uint16_t ccstat = 0;

	mmstate = (struct cicada_meas_state *)file->private_data;
	if (!mmstate) return -ENODEV;

	FLOW2("evt_read: entry, buf_count = %d", mmstate->evts.buf_count);

	if (mmstate->evts.exiting) {
		FLOW("evt_read: unplugged");
		return -ENODEV;
	}

	mutex_lock(&mmstate->evts.buf_mutex);
	if (!list_empty(&mmstate->evts.rcv_list))
		already_have_a_buf = 1;
	mutex_unlock(&mmstate->evts.buf_mutex);

	if (already_have_a_buf) {
		FLOW2("evt_read: already have some data, so don't worry about blocking state");
	}
	else {
		if (file->f_flags & O_NONBLOCK) {
			FLOW2("evt_read: non-blocking read, no data");
			/* EAGAIN and EWOULDBLOCK have the same value */
			return -EAGAIN;
		}
		else {
			FLOW2("evt_read: blocking read, wait for data");

			/* block here until the read buffer comes in */
			if (wait_event_interruptible(mmstate->evts.wait_queue,
						     mmstate->evts.buf_count >= 1)) {
				FLOW("evt_read: wait_event_interruptible exited -ERESTARTSYS");
				return -ERESTARTSYS;
			}
			FLOW("evt_read: wait_event_interruptible awake");
		}
	}

	mutex_lock(&mmstate->state_struct_mutex);
	if (!mmstate->interface) {
		/* we can end up here if the USB was disconnected while we
		   were waiting above.	If so, report that it is gone. */
		retval = -ENODEV;
		mutex_unlock(&mmstate->state_struct_mutex);
		FLOW2("evt_read: mmstate->interface == NULL");
		return retval;
	}
	mutex_unlock(&mmstate->state_struct_mutex);


	/* Return immediately if the evt callback found an error */
	mutex_lock(&mmstate->evts.buf_mutex);
	retval = mmstate->evts.error;
	mmstate->evts.error = 0;
	mutex_unlock(&mmstate->evts.buf_mutex);
	if (retval < 0) {
		ERR("evt_read: read callback returned error = %d", retval);
		return retval;
	}


	/* get input data just received */
	mutex_lock(&mmstate->evts.buf_mutex);

	do {
		if (!list_empty(&mmstate->evts.rcv_list)) {

			/* get first entry, but leave it on the recv list for now */
			dbdesc = list_first_entry(&mmstate->evts.rcv_list,
						struct evt_data_buf_desc, list);

			if (already_have_a_buf)
				bytes_available = dbdesc->nbytes - mmstate->evts.read_byte_index;
			else
				bytes_available = dbdesc->nbytes;

			bytes_this_time = count;
			if (bytes_this_time > bytes_available)
			    bytes_this_time = bytes_available;

			FLOW2("evt_read: buf_bytes = %d, read_index = %d",
			      dbdesc->nbytes,
			      mmstate->evts.read_byte_index);
			FLOW2("evt_read: bytes_this_time = %d", bytes_this_time);

			if (!bytes_this_time) {
				ERR("evt_read: error: somehow have a buffer but zero bytes");
				retval = -EFAULT;
				break;
			}

			/* Only output response id when we're at the start of a buffer */
			phdr = (struct USB_MSG_HEADER*) dbdesc->pbuf;
			if (mmstate->evts.read_byte_index == 0 &&
			    phdr->TransferSize >= (sizeof(struct USB_MSG_HEADER)+2))
			{ 
				FLOW("evt_read: respId = 0x%03x, bytes = %4d", phdr->MsgID, phdr->TransferSize);
				if (phdr->TransferSize >= (sizeof(struct USB_MSG_HEADER)+2)) {
					ccstat = *(((uint16_t*)&phdr->MsgID)+1);
					/* elevate level if ccstat is non-zero */
					if (ccstat) {  FLOW("evt_read: ccstat = 0x%x", ccstat); }
					else	    { FLOW2("evt_read: ccstat = 0x%x", ccstat); }
				}
			}

			if (copy_to_user(buffer,
					 dbdesc->pbuf + mmstate->evts.read_byte_index,
					 bytes_this_time)) {
				ERR("evt_read: copy error");
				retval = -EFAULT;
				break;
			}

			mmstate->evts.read_byte_index += bytes_this_time;

			if (mmstate->evts.read_byte_index == dbdesc->nbytes) {

				/* we're done with this one so remove from rcv list */
				list_del_init(&dbdesc->list);

				/*  delete temporary buffer or requeue to free list */
				if (dbdesc->temporary) {
					/* delete temporaries */
					FLOW2("evt_read: delete temporary buf %p", dbdesc->pbuf);
					kfree(dbdesc->pbuf);
					kfree(dbdesc);
				} else {
					/* requeue to free list */
					list_add_tail(&dbdesc->list,
						      &mmstate->evts.free_list);
				}
				mmstate->evts.buf_count--;
				mmstate->evts.read_byte_index = 0;
			}

			retval = bytes_this_time;
		}
		else {
			ERR("evt_read: error, we should have an a list entry but don't");
			retval = -EFAULT; /* somehow it got lost */
			break;
		}
	}
	while(0);

	mutex_unlock(&mmstate->evts.buf_mutex);

	FLOW2("evt_read: exit: retval = %d", retval);
	return retval;
}


static uint32_t evt_poll(struct file *filp, poll_table *wait)
{
	uint32_t mask = 0;
	struct cicada_meas_state *mmstate;
	mmstate = (struct cicada_meas_state *)filp->private_data;
	if (!mmstate) return -ENODEV;

	FLOW2("evt_poll: entry");

	mutex_lock(&mmstate->evts.buf_mutex);
	poll_wait(filp, &mmstate->evts.wait_queue, wait);
	if (!list_empty(&mmstate->evts.rcv_list))
		mask |= POLLIN | POLLRDNORM;
	if (mmstate->evts.exiting) {
		mask |= POLLIN | POLLRDNORM;
	}
	mutex_unlock(&mmstate->evts.buf_mutex);

	FLOW2("evt_poll: mask = 0x%x", mask);
	return mask;
}


void evt_delete(struct kref *kref)
{
	CONFIG("cicada_meas::evt_delete: entry");
}


static ssize_t evt_show_buf_count_max(struct device *dev,
				      struct device_attribute *attr,
				      char *buf)
{
	struct usb_interface *intf = to_usb_interface(dev);
	struct cicada_meas_state *mmstate = usb_get_intfdata(intf);
	return sprintf(buf, "%d\n", mmstate->evts.buf_count_max);
}


static DEVICE_ATTR(evt_buf_count_max, S_IRUGO,
		   evt_show_buf_count_max, NULL);

/*
 * This matlab_comm control routes all interrupt endpoint packets to
 * the event driver as a special testing configuration.
 */
static ssize_t evt_set_matlab_comm(struct device *dev,
				   struct device_attribute *attr,
				   const char *buf, size_t count)
{
	struct usb_interface *intf = to_usb_interface(dev);
	struct cicada_meas_state *mmstate = usb_get_intfdata(intf);
	int value = 0;

	if (!mmstate) {
		ERRDBG("evt_set_matlab_comm: device disabled\n");
		return -ENODEV;
	}

	if (sscanf(buf, "%d", &value) != 1 || !(value == 0 || value == 1)) {
		ERR("evt_set_matlab_comm: error: argument must be 0 or 1\n");
		return -EINVAL;
	}

	if (value == 1) {
		/* we're running matlab comm so route the net packets
		   to the evt device */
		printk("evt_set_matlab_comm: matlab, net pkts go to evt device\n");
		int_register_callback(mmstate, ROUTE_NET,
				      &evt_callback, 0);
		mmstate->matlab_int_cb_override = 1;
	}
	else {
		/* disable  */
		if (mmstate->matlab_int_cb_override &&
		    mmstate->net_callback)
		{
			printk("evt_set_matlab_comm: restoring net callback\n");
			int_register_callback(mmstate,
					      mmstate->net_routing_code,
					      mmstate->net_callback,
					      mmstate->net_context);
		}
		else {
			printk("evt_set_matlab_comm: disabling routing of net\n");
			int_register_callback(mmstate, ROUTE_NET,
					      0, 0);
		}
		mmstate->matlab_int_cb_override = 0;
	}
	return count;
}


static DEVICE_ATTR(evt_matlab_comm, 0220, NULL, evt_set_matlab_comm);


static const struct file_operations evt_fops_table = {
	.owner	 = THIS_MODULE,
	.open	 = evt_open,
	.release = evt_release,
	.read	 = evt_read,
	.poll	 = evt_poll,
};


/*----------------------------------------------------------------------------*/
/*---- interrupt subdriver ---------------------------------------------------*/
/*----------------------------------------------------------------------------*/

static struct device_attribute dev_attr_int_urb_count_max;
static int int_alloc_urbs(struct cicada_meas_state *mmstate);
static int int_destroy_urbs(struct cicada_meas_state *mmstate);


/* called by probe to perform interrupt subdriver specific actions */
int int_probe(struct cicada_meas_state *mmstate)
{
	CONFIG("int_probe: entry: mmstate = %p", mmstate);

	kref_get(&mmstate->kref);

	init_usb_anchor(&mmstate->ints.urb_anchor);
	spin_lock_init(&mmstate->ints.lock);

	INIT_LIST_HEAD(&mmstate->ints.cb_list);

	/* create our own work queue and a work struct to use with it */
	mmstate->ints.workqueue = create_singlethread_workqueue("measmod_workq");
	if (!mmstate->ints.workqueue) {
		ERR("int_probe: create_singlethread_workqueue failed\n");
		goto int_probe_error;
	}

	/* initialize struct for use by work callback */
	mmstate->ints.work_state.mmstate = mmstate;
	INIT_WORK(&mmstate->ints.work_state.work_struct, &int_work_handler);

	mmstate->ints.urb_count = 0;
	mmstate->ints.urb_count_max = 0;

	mmstate->ints.nest_count = 0;

	if (device_create_file(&mmstate->interface->dev, &dev_attr_int_urb_count_max)) {
		ERR("int_probe: device_create_file for urb count failed\n");
		goto int_probe_error_delete_workqueue;
	}

	/* Allocate urbs */
	if (int_alloc_urbs(mmstate) < 0) {
		ERR("int_probe: error allocating urbs\n");
		goto int_probe_error_remove_file;
	}

	CONFIG("int_probe: exit");

	return 0;

int_probe_error_remove_file:
	device_remove_file(&mmstate->interface->dev, &dev_attr_int_urb_count_max);

int_probe_error_delete_workqueue:
	flush_workqueue(mmstate->ints.workqueue);
	destroy_workqueue(mmstate->ints.workqueue);

int_probe_error:
	kref_put(&mmstate->kref, &cicada_meas_delete);

	CONFIG("int_probe: exit: error = %d", -ENOMEM);
	return -ENOMEM;
}


static int int_alloc_urbs(struct cicada_meas_state *mmstate)
{
	char *buf = 0;
	struct urb *urb = 0;
	int status = 0;
	struct urb_carrier *work_urb_temp;
	struct urb_carrier *urb_carrier_temp;
	int i;
	int pipe = 0;
	FLOW("int_alloc_urbs: entry: mmstate = %p", mmstate);

	if (int_urbs > INT_URBS_MAX) {
		return -EINVAL;
	}

	/* Create the interrupt endpoint URB and initialize it */
	pipe = usb_rcvintpipe(mmstate->udev,
			      mmstate->int_config.int_in_endpoint_addr);

	/* initialize internal lists */
	INIT_LIST_HEAD(&mmstate->ints.free_work_urb_list);
	INIT_LIST_HEAD(&mmstate->ints.active_work_urb_list);
	INIT_LIST_HEAD(&mmstate->ints.urbs_allocated_list);

	/* allocate the urb(s) and their buffer(s) */
	for (i = 0; i < int_urbs; i++) {
		urb = 0;
		buf = 0;

		/* get memory for the URB (kref == 1) */
		urb = usb_alloc_urb(0, GFP_KERNEL);
		if (urb == NULL) {
			ERR("Could not allocate urb");
			status = -ENOMEM;
			goto int_alloc_urb_error_loose_bits;
		}

		/* allocate buffer for data.  Use kmalloc because
		 * automatic urb/buffer free code uses kfree */
		buf = kmalloc(mmstate->int_config.int_in_size, GFP_KERNEL);
		if (!buf) {
			ERR("Could not allocate data_buffer");
			status = -ENOMEM;
			goto int_alloc_urb_error_loose_bits;
		}
		FLOW2("int_alloc_urbs: urb = %p, buf = %p", urb, buf);

		usb_fill_int_urb(urb,
				 mmstate->udev, pipe,
				 buf,
				 mmstate->int_config.int_in_size,
				 int_urb_callback,
				 mmstate,
				 mmstate->int_config.interval);

		/* enable auto-free of attached buffer when urb is freed */
		urb->transfer_flags |= URB_FREE_BUFFER;

		/* anchor before submitting so USB core aids cleanup (kref == 2) */
		usb_anchor_urb(urb, &mmstate->ints.urb_anchor);

		/* queue the urb (inc urb kref) */
		status = usb_submit_urb(urb, GFP_KERNEL);
		if (status) {
			ERR("Can't submit urb - status %d", status);
			goto int_alloc_urb_error_loose_bits;
		}

		/* Create struct to hold urb reference so we can clean them up at the end */
		urb_carrier_temp = kzalloc(sizeof(*urb_carrier_temp), GFP_KERNEL);
		if (!urb_carrier_temp) {
			ERR("can't allocate urb_carrier struct");
			status = -ENOMEM;
			goto int_alloc_urb_error_unanchor;
		}
		INIT_LIST_HEAD(&urb_carrier_temp->list);
		urb_carrier_temp->urb = urb;
		list_add_tail(&urb_carrier_temp->list, &mmstate->ints.urbs_allocated_list);

		/* Create struct to hold urb while waiting to be
		 * processed by the work callback and add to free list */
		work_urb_temp = kzalloc(sizeof(*work_urb_temp), GFP_KERNEL);
		if (!work_urb_temp) {
			ERR("can't allocate work_urb struct");
			status = -ENOMEM;
			goto int_alloc_urb_error_unanchor;
		}
		INIT_LIST_HEAD(&work_urb_temp->list);
		list_add_tail(&work_urb_temp->list, &mmstate->ints.free_work_urb_list);
		FLOW2("int_alloc_urbs: initialize and queue work_urb struct %p",
			 work_urb_temp);
	}
	return 0;

int_alloc_urb_error_loose_bits:
	usb_free_urb(urb);
	kfree(buf);

int_alloc_urb_error_unanchor:
	int_destroy_urbs(mmstate);

	return status;
}


static int int_destroy_urbs(struct cicada_meas_state *mmstate)
{
	struct urb_carrier *work_urb_temp;
	struct urb_carrier *urb_carrier_temp;
	struct list_head *ptr;
	struct list_head *next;
	unsigned long lock_flags;
	FLOW("int_destroy_urbs: entry: mmstate = %p", mmstate);

	spin_lock_irqsave(&mmstate->ints.lock, lock_flags);
	{
		list_for_each_safe(ptr, next, &mmstate->ints.free_work_urb_list) {
			work_urb_temp = list_entry(ptr, struct urb_carrier, list);
			list_del_init(&(work_urb_temp->list));
			kfree(work_urb_temp);
			DBG("int_destroy_urbs: delete free work_urb = %p",
				  work_urb_temp);
		}

		list_for_each_safe(ptr, next, &mmstate->ints.active_work_urb_list) {
			work_urb_temp = list_entry(ptr, struct urb_carrier, list);
			list_del_init(&(work_urb_temp->list));
			usb_free_urb(work_urb_temp->urb);
			kfree(work_urb_temp);
			DBG("int_destroy_urbs: delete active work_urb = %p",
				  work_urb_temp);
		}

		list_for_each_safe(ptr, next, &mmstate->ints.urbs_allocated_list) {
			urb_carrier_temp = list_entry(ptr, struct urb_carrier, list);
			list_del_init(&(urb_carrier_temp->list));
			usb_free_urb(urb_carrier_temp->urb);
			kfree(urb_carrier_temp);
			DBG("int_destroy_urbs: delete allocated urb and urb_carrier = %p",
				  urb_carrier_temp);
		}
	}
	spin_unlock_irqrestore(&mmstate->ints.lock, lock_flags);

	FLOW("int_destroy_urbs: exit: mmstate = %p", mmstate);
	return 0;
}



/* called by disconnect to perform int subdriver specific actions */
/* Note: this is run within a lock of mmstate->state_struct_mutex */
int int_disconnect(struct cicada_meas_state *mmstate)
{
	unsigned long lock_flags = 0;
	struct list_head *ptr;
	struct list_head *next;
	struct int_cb_info *cb_info = 0;

	/* Note: all urb's have been killed already */
	CONFIG("int_disconnect: entry: mmstate = %p", mmstate);

	device_remove_file(&mmstate->interface->dev, &dev_attr_int_urb_count_max);

	flush_workqueue(mmstate->ints.workqueue);
	destroy_workqueue(mmstate->ints.workqueue);

	spin_lock_irqsave(&mmstate->ints.lock, lock_flags);
	{
		/* clear out the list */
		list_for_each_safe(ptr, next, &mmstate->ints.cb_list) {
			cb_info = list_entry(ptr, struct int_cb_info, list);
			list_del_init(&cb_info->list);
			kfree(cb_info);
			DBG("int_disconnect: delete cb info = %p", cb_info);
		}
	}
	spin_unlock_irqrestore(&mmstate->ints.lock, lock_flags);

	int_destroy_urbs(mmstate);

	kref_put(&mmstate->kref, &cicada_meas_delete);

	CONFIG("int_disconnect: exit");

	return 0;
}



static void int_urb_callback(struct urb *urb)
{
	struct urb_carrier *work_urb_curr = 0;
	struct cicada_meas_state *mmstate =
			(struct cicada_meas_state *)urb->context;
	unsigned long lock_flags = 0;

	if (!mmstate) {
		ERRDBG("int_urb_callback: mmstate is NULL");
		return;
	}

	LOW2("int_urb_callback: entry: urb = %p, urb->status = %d", urb, urb->status);

	switch (urb->status) {
	case 0: /* success - copy status from device to local safe storage */

		spin_lock_irqsave(&mmstate->ints.lock, lock_flags);

		/* should always be able to get one, but just in case */
		if (!list_empty(&mmstate->ints.free_work_urb_list)) {
			/* move from free list to active list */
			work_urb_curr = list_first_entry(
						&mmstate->ints.free_work_urb_list,
						struct urb_carrier, list);
			list_del_init(&(work_urb_curr->list));
			list_add_tail(&(work_urb_curr->list),
				      &(mmstate->ints.active_work_urb_list));
			work_urb_curr->urb = urb;

			/* increment ref count while on the active list */
			usb_get_urb(urb);

			if (++mmstate->ints.urb_count >
						mmstate->ints.urb_count_max)
				mmstate->ints.urb_count_max =
						mmstate->ints.urb_count;

			spin_unlock_irqrestore(&mmstate->ints.lock, lock_flags);

			LOW2("int_urb_callback: urb_count = %d, urb_count_max = %d",
			     mmstate->ints.urb_count, mmstate->ints.urb_count_max);

			/* Don't complain if already queued.  The work callback
			   will process all entries in active_list */
			queue_work(mmstate->ints.workqueue,
				   &mmstate->ints.work_state.work_struct);

			LOW2("int_urb_callback: queued work");
		} else {
			/* TBD - should it do more than print this on error? */
			spin_unlock_irqrestore(&mmstate->ints.lock, lock_flags);
			ERR("int_urb_callback: couldn't get work urb struct\n");
		}
		break;

	case -ECONNRESET:	/* unlink */
	case -ENOENT:
	case -ESHUTDOWN:
	case -EPROTO: /* TBD: this seems to routinely happen on disconnect,
			      watch out for problems */
		return;
		/* -EPIPE:  should clear the halt */
	default:		/* error */
		ERR("Unknown int URB Status = %#x.\n", urb->status);
	}
	LOW2("int_urb_callback: exit");
}


/* handle urbs from interrupt endpoint in workqueue context */
static void int_work_handler(struct work_struct *work)
{
	struct work_state *work_state = container_of(work,
					struct work_state, work_struct);
	struct cicada_meas_state *mmstate = work_state->mmstate;
	struct list_head *ptr;
	struct urb *urb;
	int routing_code = 0;
	struct int_cb_info *cb_info = 0;
	int_callback cb_func = 0;
	void *cb_context = 0;
	struct USB_MSG_HEADER *phdr;
	unsigned long lock_flags;
	int stat = 0;
	int newUrb = 0;
	int urbError = 0;
	struct urb_carrier *work_urb_curr;

	if (!mmstate) {
		ERRDBG("int_work_handler: mmstate is NULL");
		return;
	}

	LOW2("int_work_handler: entry: work_state = %p", work_state);

	spin_lock_irqsave(&mmstate->ints.lock, lock_flags);
	{
		mmstate->ints.nest_count++;
		newUrb = !list_empty(&mmstate->ints.active_work_urb_list);
	}
	spin_unlock_irqrestore(&mmstate->ints.lock, lock_flags);

	if (mmstate->ints.nest_count > 1) {
		/* want to be able to see if this happens */
		FLOW("*********** int_work_handler: nest_count = %d",
		       mmstate->ints.nest_count);
	}

	/* keep getting them until there are no more */
	while (newUrb) {
		urbError = 0;

		spin_lock_irqsave(&mmstate->ints.lock, lock_flags);
		{
			/* get urb from active list */
			work_urb_curr = list_first_entry(
						&mmstate->ints.active_work_urb_list,
						struct urb_carrier, list);
			list_del_init(&work_urb_curr->list);
		}
		spin_unlock_irqrestore(&mmstate->ints.lock, lock_flags);

		urb = work_urb_curr->urb;
		work_urb_curr->urb = 0;

		do {
			if (urb->actual_length == 0) {
				/* ZLP is ok, but we want to ignore this urb */
				LOW("int_work_handler: received ZLP");
				urbError = 1;
				break;
			}

			if (urb->actual_length < sizeof(struct USB_MSG_HEADER)) {
				/* this shouldn't fail, just in case */
				ERR("int_work_handler: usb message too small = %d\n",
							urb->actual_length);
				urbError = 1;
				break;
			}

			/* check header and get routing code from data packet */
			phdr = (struct USB_MSG_HEADER *) urb->transfer_buffer;

			if (phdr->MagicNum != CICADA_CMD_MAGIC) {
				ERRDBG("int_work_handler: magic number = %d, expected = %d\n",
				    phdr->MagicNum, CICADA_CMD_MAGIC);
				urbError = 1;
				break;
			}

			if ((uint8_t)phdr->SeqNum != (uint8_t)~phdr->SeqInv) {
				ERRDBG("int_work_handler: SeqNum = 0x%x, seqInv = 0x%x\n",
				    phdr->SeqNum, phdr->SeqInv);
				urbError = 1;
				break;
			}

			routing_code = phdr->RouteID;
			LOW("int_work_handler: intId = 0x%03x, bytes = %d", phdr->MsgID, phdr->TransferSize);
			if (routing_code == ROUTE_NET) {
				if (phdr->TransferSize >= (sizeof(struct USB_MSG_HEADER)+sizeof(uint16_t))) {
					uint16_t commstat = *(((uint16_t*)&phdr->MsgID)+1);
					LOW("int_work_handler: rc = %d, commstat = 0x%x",
						routing_code, commstat);

					if (commstat & (COMM_STAT_MESGERR | COMM_STAT_TYPE_ERR)) {
						FLOW("int_work_handler: ************************************** commstat = 0x%x", commstat);
//						  pr_err("int_work_handler: ************************************** commstat = 0x%x", commstat);
					}
				}
			}
		}
		while(0);

		if (!urbError) {
			spin_lock_irqsave(&mmstate->ints.lock, lock_flags);
			{
				/* look at each entry in the callback list */
				list_for_each(ptr, &mmstate->ints.cb_list) {

					cb_info = list_entry(ptr,
							     struct int_cb_info, list);

					/* save the function if you match the
					   routing code (and it's non-null) */
					if (cb_info->routing_code == routing_code
					    && cb_info->func) {
						cb_func = cb_info->func;
						cb_context = cb_info->context;
						break;
					}
				}
			}
			spin_unlock_irqrestore(&mmstate->ints.lock, lock_flags);

			/* call callback function in workqueue context */
			if (cb_func) {
				cb_func(urb, cb_context);
			}
		}

		/* requeue urb if interface is still there */
		if (!mmstate->ints.shutting_down) {
			/* resubmit urb to usbcore - shouldn't
			   fail, but check just in case */
			usb_anchor_urb(urb, &mmstate->ints.urb_anchor);
			stat = usb_submit_urb(urb, GFP_KERNEL);
			if (stat)
				LOW("int_work_handler: submit err %d", stat);
		} else {
			LOW("int_work_handler: don't resched urb %p", urb);
		}

		/* Go ahead and requeue struct to free list.
		   It will be cleaned up */
		spin_lock_irqsave(&mmstate->ints.lock,
				  lock_flags);
		{
			/* put it back on the free list
			   - it's safe to do here */
			list_add_tail(
			    &(work_urb_curr->list),
			    &(mmstate->ints.free_work_urb_list));

			/* callback processing done for this urb */
			usb_free_urb(urb);

			mmstate->ints.urb_count--;

			/* Get next urb if there is one */
			newUrb = !list_empty(&mmstate->ints.active_work_urb_list);
		}
		spin_unlock_irqrestore(
				&mmstate->ints.lock,
				lock_flags);
	}

	spin_lock_irqsave(&mmstate->ints.lock, lock_flags);
	{
		mmstate->ints.nest_count--;
	}
	spin_unlock_irqrestore(&mmstate->ints.lock, lock_flags);

	LOW2("int_work_handler: exit");
}

static int int_draw_down(struct cicada_meas_state *mmstate)
{
	int time;
	if (!mmstate) return -ENODEV;

	FLOW2("int_draw_down: entry");

	time = usb_wait_anchor_empty_timeout(&mmstate->ints.urb_anchor, 100);
	FLOW2("int_draw_down: time = %d", time);
	if (!time)
		usb_kill_anchored_urbs(&mmstate->ints.urb_anchor);
	FLOW2("int_draw_down: exit = void");

	return 0;
}


static void int_delete(struct kref *kref)
{
	CONFIG("int_delete: entry");
}


/* Network and Event subdrivers call this function to register their
 * callback functions. */
static void int_register_callback(struct cicada_meas_state *mmstate,
			  int routing_code,
			  int_callback func,
			  void *context)
{
	unsigned long lock_flags;
	struct list_head *ptr;
	struct int_cb_info *cb_info;
	int found = 0;

	if (!mmstate) {
		ERRDBG("int_register_callback: mmstate is NULL");
		return;
	}

	FLOW("int_register_callback: entry: rc = %d", routing_code);
	FLOW("int_register_callback: entry: func = %p, cx = %p", func, context);

	spin_lock_irqsave(&mmstate->ints.lock, lock_flags);
	{
		/* look at each entry in the callback list for the
		 * matching routing code */
		list_for_each(ptr, &mmstate->ints.cb_list) {

			cb_info = list_entry(ptr, struct int_cb_info,
						  list);

			/* if it is already there, set the func and
			 * context (may set func to null) */
			if (cb_info->routing_code == routing_code) {
				cb_info->func = func;
				cb_info->context = context;
				found = 1;
				break;
			}
		}

		/* It's not already there so create a new entry. */
		if (found == 0) {
			if (func) {
				cb_info = kzalloc(sizeof(*cb_info),
							GFP_ATOMIC);
				if (cb_info) {
					cb_info->routing_code = routing_code;
					cb_info->func = func;
					cb_info->context = context;
					list_add_tail(&cb_info->list,
						      &mmstate->ints.cb_list);
				}
				else {
					ERR("int_register_callback: couldn't alloc cb_info");
				}
			}
		}
	}
	spin_unlock_irqrestore(&mmstate->ints.lock, lock_flags);

	FLOW("int_register_callback: exit");

	return;
}


static ssize_t int_show_urb_count_max(struct device *dev,
				      struct device_attribute *attr,
				      char *buf)
{
	struct usb_interface *intf = to_usb_interface(dev);
	struct cicada_meas_state *mmstate = usb_get_intfdata(intf);
	return sprintf(buf, "%d\n", mmstate->ints.urb_count_max);
}


static DEVICE_ATTR(int_urb_count_max, S_IRUGO,
		   int_show_urb_count_max, NULL);




/*----------------------------------------------------------------------------*/
/*---- cicada_meas Driver Version query --------------------------------------*/
/*----------------------------------------------------------------------------*/
static struct class_attribute class_attr_version;

static int version_init(struct class *class)
{
	if (class_create_file(class, &class_attr_version)) {
		ERR("version_init: device_create_file for version failed\n");
		return -ENODEV;
	}
	return 0;
}

static int version_exit(struct class *class)
{
	class_remove_file(class, &class_attr_version);
	return 0;
}


static ssize_t version_show(struct class *class,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 34)
			    struct class_attribute *attr,
#endif
			    char *buf)
{
	return sprintf(buf, "%d\n", CICADA_MEAS_VERSION);
}

static CLASS_ATTR(version, 0444,
		  version_show, NULL);



/*----------------------------------------------------------------------------*/
/*---- Module Power Control Switch on Mainframe ------------------------------*/
/*----------------------------------------------------------------------------*/
/*
 * A /sys node is created for platforms that supports control of the
 * module power supply enable by the mainframe hardware.
 */

#if defined(CONFIG_MACH_MX35_CICADA) || defined(CONFIG_MACH_MX7D_CICADA)
extern void cicada_set_module_vbat(int enable);
extern int  cicada_get_module_vbat(void);

static struct class_attribute class_attr_modpower;

/* TBD: add count so it won't shut off if more than one module and the
 * second one is removed.  Current product defn doesn't support more
 * than one module so add this long before release time */
static int modpower_init(struct class *class)
{
	cicada_set_module_vbat(0); /* make sure module power is off */

	if (class_create_file(class, &class_attr_modpower)) {
		ERR("modpower_init: device_create_file for modpower failed\n");
		return -ENODEV;
	}
	return 0;
}

static int modpower_exit(struct class *class)
{
	class_remove_file(class, &class_attr_modpower);
	cicada_set_module_vbat(0); /* make sure module power is off */
	return 0;
}

static int modpower_probe(void)
{
	FLOW("modpower_probe");
	cicada_set_module_vbat(1); /* turn module power ON */
	return 0;
}

/* Note: this is run within a lock of mmstate->state_struct_mutex */
static int modpower_disconnect(void)
{
	FLOW("modpower_disconnect");
	cicada_set_module_vbat(0); /* turn module power OFF */
	return 0;
}

static ssize_t modpower_show_enable(struct class *class,
				    struct class_attribute *attr,
				    char *buf)
{
    /* Status bit definitions:
     *	    CICADA_VBAT_ENABLE = 0x01 true when enabled
     *	    CICADA_VBAT_OC     = 0x02 true when overcurrent
    */
	return sprintf(buf, "%d\n", cicada_get_module_vbat());
}


static ssize_t modpower_set_enable(struct class *class,
				   struct class_attribute *attr,
				   const char *buf, size_t count)
{
	int enable = 0;
	if (sscanf(buf, "%d", &enable) != 1 || !(enable == 0 || enable == 1)) {
		ERR("modpower_set_enable: error: argument must be 0 or 1\n");
		return -EINVAL;
	}
	cicada_set_module_vbat(enable);
	return count;
}


static CLASS_ATTR(modpower, 0664,
		  modpower_show_enable, modpower_set_enable);

#else
/* x86 or iMX31 */
static int modpower_init(struct class *class)	{ return 0; }
static int modpower_exit(struct class *class)	{ return 0; }
static int modpower_probe(void)			{ return 0; }
static int modpower_disconnect(void)		{ return 0; }
#endif




/*----------------------------------------------------------------------------*/
/*---- Module "Special Hw Signal" Control ------------------------------------*/
/*----------------------------------------------------------------------------*/
/*
 * Two /sys nodes are created for platforms that supports control of
 * the "Special Hardware Signal" control signal.
 *
 *    - special_enable (write-only)
 *    - special	       (read/write)
 */

#ifdef CONFIG_MACH_MX35_CICADA

extern void cicada_set_special_enable(int enable);
extern int  cicada_get_special_enable(void);
static int cicada_special_enable = 0;
extern void cicada_set_special(int output);
extern int cicada_get_special(void);

static struct class_attribute class_attr_special_enable;
static struct class_attribute class_attr_special;

static int special_init(struct class *class)
{
	cicada_set_special_enable(0); /* make sure special bit is disabled */
	cicada_special_enable = 0;

	if (class_create_file(class, &class_attr_special_enable)) {
		ERR("special_init: device_create_file for special_enable failed\n");
		return -ENODEV;
	}
	if (class_create_file(class, &class_attr_special)) {
		ERR("special_init: device_create_file for special failed\n");
		return -ENODEV;
	}
	return 0;
}

static int special_exit(struct class *class)
{
	class_remove_file(class, &class_attr_special_enable);
	class_remove_file(class, &class_attr_special);
	cicada_set_special_enable(0); /* make sure special bit is disabled */
	return 0;
}


static ssize_t special_show_enable(struct class *class,
				   struct class_attribute *attr,
				   char *buf)
{
	return sprintf(buf, "%d\n", cicada_special_enable);
}


static ssize_t special_set_enable(struct class *class,
				  struct class_attribute *attr,
				  const char *buf, size_t count)
{
	int enable = 0;
	if (sscanf(buf, "%d", &enable) != 1 || !(enable == 0 || enable == 1)) {
		ERR("special_set_enable: error: argument must be 0 or 1\n");
		return -EINVAL;
	}
	cicada_set_special_enable(enable);
	cicada_special_enable = enable;
	return count;
}


static ssize_t special_show(struct class *class,
			    struct class_attribute *attr,
			    char *buf)
{
	return sprintf(buf, "%d\n", cicada_get_special());
}


static ssize_t special_set(struct class *class,
			   struct class_attribute *attr,
			   const char *buf, size_t count)
{
	int state = 0;
	if (sscanf(buf, "%d", &state) != 1 || !(state == 0 || state == 1)) {
		ERR("special_set: error: argument must be 0 or 1\n");
		return -EINVAL;
	}
	cicada_set_special(state);
	return count;
}


static CLASS_ATTR(special_enable, 0664,
		  special_show_enable, special_set_enable);

static CLASS_ATTR(special, 0664,
		  special_show, special_set);

#else  /* CONFIG_MACH_MX35_CICADA */
/* x86 or iMX31 */
static int special_init(struct class *class)	{ return 0; }
static int special_exit(struct class *class)	{ return 0; }
#endif /* CONFIG_MACH_MX35_CICADA */


/*----------------------------------------------------------------------------*/
/*---- Hub Power Port Control  -----------------------------------------------*/
/*----------------------------------------------------------------------------*/
/*
 * This provides access to the per-port hub port power control once
 * the cicada_meas driver has been loaded.
 *
 * Enter the hub port name:
 *
 *    echo "1-1.2" > hub_port_name
 *
 * The variable "hubport_name" is set to "1-1".
 * The variable "hubport_num"  is set to 2.
 *
 * To turn the power on for that port:
 *
 *    echo 1 > hub_power
 *
 * To turn the power off for that port:
 *
 *    echo 0 > hub_power
 *
 * The hub_power_set() function generates a usb control message to
 * turn the hub port power on/off and sends it to the specified hub
 * port.
 */

static struct class_attribute class_attr_hub_port_name;
static struct class_attribute class_attr_hub_power;
#define HUB_NAME_MAX  32
static char hub_name[HUB_NAME_MAX];
static char hub_port_name[HUB_NAME_MAX];
static int  hub_port_num = -1;
#define USB_PORT_FEAT_POWER  8

static int hub_port_power_init(struct class* class, struct usb_driver* usb_driver)
{
	hub_port_name[0] = '\0';

	if (class_create_file(class, &class_attr_hub_port_name)) {
		ERR("hub_port_power_init: device_create_file for hub_port_name failed\n");
		return -ENODEV;
	}
	if (class_create_file(class, &class_attr_hub_power)) {
		ERR("hub_port_power_init: device_create_file for hub_port_name failed\n");
		class_remove_file(class, &class_attr_hub_port_name);
		return -ENODEV;
	}
	return 0;
}

static int hub_port_power_exit(struct class *class, struct usb_driver* usb_driver)
{
	class_remove_file(class, &class_attr_hub_port_name);
	class_remove_file(class, &class_attr_hub_power);
	return 0;
}

static ssize_t hub_port_name_show(struct class *class,
				  struct class_attribute *attr,
				  char *buf)
{
	return sprintf(buf, "%s\n", hub_port_name);
}


#define	HUB_NAME_STATE_LOOK_FOR_DOT    0
#define	HUB_NAME_STATE_GET_PORT_NUM    1

static ssize_t hub_port_name_set(struct class *class,
				 struct class_attribute *attr,
				 const char *buf, size_t count)
{
	char *pin = 0;
	char *pout = 0;
	int state = HUB_NAME_STATE_LOOK_FOR_DOT;
	char hub_port_num_str[HUB_NAME_MAX];
	char *pport_num = 0;

	strncpy(hub_port_name, buf, HUB_NAME_MAX);
	hub_port_name[HUB_NAME_MAX-1] = '\0';

	/* the first part of the port name is the hub, the part after
	 * the dot '.' is the port id */
	pin  = hub_port_name;
	pout = hub_name;
	pport_num = hub_port_num_str;

	while (*pin != 0 && pin < (hub_port_name+HUB_NAME_MAX)) {
		switch (state) {
		case HUB_NAME_STATE_LOOK_FOR_DOT:
			if (*pin == '.') {
				*pout = '\0';
				pin++;
				state = HUB_NAME_STATE_GET_PORT_NUM;
			}
			else {
				*pout++ = *pin++;
			}
			break;

		case HUB_NAME_STATE_GET_PORT_NUM:
			/* copy chars until you run out */
			*pport_num++ = *pin++;
		}
	}
	*pport_num = '\0';

	if (sscanf(hub_port_num_str, "%d", &hub_port_num) != 1) {
		printk("hub_port_name_set: error: invalid arg %s\n", hub_port_name);
		return -EINVAL;
	}

	if (hub_port_num < 1 || hub_port_num > 8) {
		printk("error: invalid hub_port_num = %d\n", hub_port_num);
		return -EINVAL;
	}

	return count;
}


static ssize_t hub_power_set(struct class *class,
			     struct class_attribute *attr,
			     const char *buf, size_t count)
{
	struct device *device = 0;
	struct usb_device *usb_hub_dev = 0;
	int ret = 0;
	int enable = 0;
	int request;
	int feature;

	if (sscanf(buf, "%d", &enable) != 1 || !(enable == 0 || enable == 1)) {
		ERR("hub_power_set: error: argument must be 0 or 1\n");
		return -EINVAL;
	}

	device = bus_find_device_by_name(cicada_meas_driver.drvwrap.driver.bus,
					 NULL,
					 hub_name);
	if (!device) {
		printk("hub_power_set: error: couldn't find hub_name = '%s'\n", hub_name);
		return -EINVAL;
	}

	usb_hub_dev = to_usb_device(device);

	/* send usb control message to enable or disable hub power */
	feature = USB_PORT_FEAT_POWER;
	if (enable)
		request = USB_REQ_SET_FEATURE;
	else
		request = USB_REQ_CLEAR_FEATURE;

	ret = usb_control_msg(usb_hub_dev,
		      usb_sndctrlpipe(usb_hub_dev, 0),
		      request,			/* bRequest */
		      (USB_DIR_OUT | USB_TYPE_CLASS | USB_RECIP_OTHER), /* bmRequestType */
		      feature,			/* wValue */
		      hub_port_num,		/* wIndex */
		      0,			/* *data */
		      0,			/* size */
		      5000);			/* timeout */

	if (ret != 0) {
		printk(KERN_NOTICE "hub_power_set: error = %d.\n", ret);
	}
	return count;
}


static CLASS_ATTR(hub_port_name, 0664,
		  hub_port_name_show, hub_port_name_set);

static CLASS_ATTR(hub_power, 0220,
		  NULL, hub_power_set);



/*----------------------------------------------------------------------------*/
/*---- cicada_meas base driver functions  ------------------------------------*/
/*----------------------------------------------------------------------------*/

static struct device_attribute dev_attr_actual_hub_port;

static int cicada_meas_probe(struct usb_interface *interface,
			     const struct usb_device_id *id)
{
	struct cicada_meas_state *mmstate;
	struct usb_host_interface *iface_desc;
	struct usb_endpoint_descriptor *endpoint;
	size_t buffer_size;
	int i;
	int retval = -ENOMEM;
	int major, minor;
	int temp_minor;
	int rcom_enabled = 0;
	struct usb_device* temp_usb_device;

	CONFIG("cicada_meas_probe: entry: interface = %p, device_id = %p",
	       interface, id);
	printk("cicada_meas_probe: version = %d, bulk urb buf = %d, int ep = %d\n",
	       CICADA_MEAS_VERSION, CICADA_MEAS_URB_BUFBYTES, INT_URB_BUFBYTES);
	printk("cicada_meas_probe: int_urbs = %d, enable_config_logging = %d\n",
	       int_urbs, enable_config_logging);

	/* allocate memory for our device state structure and initialize it */
	mmstate = kzalloc(sizeof(*mmstate), GFP_KERNEL);
	if (!mmstate) {
		ERR("cicada_meas_probe: Out of memory");
		goto probe_error;
	}

	// kref_init initializes kref to 1.
	kref_init(&mmstate->kref);
	mutex_init(&mmstate->state_struct_mutex);

	INIT_LIST_HEAD(&mmstate->active_list);

	mmstate->udev = usb_get_dev(interface_to_usbdev(interface));
	mmstate->interface = interface;

	/* set up the endpoint information */
	/* use only the first bulk-in and bulk-out endpoints */
	iface_desc = interface->cur_altsetting;
	for (i = 0; i < iface_desc->desc.bNumEndpoints; ++i) {

		endpoint = &iface_desc->endpoint[i].desc;

		/* bulk in endpoint? */
		if (!mmstate->cmd_config.bulk_in_endpoint_addr &&
		    usb_endpoint_is_bulk_in(endpoint)) {
			buffer_size = le16_to_cpu(endpoint->wMaxPacketSize);
			mmstate->cmd_config.bulk_in_size = buffer_size;
			mmstate->cmd_config.bulk_in_endpoint_addr =
						endpoint->bEndpointAddress;
		}

		/* bulk out endpoint? */
		if (!mmstate->cmd_config.bulk_out_endpoint_addr &&
		    usb_endpoint_is_bulk_out(endpoint)) {
			/* we found a bulk out endpoint */
			mmstate->cmd_config.bulk_out_endpoint_addr =
						endpoint->bEndpointAddress;
		}

		/* interrupt endpoint? */
		if (!mmstate->int_config.int_in_endpoint_addr &&
		    usb_endpoint_is_int_in(endpoint)) {
			/* we found the interrupt IN endpoint */
			mmstate->int_config.int_in_size =
					le16_to_cpu(endpoint->wMaxPacketSize);

			if (mmstate->int_config.int_in_size >
						INT_URB_BUFBYTES) {
				ERR("cicada_meas_probe: int_size (%d) > (%d)",
					mmstate->int_config.int_in_size,
					(int)INT_URB_BUFBYTES);
				goto probe_error;
			}
			mmstate->int_config.int_in_endpoint_addr =
						endpoint->bEndpointAddress;
			mmstate->int_config.interval = le16_to_cpu(endpoint->bInterval);
			break;
		}
	}

	/* save our data pointer in this interface device */
	usb_set_intfdata(interface, mmstate);

	if (!(mmstate->cmd_config.bulk_in_endpoint_addr &&
	      mmstate->cmd_config.bulk_out_endpoint_addr)) {
		ERR("cicada_meas_probe: Could not find both bulk-in and bulk-out endpoints\n");
		goto probe_error;
	}

	if (driver_major) {
		/* If another module has already been plugged in, use
		   the same major.  Keep trying to acquire major/minor
		   combinations until you get one or you've tried too
		   many times. */
		temp_minor = CICADA_MEAS_FIRST_MINOR;
		for (i = 0; i < MAX_MODULE_INSTANCES; i++) {
			mmstate->dev_info = MKDEV(driver_major, temp_minor);
			retval = register_chrdev_region(mmstate->dev_info,
						     CICADA_MEAS_NUM_MINORS,
						     CICADA_MEAS_PROC_DEV_NAME);

			DBG("cicada_meas_probe, major = %d, minor = %d",
			       MAJOR(mmstate->dev_info), MINOR(mmstate->dev_info));
			DBG("cicada_meas_probe, retval = %d", retval);

			if (retval == 0)
				break; /* we've got them */

			temp_minor += CICADA_MEAS_NUM_MINORS;
		}
		if (retval < 0) {
			ERR("cicada_meas_probe, register_chrdev_region err\n");
			goto probe_error;
		}
	} else {
		/* This is the first/only module, get a dynamic major and
		   remember it.	 The else clause comes into play if you plug
		   in a second module.	It still has to work if you install
		   and remove the modules multiple times too. */
		retval = alloc_chrdev_region(&mmstate->dev_info,
					  CICADA_MEAS_FIRST_MINOR,
					  CICADA_MEAS_NUM_MINORS,
					  CICADA_MEAS_PROC_DEV_NAME);
		if (retval < 0) {
			ERR("cicada_meas_probe, alloc_chrdev_region failed\n");
			goto probe_error;
		}
		driver_major = MAJOR(mmstate->dev_info);
		minor_base = MINOR(mmstate->dev_info);
	}

	/* probe the hardware control for module power */
	retval = modpower_probe();
	if (retval < 0) {
		ERR("cicada_meas_probe, modpower_probe failed\n");
		/* goto net cleanup because modpower didn't do anything */
		goto probe_error_free_devnums;
	}

	/* cmd and evt use same major and use minor and minor+1.  Suffix
	   is appended to the devnode names so you can tell that cmd0/evt0
	   and cmd1/evt1 go together if you have more than one module installed.
	*/
	major = MAJOR(mmstate->dev_info);
	minor = MINOR(mmstate->dev_info);
	mmstate->suffix = (minor - minor_base)/2;
	DBG("cicada_meas_probe, major = %d, minor = %d", major, minor);
	DBG("cicada_meas_probe, suffix = %d, mmstate = %p", mmstate->suffix, mmstate);

	mmstate->net_routing_code = 0;
	mmstate->net_callback = 0;
	mmstate->net_context = 0;
	mmstate->matlab_int_cb_override = 0;

	/* initialize bulk_transaction subdriver */
	retval = bulk_probe(mmstate);
	if (retval < 0) {
		ERR("cicada_meas_probe, bulk_probe failed\n");
		goto probe_error_modpower_cleanup;
	}

	/* initialize cmd subsystem */
	retval = cmd_probe(mmstate, major, minor, mmstate->suffix);
	if (retval < 0) {
		ERR("cicada_meas_probe, cmd_probe failed\n");
		goto probe_error_modpower_cleanup;
	}

	/* initialize int/evt subsystems if requested */
	if (mmstate->int_config.int_in_endpoint_addr) {

		/* initialize int subsystem before evt becuase evt
		 * will need to register its callback */
		retval = int_probe(mmstate);
		if (retval < 0) {
			ERR("cicada_meas_probe, int_probe failed\n");
			goto probe_error_dev_cleanup;
		}
		mmstate->int_enabled = 1;

		/* initialize evt subsystem */
		retval = evt_probe(mmstate, major, minor+1, mmstate->suffix);
		if (retval < 0) {
			ERR("cicada_meas_probe, evt_probe failed\n");
			goto probe_error_dev_cleanup;
		}
		mmstate->evt_enabled = 1;

		FLOW("cicada_meas_probe: call cicada_rcom_probe()");
		retval = cicada_rcom_probe(mmstate->udev, interface,
				mmstate->cmd_config.bulk_in_endpoint_addr,
				mmstate->cmd_config.bulk_out_endpoint_addr,
				&mmstate->net_routing_code,
				&mmstate->net_callback,
				&mmstate->net_context,
				&bulk_submit_snd_urb,
				&bulk_submit_rcv_brb,
				(void*)mmstate,
				&cicada_meas_state_get,
				&cicada_meas_state_put,
				&cicada_trace_enabled_chans,
				&cicada_trace_record,
				enable_config_logging);

		if (retval < 0) {
			mmstate->rcom_enabled = 0;

			/* -EADDRNOTAVAIL is returned from rcom if serial number is empty */
			if (retval != -EADDRNOTAVAIL) {
			    ERR("cicada_meas_probe, cicada_rcom_probe failed\n");
			    goto probe_error_dev_cleanup;
			}
		}
		else {
			int_register_callback(mmstate, mmstate->net_routing_code,
					      mmstate->net_callback,
					      mmstate->net_context);
			mmstate->rcom_enabled = 1;
		}
	}

	/* extract USB bus id */
	temp_usb_device = interface_to_usbdev(interface);
	strncpy(mmstate->usb_bus_id, dev_name(&temp_usb_device->dev), USB_BUS_ID_MAX);

	if (device_create_file(&mmstate->interface->dev, &dev_attr_actual_hub_port)) {
		ERR("cicada_meas_probe: device_create_file for actual hub port failed\n");
		goto probe_error_bus_id_cleanup;
	}

	usb_disable_autosuspend(temp_usb_device);

	/* register this mmstate as being active.
	   Only deleted by cicada_meas_delete() when kref is decremented to 0. */
	add_mmstate_active(mmstate);

	/* let the user know what node this device is now attached to */
	CONFIG("cicada_meas_probe: exit");
	return 0;

	/* pretend to use this symbol so the compiler won't generate a
	 * warning.  The symbol currently doesn't get called, but it
	 * was put here in case someone adds something after the
	 * modpower_probe call later.  Remove this goto if you start
	 * going there. */
	goto probe_error_bus_id_cleanup;

probe_error_bus_id_cleanup:
	device_remove_file(&mmstate->interface->dev, &dev_attr_actual_hub_port);

probe_error_dev_cleanup:
	if (rcom_enabled)
		cicada_rcom_disconnect(mmstate->net_context);

	if (mmstate->evt_enabled)
		evt_disconnect(mmstate);

	if (mmstate->int_enabled)
		int_disconnect(mmstate);

	mutex_unlock(&mmstate->state_struct_mutex);

probe_error_modpower_cleanup:
	modpower_disconnect();

probe_error_free_devnums:
	unregister_chrdev_region(mmstate->dev_info, CICADA_MEAS_NUM_MINORS);

probe_error:
	/* free allocated memory for the "dev" struct */
	if (mmstate)
		kref_put(&mmstate->kref, cicada_meas_delete);

	/* clear the pointer to the defunct "dev" struct if it was set */
	usb_set_intfdata(interface, NULL);

	CONFIG("cicada_meas_probe: exit = %d", retval);

	return retval;
}


static void cicada_meas_disconnect(struct usb_interface *interface)
{
	struct cicada_meas_state *mmstate;
	int major, minor;

	CONFIG("cicada_meas_disconnect: entry: interface = %p", interface);
	mmstate = usb_get_intfdata(interface);
	usb_set_intfdata(interface, NULL);

	device_remove_file(&mmstate->interface->dev, &dev_attr_actual_hub_port);

	/* mutex protects against subdriver activity that would
	 * conflict with this cleanup */
	mutex_lock(&mmstate->state_struct_mutex);
	{
		if (mmstate->int_enabled) {
			/* don't start anything new */
			mmstate->ints.shutting_down = 1;

			/* make sure all interrupt urbs are done */
			int_draw_down(mmstate);

			if (mmstate->rcom_enabled) {
				FLOW("cicada_meas_disconnect: disconnect rcom");
				cicada_rcom_disconnect(mmstate->net_context);
			}

			if (mmstate->evt_enabled)
				evt_disconnect(mmstate);

			int_disconnect(mmstate);
		}
		cmd_disconnect(mmstate);
		bulk_disconnect(mmstate);

		modpower_disconnect();

		/* zeroing interface pointer tells subdrivers that
		 * it's over */
		mmstate->interface = NULL;
	}
	mutex_unlock(&mmstate->state_struct_mutex);

	major = MAJOR(mmstate->dev_info);
	minor = MINOR(mmstate->dev_info);
	unregister_chrdev_region(mmstate->dev_info, CICADA_MEAS_NUM_MINORS);

	CONFIG("cicada_meas_disconnect: call usb_put_dev()");
	usb_put_dev(mmstate->udev);

	/* decrement our usage count */
	kref_put(&mmstate->kref, cicada_meas_delete);

	CONFIG("cicada_meas_disconnect: exit");
}

void cicada_meas_delete(struct kref *kref)
{
	struct cicada_meas_state *mmstate = to_cicada_meas_state(kref);

	CONFIG("cicada_meas_delete: entry");

	cmd_delete(kref);
	bulk_delete(mmstate);

	if (mmstate->evt_enabled)
		evt_delete(kref);

	if (mmstate->int_enabled)
		int_delete(kref);

	remove_mmstate_active(mmstate);

	kfree(mmstate);

	CONFIG("cicada_meas_delete: exit");
}



struct usb_driver cicada_meas_driver = {
	.name =		"cicada_meas",
	.probe =	cicada_meas_probe,
	.disconnect =	cicada_meas_disconnect,
	.id_table =	cicada_meas_table,
	.supports_autosuspend = 0,
};


static ssize_t actual_hub_port_show(struct device *dev,
				    struct device_attribute *attr,
				    char *buf)
{
	struct usb_interface *intf = to_usb_interface(dev);
	struct cicada_meas_state *mmstate = usb_get_intfdata(intf);

	return sprintf(buf, "%s\n", mmstate->usb_bus_id);
}

static DEVICE_ATTR(actual_hub_port, S_IRUGO,
		   actual_hub_port_show, NULL);


extern int cicada_trace_start(void);

static int __init cicada_meas_init(void)
{
	int retval;

	init_mmstate_active();

	cicada_meas_class = class_create(THIS_MODULE, CICADA_MEAS_SYS_CLASS);
	if (!cicada_meas_class) {
		ERR("cicada_meas class creation failed");
		return -ENODEV;
		goto init_error;
	}

	retval = version_init(cicada_meas_class);
	if (retval < 0) {
		ERR("version_init failed");
		goto init_error_destroy_class;
	}

	retval = modpower_init(cicada_meas_class);
	if (retval < 0) {
		ERR("modpower_init failed");
		goto init_error_cleanup_version;
	}

	retval = special_init(cicada_meas_class);
	if (retval < 0) {
		ERR("special_init failed");
		goto init_error_cleanup_modpower;
	}

	/* register this driver with the USB subsystem */
	retval = usb_register(&cicada_meas_driver);
	if (retval < 0) {
		ERR("usb_register failed. Error number %d", retval);
		goto init_error_cleanup_special;
	}

	retval = hub_port_power_init(cicada_meas_class, &cicada_meas_driver);
	if (retval < 0) {
		ERR("hub_port_power_init failed");
		goto init_error_cleanup_usb;
	}

#define	CICADA_TRACE_MEAS_ENTRIES	1000
#define	CICADA_TRACE_MEAS_WRAP_MODE	true
	retval = cicada_trace_create(CICADA_TRACE_MEAS_ENTRIES,
				     CICADA_TRACE_MEAS_WRAP_MODE);
	if (retval < 0) {
		ERR("cicada_trace_create failed. Error number %d", retval);
		goto init_error_cleanup_hub_port_power;
	}
	cicada_trace_add_enabled_chans(DEFAULT_TRACE_CHANS);

	return 0;

	goto init_error_cleanup_hub_port_power;
init_error_cleanup_hub_port_power:
	hub_port_power_exit(cicada_meas_class, &cicada_meas_driver);

init_error_cleanup_usb:
	usb_deregister(&cicada_meas_driver);

init_error_cleanup_special:
	special_exit(cicada_meas_class);

init_error_cleanup_modpower:
	modpower_exit(cicada_meas_class);

init_error_cleanup_version:
	version_exit(cicada_meas_class);

init_error_destroy_class:
	class_destroy(cicada_meas_class);

init_error:
	return retval;
}

static void __exit cicada_meas_exit(void)
{
	hub_port_power_exit(cicada_meas_class, &cicada_meas_driver);

	/* deregister this driver with the USB subsystem */
	usb_deregister(&cicada_meas_driver);

	special_exit(cicada_meas_class);

	modpower_exit(cicada_meas_class);

	version_exit(cicada_meas_class);

	class_destroy(cicada_meas_class);

	cicada_trace_destroy();
}

module_init(cicada_meas_init);
module_exit(cicada_meas_exit);


MODULE_LICENSE("GPL");

#ifdef	CICADA_TRACE_DRIVER_ENABLE
#include "cicada_trace.c"
#endif
