#include <redboot.h>
#include <cyg/hal/hal_soc.h>

#define MXC_CCM_CGR1    (CCM_BASE_ADDR + 0x24)

#define KPCR    (KPP_BASE_ADDR + 0x00)
#define KPSR    (KPP_BASE_ADDR + 0x02)
#define KDDR    (KPP_BASE_ADDR + 0x04)
#define KPDR    (KPP_BASE_ADDR + 0x06)

#define KBD_STAT_KPKD         0x01   /* Key Press Interrupt Status bit */
#define KBD_STAT_KPKR         0x02   /* Key Release Interrupt Status bit */
#define KBD_STAT_KDSC         0x04   /* Key Depress Synchronizer Chain Status bit */
#define KBD_STAT_KRSS         0x08   /* Key Release Synchronizer Status bit */
#define KBD_STAT_KDIE        0x100   /* Key Depress Interrupt Enable Status bit */
#define KBD_STAT_KRIE        0x200   /* Key Release Interrupt Enable */
#define KBD_STAT_KPPEN       0x400   /* Keypad Clock Enable */

/* if the value of NUM_ROWS or NUM_COLS is changed 
 * init_iomux() must be changed to correctly configure the pins.
 */
#define NUM_ROWS    4
#define NUM_COLS    4
#define MAGIC_KEY_VAL    0x03 /* keys S10, S11 */

static void scan_keys(unsigned int *row_data)
{
    unsigned int val;
    int col;

    *row_data = 0;

    for (col = 0; col < NUM_COLS; col++) {
        /* 2. Write 1.s to KPDR[15:8] setting column data to 1.s */
        val = readw(KPDR);
        val |= 0xff00;
        writew(val, KPDR);

        /*
         * 3. Configure columns as totem pole outputs(for quick
         * discharging of keypad capacitance)
         */
        val = readw(KPCR);
        val &= 0x00ff;
        writew(val, KPCR);

        hal_delay_us(2);

        /*
         * 4. Configure columns as open-drain
         */
        val = readw(KPCR);
        val |= ((1 << NUM_COLS) - 1) << 8;
        writew(val, KPCR);

        /*
         * 5. Write a single column to 0, others to 1.
         * 6. Sample row inputs and save data. Multiple key presses
         * can be detected on a single column.
         * 7. Repeat steps 2 - 6 for remaining columns.
         */

        /* Col bit starts at 8th bit in KPDR */
        val = readw(KPDR);
        val &= ~(1 << (8 + col));
        writew(val, KPDR);

        /* Delay added to avoid propagating the 0 from column to row
         * when scanning. */

        hal_delay_us(5);

        /* Read row input */
        val = readw(KPDR);
        *row_data |= (~val & ((1 << NUM_ROWS) - 1)) << (NUM_ROWS * col);
    }

    /*
     * 8. Return all columns to 0 in preparation for standby mode.
     * 9. Clear KPKD and KPKR status bit(s) by writing to a .1.,
     * set the KPKR synchronizer chain by writing "1" to KRSS register,
     * clear the KPKD synchronizer chain by writing "1" to KDSC register
     */
    val = 0x00;
    writew(val, KPDR);
    val = readw(KPSR);
    val |= KBD_STAT_KPKD | KBD_STAT_KPKR | KBD_STAT_KRSS | KBD_STAT_KDSC;
    writew(val, KPSR);
}

#define MUX_CTL_COL_VAL        0x12     /* functional mode */
#define MUX_CTL_ROW_VAL        0x12     /* functional mode */
static void init_iomux(void)
{
    // KEY_COL 0-3, ctrl, path, pad
    // alt mode 7; input select 1; pad 1.8v
    writel(7, IOMUXC_BASE_ADDR + 0x164);
    writel(1, IOMUXC_BASE_ADDR + 0x950);
    writel(0x1 << 13, IOMUXC_BASE_ADDR + 0x5A8);
    
    writel(7, IOMUXC_BASE_ADDR + 0x168);
    writel(1, IOMUXC_BASE_ADDR + 0x954);
    writel(0x1 << 13, IOMUXC_BASE_ADDR + 0x5AC);
    
    writel(7, IOMUXC_BASE_ADDR + 0x16C);
    writel(1, IOMUXC_BASE_ADDR + 0x958);
    writel(0x1 << 13, IOMUXC_BASE_ADDR + 0x5B0);
    
    writel(7, IOMUXC_BASE_ADDR + 0x154);
    writel(1, IOMUXC_BASE_ADDR + 0x95c);
    writel(0x1 << 13, IOMUXC_BASE_ADDR + 0x598);
    
    // KEY_ROW 0-3, ctrl, path, pad
    // alt mode 7; input select 1; pad 1.8v, schmitz, 22k PU
    writel(7, IOMUXC_BASE_ADDR + 0x15C);
    writel(1, IOMUXC_BASE_ADDR + 0x970);
    writel(0x1 << 13, IOMUXC_BASE_ADDR + 0x5A0);
    
    writel(7, IOMUXC_BASE_ADDR + 0x160);
    writel(1, IOMUXC_BASE_ADDR + 0x974);
    writel(0x1 << 13, IOMUXC_BASE_ADDR + 0x5A4);
    
    writel(7, IOMUXC_BASE_ADDR + 0x14C);
    writel(1, IOMUXC_BASE_ADDR + 0x978);
    writel(0x1 << 13, IOMUXC_BASE_ADDR + 0x590);
    
    writel(7, IOMUXC_BASE_ADDR + 0x150);
    writel(1, IOMUXC_BASE_ADDR + 0x97c);
    writel(0x1 << 13, IOMUXC_BASE_ADDR + 0x594);
}

void init_keys(void)
{
    unsigned int val;

    /* enable the ipg_clk */
    val = readl(MXC_CCM_CGR1);
    val |= 3 << 20;
    writel(val, MXC_CCM_CGR1);

    init_iomux();

    /* Disable row interrupts (KPCR[7:0]) */
    val = 0;
    writew(val, KPCR);

    /* Write 0's to cols (KPDR[15:8]) */
    val = readw(KPDR);
    val &= 0x00ff;
    writew(val, KPDR);

    /* Enable columns as open-drain (KPCR[15:8]) */
    val = readw(KPCR);
    val |= (((1 << NUM_COLS) - 1) << 8);
    writew(val, KPCR);
    
    /* Configure columns as output, rows as input (KDDR[15:0]) */
    val = readw(KDDR);
    val |= 0xff00;
    val &= 0xff00;
    writew(val, KDDR);

    /* Clear interrupt enables
     * Clear KPKD and KPKR status bits by writing to a "1"
     * Set the KPKR synchronizer chain by writing "1"
     * Set the KPKD synchronizer chain by writing "1"
     */
    val = readw(KPSR);
    val &= ~(KBD_STAT_KPKR | KBD_STAT_KPKD);
    val |= KBD_STAT_KPKD | KBD_STAT_KRSS | KBD_STAT_KDSC;
    writew(val, KPSR);
}

RedBoot_init(init_keys, RedBoot_INIT_PRIO(100));

int check_keys(void)
{
    unsigned row_data = 0;

    scan_keys(&row_data);

    if (row_data == MAGIC_KEY_VAL){
        diag_printf("** Rescue Mode **\n");
        return 1;
    }
    return 0;
}

