/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/i2c.h>
#include <linux/delay.h>
#include <linux/workqueue.h>
#include <linux/debugfs.h>
#include <linux/seq_file.h>
#include <linux/mutex.h>
#include <linux/platform_device.h>
#include <linux/cicada_gasgauge.h>
#include <linux/gpio.h>

/* SMBus "command" codes from Smart Battery Data Specification. */
#define SB_MfrAccess			0x00	/* r/w word */
#define SB_RemainingCapAlarm		0x01	/* r/w word */
#define SB_RemainingTimeAlarm		0x02	/* r/w word */
#define SB_BatteryMode			0x03	/* r/w word */
#define SB_AtRate			0x04	/* r/w word */
#define SB_AtRateTimeToFull		0x05	/* read word, units of minutes */
#define SB_AtRateTimeToEmpty		0x06	/* read word, units of minutes */
#define SB_AtRateOK			0x07	/* read word */
#define SB_Temperature			0x08	/* read word, units of 0.1 K */
#define SB_Voltage			0x09	/* read word, units of mV */
#define SB_Current			0x0a	/* read word, units of mA */
#define SB_AverageCurrent		0x0b	/* read word, units of mA */
#define SB_MaxError			0x0c	/* read word, units of % */
#define SB_RelativeStateOfCharge	0x0d	/* read word, units of % */
#define SB_AbsoluteStateOfCharge	0x0e	/* read word, units of % */
#define SB_RemainingCapacity		0x0f	/* read word, units of mAh or 10mWh via mode */
#define SB_FullChargeCapacity		0x10	/* read word, units of mAh or 10mWh via mode */
#define SB_RunTimeToEmpty		0x11	/* read word, units of minutes */
#define SB_AverageTimeToEmpty		0x12	/* read word, units of minutes */
#define SB_AverageTimeToFull		0x13	/* read word, units of minutes */
#define SB_BatteryStatus		0x16	/* read word */
#define SB_CycleCount			0x17	/* read word */
#define SB_DesignCapacity		0x18	/* read word, units of mAh or 10mWh via mode */
#define SB_DesignVoltage		0x19	/* read word, units of mV */
#define SB_SpecificationInfo		0x1a	/* read word */
#define SB_ManufactureDate		0x1b	/* read word, bit mapped */
#define SB_SerialNumber			0x1c	/* read word */
#define SB_ManufacturerName		0x20	/* read block */
#define SB_DeviceName			0x21	/* read block */
#define SB_DeviceChemistry		0x22	/* read block */
#define SB_ManufacturerData		0x23	/* read block */

struct cicada_gasgauge_driver_struct {
	struct i2c_client	*client;
	struct mutex		lock;
};

static struct cicada_gasgauge_driver_struct *cicada_gassgauge;

static int cicada_gasgauge_read_word(struct i2c_client *client, u8 command)
{
	int ret;
	int i;

	for (i = 0; i < 3; ++i) {
		ret = i2c_smbus_read_word_data(client, command);
		if (ret != -1)
			break;
		else
			dev_dbg(&client->dev, "%s, retry %d\n", __func__, i+1);

		udelay(10);
	}
	if (ret < 0)
		dev_dbg(&client->dev, "%s, failed\n", __func__);

	return ret;
}

static int cicada_gasgauge_write_word(struct i2c_client *client, u8 command, u16 value)
{
	int ret;
	int i;

	for (i = 0; i < 3; ++i) {
		ret = i2c_smbus_write_word_data(client, command, value);
		if (ret == 0)
			break;
		else
			dev_dbg(&client->dev, "%s, retry %d\n", __func__, i+1);

		udelay(10);
	}
	if (ret < 0)
		dev_dbg(&client->dev, "%s, failed\n", __func__);

	return ret;
}

static int cicada_gasgauge_read_block(struct i2c_client *client, u8 command, u8 *values)
{
	char tmp[SBS_BLOCK_MAX];
	int ret;
	int i;

	for (i = 0; i < 3; ++i) {
		ret = i2c_smbus_read_i2c_block_data(client, command, I2C_SMBUS_BLOCK_MAX, tmp);
		if (ret != -1) {
			ret = tmp[0];
			tmp[ret + 1] = 0;
			strcpy(values, &tmp[1]);
			break;
		} else
			dev_dbg(&client->dev, "%s, retry %d\n", __func__, i+1);

		udelay(10);
	}
	if (ret < 0)
		dev_dbg(&client->dev, "%s, failed\n", __func__);

	return ret;
}

static int cicada_gasgauge_setmode(struct cicada_battery_data *battery)
{
	struct i2c_client *client = cicada_gassgauge->client;
	int ret;

	ret = cicada_gasgauge_read_word(client, SB_BatteryMode);
	if (ret < 0) {
		dev_dbg(&client->dev, "%s, BatteryMode read failed\n", __func__);
		return -1;
	}

	battery->mode = ret;

	/* report in mAh rather than 10mWh */
	battery->mode &= ~SB_MODE_CAPACITY_MODE;

	/* disable broadcast messages */
	battery->mode |= (SB_MODE_CHARGER_MODE | SB_MODE_ALARM_MODE);

	ret = cicada_gasgauge_write_word(cicada_gassgauge->client, SB_BatteryMode, battery->mode);
	if (ret < 0) {
		dev_dbg(&cicada_gassgauge->client->dev, "%s, BatteryMode write failed\n", __func__);
		return -1;
	}

	/* some batteries refuse to be read too soon after a write operation */
	/* so delay to avoid this problem. */
	msleep_interruptible(5);

	return 0;
}

static int cicada_gasgauge_getstate(struct cicada_battery_data *battery)
{
	struct i2c_client *client;
	int ret;

	client = cicada_gassgauge->client;

	ret = cicada_gasgauge_read_word(client, SB_Temperature);
	if (ret > -1)
		battery->temp_now = ret;

	ret = cicada_gasgauge_read_word(client, SB_Voltage);
	if (ret > -1)
		battery->voltage_now = ret;

	ret = cicada_gasgauge_read_word(client, SB_Current);
	if (ret > -1)
		/* adjust for signed 16 bit value */
		battery->current_now = ((ret > 0x8000) ? (ret - 0x10000) : ret);

	ret = cicada_gasgauge_read_word(client, SB_AverageCurrent);
	if (ret > -1)
		/* adjust for signed 16 bit value */
		battery->current_avg = ((ret > 0x8000) ? (ret - 0x10000) : ret);

	ret = cicada_gasgauge_read_word(client, SB_RemainingCapacity);
	if (ret > -1)
		battery->capacity_now = ret;

	ret = cicada_gasgauge_read_word(client, SB_RelativeStateOfCharge);
	if (ret > -1)
		battery->state_of_charge = ret;

	ret = cicada_gasgauge_read_word(client, SB_BatteryStatus);
	if (ret > -1)
		battery->status = ret;

	ret = cicada_gasgauge_read_word(client, SB_RunTimeToEmpty);
	if (ret > -1)
		battery->time_to_empty = ret;

	ret = cicada_gasgauge_read_word(client, SB_AverageTimeToEmpty);
	if (ret > -1)
		battery->time_to_empty_avg = ret;

	ret = cicada_gasgauge_read_word(client, SB_AverageTimeToFull);
	if (ret > -1)
		battery->time_to_full_avg = ret;

	ret = cicada_gasgauge_read_word(client, SB_CycleCount);
	if (ret > -1)
		battery->cycle_count = ret;

	return 0;
}

int cicada_gasgauge_get(struct cicada_battery_data *battery)
{
	struct i2c_client *client;
	int ret;

	if (!cicada_gassgauge)
		return -ENODEV;

	client = cicada_gassgauge->client;

	mutex_lock(&cicada_gassgauge->lock);

	/* broadcast modes re-enable after about 45 sec's */
	ret = cicada_gasgauge_setmode(battery);
	if (ret < 0)
		goto error1;

	if (!battery->present) {
		/* get the constant states */
		ret = cicada_gasgauge_read_block(client, SB_ManufacturerName, battery->manufacturer_name);
		if (ret < 0)
			goto error;

		ret = cicada_gasgauge_read_block(client, SB_DeviceName, battery->device_name);
		if (ret < 0)
			goto error;

		ret = cicada_gasgauge_read_block(client, SB_DeviceChemistry, battery->device_chemistry);
		if (ret < 0)
			goto error;

		ret = cicada_gasgauge_read_block(client, SB_ManufacturerData, battery->manufacturer_data);
		if (ret < 0)
			goto error;

		ret = cicada_gasgauge_read_word(client, SB_SerialNumber);
		if (ret > -1)
			sprintf(&battery->serial_number_str[0], "%05d", ret);
		else
			goto error;

		ret = cicada_gasgauge_read_word(client, SB_ManufactureDate);
		if (ret > -1) {
			sprintf(&battery->manufacturer_date_str[0], "%02d/%02d/%d",
				((ret >> 5) & 0x000f),
				(ret & 0x001f),
				(((ret >> 9) & 0x007f) + 1980));
		} else
			goto error;

		ret = cicada_gasgauge_read_word(client, SB_DesignVoltage);
		if (ret > -1)
			battery->design_voltage = ret;
		else
			goto error;

		ret = cicada_gasgauge_read_word(client, SB_DesignCapacity);
		if (ret > -1)
			battery->design_capacity = ret;
		else
			goto error;

		ret = cicada_gasgauge_read_word(client, SB_FullChargeCapacity);
		if (ret > -1)
			battery->full_charge_capacity = ret;
		else
			goto error;

		ret = cicada_gasgauge_read_word(client, SB_SpecificationInfo);
		if (ret > -1)
			battery->spec = ret;
		else
			goto error;
	}

	/* get the variable states */
	cicada_gasgauge_getstate(battery);

	battery->present = 1;

	mutex_unlock(&cicada_gassgauge->lock);

	return 0;

error:
	dev_err(&client->dev, "%s, read error\n", __func__);
error1:
	battery->present = 0;
	mutex_unlock(&cicada_gassgauge->lock);
	return ret;
}
EXPORT_SYMBOL(cicada_gasgauge_get);

static int cicada_gasgauge_probe(struct i2c_client *client,
				const struct i2c_device_id *id)
{
	struct cicada_battery_data battery;
	struct cicada_gasgauge_driver_struct *pdata;
	struct device *dev = &client->dev;
	struct device_node *np = dev->of_node;

	dev_dbg(&client->dev, "%s\n", __func__);

	if (!np) {
		dev_err(dev, "no device tree or platform data\n");
		return -ENOENT;
	}

	if (cicada_gassgauge) {
		dev_err(dev, "only one gasgauge chip allowed\n");
		return -ENODEV;
	}

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_SMBUS_WORD_DATA | I2C_FUNC_SMBUS_READ_I2C_BLOCK)) {
		dev_err(dev, "%s, i2c_check_functionality failed, %x\n", __func__, i2c_get_functionality(client->adapter));
		return -ENODEV;
	}

	pdata = devm_kzalloc(dev, sizeof *pdata, GFP_KERNEL);
	if (!pdata)
		return -ENOMEM;

	mutex_init(&pdata->lock);
	pdata->client = client;

	i2c_set_clientdata(client, pdata);
	cicada_gassgauge = pdata;

	mutex_lock(&cicada_gassgauge->lock);

	cicada_gasgauge_setmode(&battery);

	mutex_unlock(&cicada_gassgauge->lock);

	dev_info(dev, "probed, chip @ 0x%x (%s)\n",
		client->addr << 1, client->adapter->name);

	return 0;
}

static int cicada_gasgauge_remove(struct i2c_client *client)
{
	/* struct cicada_gasgauge_driver_struct *pdata = i2c_get_clientdata(client); */

	dev_dbg(&client->dev, "%s:\n", __func__);

	cicada_gassgauge = NULL;

	return 0;
}

static const struct i2c_device_id cicada_i2c_device_ids[] = {
	{ "cicada-gasgauge", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, cicada_i2c_device_ids);

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-gasgauge" },
	{},
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

struct i2c_driver cicada_gasgauge_driver = {
	.driver = {
		.name = "cicada-gasgauge",
		.owner = THIS_MODULE,
		.of_match_table = cicada_device_ids,
	},
	.probe = cicada_gasgauge_probe,
	.remove = cicada_gasgauge_remove,
	.id_table = cicada_i2c_device_ids,
};

int __init cicada_gasgauge_init(void)
{
	pr_info("cicada-gasgauge Driver\n");
	return i2c_add_driver(&cicada_gasgauge_driver);
}

void cicada_gasgauge_exit(void)
{
	i2c_del_driver(&cicada_gasgauge_driver);
}

module_init(cicada_gasgauge_init);
module_exit(cicada_gasgauge_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("cicada battery i2c driver");

