/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/printk.h>
#include <linux/module.h>
#include <linux/pm.h>
#include <linux/irq.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/reboot.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>

#define PWDN_ACTIVE_MAGIC_NUMBER	4465284

struct cicada_pwr_btn_private {
	struct tasklet_struct	tasklet;
	int wakeup;
	int suspend;
	int pwr_rdy;
	int pwr_dwn_rqst;
	int pwr_kill;
	int cpu_active;
	struct platform_device *pdev;
};

struct cicada_pwr_btn_private *privateData;

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-pwr-btn", .data = NULL },
	{}
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

static void power_kill(void)
{
	pr_debug("%s\n", __func__);
	gpio_set_value(privateData->pwr_kill, 1);
}

static ssize_t cicada_pdwn_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	return sprintf(buf, "%u\n", PWDN_ACTIVE_MAGIC_NUMBER);
}

static ssize_t cicada_pdwn_active(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t n)
{
	struct cicada_pwr_btn_private *pdata = dev_get_drvdata(dev);
	static int active_state;
	unsigned int value = simple_strtol(buf, NULL, 10);

	if (value != PWDN_ACTIVE_MAGIC_NUMBER) {
		printk(KERN_ERR "pwdn_active_store: Invalid value\n");
		return -EINVAL;
	}

	active_state = !active_state;
	pr_debug("%s %d\n", __func__, active_state);

	gpio_set_value(pdata->cpu_active, !!active_state);

	return n;
}

DEVICE_ATTR(pdwn_active, 0600, cicada_pdwn_show, cicada_pdwn_active);

static irqreturn_t cicada_pwr_btn_isr(int irqno, void *data)
{
	struct cicada_pwr_btn_private *pdata = data;

	pr_debug("%s\n", __func__);

	tasklet_schedule(&pdata->tasklet);

	return IRQ_HANDLED;
}

static void cicada_pwrdwn_tasklet(unsigned long data)
{
	struct cicada_pwr_btn_private *pdata = (struct cicada_pwr_btn_private *)data;

	pr_debug("%s\n", __func__);

	if (!pdata->suspend)
		orderly_poweroff(true);

	/* TBD: the concept of imx-active is to signal the PIC micro
	   that the imx is still alive and in the process of shutting
	   down.  a forced power off by the user would have a long timeout
	   in this case (maybe 30 seconds??).  if the imx is dead and
	   does not toggle this output, then the forced power off happens
	   much more quickly (maybe 10 seconds??) */
	gpio_set_value(pdata->cpu_active, 1);
}

static int cicada_pwr_btn_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct cicada_pwr_btn_private *pdata;
	int ret = 0;

	dev_dbg(dev, "%s\n", __func__);

	if (!np)
		return -EINVAL;

	if (!of_device_is_available(np))
		return -ENODEV;

	pdata = devm_kzalloc(dev, sizeof(*pdata),
			GFP_KERNEL);
	if (!pdata) {
		dev_err(dev, "could not allocate object\n");
		return -ENOMEM;
	}
	
	platform_set_drvdata(pdev, pdata);
	privateData = pdata;

	pdata->wakeup = 1;
	pdata->suspend = 0;

	device_init_wakeup(dev, pdata->wakeup);

	/* PS-KILL (GPIO2_29) */
	/* kill power, output, active high */
	pdata->pwr_kill = of_get_named_gpio(np, "pwr-kill-gpios", 0);
	ret = gpio_request(pdata->pwr_kill, "pwr-kill-gpios");
	if (ret < 0) {
		dev_dbg(dev, "request pwr-kill-gpios (%d) pin failed\n", pdata->pwr_kill);
		goto fail;
	}
	gpio_direction_output(pdata->pwr_kill, 0);

	/* set power off function */
	pm_power_off = power_kill;

	/* IMX-ACTIVE (GPIO2_21) */
	/* toggle during shutdown, output */
	pdata->cpu_active = of_get_named_gpio(np, "cpu-active-gpios", 0);
	ret = gpio_request(pdata->cpu_active, "cpu-active-gpios");
	if (ret < 0) {
		dev_dbg(dev, "request cpu-active-gpios pin (%d) failed\n", pdata->cpu_active);
		goto fail;
	}
	gpio_direction_output(pdata->cpu_active, 0);

	/* PS-IRQ-NOT (GPIO2_0) */
	/* pic power down request, input, active low */
	pdata->pwr_dwn_rqst = of_get_named_gpio(np, "irq-gpios", 0);
	ret = gpio_request(pdata->pwr_dwn_rqst, "irq-gpios");
	if (ret < 0) {
		dev_err(dev, "request irq-gpios pin (%d) failed\n", pdata->pwr_dwn_rqst);
		goto fail;
	}
	gpio_direction_input(pdata->pwr_dwn_rqst);

	ret = request_irq(gpio_to_irq(pdata->pwr_dwn_rqst), cicada_pwr_btn_isr,
			IRQ_TYPE_EDGE_FALLING, "pwr_dwn_rqst", pdata);
	if (ret) {
		dev_err(dev, "request_irq irq-gpios failed\n");
		goto fail;
	}

	tasklet_init(&pdata->tasklet, cicada_pwrdwn_tasklet, (unsigned long)pdata);

	ret = device_create_file(dev, &dev_attr_pdwn_active);
	if (ret)
		goto fail;

	return 0;

fail:
	dev_err(dev, "failed\n");
	return ret;
}

static int cicada_pwr_btn_remove(struct platform_device *pdev)
{
	struct cicada_pwr_btn_private *pdata = platform_get_drvdata(pdev);

	free_irq(pdata->pwr_dwn_rqst, pdev);

	device_remove_file(&pdev->dev, &dev_attr_pdwn_active);

	return 0;
}

static struct platform_driver cicada_pwr_btn_driver = {
	.driver = {
		.name = "cicada-pwr-btn",
		.owner = THIS_MODULE,
		.of_match_table = cicada_device_ids,
	},
	.probe	  = cicada_pwr_btn_probe,
	.remove   = cicada_pwr_btn_remove,
};

module_platform_driver(cicada_pwr_btn_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("Cicada Power Button");



