//==========================================================================
//
//      redboot_cmds.c
//
//      Board [platform] specific RedBoot commands
//
//==========================================================================
//####ECOSGPLCOPYRIGHTBEGIN####
// -------------------------------------------
// This file is part of eCos, the Embedded Configurable Operating System.
// Copyright (C) 1998, 1999, 2000, 2001, 2002 Red Hat, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 or (at your option) any later version.
//
// eCos is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
// for more details.
//
// You should have received a copy of the GNU General Public License along
// with eCos; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. However the source code for this file must still be made available
// in accordance with section (3) of the GNU General Public License.
//
// This exception does not invalidate any other reasons why a work based on
// this file might be covered by the GNU General Public License.
//
// Alternative licenses for eCos may be arranged by contacting Red Hat, Inc.
// at http://sources.redhat.com/ecos/ecos-license/
// -------------------------------------------
//####ECOSGPLCOPYRIGHTEND####
//==========================================================================
#include <redboot.h>
#include <cyg/hal/hal_intr.h>
#include <cyg/hal/hal_cache.h>
#include <cyg/hal/plf_mmap.h>
#include <cyg/hal/fsl_board.h>          // Platform specific hardware definitions

#ifdef CYGSEM_REDBOOT_FLASH_CONFIG
#include <flash_config.h>

RedBoot_config_option("Board specifics",
                      brd_specs,
                      ALWAYS_ENABLED,
                      true,
                      CONFIG_INT,
                      0
                     );
#endif  //CYGSEM_REDBOOT_FLASH_CONFIG

char HAL_PLATFORM_EXTRA[55] = "PASS x.x [x32 DDR]";

static void runImg(int argc, char *argv[]);

RedBoot_cmd("run",
            "Run an image at a location with MMU off",
            "[<virtual addr>]",
            runImg
           );

void launchRunImg(unsigned long addr)
{
    asm volatile ("mov r12, r0;");
    HAL_CLEAN_INVALIDATE_L2();
    HAL_DISABLE_L2();
    HAL_MMU_OFF();
    asm volatile (
                 "mov r0, #0;"
                 "mov r1, r12;"
                 "mov r11, #0;"
                 "mov r12, #0;"
                 "mrs r10, cpsr;"
                 "bic r10, r10, #0xF0000000;"
                 "msr cpsr_f, r10;"
                 "mov pc, r1"
                 );
}

extern unsigned long entry_address;

static void runImg(int argc,char *argv[])
{
    unsigned int virt_addr, phys_addr;

    // Default physical entry point for Symbian
    if (entry_address == 0xFFFFFFFF)
        virt_addr = 0x800000;
    else
    virt_addr = entry_address;

    if (!scan_opts(argc,argv,1,0,0,(void*)&virt_addr,
                   OPTION_ARG_TYPE_NUM, "virtual address"))
        return;

    if (entry_address != 0xFFFFFFFF)
        diag_printf("load entry_address=0x%lx\n", entry_address);
    HAL_VIRT_TO_PHYS_ADDRESS(virt_addr, phys_addr);

    diag_printf("virt_addr=0x%x\n",virt_addr);
    diag_printf("phys_addr=0x%x\n",phys_addr);

    hal_delay_us(10 * 1000);  // FNET addition

    launchRunImg(phys_addr);
}

RedBoot_cmd("romupdate",
            "Update Redboot with currently running image",
            "",
            romupdate
           );

extern int flash_program(void *_addr, void *_data, int len, void **err_addr);
extern int flash_erase(void *addr, int len, void **err_addr);
extern char *flash_errmsg(int err);
extern unsigned char *ram_end; //ram end is where the redboot starts FIXME: use PC value
extern cyg_uint32 mmc_data_read(cyg_uint32 *dest_ptr, cyg_uint32 length,
             cyg_uint32 offset);
extern cyg_uint32 mmc_data_write (cyg_uint32 *src_ptr, cyg_uint32 length, cyg_uint32 offset);

#ifdef CYGPKG_IO_FLASH
void romupdate(int argc, char *argv[])
{
    void *err_addr, *base_addr = 0;
    int stat;

    if (IS_FIS_FROM_MMC()) {
        diag_printf("Updating ROM in MMC/SD flash\n");
        base_addr = (void*)MXC_MMC_BASE_DUMMY;
        /* Read the first 1K from the card */
        mmc_data_read((cyg_uint32*)ram_end, 0x400, (cyg_uint32)base_addr);
        diag_printf("Programming Redboot to MMC/SD flash\n");
        mmc_data_write((cyg_uint32*)ram_end, REDBOOT_IMAGE_SIZE, (cyg_uint32)base_addr);
        return;
    } else if (IS_FIS_FROM_NAND()) {
        base_addr = (void*)MXC_NAND_BASE_DUMMY;
        diag_printf("Updating ROM in NAND flash\n");
    } else if (IS_FIS_FROM_NOR()) {
        base_addr = (void*)BOARD_FLASH_START;
        diag_printf("Updating ROM in NOR flash\n");
    } else {
        diag_printf("romupdate not supported\n");
        diag_printf("Use \"factive [NOR|NAND|MMC]\" to select either NOR, NAND or MMC flash\n");
        return;
    }

    // Erase area to be programmed
    if ((stat = flash_erase((void *)base_addr,
                            REDBOOT_IMAGE_SIZE,
                            (void **)&err_addr)) != 0) {
        diag_printf("Can't erase region at %p: %s\n",
                    err_addr, flash_errmsg(stat));
        return;
    }
    // Now program it
    if ((stat = flash_program((void *)base_addr, (void *)ram_end,
                              REDBOOT_IMAGE_SIZE,
                              (void **)&err_addr)) != 0) {
        diag_printf("Can't program region at %p: %s\n",
                    err_addr, flash_errmsg(stat));
    }
}

RedBoot_cmd("factive",
            "Enable one flash media for Redboot",
            "[NOR | NAND | MMC]",
            factive
           );

void factive(int argc, char *argv[])
{
    unsigned long phys_addr;

    if (argc != 2) {
        diag_printf("Invalid factive cmd\n");
        return;
    }

    if (strcasecmp(argv[1], "NOR") == 0) {
#ifndef MXCFLASH_SELECT_NOR
        diag_printf("Not supported\n");
        return;
#else
        MXC_ASSERT_NOR_BOOT();
#endif
    } else if (strcasecmp(argv[1], "NAND") == 0) {
#ifndef MXCFLASH_SELECT_NAND
        diag_printf("Not supported\n");
        return;
#else
        MXC_ASSERT_NAND_BOOT();
#endif
    } else if (strcasecmp(argv[1], "MMC") == 0) {
#ifndef MXCFLASH_SELECT_MMC
        diag_printf("Not supported\n");
        return;
#else
        MXC_ASSERT_MMC_BOOT();
#endif
    } else {
        diag_printf("Invalid command: %s\n", argv[1]);
        return;
    }
    HAL_VIRT_TO_PHYS_ADDRESS(ram_end, phys_addr);

    launchRunImg(phys_addr);
}
#endif //CYGPKG_IO_FLASH

RedBoot_cmd("mmc_copy", 
            "load kernel from mmc card",
            "-s <src sector addr> -l <bytes length>",
            do_mmc_copy
    );

extern int mmcflash_hwr_init(void);
static void mmc_copy_init(void)
{
    static bool inited = false;
    
    if (!inited) {
        mmcflash_hwr_init();
        inited = true;
    }
}

#define MMC_SDRAM_START_OFFSET  0x800000
#define MMC_BYTES_PER_SECTOR    512

void do_mmc_copy(int argc, char *argv[])
{
    struct option_info opts[2];
    bool src_set;
    bool len_set;
    unsigned long src;
    unsigned long len;
    
    init_opts(&opts[0], 's', true, OPTION_ARG_TYPE_NUM, 
              (void *)&src, (bool *)&src_set, "src sector addr");
    init_opts(&opts[1], 'l', true,  OPTION_ARG_TYPE_NUM, 
              (void *)&len, (bool *)&len_set, "bytes length");
    if (!scan_opts(argc, argv, 1, opts, 2, 0, 0, "")) {
        return;
    }
    
    if (!src_set || !len_set) {
        diag_printf("usage: mmc_copy -s <src sector addr> -l <bytes length>\n");
        return;
    }
    
    mmc_copy_init();
    
    diag_printf("Reading from MMC byte address 0x%08lx to SDRAM...\n", src);

    src *= MMC_BYTES_PER_SECTOR;
    mmc_data_read((cyg_uint32 *)(SDRAM_BASE_ADDR + MMC_SDRAM_START_OFFSET),
                  len, src);
}

#if 0
extern int norflash_erase_block(void* addr, unsigned int size);
extern int norflash_program_buf(void* addr, void* data, int len);

#define POST_SDRAM_START_OFFSET         0x800000
#define POST_MMC_OFFSET                 0x100000
#define POST_SIZE                       0x100000
#define POST_MAGIC_MARKER               0x43

#define POST_MAGIC_MARKER_LINUX         0x47
#define POST_MMC_REDBOOT_NOR_OFFSET     0x500000    // offset in MMC for NOR flash redboot
#define POST_MMC_KERNEL_OFFSET          0x600000    // offset in MMC for Linux kernel
#define POST_MMC_ROOTFS_OFFSET          0x800000    // offset in MMC for Linux root
#define NOR_REDBOOT_OFFSET              0x0         // offset in NOR flash for redboot
#define NOR_KERNEL_OFFSET               0x080000    // offset in NOR flash for kernel
#define NOR_ROOTFS_OFFSET               0x480000    // offset in NOR flash for root

void imx_launch_post(void)
{
    if (POST_MODE != POST_MAGIC_MARKER &&
        POST_MODE != POST_MAGIC_MARKER_LINUX)
        return;
    if (POST_MODE == POST_MAGIC_MARKER_LINUX) {
        if (readl(IRAM_BASE_ADDR) != 0x12345678) {
            diag_printf("Reading Redboot from MMC to SDRAM...\n");
            mmc_data_read((cyg_uint32 *)(SDRAM_BASE_ADDR + POST_SDRAM_START_OFFSET),
                          0x60000,
                          (cyg_uint32)POST_MMC_REDBOOT_NOR_OFFSET);
            diag_printf("Reading Linux kernel from MMC to SDRAM...\n");
            mmc_data_read((cyg_uint32 *)(SDRAM_BASE_ADDR + POST_SDRAM_START_OFFSET + 0x100000),
                          0x200000,
                          (cyg_uint32)POST_MMC_KERNEL_OFFSET);
            diag_printf("Reading Linux filesystem from MMC to SDRAM...\n");
            mmc_data_read((cyg_uint32 *)(SDRAM_BASE_ADDR + POST_SDRAM_START_OFFSET + 0x300000),
                          0x1E00000,
                          (cyg_uint32)POST_MMC_ROOTFS_OFFSET);

            writel(0x12345678, IRAM_BASE_ADDR);
            MXC_ASSERT_NOR_BOOT();
            launchRunImg(0x87f00000);
        }
        return;
    }
    diag_printf("Launching POST\n");
    if (mmc_data_read((cyg_uint32 *)(SDRAM_BASE_ADDR + POST_SDRAM_START_OFFSET),
                      POST_SIZE, (cyg_uint32)POST_MMC_OFFSET) != 0) {
        while (1)
            diag_printf("MMC read failed: %s, %d\n", __FILE__, __LINE__);
    } else {
        diag_printf("MMC read successful\n");
        launchRunImg(SDRAM_BASE_ADDR + POST_SDRAM_START_OFFSET);
    }
}
RedBoot_init(imx_launch_post, RedBoot_INIT_BEFORE_NET);

void imx_launch_post_linux(void)
{
    void *err_addr;
    if (POST_MODE != POST_MAGIC_MARKER_LINUX)
        return;
    if (readl(IRAM_BASE_ADDR) == 0x87654321)
        return;

    writel(0x87654321, IRAM_BASE_ADDR);

    // redboot
    diag_printf("erasing NOR flash for Redboot\n");
    flash_erase((void *)(CS0_BASE_ADDR + NOR_REDBOOT_OFFSET), 0x60000,(void **)&err_addr);
    diag_printf("programming NOR flash for Redboot\n");
    flash_program((void *)(CS0_BASE_ADDR + NOR_REDBOOT_OFFSET),
                  (void *)(SDRAM_BASE_ADDR + POST_SDRAM_START_OFFSET),
                  0x60000, (void **)&err_addr);
    // linux kernel
    diag_printf("erasing NOR flash for Linux Kernel\n");
    flash_erase((void *)(CS0_BASE_ADDR + NOR_KERNEL_OFFSET), 0x200000,(void **)&err_addr);
    diag_printf("programming NOR flash for Linux Kernel\n");
    flash_program((void *)(CS0_BASE_ADDR + NOR_KERNEL_OFFSET),
                  (void *)(SDRAM_BASE_ADDR + POST_SDRAM_START_OFFSET + 0x100000),
                  0x200000, (void **)&err_addr);

    // linux root filesystem
    diag_printf("erasing NOR flash for Linux file system\n");
    flash_erase((void *)(CS0_BASE_ADDR + NOR_ROOTFS_OFFSET), 0x1E00000,(void **)&err_addr); //30MB
    diag_printf("programming NOR flash for Linux file system\n");
    flash_program((void *)(CS0_BASE_ADDR + NOR_ROOTFS_OFFSET),
                  (void *)(SDRAM_BASE_ADDR + POST_SDRAM_START_OFFSET + 0x300000),
                  0x1E00000, (void **)&err_addr);
}
//RedBoot_init(imx_launch_post_linux, RedBoot_INIT_LAST);
RedBoot_init(imx_launch_post_linux, RedBoot_INIT_BEFORE_NET);
#endif

#if 1
// FNET special DDR memory test - originally used for testing cold boot
// problem on the Cicada dev board.  Problem turned out to be violation
// of the tRFC timing on the Micron 1Gbit Mobile DDR device.  The problem
// showed up with caches turned off when running at max DDR clock rate of
// 133 MHz.
//
// FIXME: TBD: If necessary, this may be if'ed of of the build at some point.
//
RedBoot_cmd("memtest", 
            "Test DDR memory",
            "-b <location> -l <length> [-f] [-m] [-r]",
            do_memtest
    );

/* Table of random data that is 131 bytes long.  The length of this table */
/* is intended to be a prime number so that the data in memory is skewed  */
/* and not repeated in a binary fashion.                                   */
const cyg_uint32 randMemTable[] = {
    0xc92dbd26, 0x6db4174b, 0x82d08f66, 0x22339d2f,  /* 0 - 3 */
    0xab1134ef, 0xef4976eb, 0x3681665c, 0xba86ceb7,  /* 4 -7 */
    0x8a543dab, 0xdf7463b3, 0x2772d35a, 0x3202d2a3,  /* 8-11 */
    0x5e61415f, 0x99bce68f, 0xa9d6c12b, 0xe8f7b6d2,  /* 12-15 */
    0x5fe1db6a, 0x8588c821, 0x0a1c7e4e, 0x91a286e4,  /* 16-19 */
    0xb69b2b20, 0x57986334, 0x0035eb79, 0xeb21fa2d,  /* 20-23 */
    0x3345bf5f, 0x4ff1734f, 0xeb85a146, 0xbf0d8adb,  /* 24-27 */
    0xb78b2cd1, 0xd73732f5, 0xb1e84dd5, 0x3159ae11,  /* 28-31 */
    0x6460c7c9, 0x2dbd266d, 0xb4174b82, 0xd08f6622,  /* 32-35 */
    0x339d2fab, 0x1134efef, 0x4976eb36, 0x81665cba,  /* 36-39 */
    0x86ceb78a, 0x543dabdf, 0x7463b327, 0x72d35a32,  /* 40-43 */
    0x02d2a35e, 0x61415f99, 0xbce68fa9, 0xd6c12be8,  /* 44-47 */
    0xf7b6d25f, 0xe1db6a85, 0x88c8210a, 0x1c7e4e91,  /* 48-51 */
    0xa286e4b6, 0x9b2b2057, 0x98633400, 0x35eb79eb,  /* 52-55 */
    0x21fa2d33, 0x45bf5f4f, 0xf1734feb, 0x85a146bf,  /* 56-59 */
    0x0d8adbb7, 0x8b2cd1d7, 0x3732f5b1, 0xe84dd531,  /* 60-63 */
    0x59ae1164, 0x60c7c92d, 0xbd266db4, 0x174b82d0,  /* 64-67 */
    0x8f662233, 0x9d2fab11, 0x34efef49, 0x76eb3681,  /* 68-71 */
    0x665cba86, 0xceb78a54, 0x3dabdf74, 0x63b32772,  /* 72-75 */
    0xd35a3202, 0xd2a35e61, 0x415f99bc, 0xe68fa9d6,  /* 76-79 */
    0xc12be8f7, 0xb6d25fe1, 0xdb6a8588, 0xc8210a1c,  /* 80-83 */
    0x7e4e91a2, 0x86e4b69b, 0x2b205798, 0x63340035,  /* 84-87 */
    0xeb79eb21, 0xfa2d3345, 0xbf5f4ff1, 0x734feb85,  /* 88-91 */
    0xa146bf0d, 0x8adbb78b, 0x2cd1d737, 0x32f5b1e8,  /* 92-95 */
    0x4dd53159, 0xae116460, 0xc7c92dbd, 0x266db417,  /* 96-99 */
    0x4b82d08f, 0x6622339d, 0x2fab1134, 0xefef4976,  /* 100-103 */
    0xeb368166, 0x5cba86ce, 0xb78a543d, 0xabdf7463,  /* 104-107 */
    0xb32772d3, 0x5a3202d2, 0xa35e6141, 0x5f99bce6,  /* 108-111 */
    0x8fa9d6c1, 0x2be8f7b6, 0xd25fe1db, 0x6a8588c8,  /* 112-115 */
    0x210a1c7e, 0x4e91a286, 0xe4b69b2b, 0x20579863,  /* 116-119 */
    0x340035eb, 0x79eb21fa, 0x2d3345bf, 0x5f4ff173,  /* 120-123 */
    0x4feb85a1, 0x46bf0d8a, 0xdbb78b2c, 0xd1d73732,  /* 124-127 */
    0xf5b1e84d, 0xd53159ae, 0x116460c7               /* 128-130 */
};

void
do_memtest(int argc, char *argv[])
{
    // Fill a region of memory then read it back and verify it
    struct option_info opts[5];
    bool multi_set = false;
    bool flash_test_set = false;
    bool range_set = false;
    unsigned long dst_base;
    long len;
    long passCnt = 0;
    int i = 0;
    int errCnt = 0;
    bool dst_base_set, len_set;
    register cyg_uint32 *s;
    cyg_uint32 *ss;
    cyg_uint32 *se;
    register cyg_uint32 *d;
    register cyg_uint32 *de;
    cyg_uint32 *tblBase;

    init_opts(&opts[0], 'b', true, OPTION_ARG_TYPE_NUM, 
              (void *)&dst_base, (bool *)&dst_base_set, "base address");
    init_opts(&opts[1], 'l', true, OPTION_ARG_TYPE_NUM, 
              (void *)&len, (bool *)&len_set, "length");
    init_opts(&opts[2], 'f', false, OPTION_ARG_TYPE_FLG, 
              (void *)&flash_test_set, (bool *)0, "do Flash->DDR test");
    init_opts(&opts[3], 'm', false, OPTION_ARG_TYPE_FLG, 
              (void *)&multi_set, (bool *)0, "do multiple rd/wr test");
    init_opts(&opts[4], 'r', false, OPTION_ARG_TYPE_FLG, 
              (void *)&range_set, (bool *)0, "range error output");
    if (!scan_opts(argc, argv, 1, opts, 5, 0, 0, "")) {
        return;
    }
    if (!dst_base_set || !len_set) {
        diag_printf("usage: memtest -b <addr> -l <length> [-f] [-m] [-r]\n");
        return;
    }

#define SIZE_RANDMTBL        (sizeof(randMemTable) / sizeof(randMemTable[0]))
#define SIZE_MULTITBL        (8)

    // set pointer to the base of the memory pattern table
    if (flash_test_set)
        tblBase = (cyg_uint32 *)( (unsigned char *)&randMemTable +
                                  (0xa0000000 - 0x87f00000) );
    else
        tblBase = (cyg_uint32 *)&randMemTable;

    while (1) {
        i = passCnt % (multi_set ? SIZE_MULTITBL : SIZE_RANDMTBL);
        d  = (cyg_uint32 *)dst_base;
        de = d + (len / sizeof(cyg_uint32));

        if (multi_set) {
            s  = tblBase + i;
            se = s + SIZE_MULTITBL;
            ss = s;
            asm volatile ( \
                  "FstMemTstLbl: "              \
                  "ldmia %0,  {r3 - r10}; "    \
                  "stmia %1!, {r3 - r10}; "    \
                  "cmp   %1, %2; "                                 \
                  "bcc   FstMemTstLbl; "                           \
                   :                                                  \
                   : "r"(s), "r"(d),  "r"(de)                         \
                   : "r3", "r4", "r5", "r6", "r7", "r8", "r9", "r10"  \
                );

/*             diag_printf("ss=%p s=%p se=%p d=%p de=%p i=%d pass=%ld\n", */
/*                         ss, s, se, d, de, i, passCnt); */

        }
        else {
            s  = tblBase + i;
            ss = tblBase;
            se = tblBase + SIZE_RANDMTBL;
            while (d < de) {
                *d++ = *s++;
                if (s >= se)
                    s = ss;
            }
        }

        mon_write_char('w');

        d = (cyg_uint32 *)dst_base;
        s = tblBase + i;
        de = d + (len / sizeof(cyg_uint32));

/*         diag_printf("ss=%p s=%p se=%p d=%p de=%p\n", */
/*                 ss, s, se, d, de); */

        if (range_set) {
            cyg_uint32 * firstAdd = 0;
            while (d < de) {
                if (*d++ != *s++) {
                    if (firstAdd == 0)
                        firstAdd = d - 1;

                }
                else if (firstAdd != 0) {
                    d--;
                    diag_printf("\n%p-%p  0x%08x\n",
                                firstAdd, d, (d - firstAdd) * sizeof(cyg_uint32));
                    firstAdd = 0;
                    d++;
                }

                if (s >= se)
                    s = ss;
            }
        }
        else {
            while (d < de) {
                if (*d++ != *s++) {
                    s--;
                    d--;
                    diag_printf("\nMemory mis-match - %p=0x%08x, %p=0x%08x\n",
                                s, *s, d, *d);
                    s++;
                    d++;
                    if (++errCnt > 8)
                        return;
                }

                if (s >= se)
                    s = ss;
            }
        }

        mon_write_char('r');

        if ((++passCnt % 10) == 0) {
            diag_printf(" pass = %ld, i = %d\n", passCnt, i);
        }

        if (_rb_break(1))
            break;

    }
    diag_printf("\nTotal passes = %ld\n", passCnt);
}
#endif
