#include <redboot.h>
#include <cyg/hal/hal_soc.h>
#include <cyg/io/mxc_i2c.h>
#include <cyg/hal/version.h>
#include <cyg/hal/fsl_board.h>

#define EEPROM_ADR_OFFSET 0

#define EEPROM_READ     true
#define EEPROM_WRITE    false

#define MAX_KERNEL_SIZE         0x600000

// zero based buss number
#define EEPROM_BUSS_NUMBER      0
#define EEPROM_BUSS_ADDRESS     0x54
#define EEPROM_PAGE_SIZE        ((unsigned int)64)

// Internal structure used to hold config data
#define NVHDR_MAGIC     0xcafefeed
#define NVHDR_HDRVER    1
#define NVHDR_TYPE_INVALID 0
#define NVHDR_TYPE_NVDATA_OLD_MANTIS 1
#define NVHDR_TYPE_BOOTPARM 2
#define NVHDR_TYPE_NVDATA_SHARED 3
struct NvHdr
{
    unsigned long magic;      // the same for all NvHdr's
    unsigned long hdrVersion;
    unsigned long dataType;   // identifies the containing struct (e.g. NvDataShared)
    unsigned long size;       // header plus data
    unsigned long dataCsum;
    unsigned long hdrCsum;
};

#define MAX_DATA_LEN    512
struct BootParm
{
    struct NvHdr nvHdr;
    char config_data[MAX_DATA_LEN];
    unsigned long kernAddr;
    unsigned long kernLen;
    unsigned long kernCrc;
};

// group shared data
#define ENETADDR_LEN 17
#define INSTRSTRING_LEN 255
struct NvDataShared
{
    struct NvHdr nvHdr;
    unsigned long bootGroup;
    char enetAddr[ENETADDR_LEN + 3]; // long aligned
    char instrString[INSTRSTRING_LEN + 1];
    char unused[208];
};

//
// this represents the layout of data in EEPROM, starting at address 0
//
#define BOOT_SLOTS_PER_GROUP 2
#define MAX_BOOT_GROUPS 4
#define MAX_BOOT_SLOTS (MAX_BOOT_GROUPS * BOOT_SLOTS_PER_GROUP)
struct CicadaEeprom
{
    struct BootParm bootParms[MAX_BOOT_SLOTS];
    struct NvDataShared nvDataShared;
}cicada_eeprom;

char fnet_version[] = BUILD_DATE;
static char bootline[MAX_DATA_LEN];
static bool corrupt = false;

#define SDRAM_LOAD_ADDR  (SDRAM_BASE_ADDR + 0x800000)
#define DEFAULT_SECTOR_SIZE    512

struct partition {
	unsigned char boot_ind;         /* 0x80 - active */
	unsigned char head;             /* starting head */
	unsigned char sector;           /* starting sector */
	unsigned char cyl;              /* starting cylinder */
	unsigned char sys_ind;          /* what partition type */
	unsigned char end_head;         /* end head */
	unsigned char end_sector;       /* end sector */
	unsigned char end_cyl;          /* end cylinder */
	unsigned char start4[4];        /* starting sector counting from 0 */
	unsigned char size4[4];         /* nr of sectors in partition */
};

char MBRbuffer[DEFAULT_SECTOR_SIZE];
cyg_uint32 numSectors;
cyg_uint32 startSector;

extern int check_keys(void);
extern cyg_uint32 mmc_data_read(cyg_uint32 *dest_ptr, cyg_uint32 length,
             cyg_uint32 offset);
static int rw_Data(char *buff, unsigned int offset, int size, bool rw);

static int
str_cmp(const char *a, const char *b)
{
    for (; *a == *b; a++, b++) {
        if (*a == 0 && *b == 0)
            return 0;
    }
    return 1;
}

static cyg_uint32
crc_NvHdr(struct NvHdr *nvHdr)
{
    cyg_uint32 crc;
    unsigned char *addr = (unsigned char *)nvHdr;
 
    crc = cyg_crc32(addr, sizeof(*nvHdr) - sizeof(nvHdr->hdrCsum));

    return crc;
}

// excluding the header, nvHdr
static cyg_uint32
crc_NvHdr_data(struct NvHdr *nvHdr)
{
    cyg_uint32 crc;
    unsigned char *addr = (unsigned char *)nvHdr + sizeof(*nvHdr);

    crc = cyg_crc32(addr, nvHdr->size - sizeof(*nvHdr));

    return crc;
}

static int
chk_NvData(struct NvHdr *nvHdr)
{
    if (crc_NvHdr(nvHdr) != nvHdr->hdrCsum ||
        nvHdr->magic != NVHDR_MAGIC ||
        nvHdr->dataType == 0 ||
        nvHdr->size == 0) {
        // header error
        return 1;
    }
    if (crc_NvHdr_data(nvHdr) != nvHdr->dataCsum) {
        // data error
        return 2;
    }
    return 0;
}

static void
set_NvData(struct NvHdr *nvHdr, unsigned int offset)
{
    nvHdr->magic = NVHDR_MAGIC;
    nvHdr->hdrVersion = NVHDR_HDRVER;
    
    switch (nvHdr->dataType) {
    case NVHDR_TYPE_BOOTPARM:
        nvHdr->size = sizeof(struct BootParm);
        break;
    case NVHDR_TYPE_NVDATA_SHARED:
        nvHdr->size = sizeof(struct NvDataShared);
        break;
    default:
        diag_printf("** EEPROM Error invalid header dataType\n");
        nvHdr->size = 0;
        break;
    }
    
    nvHdr->dataCsum = crc_NvHdr_data(nvHdr);
    nvHdr->hdrCsum = crc_NvHdr(nvHdr);
    
    if (corrupt) {
        corrupt = false;
        nvHdr->magic++;
    }
    
    rw_Data((char *)nvHdr, offset, nvHdr->size, EEPROM_WRITE);
}

static int
rw_eeprom(bool read, unsigned int addr,
        unsigned char *data, unsigned int bytes)
{
    int dir = (read ? I2C_READ : I2C_WRITE);
    unsigned int start_addr = addr;
    int total_bytes = bytes;
    unsigned char *buf = data;
    
    struct mxc_i2c_request rq;
    int i;
    unsigned int end_addr;
    int num_bytes;

    rq.dev_addr = EEPROM_BUSS_ADDRESS;
    rq.reg_addr_sz = 2;

    if (dir == I2C_WRITE){
        // write
        end_addr = (start_addr + EEPROM_PAGE_SIZE) & ~(EEPROM_PAGE_SIZE - 1);
        num_bytes = end_addr - start_addr;

        while (total_bytes > 0){
            if (num_bytes > total_bytes)
                num_bytes = total_bytes;

            // address should go out ms byte first, ls byte last
            // i2c_xfer sends out ls byte first
            rq.reg_addr = ((start_addr & 0xff00) >> 8) | ((start_addr & 0xff) << 8);
            rq.buffer_sz = num_bytes;
            rq.buffer = buf;

            for (i = 0; i < 3; ++i) {
                if (i2c_xfer(EEPROM_BUSS_NUMBER, &rq, dir) == 0)
                    break;
                hal_delay_us(5000);
            }
            hal_delay_us(5000);

            if (i == 3) {
                diag_printf("I2C transfer %s error\n", (read ? "read":"write"));
                return -1;
            }
            
            start_addr += num_bytes;
            buf += num_bytes;
            total_bytes -= num_bytes;
            num_bytes = EEPROM_PAGE_SIZE;
        } 
    }else{
        // read
        // address should go out ms byte first, ls byte last
        // i2c_xfer sends out ls byte first
        rq.reg_addr = ((start_addr & 0xff00) >> 8) | ((start_addr & 0xff) << 8);
        rq.buffer_sz = total_bytes;
        rq.buffer = buf;
        
        if (i2c_xfer(EEPROM_BUSS_NUMBER, &rq, dir) != 0) {
            diag_printf("I2C transfer %s error\n", (read ? "read":"write"));
            return -1;
        }
    }

    return 0;
}

static int
rw_Data(char *buff, unsigned int offset, int size, bool rw)
{
    int dir;
    unsigned int start_addr = EEPROM_ADR_OFFSET;
    unsigned char *buf;

    dir = rw ? I2C_READ : I2C_WRITE;
    start_addr += offset;
    buf = (unsigned char *)buff;

    return rw_eeprom(dir, start_addr, buf, size);
}

RedBoot_cmd("eeconfig",
            "Manage configuration kept in EEPROM memory, for help:",
            "-h",
            do_eeconfig
    );

void do_eeconfig(int argc, char *argv[])
{
    static bool eedata = false;
    int ret;
    char *name;
    bool list_set = false;
    bool init_set = false;
    bool test_set = false;
    bool help = false;
    bool str_set = false;
    char *str = (char *)0;
    bool val_set = false;
    long val = 0;
    bool x_set = false;
    int x = 0;
    int i;
    int num_options = 0;
    unsigned int offset;

    struct option_info opts[8];
    
    init_opts(&opts[num_options++], 'l', false, OPTION_ARG_TYPE_FLG, 
              (void *)&list_set, (bool *)0, "list configuration only");
    init_opts(&opts[num_options++], 'i', false, OPTION_ARG_TYPE_FLG, 
              (void *)&init_set, (bool *)0, "initialize configuration database");
    init_opts(&opts[num_options++], 't', false, OPTION_ARG_TYPE_FLG, 
              (void *)&test_set, (bool *)0, "test configuration database");
    init_opts(&opts[num_options++], 's', true, OPTION_ARG_TYPE_STR, 
              (void *)&str, (bool *)&str_set, "string");
    init_opts(&opts[num_options++], 'v', true, OPTION_ARG_TYPE_NUM, 
              (void *)&val, (bool *)&val_set, "value");
    init_opts(&opts[num_options++], 'x', true, OPTION_ARG_TYPE_NUM, 
              (void *)&x, (bool *)&x_set, "index");
    init_opts(&opts[num_options++], 'c', false, OPTION_ARG_TYPE_FLG, 
              (void *)&corrupt, (bool *)0, "corrupt");
    init_opts(&opts[num_options++], 'h', false, OPTION_ARG_TYPE_FLG, 
              (void *)&help, (bool *)0, "help");
    
    if (!scan_opts(argc, argv, 1, opts, num_options, (void *)&name, OPTION_ARG_TYPE_STR, "value name")) {
        diag_printf("** Error: invalid arguments\n");
        return;
    }
    
    if (!eedata) {
        eedata = true;
        offset = 0;
        rw_Data((char *)&cicada_eeprom, offset, sizeof(cicada_eeprom), EEPROM_READ);
    }

    if (init_set) {
        if (verify_action("Initialize non-volatile configuration")) {
            memset(&cicada_eeprom, 0, sizeof(cicada_eeprom));

            for(i = 0; i < MAX_BOOT_SLOTS; ++i) {
                cicada_eeprom.bootParms[i].nvHdr.dataType = NVHDR_TYPE_INVALID;
                
                offset = (unsigned int)&cicada_eeprom.bootParms[i].nvHdr - (unsigned int)&cicada_eeprom;
                set_NvData(&cicada_eeprom.bootParms[i].nvHdr, offset);
            }
            
            cicada_eeprom.nvDataShared.nvHdr.dataType = NVHDR_TYPE_INVALID;
            offset = (unsigned int)&cicada_eeprom.nvDataShared.nvHdr - (unsigned int)&cicada_eeprom;
            set_NvData(&cicada_eeprom.nvDataShared.nvHdr, offset);
        }
        
        return;
    }

    if (test_set) {
        ret = 0;
        for (i = 0; i < MAX_BOOT_SLOTS; ++i) {
            switch (chk_NvData(&cicada_eeprom.bootParms[i].nvHdr)) {
            case 0:
                diag_printf("** EEPROM bootParms[%d]: ok\n", i);
                break;
            case 1:
                diag_printf("** EEPROM bootParms[%d]: header corrupt\n", i);
                break;
            case 2:
                diag_printf("** EEPROM bootParms[%d]: data corrupt\n", i);
                break;
            default:
                diag_printf("** EEPROM bootParms[%d]: corrupt\n", i);
                break;
            }
        }

        switch (chk_NvData(&cicada_eeprom.nvDataShared.nvHdr)) {
        case 0:
            diag_printf("** EEPROM nvDataShared: ok\n");
            break;
        case 1:
            diag_printf("** EEPROM nvDataShared: header corrupt\n");
            break;
        case 2:
            diag_printf("** EEPROM nvDataShared: data corrupt\n");
            break;
        default:
            diag_printf("** EEPROM nvDataShared: corrupt\n");
            break;
        }
        
        return;
    }
    
    if (list_set) {
        for (i = 0; i < MAX_BOOT_SLOTS; ++i) {
            diag_printf("=====================\n");
            switch (chk_NvData(&cicada_eeprom.bootParms[i].nvHdr)) {
            case 0:
                diag_printf("** EEPROM bootParms[%d]: ok\n", i);
                break;
            case 1:
                diag_printf("** EEPROM bootParms[%d]: header corrupt\n", i);
                break;
            case 2:
                diag_printf("** EEPROM bootParms[%d]: data corrupt\n", i);
                break;
            default:
                diag_printf("** EEPROM bootParms[%d]: corrupt\n", i);
                break;
            }
            
            diag_printf("-script-\n%s\n", cicada_eeprom.bootParms[i].config_data);
            diag_printf("-kernAddr-\n%p\n", (void *)cicada_eeprom.bootParms[i].kernAddr);
            diag_printf("-kernLen-\n%ld\n", cicada_eeprom.bootParms[i].kernLen);
            diag_printf("-kernCrc-\n0x%lx\n", cicada_eeprom.bootParms[i].kernCrc);
        }

        diag_printf("=====================\n");
        switch (chk_NvData(&cicada_eeprom.nvDataShared.nvHdr)) {
        case 0:
            diag_printf("** EEPROM nvDataShared: ok\n");
            break;
        case 1:
            diag_printf("** EEPROM nvDataShared: header corrupt\n");
            break;
        case 2:
            diag_printf("** EEPROM nvDataShared: data corrupt\n");
            break;
        default:
            diag_printf("** EEPROM nvDataShared: corrupt\n");
            break;
        }

        diag_printf("bootGroup:   %ld\n", cicada_eeprom.nvDataShared.bootGroup);
        diag_printf("enetAddr:    %s\n", cicada_eeprom.nvDataShared.enetAddr);
        diag_printf("instrString: %s\n", cicada_eeprom.nvDataShared.instrString);

        return;
    }
    
    if (x_set) {
        if (x >= (MAX_DATA_LEN - 1) || x < 0) {
            diag_printf ("** EEPROM Error: invalid index\n");
            return;
        }
        
        if (val_set) {
            if (str_cmp(name, "kernAddr") == 0) {
                cicada_eeprom.bootParms[x].kernAddr = val;
                cicada_eeprom.bootParms[x].nvHdr.dataType = NVHDR_TYPE_BOOTPARM;
                
                offset = (unsigned int)&cicada_eeprom.bootParms[x].nvHdr - (unsigned int)&cicada_eeprom;
                set_NvData(&cicada_eeprom.bootParms[x].nvHdr, offset);
            } else if (str_cmp(name, "kernLen") == 0) {
                cicada_eeprom.bootParms[x].kernLen = val;
                cicada_eeprom.bootParms[x].nvHdr.dataType = NVHDR_TYPE_BOOTPARM;
                
                offset = (unsigned int)&cicada_eeprom.bootParms[x].nvHdr - (unsigned int)&cicada_eeprom;
                set_NvData(&cicada_eeprom.bootParms[x].nvHdr, offset);
            } else if (str_cmp(name, "kernCrc") == 0) {
                cicada_eeprom.bootParms[x].kernCrc = val;
                cicada_eeprom.bootParms[x].nvHdr.dataType = NVHDR_TYPE_BOOTPARM;
                
                offset = (unsigned int)&cicada_eeprom.bootParms[x].nvHdr - (unsigned int)&cicada_eeprom;
                set_NvData(&cicada_eeprom.bootParms[x].nvHdr, offset);
            } else {
                    diag_printf ("** EEPROM Error: invalid param name\n");
            }
        } else {
            if (str_cmp(name, "script") == 0) {
                // script
                char script[MAX_DATA_LEN];
                char line[256], *sp, *lp;
                int script_len = 0;

                sp = script;
                diag_printf("Enter script, terminate with empty line\n");
                while (true) {
                    diag_printf(">> ");
                    ret = _rb_gets(line, sizeof(line), 0);
                    if (ret < 0) {
                        diag_printf ("** EEPROM invalid entry\n");
                        return;
                    }

                    if (strlen(line) == 0)
                        break;

                    script_len += strlen(line);
                    if (script_len >= (MAX_DATA_LEN - 1)) {
                        diag_printf("** EEPROM Error: script longer than %d not allowed!\n", (MAX_DATA_LEN - 1));
                        return;
                    }

                    lp = line;
                    while (*lp) {
                        *sp++ = *lp++;
                    }
                    *sp = 0;
                }
                strcpy(cicada_eeprom.bootParms[x].config_data, script);
                cicada_eeprom.bootParms[x].nvHdr.dataType = NVHDR_TYPE_BOOTPARM;
                
                offset = (unsigned int)&cicada_eeprom.bootParms[x].nvHdr - (unsigned int)&cicada_eeprom;
                set_NvData(&cicada_eeprom.bootParms[x].nvHdr, offset);
            } else {
                diag_printf ("** EEPROM Error: syntax\n");
            }
        }
        return;
    }

    if (val_set) {
        if (str_cmp(name, "bootGroup") == 0) {
            cicada_eeprom.nvDataShared.bootGroup = val;
            cicada_eeprom.nvDataShared.nvHdr.dataType = NVHDR_TYPE_NVDATA_SHARED;

            offset = (unsigned int)&cicada_eeprom.nvDataShared.nvHdr - (unsigned int)&cicada_eeprom;
            set_NvData(&cicada_eeprom.nvDataShared.nvHdr, offset);
        } else {
            diag_printf ("** EEPROM Error: invalid param name\n");
        }
        
        return;
    }

    if (str_set) {
        if (str_cmp(name, "enetAddr") == 0) {
            if (strlen(str) == ENETADDR_LEN){
                strcpy(cicada_eeprom.nvDataShared.enetAddr, str);
                cicada_eeprom.nvDataShared.nvHdr.dataType = NVHDR_TYPE_NVDATA_SHARED;

                offset = (unsigned int)&cicada_eeprom.nvDataShared.nvHdr - (unsigned int)&cicada_eeprom;
                set_NvData(&cicada_eeprom.nvDataShared.nvHdr, offset);
            } else {
                diag_printf("** EEPROM Error: invalid string, format is xx:xx:xx:xx:xx:xx\n");
            }
        } else if (str_cmp(name, "instrString") == 0) {
            if (strlen(str) < 256){
                strcpy(cicada_eeprom.nvDataShared.instrString, str);
                cicada_eeprom.nvDataShared.nvHdr.dataType = NVHDR_TYPE_NVDATA_SHARED;

                offset = (unsigned int)&cicada_eeprom.nvDataShared.nvHdr - (unsigned int)&cicada_eeprom;
                set_NvData(&cicada_eeprom.nvDataShared.nvHdr, offset);
            } else {
                diag_printf("** EEPROM Error: invalid string\n");
            }
        } else {
            diag_printf ("** EEPROM Error: invalid param name\n");
        }
        return;
    }

    diag_printf("eeconfig\n"
        "Manage configuration kept in EEPROM memory\n"
        "Only one option at a time\n"
        "Options:\n"
        "  -c                                    corrupt the data write\n"
        "  -i                                    initalize\n"
        "  -l                                    list\n"
        "  -t                                    test crc\n"
        "  -h                                    help\n"
        "  -x <index> script                    change script\n"
        "  -x <index> -v <value> <param name>    change parameter value\n"
        "      <param name>: kernAddr | kernLen | kernCrc\n"
        "      Note for kernCrc: 0xdeadbeef disables kernel crc check\n"
        "  -s <string> <param name>              change parameter string\n"
        "      <param name>: enetAddr | instrString\n"
        "  -v <value> <param name>               change parameter value\n"
        "      <param name>: bootGroup\n"
        "\n");
            
    return;
}

static void init_i2c(void)
{
    // 100kHz data rate
    i2c_init(i2c_base_addr[EEPROM_BUSS_NUMBER], 100000);
}

RedBoot_init(init_i2c, RedBoot_INIT_PRIO(120));

#define move_from_unaligned32(v, u32p) (memcpy(&(v), (u32p), 4))

static inline unsigned
read4_little_endian(const unsigned char *cp)
{
	cyg_uint32 v;
	move_from_unaligned32(v, cp);
	return v;
}

static cyg_uint32
get_start_sect(const struct partition *p)
{
	return read4_little_endian(p->start4);
}

static cyg_uint32
get_nr_sects(const struct partition *p)
{
	return read4_little_endian(p->size4);
}

#define pt_offset(b, n) \
	((struct partition *)((b) + 0x1be + (n) * sizeof(struct partition)))

static int valid_part_table_flag(const char *mbuffer)
{
	return (mbuffer[510] == 0x55 && (unsigned char)mbuffer[511] == 0xaa);
}

static int is_cleared_partition(const struct partition *p)
{
    /* We consider partition "cleared" only if it has only zeros */
    const char *cp = (const char *)p;
    int cnt = sizeof(*p);
    char bits = 0;
    while (--cnt >= 0)
	    bits |= *cp++;
    return (bits == 0);
}

static int get_sectors(void)
{
//    char MBRbuffer[DEFAULT_SECTOR_SIZE * 2];
    cyg_uint32 *dst = (cyg_uint32 *)MBRbuffer;
    unsigned long src = 0;
    struct partition *part_table;
    int i;

    numSectors = 0;
    startSector = 0;

    // read the MBR
    mmc_data_read(dst, DEFAULT_SECTOR_SIZE, src);
    
    if (!valid_part_table_flag(MBRbuffer)) {
        diag_printf("** ERROR Invalid partition table\n");
        return -1;
    }
    
    for (i = 0; i < 4; ++i) {
        part_table = pt_offset(MBRbuffer, i);
        
        if (is_cleared_partition(part_table)) {
            continue;
        }
        
        if (part_table->sys_ind == 0) {
            break;
        }
    }
    
    if (i == 4) {
        diag_printf("** ERROR Kernel partition not found\n");
        return -1;
    }
    
    numSectors = get_nr_sects(part_table);
    startSector = get_start_sect(part_table);    

    diag_printf("** MMC Kernel partition %d, sector start %lu end %lu\n", i + 1,
        (unsigned long)startSector, (unsigned long)(startSector + numSectors));
        
    return 0;
}

extern int mmcflash_hwr_init(void);
static void mmc_copy_init(void)
{
    static bool inited = false;
    
    if (!inited) {
        mmcflash_hwr_init();
        inited = true;
        
        get_sectors();
    }
}

RedBoot_cmd("linux",
            "run a linux kernel",
            "-h",
            exec_linux
    );
 
//
// Attempt to get configuration information from the eeprom.
// If available (i.e. good checksum, etc), initialize "known"
// values for later use.
//
void exec_linux(int argc, char *argv[])
{
    int i, j, k;
    unsigned long len;
    int grpSlot = MAX_BOOT_SLOTS;
    int bootSlot = MAX_BOOT_SLOTS;
    unsigned int offset;
    int rescueMode = 0;
    
    enum {
        SLOT_NORMAL = 0,
        SLOT_RESCUE,
        SLOT_FALLBACK,
        SLOT_AH_CRAP,
        SLOT_DESPERATE,
    } status = SLOT_NORMAL;
    
    char normalStr[] = "";
    char rescueStr[] = " cicada_rescue";
    char fallbackStr[] = " cicada_fallback";
    char ahCrapStr[] = " cicada_group0";
    char desperateStr[] = " cicada_desperate";
    char *statusStr = NULL;
    
    rescueMode = check_keys();
    
    // get and check shared nv data
    for (i = 0; i < 2; ++i) {
        offset = (unsigned int)&cicada_eeprom.nvDataShared - (unsigned int)&cicada_eeprom;
        rw_Data((char *)&cicada_eeprom.nvDataShared, offset, sizeof(struct NvDataShared), EEPROM_READ);
        if (!chk_NvData(&cicada_eeprom.nvDataShared.nvHdr)) {
            grpSlot = cicada_eeprom.nvDataShared.bootGroup * BOOT_SLOTS_PER_GROUP;
            break;
        } else {
            diag_printf("** ERROR Boot slot %d, nvDataShared corrupt\n", i + 1);
        }
    }
    
    if (i == 2) {
        // nvDataShared corrupt
        // try mantis boot slots
        status = SLOT_AH_CRAP;
        grpSlot = 0;
    }
    
    i = grpSlot;
    
    if (status == SLOT_NORMAL && rescueMode) {
        status = SLOT_RESCUE;
	i++;
    }
    
    j = 0;
    while (1) {
        // try 2 times to read eeprom
        for (k = 0; k < 2; ++k) {
            offset = (unsigned int)&cicada_eeprom.bootParms[i] - (unsigned int)&cicada_eeprom;
            rw_Data((char *)&cicada_eeprom.bootParms[i], offset, sizeof(struct BootParm), EEPROM_READ);
            if (!chk_NvData(&cicada_eeprom.bootParms[i].nvHdr)) {
                bootSlot = i;
                break;
            }
        }
        
        if (bootSlot < MAX_BOOT_SLOTS) {
            // found good boot parameters
            unsigned long crc;
            if (bootSlot < 2) {
                // load kernel from flash
                unsigned long src = cicada_eeprom.bootParms[bootSlot].kernAddr;
                unsigned long dst = SDRAM_LOAD_ADDR;
                unsigned long length = (cicada_eeprom.bootParms[bootSlot].kernLen + 3) & ~3;
                unsigned long end = src + length;

                // length > 1MB & <= MAX_KERNEL_SIZE
                // src > end of bootloader & end < end of kernel space
                if (length > 0x100000 && length <= MAX_KERNEL_SIZE &&
                    src > 0xa0040000 && end < 0xa0d80000) {
                    while (src != end) {
	                *(unsigned long *)dst = *(unsigned long *)src;
	                src += 4;
	                dst += 4;
	            }
                } else {
                    diag_printf("** ERROR Boot slot %d invalid kernel parameter\n", bootSlot);
                    bootSlot = MAX_BOOT_SLOTS;
                }
            } else {
                // load kernel from mmc
                // kernAddr is a sector offset from partition, 1 sector = 512B
                // src (bytes)
                unsigned long src = cicada_eeprom.bootParms[bootSlot].kernAddr * DEFAULT_SECTOR_SIZE;
                unsigned long dst = SDRAM_LOAD_ADDR;
                // length (bytes)
                unsigned long length = cicada_eeprom.bootParms[bootSlot].kernLen;
                
                mmc_copy_init();
                
                // length > 1MB & length < MAX_KERNEL_SIZE
                if (length > 0x100000 && length < MAX_KERNEL_SIZE &&
                    numSectors > 0 &&
                    (src + length) < (numSectors * DEFAULT_SECTOR_SIZE)) {
                    diag_printf("** Reading Linux kernel from MMC to SDRAM...\n");

                    // length (bytes), src (bytes)
                    src += startSector * DEFAULT_SECTOR_SIZE;
                    mmc_data_read((cyg_uint32 *)dst, length, src);
                } else {
                    diag_printf("** ERROR Boot slot %d invalid kernel parameter\n", bootSlot);
                    bootSlot = MAX_BOOT_SLOTS;
                }
            }

            if (bootSlot < MAX_BOOT_SLOTS) {
                // check the crc
                crc = cyg_crc32((unsigned char *)0x00800000, cicada_eeprom.bootParms[bootSlot].kernLen);
                if (cicada_eeprom.bootParms[bootSlot].kernCrc == 0xdeadbeef) {
                    diag_printf("** INFO Boot slot %d, kernel crc ignored, crc is 0x%lx\n", bootSlot, crc);
                    break;
                }
                if (cicada_eeprom.bootParms[bootSlot].kernCrc != crc) {
                    // error
                    diag_printf("** ERROR Boot slot %d, kernel crc 0x%lx (actual) != 0x%lx (stated)\n", bootSlot, crc, cicada_eeprom.bootParms[bootSlot].kernCrc);
                } else {
                    // boot slot ok
                    break;
                }
            }
        } else {
            diag_printf("** ERROR Boot slot %d, parameter corrupt\n", i);
        }

        // fail to get requested slot
        if (status == SLOT_RESCUE) {
            status = SLOT_FALLBACK;
        } else if (status == SLOT_NORMAL) {
            status = SLOT_RESCUE;
        }
        
        if (++j == BOOT_SLOTS_PER_GROUP) {
            // all group slots failed
            if (grpSlot == 0) {
                status = SLOT_DESPERATE;
                
                // try hardcoded boot parms
                diag_printf("**** Trying to boot fixed location *****\n");
#if 0
                // productin code                
                strcpy(bootline,
                    "mcopy -s 0xa0060000 -d 0x00800000 -l 0x00600000\n"
                    "exec -b 0x00800000 -l 0x00600000 -c \""
                    "console=ttymxc0 "
                    "rootfstype=ubifs ubi.mtd=6 ro root=ubi0_0 "
                    "logo_index=1 cicada_desperate\"\n");
#else
                // factory code                
                strcpy(bootline,
                    "mcopy -s 0xa0060000 -d 0x00800000 -l 0x00600000\n"
                    "exec -b 0x00800000 -l 0x00600000 -c \""
                    "console=ttymxc0 "
                    "rootfstype=ext3 root=/dev/sda1 rootwait "
                    "logo_index=1 cicada_factory redboot=");
                j = strlen(bootline);
                strcpy(&bootline[j], fnet_version);
                j = strlen(bootline);
                strcpy(&bootline[j], "\"\n");
#endif
                entry_address = 1; // just to squelch an exec warning
                script_timeout = 1; // minimium, 10 msec
                script = (unsigned char *)bootline;
                return;
            }
            
            // try the mantis slots
            status = SLOT_AH_CRAP;
            grpSlot = 0;
            i = 0;
            j = 0;
            continue;
        }
        
        if (++i == (grpSlot + BOOT_SLOTS_PER_GROUP)) {
            // wrap around and try next slot
            i = grpSlot;
        }
    }

    diag_printf("** Booting slot %d\n", bootSlot);
    
    //-----------------------------
    // assemble the boot command
    strcpy(&bootline[0], "exec -b 0x00800000 -l 0xa00000 -c \"");
    len = 35;
    
    // add the kernel boot line
    j = strlen(cicada_eeprom.bootParms[bootSlot].config_data);
    strcpy(&bootline[len], cicada_eeprom.bootParms[bootSlot].config_data);
    len += j;
    
    // add the ethernet address
    j = strlen(cicada_eeprom.nvDataShared.enetAddr);
    if (j == ENETADDR_LEN) {
        if ((len + 9 + ENETADDR_LEN + 3) < MAX_DATA_LEN) {
            strcpy(&bootline[len], " fec_mac=");
            len += 9;
            strcpy(&bootline[len], cicada_eeprom.nvDataShared.enetAddr);
            len += ENETADDR_LEN;
        } else {
            diag_printf("** WARNING Boot line too long for enet address string.\n");
        }
    }

    // add redboot version
    j = strlen(fnet_version);
    if (j > 0) {
        if ((len + 9 + j + 3) < MAX_DATA_LEN) {
            strcpy(&bootline[len], " redboot=");
            len += 9;
            strcpy(&bootline[len], fnet_version);
            len += j;
        } else {
            diag_printf("** WARNING Boot boot line too long for bootloader version string.\n");
        }
    }

    // add boot status
    switch (status) {
    case SLOT_NORMAL:
        statusStr = normalStr;
        break;
    case SLOT_RESCUE:
        statusStr = rescueStr;
        break;
    case SLOT_FALLBACK:
        statusStr = fallbackStr;
        break;
    case SLOT_AH_CRAP:
        statusStr = ahCrapStr;
        break;
    case SLOT_DESPERATE:
        statusStr = desperateStr;
        break;
    }
    j = strlen(statusStr);
    if (j > 0) {
        if ((len + j + 3) < MAX_DATA_LEN) {
            strcpy(&bootline[len], statusStr);
            len += j;
        } else {
            diag_printf("** WARNING Boot boot line too long for status string.\n");
        }
    }

    // terminate string
    strcpy(&bootline[len], "\"\n");
  
    entry_address = 1; // just to squelch an exec warning
    script_timeout = 1; // minimium, 10 msec
    script = (unsigned char *)bootline;

    return;
}

// EOF econfig.c

