#ifndef CYGONCE_HAL_PLATFORM_SETUP_H
#define CYGONCE_HAL_PLATFORM_SETUP_H

//=============================================================================
//
//      hal_platform_setup.h
//
//      Platform specific support for HAL (assembly code)
//
//=============================================================================
//####ECOSGPLCOPYRIGHTBEGIN####
// -------------------------------------------
// This file is part of eCos, the Embedded Configurable Operating System.
// Copyright (C) 1998, 1999, 2000, 2001, 2002 Red Hat, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 or (at your option) any later version.
//
// eCos is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
// for more details.
//
// You should have received a copy of the GNU General Public License along
// with eCos; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. However the source code for this file must still be made available
// in accordance with section (3) of the GNU General Public License.
//
// This exception does not invalidate any other reasons why a work based on
// this file might be covered by the GNU General Public License.
//
// Alternative licenses for eCos may be arranged by contacting Red Hat, Inc.
// at http://sources.redhat.com/ecos/ecos-license/
// -------------------------------------------
//####ECOSGPLCOPYRIGHTEND####
//===========================================================================

#include <pkgconf/system.h>             // System-wide configuration info
#include CYGBLD_HAL_VARIANT_H           // Variant specific configuration
#include CYGBLD_HAL_PLATFORM_H          // Platform specific configuration
#include <cyg/hal/hal_soc.h>            // Variant specific hardware definitions
#include <cyg/hal/hal_mmu.h>            // MMU definitions
#include <cyg/hal/fsl_board.h>          // Platform specific hardware definitions

#if defined(CYG_HAL_STARTUP_ROM) || defined(CYG_HAL_STARTUP_ROMRAM)
#define PLATFORM_SETUP1 _platform_setup1

//#define BOOT_FROM_MMC

#if defined(BOOT_FROM_MMC)
#define PLATFORM_PREAMBLE setup_flash_header
//#define MEMORY_MDDR_ENABLE
#endif

#define CYGHWR_HAL_ARM_HAS_MMU

#ifdef CYG_HAL_STARTUP_ROMRAM
#define CYGSEM_HAL_ROM_RESET_USES_JUMP
#endif

#define CYGHWR_HAL_ROM_VADDR    0x0

#define UNALIGNED_ACCESS_ENABLE
#define LOW_INT_LATENCY_ENABLE
#define BRANCH_PREDICTION_ENABLE

#if 0
#define SET_T_BIT_DISABLE
#define TURN_OFF_IMPRECISE_ABORT
#endif

/* CS0_CSCRx Defines for FNET Mantis Board, ASYNC Mode ROM Access 
 *  U = 0x0000_D403, L = 0x5022_0D01, A = 0x0022_0000                      */
#define CS0_CSCRU_CIC_ASYNC (CSCRU_CNC(3) + CSCRU_WSC(18) + CSCRU_EDC(3) )
#define CS0_CSCRL_CIC_ASYNC (CSCRL_OEA(5) + CSCRL_EBWA(2) + CSCRL_EBWN(2) + \
    CSCRL_EBC + CSCRL_DSZ(5) + CSCRL_CSEN )
#define CS0_CSCRA_CIC_ASYNC (CSCRA_RWA(2) + CSCRA_RWN(2) )

/* CS0_CSCRx Defines for FNET Mantis Dev Board, Page Mode ROM Access 
 *  U = 0x0076_D203, L = 0x5022_0D01, A = 0x0022_0000                      */
#define CS0_CSCRU_CIC_PAGE  (CSCRU_PSZ(1) + CSCRU_PME + CSCRU_SYNC + \
    CSCRU_DOL(6) + CSCRU_CNC(3) + CSCRU_WSC(18) + CSCRU_EDC(3) )
#define CS0_CSCRL_CIC_PAGE  (CSCRL_OEA(5) + CSCRL_EBWA(2) + CSCRL_EBWN(2) + \
    CSCRL_EBC + CSCRL_DSZ(5) + CSCRL_CSEN )
#define CS0_CSCRA_CIC_PAGE  (CSCRA_RWA(2) + CSCRA_RWN(2) )

// This macro represents the initial startup code for the platform
    .macro  _platform_setup1
FSL_BOARD_SETUP_START:
/*
 *       ARM1136 init
 *       - invalidate I/D cache/TLB and drain write buffer;
 *       - invalidate L2 cache
 *       - unaligned access
 *       - branch predictions
 */
#ifdef TURN_OFF_IMPRECISE_ABORT
    mrs r0, cpsr
    bic r0, r0, #0x100
    msr cpsr, r0
#endif

    mrc 15, 0, r1, c1, c0, 0
    bic r1, r1, #(0x3<<21)
    bic r1, r1, #(0x3<<11)
    bic r1, r1, #0x5

#ifndef BRANCH_PREDICTION_ENABLE
    mrc 15, 0, r0, c1, c0, 1
    bic r0, r0, #7
    mcr 15, 0, r0, c1, c0, 1
#else
    mrc 15, 0, r0, c1, c0, 1
    orr r0, r0, #7
    mcr 15, 0, r0, c1, c0, 1
    orr r1, r1, #(1<<11)
#endif

#ifdef UNALIGNED_ACCESS_ENABLE
    orr r1, r1, #(1<<22)
#endif

#ifdef LOW_INT_LATENCY_ENABLE
    orr r1, r1, #(1<<21)
#endif
    mcr 15, 0, r1, c1, c0, 0

#ifdef BRANCH_PREDICTION_ENABLE
    mov r0, #0
    mcr 15, 0, r0, c15, c2, 4
#endif

    mov r0, #0
    mcr 15, 0, r0, c7, c7, 0        /* invalidate I cache and D cache */
    mcr 15, 0, r0, c8, c7, 0        /* invalidate TLBs */
    mcr 15, 0, r0, c7, c10, 4       /* Drain the write buffer */

    /* Also setup the Peripheral Port Remap register inside the core */
    ldr r0, ARM_PPMRR        /* start from AIPS 2GB region */
    mcr p15, 0, r0, c15, c2, 4

    /*** L2 Cache setup/invalidation/disable ***/
    /* Disable L2 cache first */
    mov r0, #L2CC_BASE_ADDR
    ldr r2, [r0, #L2_CACHE_CTL_REG]
    bic r2, r2, #0x1
    str r2, [r0, #L2_CACHE_CTL_REG]
    /*
     * Configure L2 Cache:
     * - 128k size(16k way)
     * - 8-way associativity
     * - 0 ws TAG/VALID/DIRTY
     * - 4 ws DATA R/W
     */
    ldr r1, [r0, #L2_CACHE_AUX_CTL_REG]
    and r1, r1, #0xFE000000
    ldr r2, L2CACHE_PARAM
    orr r1, r1, r2
    str r1, [r0, #L2_CACHE_AUX_CTL_REG]
    ldr r1, ROM_VER_ADDR_W
    ldr r2, [r1]
    cmp r2, #0x1
    // if version 1 silicon
    /* Workaournd for DDR issue: force write through cache behavior*/
    ldreq r1, [r0, #L2_CACHE_DBG_CTL_REG]
    orreq r1, r1, #2
    streq r1, [r0, #L2_CACHE_DBG_CTL_REG]

    /* Invalidate L2 */
    mov r1, #0x000000FF
    str r1, [r0, #L2_CACHE_INV_WAY_REG]
L2_loop:
    /* Poll Invalidate By Way register */
    ldr r2, [r0, #L2_CACHE_INV_WAY_REG]
    cmp r2, #0
    bne L2_loop
    /*** End of L2 operations ***/

#if defined(BOOT_FROM_MMC)
    mov r0, #MMC_FLASH_BOOT
    ldr r1, AVIC_VECTOR0_ADDR_W
    str r0, [r1] // if MMC is selected, set MXCFIS_FLAG_REG
#else
    mov r0, #SDRAM_NON_FLASH_BOOT
    ldr r1, AVIC_VECTOR0_ADDR_W
    str r0, [r1] // for checking boot source from nand, nor or sdram
#endif
/*
 * End of ARM1136 init
 */

    /* Do this early for NOR boot. */
init_dbgo_start:
    init_wdogo
    init_usbh2
    init_dbgo
    dbgo_off TP23  // modifies r0, r1, r2
    dbgo_off TP24  // modifies r0, r1, r2
    dbgo_off TP26  // modifies r0, r1, r2

init_spba_start:
    init_spba

init_aips_start:
    /* AIPS setup */
    init_aips
init_max_start:
    /* MAX (Multi-Layer AHB Crossbar Switch) setup */
    init_max
init_m3if_start:
    /* Configure M3IF registers */
    init_m3if

#ifndef BOOT_FROM_MMC
    /* If SDRAM has been setup, bypass clock/WEIM setup */
    cmp pc, #SDRAM_BASE_ADDR
    blo init_clock_start
    cmp pc, #(SDRAM_BASE_ADDR + SDRAM_SIZE)
    blo HWInitialise_skip_SDRAM_setup

    mov r0, #NOR_FLASH_BOOT
    ldr r1, AVIC_VECTOR0_ADDR_W
    str r0, [r1]
#endif

    dbgo_on TP23  // modifies r0, r1, r2

init_clock_start:
    init_clock
    dbgo_off TP23  // modifies r0, r1, r2

#ifndef BOOT_FROM_MMC
/* TODO::
 * Use default setting is more stable then the settinig from IC team for EINCE
 */
init_iomuxc_start:
    init_iomuxc
    init_hw_pins
    dbgo_on TP23  // modifies r0, r1, r2

init_cs0_start:
    // the parameter to init_cs0 gets prepended with
    // CS0_CSCRU_LOC_ and CS0_CSCRL_LOC_ and CS0_CSCRA_LOC_ within the macro
    // ex. CS0_CSCRU_LOC_PAGE
//    init_cs0 PAGE   // Spansion & Numonyx use different page sizes so don't
                    // trust page mode to work until some testing can be done.
    init_cs0 ASYNC  // use Spansion & Numonyx ASYNC mode
    dbgo_off TP23  // modifies r0, r1, r2

init_sdram_start:
    dbgo_on TP24  // modifies r0, r1, r2
    setup_sdram
    cmp r3, #0x0
    bne HWInitialise_skip_SDRAM_setup
    dbgo_off TP24  // modifies r0, r1, r2
#endif

HWInitialise_skip_SDRAM_setup:
    dbgo_on TP26  // modifies r0, r1, r2

#ifdef CYG_HAL_STARTUP_ROMRAM     /* enable running from RAM */
    /* Copy image from flash to SDRAM first */
    ldr r0, =0xFFFFF000
    and r0, r0, pc
    ldr r1, MXC_REDBOOT_ROM_START
    cmp r0, r1        // Skip copy if already in high RAM
    beq HWInitialise_skip_SDRAM_copy

    add r2, r0, #REDBOOT_IMAGE_SIZE

1:  ldmia r0!, {r3-r10}
    stmia r1!, {r3-r10}
    cmp r0, r2
    ble 1b
    /* Jump to SDRAM */

    ldr r1, =0xFFFF
    and r0, pc, r1        /* offset of pc */
    ldr r1, =(SDRAM_BASE_ADDR + SDRAM_SIZE - 0x100000 + 0x8)
    add pc, r0, r1
    nop
    nop
    nop
    nop
#endif /* CYG_HAL_STARTUP_ROMRAM */

HWInitialise_skip_SDRAM_copy:

/*
 * Note:
 *     IOMUX setup is done in C function plf_hardware_init() for simplicity
 */

STACK_Setup:
    // Set up a stack [for calling C code]
    ldr r1, =__startup_stack
    ldr r2, =SRAM_BASE_ADDR
    orr sp, r1, r2

    // Create MMU tables
    bl hal_mmu_init

    // Enable MMU
    ldr r2, =10f
    mrc MMU_CP, 0, r1, MMU_Control, c0      // get c1 value to r1 first
    orr r1, r1, #7                          // enable MMU bit
    orr r1, r1, #0x800                      // enable z bit
    mcr MMU_CP, 0, r1, MMU_Control, c0
    mov pc,r2    /* Change address spaces */
    nop
    nop
    nop
10:

    // Save shadow copy of BCR, also hardware configuration
    ldr r1, =_board_BCR
    str r2, [r1]
    ldr r1, =_board_CFG
    str r9, [r1]                // Saved far above...

    dbgo_off TP26  // modifies r0, r1, r2

    .endm                       // _platform_setup1

#else // defined(CYG_HAL_STARTUP_ROM) || defined(CYG_HAL_STARTUP_ROMRAM)
#define PLATFORM_SETUP1
#endif

    /* Do nothing */
    .macro  init_spba
    .endm  /* init_spba */

    /* AIPS setup - Only setup MPROTx registers. The PACR default values are good.*/
    .macro init_aips
        /*
         * Set all MPROTx to be non-bufferable, trusted for R/W,
         * not forced to user-mode.
         */
        ldr r0, AIPS1_CTRL_BASE_ADDR_W
        ldr r1, AIPS1_PARAM_W
        str r1, [r0, #0x00]
        str r1, [r0, #0x04]
        ldr r0, AIPS2_CTRL_BASE_ADDR_W
        str r1, [r0, #0x00]
        str r1, [r0, #0x04]

        /*
         * Clear the on and off peripheral modules Supervisor Protect bit
         * for SDMA to access them. Did not change the AIPS control registers
         * (offset 0x20) access type
         */
        ldr r0, AIPS1_CTRL_BASE_ADDR_W
        ldr r1, =0x0
        str r1, [r0, #0x40]
        str r1, [r0, #0x44]
        str r1, [r0, #0x48]
        str r1, [r0, #0x4C]
        ldr r1, [r0, #0x50]
        and r1, r1, #0x00FFFFFF
        str r1, [r0, #0x50]

        ldr r0, AIPS2_CTRL_BASE_ADDR_W
        ldr r1, =0x0
        str r1, [r0, #0x40]
        str r1, [r0, #0x44]
        str r1, [r0, #0x48]
        str r1, [r0, #0x4C]
        ldr r1, [r0, #0x50]
        and r1, r1, #0x00FFFFFF
        str r1, [r0, #0x50]
    .endm /* init_aips */

    /* MAX (Multi-Layer AHB Crossbar Switch) setup */
    .macro init_max
        ldr r0, MAX_BASE_ADDR_W
        /* MPR - priority is M4 > M2 > M3 > M5 > M0 > M1 */
        ldr r1, MAX_PARAM1
        str r1, [r0, #0x000]        /* for S0 */
        str r1, [r0, #0x100]        /* for S1 */
        str r1, [r0, #0x200]        /* for S2 */
        str r1, [r0, #0x300]        /* for S3 */
        str r1, [r0, #0x400]        /* for S4 */
        /* SGPCR - always park on last master */
        ldr r1, =0x10
        str r1, [r0, #0x010]        /* for S0 */
        str r1, [r0, #0x110]        /* for S1 */
        str r1, [r0, #0x210]        /* for S2 */
        str r1, [r0, #0x310]        /* for S3 */
        str r1, [r0, #0x410]        /* for S4 */
        /* MGPCR - restore default values */
        ldr r1, =0x0
        str r1, [r0, #0x800]        /* for M0 */
        str r1, [r0, #0x900]        /* for M1 */
        str r1, [r0, #0xA00]        /* for M2 */
        str r1, [r0, #0xB00]        /* for M3 */
        str r1, [r0, #0xC00]        /* for M4 */
        str r1, [r0, #0xD00]        /* for M5 */
    .endm /* init_max */

    /* Clock setup */
    .macro    init_clock
        ldr r0, CCM_BASE_ADDR_W

        /* default CLKO to the 24MHz clock used for PLL reference*/
        /*    lower two bytes should be 0x0021 when done */
        ldr r1, [r0, #CLKCTL_COSR]
        bic r1, r1, #0x00000FF00
        bic r1, r1, #0x0000000FF
        mov r2, #0x21
        orr r1, r1, r2
        str r1, [r0, #CLKCTL_COSR]

        /* TBD: RAMW set to "reserved" state and LPM set to */
        /* "wait mode" although no WFI instructions in RedBoot */
        ldr r2, CCM_CCMR_W
        str r2, [r0, #CLKCTL_CCMR]

        /*check clock path*/
        ldr r3, ROM_VER_ADDR_W
        ldr r4, [r3]
        cmp r4, #0x1
        ldreq r2, [r0, #CLKCTL_PDR0]
        movne r2, #0x1
        tst r2, #0x1
        ldrne r3, MPCTL_PARAM_532_W  /* consumer path*/
        ldreq r3, MPCTL_PARAM_399_W  /* auto path*/

        /*Set core PLL frequency*/
        str r3, [r0, #CLKCTL_MPCTL]

        /*Set peripheral PLL frequency*/
        ldr r1, PPCTL_PARAM_W
        str r1, [r0, #CLKCTL_PPCTL]

        /*Set arm clock and ahb clock dividers*/
        cmp   r4, #0x1
        ldreq r1, [r0, #CLKCTL_PDR0]
        orreq r1, r1, #0x00800000
        streq r1, [r0, #CLKCTL_PDR0]
        tst   r2, #0x1
        ldrne r1, CCM_PDR0_532_W
        ldreq r1, CCM_PDR0_399_W
        cmp   r4, #0x1
        bicne r1, r1, #0x00800000
        str   r1, [r0, #CLKCTL_PDR0]

	/*Enable clock to EPIT1 module -- note reset default*/
	/*value is for this to already be enabled */
        ldr r1, [r0, #CLKCTL_CGR0]
        orr r1, r1, #0x00300000
        str r1, [r0, #CLKCTL_CGR0]

	/*Enable clocks to I2C1 and FEC modules -- note reset*/
	/*default value is for this to already be enabled */
        ldr r1, [r0, #CLKCTL_CGR1]
        orr r1, r1, #0x00000C00
        orr r1, r1, #0x00000003
        str r1, [r0, #CLKCTL_CGR1]
    .endm /* init_clock */

    /* M3IF setup */
    .macro init_m3if
        /* Configure M3IF registers */
        ldr r1, M3IF_BASE_W
        /*
        * M3IF Control Register (M3IFCTL)
        * MRRP[0] = L2CC0 not on priority list (0 << 0)  = 0x00000000
        * MRRP[1] = MAX1 not on priority list (0 << 0)   = 0x00000000
        * MRRP[2] = L2CC1 not on priority list (0 << 0)  = 0x00000000
        * MRRP[3] = USB  not on priority list (0 << 0)   = 0x00000000
        * MRRP[4] = SDMA not on priority list (0 << 0)   = 0x00000000
        * MRRP[5] = GPU not on priority list (0 << 0)    = 0x00000000
        * MRRP[6] = IPU1 on priority list (1 << 6)       = 0x00000040
        * MRRP[7] = IPU2 not on priority list (0 << 0)   = 0x00000000
        *                                                ------------
        *                                                  0x00000040
        */
        ldr r0, =0x00000040
        str r0, [r1]  /* M3IF control reg */
    .endm /* init_m3if */

    /* CS0 mode setup */
    .macro init_cs0 mode
        ldr r0, WEIM_CTRL_CS0_W     // R0 to base address  for CS0
        ldr r1, CS0_CSCRU_LOC_\mode   // Cicada Dev Board, PAGE or ASYNC mode
        str r1, [r0, #CSCRU]
        ldr r1, CS0_CSCRL_LOC_\mode   // Cicada Dev Board, PAGE or ASYNC mode
        str r1, [r0, #CSCRL]
        ldr r1, CS0_CSCRA_LOC_\mode   // Cicada Dev Board, PAGE or ASYNC mode
        str r1, [r0, #CSCRA]
    .endm /* init_cs0 mode */


    .macro setup_sdram
        get_hwid  // modifies r0, r1
        // if hwid (r0) == 1 then ddr_128
        cmp     r0, #1
        beq     ddr_128

        get_hwrev  // modifies r0
        // if hwrev (r0) < 2 then ddr_128
        cmp     r0, #1
	bhi     ddr_256
	
ddr_128:
        dbgo_on TP23	// modifies r0, r1, r2
	mov r5, #0	// LPDDR size is 128Mb
	b ddr_common

ddr_256:
        dbgo_on TP26  	// modifies r0, r1, r2
	mov r5, #1	// LPDDR size is 256Mb

ddr_common:
        // disable enhanced SDRAM controller, and set row refresh
        // rate one row per refresh clock and 2048 rows every 64ms
        // (i.e. 31.25usec per row)
        ldr r0, ESDCTL_BASE_W
        mov r3, #ESDCTL_SREFR(ESD_SREFR_1_2048)
        str r3, [r0, #ESDCTL_ESDCTL0]
        ldr r2, ROM_VER_ADDR_W
        ldr r4, [r2]
        cmp r4, #0x1
        streq r3, [r0, #ESDCTL_ESDCTL1]

        // r1 == pointer to start of mobile DDR
        ldr r0, ESDCTL_BASE_W
        mov r1, #SRAM_BASE_ADDR

        // ESDCFG <- SDRAM configuration
        ldr r3, ESDCFG_CONFIG
        str r3, [r0, #ESDCTL_ESDCFG0]

        // ESDMISC <- soft reset, MDDR enable and LPDDR dly line soft reset
        mov r3, #(ESDMISC_RST | ESDMISC_MDDREN | ESDMISC_MDDR_DL_RST)
        str r3, [r0, #ESDCTL_ESDMISC]
        // clear reset bits leaving only MDDR enable active
        bic r3, r3, #(ESDMISC_RST | ESDMISC_MDDR_DL_RST)
        str r3, [r0, #ESDCTL_ESDMISC]

        // Wait for 200us SDRAM/LPDDR external memory wakeup period
0:        ldr  r3, [r0, #ESDCTL_ESDMISC]
        ands r3, r3, #ESDMISC_SDRAMDRY
        beq  0b

        // Hold for more than 200ns
	// NOTE: executing out of flash at this time so 200ns is a very short
	// counted loop.
        mov r3, #20
1:        subs r3, r3, #0x1
        bne 1b

        // ESDCTL <- Enable Precharge Command
        cmp   r5, #0
        ldreq r3, ESDCTL_PRECHARGE_128
        ldrne r3, ESDCTL_PRECHARGE_256
        str   r3, [r0, #ESDCTL_ESDCTL0]
        
        // Precharge all banks (requires driving A10 high on mobile DDR
        // part so that all banks get precharged)
        ldr r4, RAM_PARAM1_MDDR
        mov r3, #0xDA
        strb r3, [r1, r4]

        // ESDCTL <- Enable Auto Refresh Command
        cmp   r5, #0
        ldreq r3, ESDCTL_AUTO_REFRESH_128
        ldrne r3, ESDCTL_AUTO_REFRESH_256
        str r3, [r0, #ESDCTL_ESDCTL0]
        mov r3, #0xDA
        
        // auto refresh 8 times
        mov r4, #0x8
2:        strb r3, [r1]
        subs r4, r4, #1
        bne 2b

        // ESDCTL <- Enable Load Mode Register Command
        cmp   r5, #0
        ldreq r3, ESDCTL_LOAD_MODE_128
        ldrne r3, ESDCTL_LOAD_MODE_256
        str r3, [r0, #ESDCTL_ESDCTL0]
        
        // CAS Latency, burst length
        ldr r4, RAM_PARAM6_MDDR
        mov r3, #0xDA
        strb r3, [r1, r4]
        
        // init LPDDR extended mode register
        ldr r4, EMRS_MDDR_2QTR		// RAM_PARAM3_MDDR
        strb r3, [r1, r4]

        // ESDCTL <- final, activate normal mode operation
        cmp   r5, #0
        ldreq r3, ESDCTL_NORMAL_128
        ldrne r3, ESDCTL_NORMAL_256
        str r3, [r0, #ESDCTL_ESDCTL0]

        // short delay before read / write test
        mov r4, #10
3:        subs r4, r4, #1
        bne 3b

        // test if memory works
        str r3, [r1, #0x100]
	ldr r4, DEADBEEF_W
        str r4, [r1, #0x104]
        ldr r4, [r1, #0x100]
        cmp r3, r4
        movne r3, #1 /* return 1 on mem check fail */
        moveq r3, #0 /* return 0 on mem check pass */

        dbgo_off TP23  // modifies r0, r1, r2
        dbgo_off TP26  // modifies r0, r1, r2

#if 0
        // this may be specific to the 3stack pdk
        ldr r3, ESDCTL_DELAY5
        str r3, [r0, #0x30]
#endif
    .endm
    
    /* To support 133MHz mobile DDR.  Assume high drive strength,*/
    /* pull/keeper enabled (where available), no hysterisis, and */
    /* 1.8v drive voltage */
    .macro  init_iomuxc
        ldr r0, =0x000020E2
        ldr r1, IOMUXC_BASE_ADDR_W
        add r1, r1, #0x368
        add r2, r1, #0x4C8 - 0x368
1:      str r0, [r1], #4
        cmp r1, r2
        ble 1b
    .endm /* init_iomuxc */


#define PLATFORM_VECTORS         _platform_vectors
    .macro  _platform_vectors
        .globl  _board_BCR, _board_CFG
_board_BCR:   .long   0       // Board Control register shadow
_board_CFG:   .long   0       // Board Configuration (read at RESET)
    .endm

    /* IO Mux control and data for WDOG_RST_B outputs */
    .macro init_wdogo
        // initialize GPIO1-6 direction and data for output low to
	// signal that bootloader is starting to run.  when the
	// internal watchdog timer goes off on a RedBoot "reset" command
	// or a Linux "reboot" command, this will reset the IOMUX+GPIO
	// causing a low to high transition to happen on the WDOG_RST_B
	// pin so the PIC micro can detect this and force a hard system
	// reset to happen.
        ldr r0, =GPIO1_BASE_ADDR  // R0 to base address  for GPIO 1
        ldr r1, [r0]              
        bic r1, r1, #0x00000040   // init data reg for low output
        str r1, [r0]              // write Data Register

        ldr r1, [r0, #0x4]        // read Direction Register
        orr r1, r1, #0x00000040   // make it an output
        str r1, [r0, #0x04]       // write Direction Register

        // set IOMUX to route the GPIO out to the test points
        ldr r0, =IOMUXC_BASE_ADDR // R0 to base address for IO Mux Control
        mov r1, #0x05             // GPIO is alt mode 5
        str r1, [r0, #0x00C]      // WDOG_RST_B as GPIO
    .endm /* init_wdogo */

    /* Grab IO to external ULPI phy.  Workaround for mysterious
    interaction between input on console serial port prior to initialization
    of ULPI phy interface in Linux that causes reading of the ULPI phy ID to
    never complete thus hanging the Linux boot indefinitely. Unless we do
    this, pull-up/downs in the Phy conflict with pull-up/downs in i.MX35 and
    some I/O sit at about 1.0 volts until the Linux GPIO init is done for the
    Phy interface. */
    .macro init_usbh2
        ldr r0, =0x00000001       // MUX_CTL ALT1 for NFC pins -> USBH2
	ldr r3, =0x0              // PAD_CTL value
        ldr r1, IOMUXC_BASE_ADDR_W
        str r0, [r1, #0x01C]      // USB-HST-CLK in GPIO3_0
	ldr r4, =IOMUXC_BASE_ADDR + 0x04cc   // ptr to PAD_CTL
        add r1, r1, #0x00c8       // ptr to MUX_CTL
        add r2, r1, #0x00d8 - 0x00c8
1:      str r0, [r1], #4          // MUX_CTL NFC pins for ULPI interface
        str r3, [r4], #4          // PAD_CTL NFC pins for ULPI interface
        cmp r1, r2
        ble 1b

        ldr r0, =0x00000004       // MUX_CTL ALT4 for SDHC2 pins -> USBH2
	ldr r3, =0x0              // PAD_CTL value
        ldr r1, IOMUXC_BASE_ADDR_W
	ldr r4, =IOMUXC_BASE_ADDR + 0x06ac   // ptr to PAD_CTL
        add r1, r1, #0x0248       // ptr to MUX_CTL
        add r2, r1, #0x025c - 0x0248
2:      str r0, [r1], #4          // MUX_CTL SDHC2 pins for ULPI interface
        str r3, [r4], #4          // PAD_CTL SDHC2 pins for ULPI interface
        cmp r1, r2
        ble 2b
    .endm /* init_usbh2 */

    /* IO Mux control and data for debug outputs */
    .macro init_dbgo
        // initialize GPIO direction and data for output high
        ldr r0, =GPIO1_BASE_ADDR  // R0 to base address  for GPIO 1
        ldr r1, [r0]              
        bic r1, r1, #0x00000007   // init data reg for low outputs
        str r1, [r0]              // write Data Register

        ldr r1, [r0, #0x4]        // read Direction Register
        orr r1, r1, #0x00000007   // make them outputs
        str r1, [r0, #0x04]       // write Direction Register

        // set IOMUX to route the GPIO out to the test points
        ldr r0, =IOMUXC_BASE_ADDR // R0 to base address for IO Mux Control
        mov r1, #0x05             // GPIO is alt mode 5
        str r1, [r0, #0x130]      // STXD5(TP23) as GPIO
        str r1, [r0, #0x134]      // SRXD5(TP24) as GPIO
        str r1, [r0, #0x138]      // SCK5(TP26) as GPIO
    .endm /* init_dbgo */

    /* Toggle a debug output test point */
    /* Legal arguments are TP23, TP24, and TP26 */
    .macro dbgo_toggle tpxx
        ldr r0, =GPIO1_BASE_ADDR  // R0 to base address  for GPIO 1
        ldr r1, [r0]
        ldr r2, BIT_\tpxx
        eor r1, r1, r2
        str r1, [r0]
    .endm /* dbgo_toggle */

    /* Turn on test point for debugging boot up status */
    /* Legal arguments are TP23, TP24, and TP26 */
    .macro dbgo_on tpxx
        ldr r0, =GPIO1_BASE_ADDR  // R0 to base address for GPIO 1
        ldr r1, [r0]
        ldr r2, BIT_\tpxx
        orr r1, r1, r2
        str r1, [r0]
    .endm /* dbgo_on */

    /* Turn off test point for debugging boot up status */
    /* Legal arguments are TP23, TP24, and TP26 */
    .macro dbgo_off tpxx
        ldr r0, =GPIO1_BASE_ADDR  // R0 to base address for GPIO 1
        ldr r1, [r0]
        ldr r2, BIT_\tpxx
        bic r1, r1, r2
        str r1, [r0]
    .endm /* dbgo_off */

    /* initialize gpio for hardware and version input pins */
    .macro init_hw_pins
        // GPIO direction is input at reset
        // set IOMUX to route the GPIO
        mov r1, #0x05          // GPIO alt mode 5

        ldr r2, =0x000001e0    // pad 3.3v, schmitz, 100k PU
        ldr r0, =IOMUXC_BASE_ADDR + 0x02a8
        str r1, [r0]     // ATA_DATA11, GPIO2_24, HWID-2, mode
        ldr r0, =IOMUXC_BASE_ADDR + 0x070c
        str r2, [r0]     // ATA_DATA11, GPIO2_24, HWID-2, pad
        ldr r2, =0x00000001    // select input daisy chain
        ldr r0, =IOMUXC_BASE_ADDR + 0x08a8
        str r2, [r0]

        ldr r2, =0x000021e0    // pad 1.8v, schmitz, 100k PU
        ldr r0, =IOMUXC_BASE_ADDR + 0x010c
        str r1, [r0]     // CSI_PIXCLK, GPIO1_31, HWID-1, mode
        ldr r0, =IOMUXC_BASE_ADDR + 0x0550
        str r2, [r0]     // CSI_PIXCLK, GPIO1_31, HWID-1, pad
	ldr r0, =IOMUXC_BASE_ADDR + 0x0100
        str r1, [r0]     // CSI_MCLK, GPIO1_28, HWID-0, mode
        ldr r0, =IOMUXC_BASE_ADDR + 0x0544
        str r2, [r0]     // CSI_MCLK, GPIO1_28, HWID-0, pad

        ldr r0, =IOMUXC_BASE_ADDR + 0x01a4
        str r1, [r0]     // CTS2, GPIO3_13, HWREV-3, mode
        ldr r0, =IOMUXC_BASE_ADDR + 0x05e8
        str r2, [r0]     // CTS2, GPIO3_13, HWREV-3, pad
        ldr r0, =IOMUXC_BASE_ADDR + 0x01a0
        str r1, [r0]     // RTS2, GPIO3_12, HWREV-2, mode
        ldr r0, =IOMUXC_BASE_ADDR + 0x05e4
        str r2, [r0]     // RTS2, GPIO3_12, HWREV-2, pad
        ldr r0, =IOMUXC_BASE_ADDR + 0x019c
        str r1, [r0]     // TXD2, GPIO3_11, HWREV-1, mode
        ldr r0, =IOMUXC_BASE_ADDR + 0x05e0
        str r2, [r0]     // TXD2, GPIO3_11, HWREV-1, pad
        ldr r0, =IOMUXC_BASE_ADDR + 0x0198
        str r1, [r0]     // RXD2, GPIO3_10, HWREV-0, mode
        ldr r0, =IOMUXC_BASE_ADDR + 0x05dc
        str r2, [r0]     // RXD2, GPIO3_10, HWREV-0, pad
    .endm
    
    /* results returned in r0 */
    .macro get_hwrev
        ldr r0, =GPIO3_BASE_ADDR
        ldr r0, [r0]
        mvn r0, r0
	lsr r0, r0, #10
	and r0, r0, #15
    .endm

    /* results returned in r0 */
    .macro get_hwid
        ldr r1, =GPIO1_BASE_ADDR
        ldr r0, [r1]
        mvn r0, r0
	lsr r0, r0, #30
	and r0, r0, #2
        ldr r1, [r1]
        mvn r1, r1
	lsr r1, r1, #28
	and r1, r1, #1
	orr r0, r1, r0
        ldr r1, =GPIO2_BASE_ADDR
        ldr r1, [r1]
        mvn r1, r1
	lsr r1, r1, #22
	and r1, r1, #4
	orr r0, r1, r0
    .endm

//Internal Boot, from MMC/SD cards
#ifdef MXCFLASH_SELECT_MMC
#warning "TBD: Code not modified for Cicada/Mantis platform."
#define DCDGEN(i,type, addr, data) \
dcd_##i:                         ;\
    .long type                   ;\
    .long addr                   ;\
    .long data

#define FHEADER_OFFSET 0x400
#if 0
#define GEN_FHEADERADDR(x) ((x) + FHEADER_OFFSET)
#else
#define GEN_FHEADERADDR(x) (x)
#endif

     .macro setup_flash_header
     b reset_vector
#if defined(FHEADER_OFFSET)
     .org FHEADER_OFFSET
#endif
app_code_jump_v:    .long GEN_FHEADERADDR(reset_vector)
app_code_barker:    .long 0xB1
app_code_csf:       .long 0
hwcfg_ptr_ptr:      .long GEN_FHEADERADDR(hwcfg_ptr)
super_root_key:     .long 0
hwcfg_ptr:          .long GEN_FHEADERADDR(dcd_data)
app_dest_ptr:       .long SDRAM_BASE_ADDR + SDRAM_SIZE - 0x100000
dcd_data:           .long 0xB17219E9
#ifdef MEMORY_MDDR_ENABLE
                    .long (51 * 12)
//real dcd data

//arm clock is 266Mhz and ahb clock is 133Mhz
//DCDGEN(1, 4, 0x53F80004, 0x00821000)

//WEIM config-CS5 init
DCDGEN(1, 4, 0xB8002054, 0x444a4541)
DCDGEN(2, 4, 0xB8002050, 0x0000dcf6)
DCDGEN(3, 4, 0xB8002058, 0x44443302)
//MDDR init
//enable mDDR
DCDGEN(4, 4, 0xB8001010, 0x00000004)
//reset delay time
DCDGEN(5, 4, 0xB8001010, 0x0000000C)
DCDGEN(6, 4, 0xB800100C, 0x007ffc3f)
DCDGEN(7, 4, 0xB800100C, 0x007ffc3f)
DCDGEN(8, 4, 0xB8001004, 0x007ffc3f)
DCDGEN(9, 4, 0xB8001000, 0x92220000)
DCDGEN(10, 1, 0x80000400, 0xda)
DCDGEN(11, 4, 0xB8001000, 0xA2220000)
DCDGEN(12, 4, 0x80000000, 0x87654321)
DCDGEN(13, 4, 0x80000000, 0x87654321)
DCDGEN(14, 4, 0xB8001000, 0xB2220000)
DCDGEN(15, 1, 0x80000033, 0xda)
DCDGEN(16, 1, 0x82000000, 0xda)
DCDGEN(17, 4, 0xB8001000, 0x82226080)
DCDGEN(18, 4, 0xB8001010, 0x00000004)
DCDGEN(19, 4, 0xB8001008, 0x00002000)
// Setting up nominal drive strength/keeper for DDR data 
DCDGEN(20, 4, 0x43FAC3DC, 0x00000080)
DCDGEN(21, 4, 0x43FAC3E0, 0x00000080)
DCDGEN(22, 4, 0x43FAC3E4, 0x00000080)
DCDGEN(23, 4, 0x43FAC3E8, 0x00000080)
DCDGEN(24, 4, 0x43FAC3EC, 0x00000080)
DCDGEN(25, 4, 0x43FAC3F0, 0x00000080)
DCDGEN(26, 4, 0x43FAC3F4, 0x00000080)
DCDGEN(27, 4, 0x43FAC3F8, 0x00000080)
DCDGEN(28, 4, 0x43FAC3FC, 0x00000080)
DCDGEN(29, 4, 0x43FAC400, 0x00000080)
DCDGEN(30, 4, 0x43FAC404, 0x00000080)
DCDGEN(31, 4, 0x43FAC408, 0x00000080)
DCDGEN(32, 4, 0x43FAC40C, 0x00000080)
DCDGEN(33, 4, 0x43FAC410, 0x00000080)
DCDGEN(34, 4, 0x43FAC414, 0x00000080)
DCDGEN(35, 4, 0x43FAC418, 0x00000080)
DCDGEN(36, 4, 0x43FAC41C, 0x00000080)
DCDGEN(37, 4, 0x43FAC420, 0x00000080)
DCDGEN(38, 4, 0x43FAC424, 0x00000080)
DCDGEN(39, 4, 0x43FAC428, 0x00000080)
DCDGEN(40, 4, 0x43FAC42C, 0x00000080)
DCDGEN(41, 4, 0x43FAC430, 0x00000080)
DCDGEN(42, 4, 0x43FAC434, 0x00000080)
DCDGEN(43, 4, 0x43FAC438, 0x00000080)
DCDGEN(44, 4, 0x43FAC43C, 0x00000080)
DCDGEN(45, 4, 0x43FAC440, 0x00000080)
DCDGEN(46, 4, 0x43FAC444, 0x00000080)
DCDGEN(47, 4, 0x43FAC448, 0x00000080)
DCDGEN(48, 4, 0x43FAC44C, 0x00000080)
DCDGEN(49, 4, 0x43FAC450, 0x00000080)
DCDGEN(50, 4, 0x43FAC454, 0x00000080)
DCDGEN(51, 4, 0x43FAC458, 0x00000080)

#else
                    .long (52 * 12)

//arm clock is 266Mhz and ahb clock is 133Mhz
//DCDGEN(1, 4, 0x53F80004, 0x00821000)

//WEIM config-CS5 init
DCDGEN(1, 4, 0xB8002050, 0x0000d843)
DCDGEN(2, 4, 0xB8002054, 0x22252521)
DCDGEN(3, 4, 0xB8002058, 0x22220a00)

//DDR2 init
DCDGEN(4, 4, 0xB8001010, 0x00000304)
DCDGEN(5, 4, 0xB8001010, 0x0000030C)
DCDGEN(6, 4, 0xB8001004, 0x007ffc3f)
DCDGEN(7, 4, 0xB8001000, 0x92220000)
DCDGEN(8, 4, 0x80000400, 0x12345678)
DCDGEN(9, 4, 0xB8001000, 0xA2220000)
DCDGEN(10, 4, 0x80000000, 0x87654321)
DCDGEN(11, 4, 0x80000000, 0x87654321)
DCDGEN(12, 4, 0xB8001000, 0xB2220000)
DCDGEN(13, 1, 0x80000233, 0xda)
DCDGEN(14, 1, 0x82000780, 0xda)
DCDGEN(15, 1, 0x82000400, 0xda)
DCDGEN(16, 4, 0xB8001000, 0x82226080)
DCDGEN(17, 4, 0xB8001004, 0x007ffc3f)
DCDGEN(18, 4, 0xB800100C, 0x007ffc3f)
DCDGEN(19, 4, 0xB8001010, 0x00000304)
DCDGEN(20, 4, 0xB8001008, 0x00002000)

// Setting up nominal drive strength/keeper for DDR data 
DCDGEN(21, 4, 0x43FAC3DC, 0x00000080)
DCDGEN(22, 4, 0x43FAC3E0, 0x00000080)
DCDGEN(23, 4, 0x43FAC3E4, 0x00000080)
DCDGEN(24, 4, 0x43FAC3E8, 0x00000080)
DCDGEN(25, 4, 0x43FAC3EC, 0x00000080)
DCDGEN(26, 4, 0x43FAC3F0, 0x00000080)
DCDGEN(27, 4, 0x43FAC3F4, 0x00000080)
DCDGEN(28, 4, 0x43FAC3F8, 0x00000080)
DCDGEN(29, 4, 0x43FAC3FC, 0x00000080)
DCDGEN(30, 4, 0x43FAC400, 0x00000080)
DCDGEN(31, 4, 0x43FAC404, 0x00000080)
DCDGEN(32, 4, 0x43FAC408, 0x00000080)
DCDGEN(33, 4, 0x43FAC40C, 0x00000080)
DCDGEN(34, 4, 0x43FAC410, 0x00000080)
DCDGEN(35, 4, 0x43FAC414, 0x00000080)
DCDGEN(36, 4, 0x43FAC418, 0x00000080)
DCDGEN(37, 4, 0x43FAC41C, 0x00000080)
DCDGEN(38, 4, 0x43FAC420, 0x00000080)
DCDGEN(39, 4, 0x43FAC424, 0x00000080)
DCDGEN(40, 4, 0x43FAC428, 0x00000080)
DCDGEN(41, 4, 0x43FAC42C, 0x00000080)
DCDGEN(42, 4, 0x43FAC430, 0x00000080)
DCDGEN(43, 4, 0x43FAC434, 0x00000080)
DCDGEN(44, 4, 0x43FAC438, 0x00000080)
DCDGEN(45, 4, 0x43FAC43C, 0x00000080)
DCDGEN(46, 4, 0x43FAC440, 0x00000080)
DCDGEN(47, 4, 0x43FAC444, 0x00000080)
DCDGEN(48, 4, 0x43FAC448, 0x00000080)
DCDGEN(49, 4, 0x43FAC44C, 0x00000080)
DCDGEN(50, 4, 0x43FAC450, 0x00000080)
DCDGEN(51, 4, 0x43FAC454, 0x00000080)
DCDGEN(52, 4, 0x43FAC458, 0x00000080)

#endif

//CARD_FLASH_CFG_PARMS_T---length
card_cfg:           .long REDBOOT_IMAGE_SIZE
     .endm
#endif

ARM_PPMRR:              .word   0x40000015
L2CACHE_PARAM:          .word   0x00030024
IIM_SREV_REG_VAL:       .word   IIM_BASE_ADDR + IIM_SREV_OFF
AIPS1_CTRL_BASE_ADDR_W: .word   AIPS1_CTRL_BASE_ADDR
AIPS2_CTRL_BASE_ADDR_W: .word   AIPS2_CTRL_BASE_ADDR
AIPS1_PARAM_W:          .word   0x77777777
MAX_BASE_ADDR_W:        .word   MAX_BASE_ADDR
MAX_PARAM1:             .word   0x00302154
ESDCTL_BASE_W:          .word   ESDCTL_BASE
M3IF_BASE_W:            .word   M3IF_BASE
RAM_PARAM1_MDDR:        .word     0x00000400
RAM_PARAM3_MDDR:        .word     (SDRAM_SIZE / 2) // full drive
EMRS_MDDR_4QTR:         .word     ((SDRAM_SIZE / 2) + 0x00000000) // full drive
EMRS_MDDR_3QTR:         .word     ((SDRAM_SIZE / 2) + 0x00000080) // 3/4 drive
EMRS_MDDR_2QTR:         .word     ((SDRAM_SIZE / 2) + 0x00000020) // 1/2 drive
EMRS_MDDR_1QTR:         .word     ((SDRAM_SIZE / 2) + 0x00000040) // 1/4 drive
RAM_PARAM6_MDDR:        .word     0x00000033 //burst
//ESDCTL_0x92220000:      .word   0x92220000
//ESDCTL_0xA2220000:      .word   0xA2220000
//ESDCTL_0xB2220000:      .word   0xB2220000
//ESDCTL_0x82226080:      .word   0x82226080
ESDCFG_CONFIG:          .word   (ESDCFG_tXP(3) + ESDCFG_tWTR + \
                                 ESDCFG_tRP(2) + ESDCFG_tMRD(1) + \
                                 ESDCFG_tWR + ESDCFG_tRAS(6) + \
                                 ESDCFG_tRRD(1) + ESDCFG_tCAS(3) + \
                                 ESDCFG_tRCD(3) + ESDCFG_tRC(15) )
ESDCTL_PRECHARGE_128:    .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_PRECHARGE) + \
                                 ESDCTL_ROW(ESD_ROW_13) + \
                                 ESDCTL_COL(ESD_COL_10) + \
                                 ESDCTL_DSIZ(ESD_DSIZ_32) )
ESDCTL_AUTO_REFRESH_128: .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_AUTO_REFSH) + \
                                 ESDCTL_ROW(ESD_ROW_13) + \
                                 ESDCTL_COL(ESD_COL_10) + \
                                 ESDCTL_DSIZ(ESD_DSIZ_32) )
ESDCTL_LOAD_MODE_128:    .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_LOAD_MODE) + \
                                 ESDCTL_ROW(ESD_ROW_13) + \
                                 ESDCTL_COL(ESD_COL_10) + \
                                 ESDCTL_DSIZ(ESD_DSIZ_32) )
ESDCTL_NORMAL_128:       .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_NORMAL) + \
                                 ESDCTL_ROW(ESD_ROW_13) + \
                                 ESDCTL_COL(ESD_COL_10) + \
                                 ESDCTL_DSIZ(ESD_DSIZ_32) + \
                                 ESDCTL_SREFR(ESD_SREFR_4_8192) + \
                                 ESDCTL_PWDT(ESD_PWDT_DISABLE) + \
                                 ESDCTL_BL_8 )

ESDCTL_PRECHARGE_256:    .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_PRECHARGE) + \
                                 ESDCTL_ROW(ESD_ROW_14) + \
                                 ESDCTL_COL(ESD_COL_10) + \
                                 ESDCTL_DSIZ(ESD_DSIZ_32) )
ESDCTL_AUTO_REFRESH_256: .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_AUTO_REFSH) + \
                                 ESDCTL_ROW(ESD_ROW_14) + \
                                 ESDCTL_COL(ESD_COL_10) + \
                                 ESDCTL_DSIZ(ESD_DSIZ_32) )
ESDCTL_LOAD_MODE_256:    .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_LOAD_MODE) + \
                                 ESDCTL_ROW(ESD_ROW_14) + \
                                 ESDCTL_COL(ESD_COL_10) + \
                                 ESDCTL_DSIZ(ESD_DSIZ_32) )
ESDCTL_NORMAL_256:       .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_NORMAL) + \
                                 ESDCTL_ROW(ESD_ROW_14) + \
                                 ESDCTL_COL(ESD_COL_10) + \
                                 ESDCTL_DSIZ(ESD_DSIZ_32) + \
                                 ESDCTL_SREFR(ESD_SREFR_4_8192) + \
                                 ESDCTL_PWDT(ESD_PWDT_DISABLE) + \
                                 ESDCTL_BL_8 )
//ESDCTL_DELAY5:          .word   0x00F49F00
IOMUXC_BASE_ADDR_W:     .word   IOMUXC_BASE_ADDR
CCM_CCMR_W:             .word   0x003F4208
CCM_PDR0_399_W:         .word   0x00811000
CCM_PDR0_532_W:         .word   0x00801000
MPCTL_PARAM_399_W:      .word   MPCTL_PARAM_399
MPCTL_PARAM_532_W:      .word   MPCTL_PARAM_532
PPCTL_PARAM_W:          .word   PPCTL_PARAM_300
AVIC_VECTOR0_ADDR_W:    .word   MXCBOOT_FLAG_REG
AVIC_VECTOR1_ADDR_W:    .word   MXCFIS_FLAG_REG
MXC_REDBOOT_ROM_START:  .word   SDRAM_BASE_ADDR + SDRAM_SIZE - 0x100000
CONST_0x0FFF:           .word   0x0FFF
CCM_BASE_ADDR_W:        .word   CCM_BASE_ADDR
WEIM_CTRL_CS0_W:        .word   WEIM_CTRL_CS0
CS0_CSCRU_LOC_ASYNC:    .word   CS0_CSCRU_CIC_ASYNC
CS0_CSCRL_LOC_ASYNC:    .word   CS0_CSCRL_CIC_ASYNC
CS0_CSCRA_LOC_ASYNC:    .word   CS0_CSCRA_CIC_ASYNC
CS0_CSCRU_LOC_PAGE:     .word   CS0_CSCRU_CIC_PAGE
CS0_CSCRL_LOC_PAGE:     .word   CS0_CSCRL_CIC_PAGE
CS0_CSCRA_LOC_PAGE:     .word   CS0_CSCRA_CIC_PAGE
ROM_VER_ADDR_W:         .word        ROM_BASE_ADDR + ROM_SI_REV_OFFSET
BIT_TP23:               .word   (1 << 0)
BIT_TP24:               .word   (1 << 1)
BIT_TP26:               .word   (1 << 2)
DEADBEEF_W:             .word   0xdeadbeef
        
/*---------------------------------------------------------------------------*/
/* end of hal_platform_setup.h                                               */
#endif /* CYGONCE_HAL_PLATFORM_SETUP_H */
