#include <redboot.h>
#include <cyg/hal/hal_soc.h>

#define MXC_CCM_CGR1    (CCM_BASE_ADDR + 0x24)

#define KPCR    (KPP_BASE_ADDR + 0x00)
#define KPSR    (KPP_BASE_ADDR + 0x02)
#define KDDR    (KPP_BASE_ADDR + 0x04)
#define KPDR    (KPP_BASE_ADDR + 0x06)

#define KBD_STAT_KPKD         0x01   /* Key Press Interrupt Status bit */
#define KBD_STAT_KPKR         0x02   /* Key Release Interrupt Status bit */
#define KBD_STAT_KDSC         0x04   /* Key Depress Synchronizer Chain Status bit */
#define KBD_STAT_KRSS         0x08   /* Key Release Synchronizer Status bit */
#define KBD_STAT_KDIE        0x100   /* Key Depress Interrupt Enable Status bit */
#define KBD_STAT_KRIE        0x200   /* Key Release Interrupt Enable */
#define KBD_STAT_KPPEN       0x400   /* Keypad Clock Enable */

/* if the value of NUM_ROWS or NUM_COLS is changed 
 * init_iomux() must be changed to correctly configure the pins.
 */
#define NUM_ROWS    4
#define NUM_COLS    4
#define MAGIC_KEY_VAL    0x05 /* keys S5, S4 */

extern int g_eebootval;

static void scan_keys(unsigned int *row_data)
{
    unsigned int val;
    int col;

    *row_data = 0;

    for (col = 0; col < NUM_COLS; col++) {
        /* 2. Write 1.s to KPDR[15:8] setting column data to 1.s */
        val = readw(KPDR);
        val |= 0xff00;
        writew(val, KPDR);

        /*
         * 3. Configure columns as totem pole outputs(for quick
         * discharging of keypad capacitance)
         */
        val = readw(KPCR);
        val &= 0x00ff;
        writew(val, KPCR);

        hal_delay_us(2);

        /*
         * 4. Configure columns as open-drain
         */
        val = readw(KPCR);
        val |= ((1 << NUM_COLS) - 1) << 8;
        writew(val, KPCR);

        /*
         * 5. Write a single column to 0, others to 1.
         * 6. Sample row inputs and save data. Multiple key presses
         * can be detected on a single column.
         * 7. Repeat steps 2 - 6 for remaining columns.
         */

        /* Col bit starts at 8th bit in KPDR */
        val = readw(KPDR);
        val &= ~(1 << (8 + col));
        writew(val, KPDR);

        /* Delay added to avoid propagating the 0 from column to row
         * when scanning. */

        hal_delay_us(5);

        /* Read row input */
        val = readw(KPDR);
        *row_data |= (~val & ((1 << NUM_ROWS) - 1)) << (NUM_ROWS * col);
    }

    /*
     * 8. Return all columns to 0 in preparation for standby mode.
     * 9. Clear KPKD and KPKR status bit(s) by writing to a .1.,
     * set the KPKR synchronizer chain by writing "1" to KRSS register,
     * clear the KPKD synchronizer chain by writing "1" to KDSC register
     */
    val = 0x00;
    writew(val, KPDR);
    val = readw(KPSR);
    val |= KBD_STAT_KPKD | KBD_STAT_KPKR | KBD_STAT_KRSS | KBD_STAT_KDSC;
    writew(val, KPSR);
}

#define MUX_CTL_COL_VAL        0x12     /* functional mode */
#define MUX_CTL_ROW_VAL        0x12     /* functional mode */
static void init_iomux(void)
{
    unsigned int val, reg;

    // KEY_COL 0-3 mux ctl
    reg = IOMUXC_BASE_ADDR + 0x64;
    val = ((MUX_CTL_COL_VAL & 0xff) << 24) | ((MUX_CTL_COL_VAL & 0xff) << 16) |
        ((MUX_CTL_COL_VAL & 0xff) << 8) | ((MUX_CTL_COL_VAL & 0xff) << 0);
    writel(val, reg);

    // KEY_ROW 0-3 mux ctl
    reg = IOMUXC_BASE_ADDR + 0x6c;
    val = ((MUX_CTL_ROW_VAL & 0xff) << 24) | ((MUX_CTL_ROW_VAL & 0xff) << 16) |
        ((MUX_CTL_ROW_VAL & 0xff) << 8) | ((MUX_CTL_ROW_VAL & 0xff) << 0);
    writel(val, reg);
}

void init_keys(void)
{
    unsigned int val;

    /* enable the ipg_clk */
    val = readl(MXC_CCM_CGR1);
    val |= 3 << 20;
    writel(val, MXC_CCM_CGR1);

    init_iomux();

    /* Disable row interrupts (KPCR[7:0]) */
    val = 0;
    writew(val, KPCR);

    /* Write 0's to cols (KPDR[15:8]) */
    val = readw(KPDR);
    val &= 0x00ff;
    writew(val, KPDR);

    /* Enable columns as open-drain (KPCR[15:8]) */
    val = readw(KPCR);
    val |= (((1 << NUM_COLS) - 1) << 8);
    writew(val, KPCR);
    
    /* Configure columns as output, rows as input (KDDR[15:0]) */
    val = readw(KDDR);
    val |= 0xff00;
    val &= 0xff00;
    writew(val, KDDR);

    /* Clear interrupt enables
     * Clear KPKD and KPKR status bits by writing to a "1"
     * Set the KPKR synchronizer chain by writing "1"
     * Set the KPKD synchronizer chain by writing "1"
     */
    val = readw(KPSR);
    val &= ~(KBD_STAT_KPKR | KBD_STAT_KPKD);
    val |= KBD_STAT_KPKD | KBD_STAT_KRSS | KBD_STAT_KDSC;
    writew(val, KPSR);
}

RedBoot_init(init_keys, RedBoot_INIT_PRIO(100));

void check_keys(void)
{
    unsigned row_data = 0;

    scan_keys(&row_data);

    if (row_data == MAGIC_KEY_VAL){
        diag_printf("check_keys, MAGIC\n");
        g_eebootval = 1;
    }
}

RedBoot_init(check_keys, RedBoot_INIT_PRIO(110));
