#include <redboot.h>
#include <cyg/hal/hal_soc.h>
#include <cyg/io/mxc_i2c.h>

// zero based buss number
#define EEPROM_BUSS_NUMBER      1
#define EEPROM_BUSS_ADDRESS     0x50
#define EEPROM_PAGE_SIZE        ((unsigned int)64)

// Internal structure used to hold config data
#define MAX_NAME_LEN    12
#define MAX_DATA_LEN    256
#define CONFIG_KEY1     0x0BADFACE
#define CONFIG_KEY2     0xDEADDEAD
#define MAX_CONFIGS     4
struct ee_config {
    char name[MAX_NAME_LEN];
    unsigned long key1;
    char config_data[MAX_DATA_LEN];
    unsigned long key2;
    unsigned long kernFlashAddr;
    unsigned long kernLen;
    unsigned long kernCrc;
    unsigned long cksum;
};

static struct ee_config ee_configs[MAX_CONFIGS];

int g_eebootval = 0;

static hal_jmp_buf error_jmpbuf;

static cyg_uint32
eeprom_crc(struct ee_config *conf)
{
    cyg_uint32 crc;
 
    crc = cyg_crc32((unsigned char *)conf, sizeof(*conf)-sizeof(conf->cksum));

    return crc;
}

static int
str_cmp(const char *a, const char *b)
{
    for (; *a == *b; a++, b++) {
        if (*a == 0 && *b == 0)
            return 0;
    }
    return 1;
}

#define EEPROM_READ     true
#define EEPROM_WRITE    false

static int
rw_eeprom(bool read, unsigned int addr,
        unsigned char *data, unsigned int bytes)
{
    int dir = (read ? I2C_READ : I2C_WRITE);
    unsigned int start_addr = addr;
    int total_bytes = bytes;
    unsigned char *buf = data;
    
    struct mxc_i2c_request rq;
    int i;
    unsigned int end_addr;
    int num_bytes;

    rq.dev_addr = EEPROM_BUSS_ADDRESS;
    rq.reg_addr_sz = 2;

    if (dir == I2C_WRITE){
        // write
        end_addr = (start_addr + EEPROM_PAGE_SIZE) & ~(EEPROM_PAGE_SIZE - 1);
        num_bytes = end_addr - start_addr;

        while (total_bytes > 0){
            if (num_bytes > total_bytes)
                num_bytes = total_bytes;

            // address should go out ms byte first, ls byte last
            // i2c_xfer sends out ls byte first
            rq.reg_addr = ((start_addr & 0xff00) >> 8) | ((start_addr & 0xff) << 8);
            rq.buffer_sz = num_bytes;
            rq.buffer = buf;

            for (i = 0; i < 3; ++i) {
                if (i2c_xfer(EEPROM_BUSS_NUMBER, &rq, dir) == 0)
                    break;
                hal_delay_us(5000);
            }
            hal_delay_us(5000);

            if (i == 3) {
                diag_printf("I2C transfer %s error\n", (read ? "read":"write"));
                return -1;
            }
            
            start_addr += num_bytes;
            buf += num_bytes;
            total_bytes -= num_bytes;
            num_bytes = EEPROM_PAGE_SIZE;
        } 
    }else{
        // read
        // address should go out ms byte first, ls byte last
        // i2c_xfer sends out ls byte first
        rq.reg_addr = ((start_addr & 0xff00) >> 8) | ((start_addr & 0xff) << 8);
        rq.buffer_sz = total_bytes;
        rq.buffer = buf;
        
        if (i2c_xfer(EEPROM_BUSS_NUMBER, &rq, dir) != 0) {
            diag_printf("I2C transfer %s error\n", (read ? "read":"write"));
            return -1;
        }
    }

    return 0;
}

static int
rw_eeprom_data(bool read, int index)
{
    int dir = (read ? I2C_READ : I2C_WRITE);
    unsigned int start_addr = sizeof(struct ee_config) * index;
    int total_bytes = sizeof(struct ee_config);
    unsigned char *buf = (unsigned char *)&ee_configs[index];
    
    return rw_eeprom(dir, start_addr, buf, total_bytes);
}

static int
find_name(char *scriptname)
{
    int i;

    for (i = 0; i < MAX_CONFIGS; ++i) {
        if (ee_configs[i].name[0] == 0)
            continue;

        if (str_cmp(ee_configs[i].name, scriptname) == 0)
            return i;
    }
    
    return -1;
}

RedBoot_cmd("eerun",
            "run a script from EEPROM memory",
            "scriptname",
            do_ee_run
    );

void do_ee_run(int argc, char *argv[])
{
    int ret;
    int ret1;
    int i;
    char line[MAX_DATA_LEN];
    char *command;
    char *scriptname = argv[1];
    int argcc;
    char *argvv[MAX_ARGV];
    struct cmd *cmd;
    
    if (argc != 2) {
        diag_printf("** Error: Invalid entry\n");
        return;
    }

    ret = find_name(scriptname);
    if (ret < 0) {
        diag_printf("** Error: Invalid script name\n");
        return;
    }
    
    // see if the script is a script name
    strcpy(line, ee_configs[ret].config_data);
    for (i = 0; i < MAX_DATA_LEN && line[i] != 0; ++i) {
        if (line[i] == '\n') {
            line[i] = 0;
            break;
        }
    }
    ret1 = find_name(line);
    if (ret1 >= 0) {
        ret = ret1;
    }

    strcpy(line, ee_configs[ret].config_data);
    for (i = 0; i < MAX_DATA_LEN && line[i] != 0; ++i) {
        if (line[i] == '\n')
            line[i] = ';';
    }
        
    command = line;
    while (strlen(command) > 0) {
        cmd = parse(&command, &argcc, &argvv[0]);
        if (cmd != (struct cmd *)0) {
            // Try to handle aborts - messy because of the stack unwinding...
            if (hal_setjmp(error_jmpbuf))
                diag_printf("** command abort - illegal memory access?\n");
            else
                (cmd->fun)(argcc, argvv);
        } else {
            diag_printf("** Error: Illegal command: \"%s\"\n", argv[1]);
        }
    }
}

RedBoot_cmd("eeconfig",
            "Manage configuration kept in EEPROM memory",
            "[-i] | [-t] | [-l] | scriptname",
            do_ee_config
    );

void do_ee_config(int argc, char *argv[])
{
    struct option_info opts[3];
    char *scriptname = (char *)0;
    int ret;
    bool list = false;
    bool init = false;
    bool test = false;
    int i;
    char line[256], *sp, *lp;
    int script_len;
    bool match = false;
    int empty = -1;
    struct ee_config tmp;

    init_opts(&opts[0], 'l', false, OPTION_ARG_TYPE_FLG, 
              (void *)&list, (bool *)0, "list configuration only");
    init_opts(&opts[1], 'i', false, OPTION_ARG_TYPE_FLG, 
              (void *)&init, (bool *)0, "initialize configuration database");
    init_opts(&opts[2], 't', false, OPTION_ARG_TYPE_FLG, 
              (void *)&test, (bool *)0, "initialize configuration database");

    if (!scan_opts(argc, argv, 1, opts, 3, (void *)&scriptname, OPTION_ARG_TYPE_STR, "scriptname"))
        return;

    if (init) {
        if (test || list || scriptname) {
            diag_printf("** entry error\n");
            return;
        }

        if (verify_action("Initialize non-volatile configuration")) {
            for(i = 0; i < MAX_CONFIGS; ++i) {
                memset(&ee_configs[i], 0, sizeof(struct ee_config));
                ee_configs[i].key1 = CONFIG_KEY1;
                ee_configs[i].key2 = CONFIG_KEY2;
                ee_configs[i].cksum = eeprom_crc(&ee_configs[i]);
                rw_eeprom_data(EEPROM_WRITE, i);
            }
        }
        init = false;
        test = true;
    }

    if (test) {
        if (init || list || scriptname) {
            diag_printf("** entry error\n");
            return;
        }
        
        ret = 0;
        for (i = 0; i < MAX_CONFIGS; ++i) {
            rw_eeprom_data(EEPROM_READ, i);
            if ((eeprom_crc(&ee_configs[i]) != ee_configs[i].cksum) ||
                (ee_configs[i].key1 != CONFIG_KEY1)||
                (ee_configs[i].key2 != CONFIG_KEY2)) {
                ret = -1;
                diag_printf("** Warning EEPROM configuration %d corrupt\n", i);
            }
        }
        if (!ret)
            diag_printf("** EEPROM configuration ok\n");
        
        return;
    }
    
    if (list) {
        if (test || scriptname) {
            diag_printf("** entry error\n");
            return;
        }
        
        for (i = 0; i < MAX_CONFIGS; ++i) {
            if ((eeprom_crc(&ee_configs[i]) != ee_configs[i].cksum) ||
                (ee_configs[i].key1 != CONFIG_KEY1)||
                (ee_configs[i].key2 != CONFIG_KEY2)) {
                 continue;
            }
            
            if (ee_configs[i].name[0] == 0) {
                 continue;
            }

            diag_printf("%s\n", ee_configs[i].name);
            diag_printf("%s\n", ee_configs[i].config_data);
        }
        return;
    }
    
    if (scriptname) {
        for (i = 0; i < MAX_CONFIGS; ++i) {
            if ((eeprom_crc(&ee_configs[i]) != ee_configs[i].cksum) ||
                (ee_configs[i].key1 != CONFIG_KEY1)||
                (ee_configs[i].key2 != CONFIG_KEY2)) {
                // corrupt
                continue;
            }

            if (ee_configs[i].name[0] == 0) {
                if (empty < 0) {
                    empty = i;
                }
                continue;
            }

            if (str_cmp(scriptname, ee_configs[i].name) == 0) {
                diag_printf("%s\n", ee_configs[i].name);
                diag_printf("%s\n", ee_configs[i].config_data);
                match = true;
                break;
            }
        }
        
        if (!match)
            i = empty;

        if (i < 0 || i == MAX_CONFIGS) {
            diag_printf("** no space available for a new entry\n");
            return;
        }

        // fill out the config struct
        memset(&tmp, 0, sizeof(struct ee_config));
        tmp.key1 = CONFIG_KEY1;  
        tmp.key2 = CONFIG_KEY2;

        // name
        if (strlen(scriptname) >= MAX_NAME_LEN){
                diag_printf("** script name longer than %d not allowed!\n", MAX_NAME_LEN - 1);
                return;
        }
        strcpy(tmp.name, scriptname);

        // script
        script_len = 0;
        sp = tmp.config_data;
        diag_printf("Enter script, terminate with empty line\n");
        while (true) {
            diag_printf(">> ");
            ret = _rb_gets(line, sizeof(line), 0);
            if (ret < 0) {
                diag_printf ("** invalid entry\n");
                return;
            }

            if (strlen(line) == 0)
                break;

            script_len += strlen(line);
            if (script_len >= MAX_DATA_LEN) {
                diag_printf("** script longer than %d not allowed!\n", MAX_DATA_LEN - 1);
                return;
            }
        
            lp = line;
            while (*lp) {
                *sp++ = *lp++;
            }
            *sp++ = '\n';
        }
    
        // check sum
        tmp.cksum = eeprom_crc(&tmp);

        memcpy(&ee_configs[i], &tmp, sizeof(struct ee_config));
        
        rw_eeprom_data(EEPROM_WRITE, i);
    }

    return;
}

static void init_i2c(void)
{
    // 40kHz data rate
    i2c_init(i2c_base_addr[EEPROM_BUSS_NUMBER], 40000);
}

RedBoot_init(init_i2c, RedBoot_INIT_PRIO(120));

//
// Attempt to get configuration information from the eeprom.
// If available (i.e. good checksum, etc), initialize "known"
// values for later use.
//
void load_ee_config(void)
{
    int i;
    int ret = 0;
    
    for(i = 0; i < MAX_CONFIGS; ++i) {
        rw_eeprom_data(EEPROM_READ, i);
        if ((eeprom_crc(&ee_configs[i]) != ee_configs[i].cksum) ||
            (ee_configs[i].key1 != CONFIG_KEY1)||
            (ee_configs[i].key2 != CONFIG_KEY2)) {
            ret = -1;
            diag_printf("**Warning** EEPROM configuration %d corrupt\n", i);
            memset(&ee_configs[i], 0, sizeof(struct ee_config));
        } else if (i == g_eebootval && ee_configs[i].config_data[0] != 0)
            script = (unsigned char *)ee_configs[i].config_data;
    }
    
    if (ret) {
        diag_printf("Use 'eeconfig -i' to [re]initialize database\n");
        return;
    }
}

RedBoot_init(load_ee_config, RedBoot_INIT_PRIO(130));

// EOF econfig.c

