#ifndef CYGONCE_HAL_PLATFORM_SETUP_H
#define CYGONCE_HAL_PLATFORM_SETUP_H

//=============================================================================
//
//      hal_platform_setup.h
//
//      Platform specific support for HAL (assembly code)
//
//=============================================================================
//####ECOSGPLCOPYRIGHTBEGIN####
// -------------------------------------------
// This file is part of eCos, the Embedded Configurable Operating System.
// Copyright (C) 1998, 1999, 2000, 2001, 2002 Red Hat, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 or (at your option) any later version.
//
// eCos is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
// for more details.
//
// You should have received a copy of the GNU General Public License along
// with eCos; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. However the source code for this file must still be made available
// in accordance with section (3) of the GNU General Public License.
//
// This exception does not invalidate any other reasons why a work based on
// this file might be covered by the GNU General Public License.
//
// Alternative licenses for eCos may be arranged by contacting Red Hat, Inc.
// at http://sources.redhat.com/ecos/ecos-license/
// -------------------------------------------
//####ECOSGPLCOPYRIGHTEND####
//===========================================================================

#include <pkgconf/system.h>             // System-wide configuration info
#include CYGBLD_HAL_VARIANT_H           // Variant specific configuration
#include CYGBLD_HAL_PLATFORM_H          // Platform specific configuration
#include <cyg/hal/hal_soc.h>            // Variant specific hardware definitions
#include <cyg/hal/hal_mmu.h>            // MMU definitions
#include <cyg/hal/fsl_board.h>          // Platform specific hardware definitions

//#define BOOT_FROM_MMC

#if defined(CYG_HAL_STARTUP_ROM) || defined(CYG_HAL_STARTUP_ROMRAM)
#define PLATFORM_SETUP1 _platform_setup1
#if defined(BOOT_FROM_MMC)
#define PLATFORM_PREAMBLE flash_header
#endif
#define CYGHWR_HAL_ARM_HAS_MMU

#ifdef CYG_HAL_STARTUP_ROMRAM
#define CYGSEM_HAL_ROM_RESET_USES_JUMP
#endif

#define NFC_2K_BI_SWAP
#define SDRAM_FULL_PAGE_BIT     0x100
#define SDRAM_FULL_PAGE_MODE    0x37
#define SDRAM_BURST_MODE        0x33

#define MMC_BLK_LEN                    0x200
#define MMC_START_ADDR             0x0
#define MMC_LOAD_SIZE                0x30000
#define CYGHWR_HAL_ROM_VADDR    0x0

#if 0
#define UNALIGNED_ACCESS_ENABLE
#define SET_T_BIT_DISABLE
#define BRANCH_PREDICTION_ENABLE
#endif

/* CSCR0x Defines for FNET Cicada Dev Board, ASYNC Mode ROM Access 
 *  U = 0x0044_D203, L = 0xA022_0D01, A = 0x0022_0000                      */
#define CSCR0U_CIC_ASYNC (CSCRU_CNC(3) + CSCRU_WSC(20) + CSCRU_EDC(3) )
#define CSCR0L_CIC_ASYNC (CSCRL_OEA(5) + CSCRL_EBWA(2) + CSCRL_EBWN(2) + \
    CSCRL_EBC + CSCRL_DSZ(5) + CSCRL_CSEN )
#define CSCR0A_CIC_ASYNC (CSCRA_RWA(2) + CSCRA_RWN(2) )

/* CSCR0x Defines for FNET Cicada Dev Board, Page Mode ROM Access 
 *  U = 0x0074_D203, L = 0x5022_0D01, A = 0x0022_0000                      */
#define CSCR0U_CIC_PAGE  (CSCRU_PSZ(1) + CSCRU_PME + CSCRU_SYNC + \
    CSCRU_DOL(6) + CSCRU_CNC(3) + CSCRU_WSC(18) + CSCRU_EDC(3) )
#define CSCR0L_CIC_PAGE  (CSCRL_OEA(5) + CSCRL_EBWA(2) + CSCRL_EBWN(2) + \
    CSCRL_EBC + CSCRL_DSZ(5) + CSCRL_CSEN )
#define CSCR0A_CIC_PAGE  (CSCRA_RWA(2) + CSCRA_RWN(2) )

/* CSCR1x Defines for FNET Cicada Dev Board, ASYNC Mode MDOC Access */
#define CSCR1U_CIC_ASYNC (CSCRU_CNC(2) + CSCRU_WSC(5) + CSCRU_EDC(3) )
#define CSCR1L_CIC_ASYNC (CSCRL_OEA(2) + CSCRL_EBWA(2) + CSCRL_EBWN(2) + \
    CSCRL_EBC + CSCRL_CSA(2) + CSCRL_DSZ(5) + CSCRL_CSEN )
#define CSCR1A_CIC_ASYNC (CSCRA_RWA(2) + CSCRA_RWN(2) )

/* CSCR1x Defines for FNET Cicada Dev Board, Burst Mode MDOC Access */
/*   WARNING: this is untested and simply a copy of the NOR config */
#define CSCR1U_CIC_BURST  (CSCRU_PSZ(1) + CSCRU_PME + CSCRU_SYNC + \
    CSCRU_DOL(6) + CSCRU_CNC(3) + CSCRU_WSC(18) + CSCRU_EDC(3) )
#define CSCR1L_CIC_BURST  (CSCRL_OEA(5) + CSCRL_EBWA(2) + CSCRL_EBWN(2) + \
    CSCRL_EBC + CSCRL_DSZ(5) + CSCRL_CSEN )
#define CSCR1A_CIC_BURST  (CSCRA_RWA(2) + CSCRA_RWN(2) )

/* CSCR4x Defines for FNET Cicada Dev Board, CS4 - to FPGA Internal Registers 
 *  U = 0x0000_D822, L = 0x2146_2511, A = 0x2222_0000                       */
#define CSCR4U_CIC_CS4   (CSCRU_CNC(3) + CSCRU_WSC(20) + CSCRU_WWS(2) + \
    CSCRU_EDC(2) )
#define CSCR4L_CIC_CS4   (CSCRL_OEA(2) + CSCRL_OEN(1) + CSCRL_EBWA(4) + \
    CSCRL_EBWN(6) + CSCRL_CSA(2) + CSCRL_DSZ(5) + CSCRL_CSN(1) + CSCRL_CSEN)
#define CSCR4A_CIC_CS4   (CSCRA_EBRA(2) + CSCRA_EBRN(2) + CSCRA_RWA(2) + \
    CSCRA_RWN(2))

/* CSCR5x Defines for FNET Cicada Dev Board, CS5 - to module via FPGA
 *  U = 0x0000_DE22, L = 0x2146_2511, A = 0x2222_0000                       */
#define CSCR5U_CIC_CS5   (CSCRU_CNC(3) + CSCRU_WSC(30) + CSCRU_WWS(2) + \
    CSCRU_EDC(2) )
#define CSCR5L_CIC_CS5   (CSCRL_OEA(2) + CSCRL_OEN(1) + CSCRL_EBWA(4) + \
    CSCRL_EBWN(6) + CSCRL_CSA(2) + CSCRL_DSZ(5) + CSCRL_CSN(1) + CSCRL_CSEN)
#define CSCR5A_CIC_CS5   (CSCRA_EBRA(2) + CSCRA_EBRN(2) + CSCRA_RWA(2) + \
    CSCRA_RWN(2))

/* DDR controller defines for FNET Cicada Dev Board, CS2 - Micron 128Mbyte
   Mobile DDR */
#define ESDCFG_CIC_DDR   (ESDCFG_tXP(3) + ESDCFG_tWTR + ESDCFG_tRP(2) + \
                          ESDCFG_tMRD(1) + ESDCFG_tWR + ESDCFG_tRAS(6) + \
                          ESDCFG_tRRD(1) + ESDCFG_tCAS(3) + \
                          ESDCFG_tRCD(3) + ESDCFG_tRC(15) )

//#define TURN_OFF_IMPRECISE_ABORT

    .macro flash_header
        b 1f
        //Start code from offset 1K, gap left for MBR

        .org 0x400
        .long 0
        .long 0
        MMC_SDHC1_BASE_ADDR_W:	.word	MMC_SDHC1_BASE_ADDR
        ESDHC_INTERRUPT_ENABLE_W:	.word	ESDHC_INTERRUPT_ENABLE
        ESDHC_CLEAR_INTERRUPT_W:	.word	ESDHC_CLEAR_INTERRUPT
        MXC_REDBOOT_ROM_ST_ADDR:	.word	SDRAM_BASE_ADDR + SDRAM_SIZE - 0x100000
        REDBOOT_RESET_VECTOR:	.word	reset_vector
1:
        /* Check if booting from IRAM for MMC boot */
        mov r0, #SDRAM_BASE_ADDR
        cmp pc, r0
        bhs 100f
        setup_sdram ddr X32 DDR 0
        mmcsd_read
        mov r12, #MMC_BOOT
100:
        ldr r0, REDBOOT_RESET_VECTOR
        mov pc, r0


    .endm

    .macro mmcsd_read
        //Configure interface block and number of blocks 1 block and size is 512 Bytes
        mov r2, #MMC_BLK_LEN
        ldr r3, MMC_SDHC1_BASE_ADDR_W
        str r2, [r3, #ESDHC_REG_BLK_LEN]
        mov r2, #1
        str r2, [r3, #ESDHC_REG_NOB]
        //set block size and number of blocks of card
        mov r1, #MMC_START_ADDR
        mov r2, #MMC_BLK_LEN
        sub r10, r1, r2
        ldr r11, MXC_REDBOOT_ROM_ST_ADDR
        mov r12, #MMC_LOAD_SIZE
        add r12, r11, r12

        //set read data length, Comfigure command CMD16 for single block read
        mov r0, #MMC_BLK_LEN
        mov r1, #0x10
        mov r2, #0x1
        send_cmd_wait_resp

read_a_blk:
        //set read data address
        //CMD17 data_present Y
        mov r2, #MMC_BLK_LEN
        add r10, r10, r2
        mov r0, r10
        mov r1, #0x11
        mov r2, #0x9
        send_cmd_wait_resp
        mov r5, #MMC_BLK_LEN
        add r5, r11, r5

        //enable interrupt
        ldr r4, ESDHC_INTERRUPT_ENABLE_W
        str r4, [r3, #ESDHC_REG_INT_STATUS_ENABLE]
read_from_buffer:
        ldr r4, [r3, #ESDHC_REG_INT_STATUS]
        mov r2, #0x80   	//ESDHC_STATUS_BUF_READ_RDY_MSK
        ands r4, r4, r2
        beq read_from_buffer

four_times:	//transfer data from SDHC buffer to ddr(4 words once)
        ldr r4, [r3, #ESDHC_REG_BUFFER_DATA]
        str r4, [r11]
        add r11, r11, #0x4
        ldr r4, [r3, #ESDHC_REG_BUFFER_DATA]
        str r4, [r11]
        add r11, r11, #0x4
        ldr r4, [r3, #ESDHC_REG_BUFFER_DATA]
        str r4, [r11]
        add r11, r11, #0x4
        ldr r4, [r3, #ESDHC_REG_BUFFER_DATA]
        str r4, [r11]
        add r11, r11, #0x4
        cmp r11, r5
        blo read_from_buffer

check_tran_done:      //check if the transfer is over
        ldr r4, [r3, #ESDHC_REG_INT_STATUS]
        mov r2, #0x800     //ESDHC_STATUS_TRANSFER_COMPLETE_MSK
        ands r2, r4, r2
        beq check_tran_done
        ands r2, r2, #0x8
        bne check_tran_done
        cmp r11, r12
        blo read_a_blk
    .endm

    //r0~r2 are reserved
    .macro send_cmd_wait_resp
        //start clk
        ldr r3, MMC_SDHC1_BASE_ADDR_W
        mov r4, #0x2
        str r4, [r3, #ESDHC_REG_CLK]

        //wait until the clk has started
1:
        ldr r4, [r3, #ESDHC_REG_INT_STATUS]
        mov r5,  #0x100
        ands r4, r4, r5
        beq 1b

        //Clear Interrupt status register
        ldr r4, ESDHC_CLEAR_INTERRUPT_W
        str r4, [r3, #ESDHC_REG_INT_STATUS]
        /* Enable Interrupt */
        ldr r4, [r3, #ESDHC_REG_INT_STATUS_ENABLE]
        ldr r5, ESDHC_INTERRUPT_ENABLE_W
        orr r4, r4, r5
        str r4, [r3, #ESDHC_REG_INT_STATUS_ENABLE]

        /* Write Command Argument in Command Argument Register */
        str r1, [r3, #ESDHC_REG_COMMAND]
        str r0, [r3, #ESDHC_REG_COMMAND_TRANS_TYPE]
        str r2, [r3, #ESDHC_REG_COMMAND_DAT_CONT]

2:   //wait for responds
        mov r0, #0
        mov r1, #0x1000
3:
        add r0,r0,#1
        cmp r0,r1
        bne 3b

        ldr r0, [r3, #ESDHC_REG_INT_STATUS]
        mov r1, #0x2000
        ands r1, r0, r1
        beq 2b

        //mask all int
        mov r4, #0
        str r4, [r3, #ESDHC_REG_INT_STATUS_ENABLE]
    .endm

// This macro represents the initial startup code for the platform
    .macro  _platform_setup1
FSL_BOARD_SETUP_START:
/*
 *       ARM1136 init
 *       - invalidate I/D cache/TLB and drain write buffer;
 *       - invalidate L2 cache
 *       - unaligned access
 *       - branch predictions
 */
#ifdef TURN_OFF_IMPRECISE_ABORT
    mrs r0, cpsr
    bic r0, r0, #0x100
    msr cpsr, r0
#endif

    ldr r0, =CP_CONTROL_STARTUP    // R0 Startup value for CP15 Control Reg
    mcr 15, 0, r0, c1, c0, 0           // inits co-processor control reg
    mov r0, #0
    mcr 15, 0, r0, c7, c7, 0        /* invalidate I cache and D cache */
    mcr 15, 0, r0, c8, c7, 0        /* invalidate TLBs */
    mcr 15, 0, r0, c7, c10, 4       /* Drain the write buffer */

    /* Also setup the Peripheral Port Remap register inside the core */
    ldr r0, ARM_PPMRR        /* start from AIPS 2GB region */
    mcr p15, 0, r0, c15, c2, 4

    /* Reload data from spare area to 0x400 of main area if booting from NAND */
    mov r0, #NFC_BASE
    add r1, r0, #0x400
    cmp pc, r0
    blo 1f
    cmp pc, r1
    bhi 1f
#ifdef NFC_2K_BI_SWAP
    ldr r3, [r0, #0x7D0]    // load word at addr 464 of last 512 RAM buffer
    and r3, r3, #0xFFFFFF00 // mask off the LSB
    ldr r4, [r0, #0x834]    // load word at addr 4 of the 3rd spare area buffer
    mov r4, r4, lsr #8      // shift it to get the byte at addr 5
    and r4, r4, #0xFF       // throw away upper 3 bytes
    add r3, r4, r3          // construct the word
    str r3, [r0, #0x7D0]    // write back
#endif

1:
    /*** L2 Cache setup/invalidation/disable ***/
    /* Disable L2 cache first */
    mov r0, #L2CC_BASE_ADDR
    ldr r2, [r0, #L2_CACHE_CTL_REG]
    bic r2, r2, #0x1
    str r2, [r0, #L2_CACHE_CTL_REG]
    /*
     * Configure L2 Cache:
     * - 128k size(16k way)
     * - 8-way associativity
     * - 0 ws TAG/VALID/DIRTY
     * - 4 ws DATA R/W
     */
    ldr r1, [r0, #L2_CACHE_AUX_CTL_REG]
    and r1, r1, #0xFE000000
    ldr r2, L2CACHE_PARAM
    orr r1, r1, r2
    str r1, [r0, #L2_CACHE_AUX_CTL_REG]

    /* Invalidate L2 */
    mov r1, #0x000000FF
    str r1, [r0, #L2_CACHE_INV_WAY_REG]
L2_loop:
    /* Poll Invalidate By Way register */
    ldr r2, [r0, #L2_CACHE_INV_WAY_REG]
    cmp r2, #0
    bne L2_loop
    /*** End of L2 operations ***/

    mov r0, #SDRAM_NON_FLASH_BOOT
    ldr r1, AVIC_VECTOR0_ADDR_W
    str r0, [r1] // for checking boot source from nand, nor or sdram
/*
 * End of ARM1136 init
 */

    /* Do this early for NOR boot.  Doesn't seem to work for NAND boot,
       so we'll repeat it in a place that does work for NAND boot. */
    init_leds

init_spba_start:
    init_spba

init_aips_start:
    /* AIPS setup */
    init_aips
init_max_start:
    /* MAX (Multi-Layer AHB Crossbar Switch) setup */
    init_max
init_m3if_start:
    /* Configure M3IF registers */
    init_m3if

    ldr r11, =CHIP_REV_1_0
    ldr r0, IIM_SREV_REG_VAL
    ldr r1, [r0, #0x0]
    cmp r1, #0x0
    ldrne r11, =CHIP_REV_1_1
    init_drive_strength

init_cs0_start:
    // the parameter to init_cs0 gets prepended with
    // CSCR0U_LOC_ and CSCR0L_LOC_ and CSCR0A_LOC_ within the macro
    // ex. CSCR0U_LOC_PAGE
//  init_cs0 LPD    // Just in case we want to try our code on LPD SOM
    init_cs0 PAGE   // Cicada uses Spansion S29GL01P ASYNC or PAGE mode
//  init_cs0 ASYNC  // Cicada uses Spansion S29GL01P ASYNC or PAGE mode

init_cs1_start:
    init_cs1 ASYNC  // Cicada uses MDOC H3 in ASYNC or BURST mode
        
    /* If SDRAM has been setup, bypass clock/WEIM setup */
#ifdef BOOT_FROM_MMC
    cmp r12, #MMC_BOOT
    ldreq r1, AVIC_VECTOR0_ADDR_W
    streq r12, [r1]
    beq 5f
#endif

    cmp pc, #SDRAM_BASE_ADDR
    blo init_clock_start
    cmp pc, #(SDRAM_BASE_ADDR + SDRAM_SIZE)
    blo HWInitialise_skip_SDRAM_setup

    mov r0, #NOR_FLASH_BOOT
    ldr r1, AVIC_VECTOR0_ADDR_W
    str r0, [r1]

5:

init_clock_start:
    // the parameter to init_clock gets prepended with
    // CCM_PDR0_ and MPCTL_PARAM_ within the macro
    // ex. CCM_PDR0_399_133
    // Legal values = 532_133, 399_133, 360_120, 480_120
//  init_clock 399_133
    init_clock 532_133
//  init_clock 360_120
//  init_clock 480_120

#ifdef BOOT_FROM_MMC
    cmp r12, #MMC_BOOT
    beq HWInitialise_skip_SDRAM_setup
#endif

    /* Based on chip rev, setup params for SDRAM controller */
    ldr r10, =0
    mov r4, #SDRAM_BURST_MODE

init_sdram_start:
#ifndef BOOT_FROM_MMC
    /* Assuming DDR memory first */
    /* setup_sdram( name, bus_width, mode, full_page ) */
    setup_sdram ddr X32 DDR 0
#endif

HWInitialise_skip_SDRAM_setup:

    mov r0, #NFC_BASE
    add r2, r0, #0x800      // 2K window
    cmp pc, r0
    blo Normal_Boot_Continue
    cmp pc, r2
    bhi Normal_Boot_Continue
NAND_Boot_Start:
    /* Copy image from flash to SDRAM first */
    ldr r1, MXC_REDBOOT_ROM_START

1:  ldmia r0!, {r3-r10}
    stmia r1!, {r3-r10}
    cmp r0, r2
    blo 1b
    /* Jump to SDRAM */
    ldr r1, CONST_0x0FFF
    and r0, pc, r1     /* offset of pc */
    ldr r1, MXC_REDBOOT_ROM_START
    add r1, r1, #0x10
    add pc, r0, r1
    nop
    nop
    nop
    nop

NAND_Copy_Main:
    /* Repeating this for NAND boot since the first one doesn't do anything,
       apparently because we are executing from the NFC 2k buffer still. */
    init_leds

    // Check if x16/2kb page
    ldr r7, CCM_BASE_ADDR_W
    ldr r7, [r7, #0xC]
    ands r7, r7, #(1 << 30)

    mov r0, #NAND_FLASH_BOOT
    ldr r1, AVIC_VECTOR0_ADDR_W
    str r0, [r1]
    mov r0, #MXCFIS_NAND
    ldr r1, AVIC_VECTOR1_ADDR_W
    str r0, [r1]

    mov r0, #NFC_BASE;   //r0: nfc base. Reloaded after each page copying
    mov r1, #0x800       //r1: starting flash addr to be copied. Updated constantly
    add r2, r0, #0x800   //r2: end of 3rd RAM buf. Doesn't change
    addeq r2, r0, #0x200   //r2: end of 1st RAM buf. Doesn't change
    add r12, r0, #0xE00  //r12: NFC register base. Doesn't change
    ldr r11, MXC_REDBOOT_ROM_START
    add r13, r11, #REDBOOT_IMAGE_SIZE //r13: end of SDRAM address for copying. Doesn't change
    add r11, r11, r1     //r11: starting SDRAM address for copying. Updated constantly

    //unlock internal buffer
    mov r3, #0x2
    strh r3, [r12, #0xA]

Nfc_Read_Page:
//  NFC_CMD_INPUT(FLASH_Read_Mode1);
    mov r3, #0x0
    nfc_cmd_input

    // Check if x16/2kb page
    ldr r7, CCM_BASE_ADDR_W
    ldr r7, [r7, #0xC]
    ands r7, r7, #(1 << 30)
    bne nfc_addr_ops_2kb
//    start_nfc_addr_ops(ADDRESS_INPUT_READ_PAGE, addr, nflash_dev_info->base_mask);
    mov r3, r1
    do_addr_input       //1st addr cycle
    mov r3, r1, lsr #9
    do_addr_input       //2nd addr cycle
    mov r3, r1, lsr #17
    do_addr_input       //3rd addr cycle
    mov r3, r1, lsr #25
    do_addr_input       //4th addr cycle
    b end_of_nfc_addr_ops

nfc_addr_ops_2kb:
//    start_nfc_addr_ops(ADDRESS_INPUT_READ_PAGE, addr, nflash_dev_info->base_mask);
    mov r3, #0
    do_addr_input       //1st addr cycle
    mov r3, #0
    do_addr_input       //2nd addr cycle
    mov r3, r1, lsr #11
    do_addr_input       //3rd addr cycle
    mov r3, r1, lsr #19
    do_addr_input       //4th addr cycle
    mov r3, r1, lsr #27
    do_addr_input       //5th addr cycle

//    NFC_CMD_INPUT(FLASH_Read_Mode1_2K);
    mov r3, #0x30
    nfc_cmd_input

end_of_nfc_addr_ops:
//    NFC_DATA_OUTPUT(buf, FDO_PAGE_SPARE_VAL);
//        writew(NAND_FLASH_CONFIG1_INT_MSK | NAND_FLASH_CONFIG1_ECC_EN,
//               NAND_FLASH_CONFIG1_REG);
    mov r8, #0
    bl nfc_data_output
    bl do_wait_op_done
    // Check if x16/2kb page
    ldr r7, CCM_BASE_ADDR_W
    ldr r7, [r7, #0xC]
    ands r7, r7, #(1 << 30)
    beq nfc_addr_data_output_done_512

// For 2K page - 2nd 512
    mov r8, #1
    bl nfc_data_output
    bl do_wait_op_done

// 3rd 512
    mov r8, #2
    bl nfc_data_output
    bl do_wait_op_done

// 4th 512
    mov r8, #3
    bl nfc_data_output
    bl do_wait_op_done
// end of 4th
#ifdef NFC_2K_BI_SWAP
   ldr r3, [r0, #0x7D0]    // load word at addr 464 of last 512 RAM buffer
//   and r7, r3, #0xFF       // load the BBI data
   and r3, r3, #0xFFFFFF00 // mask off the LSB
   ldr r4, [r0, #0x834]    // load word at addr 4 of the 3rd spare area buffer
   mov r4, r4, lsr #8      // shift it to get the byte at addr 5
   and r4, r4, #0xFF       // throw away upper 3 bytes
   add r3, r4, r3          // construct the word
   str r3, [r0, #0x7D0]    // write back
#endif
    // check for bad block
    mov r3, r1, lsl #(32-17)    // get rid of block number
    cmp r3, #(0x800 << (32-17)) // check if not page 0 or 1
    b nfc_addr_data_output_done

nfc_addr_data_output_done_512:
    // check for bad block
    mov r3, r1, lsl #(32-5-9)    // get rid of block number
    cmp r3, #(512 << (32-5-9))   // check if not page 0 or 1

nfc_addr_data_output_done:
    bhi Copy_Good_Blk
#ifdef NFC_2K_BI_SWAP
    mov r4, r7, lsl #8
#else
    add r4, r0, #0x800  //r4 -> spare area buf 0
    ldrh r4, [r4, #0x4]
#endif
    and r4, r4, #0xFF00
    cmp r4, #0xFF00
    beq Copy_Good_Blk
//    b Copy_Good_Blk //b01969: just for test
    // really sucks. Bad block!!!!
    cmp r3, #0x0
    beq Skip_bad_block
    // even suckier since we already read the first page!
    // Check if x16/2kb page
    ldr r7, CCM_BASE_ADDR_W
    ldr r7, [r7, #0xC]
    ands r7, r7, #(1 << 30)

    subeq r11, r11, #512  //rewind 1 page for the sdram pointer
    subeq r1, r1, #512    //rewind 1 page for the flash pointer

    // for 2k page
    subne r11, r11, #0x800  //rewind 1 page for the sdram pointer
    subne r1, r1, #0x800    //rewind 1 page for the flash pointer

Skip_bad_block:
    // Check if x16/2kb page
    ldr r7, CCM_BASE_ADDR_W
    ldr r7, [r7, #0xC]
    ands r7, r7, #(1 << 30)

    addeq r1, r1, #(32*512)
    addne r1, r1, #(64*2048)

    b Nfc_Read_Page
Copy_Good_Blk:
    //copying page
1:  ldmia r0!, {r3-r10}
    stmia r11!, {r3-r10}
    cmp r0, r2
    blo 1b
    cmp r11, r13
    bge NAND_Copy_Main_done
    // Check if x16/2kb page
    ldr r7, CCM_BASE_ADDR_W
    ldr r7, [r7, #0xC]
    ands r7, r7, #(1 << 30)
    addeq r1, r1, #0x200
    addne r1, r1, #0x800
    mov r0, #NFC_BASE
    b Nfc_Read_Page

NAND_Copy_Main_done:
    led_off LED1                // only on NAND boot

Normal_Boot_Continue:

#ifdef CYG_HAL_STARTUP_ROMRAM     /* enable running from RAM */
    /* Copy image from Present location to High RAM first */
    ldr r0, =0xFFFFF000
    and r0, r0, pc
    ldr r1, MXC_REDBOOT_ROM_START
    cmp r0, r1                    // Skip copy if already in high RAM
    beq HWInitialise_skip_SDRAM_copy

    add r2, r0, #REDBOOT_IMAGE_SIZE

1:  ldmia r0!, {r3-r10}
    stmia r1!, {r3-r10}
    cmp r0, r2
    ble 1b

    /* Jump to SDRAM */

    ldr r1, =0xFFFF
    and r0, pc, r1         /* offset of pc */
    ldr r1, =(SDRAM_BASE_ADDR + SDRAM_SIZE - 0x100000 + 0x8)
    add pc, r0, r1
    nop
    nop
    nop
    nop

#endif /* CYG_HAL_STARTUP_ROMRAM */

HWInitialise_skip_SDRAM_copy:
    led_off LED1                // only on NOR, SRAM boot

STACK_Setup:
    // Set up a stack [for calling C code]
    ldr r1, =__startup_stack
    ldr r2, =RAM_BANK0_BASE
    orr sp, r1, r2

    // Create MMU tables
    bl hal_mmu_init

    // Enable MMU
    ldr r2, =10f
    mrc MMU_CP, 0, r1, MMU_Control, c0      // get c1 value to r1 first
    orr r1, r1, #7                          // enable MMU bit
    orr r1, r1, #0x800                      // enable z bit
    mcr MMU_CP, 0, r1, MMU_Control, c0
    mov pc,r2    /* Change address spaces */
    nop
    nop
    nop
10:

// init extra chip selects for hardware not used during early boot
init_cs4_start:
    init_cs4
init_cs5_start:
    init_cs5

#if 0
    // Save shadow copy of BCR, also hardware configuration
    ldr r1, =_board_BCR
    str r2, [r1]
    ldr r1, =_board_CFG
    str r9, [r1]                // Saved far above...
#endif


    led_off LED2                // for both NOR, NAND and SRAM boot

    .endm                       // _platform_setup1

do_wait_op_done:
    1:
        ldrh r3, [r12, #NAND_FLASH_CONFIG2_REG_OFF]
        ands r3, r3, #NAND_FLASH_CONFIG2_INT_DONE
        beq 1b
    bx lr     // do_wait_op_done

nfc_data_output:
    mov r3, #(NAND_FLASH_CONFIG1_INT_MSK | NAND_FLASH_CONFIG1_ECC_EN)
    strh r3, [r12, #NAND_FLASH_CONFIG1_REG_OFF]

    // writew(buf_no, RAM_BUFFER_ADDRESS_REG);
    strh r8, [r12, #RAM_BUFFER_ADDRESS_REG_OFF]
    // writew(FDO_PAGE_SPARE_VAL & 0xFF, NAND_FLASH_CONFIG2_REG);
    mov r3, #FDO_PAGE_SPARE_VAL
    strh r3, [r12, #NAND_FLASH_CONFIG2_REG_OFF]
    bx lr

#else // defined(CYG_HAL_STARTUP_ROM) || defined(CYG_HAL_STARTUP_ROMRAM)
#define PLATFORM_SETUP1
#endif

    /* Do nothing */
    .macro  init_spba
    .endm  /* init_spba */

    /* AIPS setup - Only setup MPROTx registers. The PACR default values are good.*/
    .macro init_aips
        /*
         * Set all MPROTx to be non-bufferable, trusted for R/W,
         * not forced to user-mode.
         */
        ldr r0, AIPS1_CTRL_BASE_ADDR_W
        ldr r1, AIPS1_PARAM_W
        str r1, [r0, #0x00]
        str r1, [r0, #0x04]
        ldr r0, AIPS2_CTRL_BASE_ADDR_W
        str r1, [r0, #0x00]
        str r1, [r0, #0x04]

        /*
         * Clear the on and off peripheral modules Supervisor Protect bit
         * for SDMA to access them. Did not change the AIPS control registers
         * (offset 0x20) access type
         */
        ldr r0, AIPS1_CTRL_BASE_ADDR_W
        ldr r1, =0x0
        str r1, [r0, #0x40]
        str r1, [r0, #0x44]
        str r1, [r0, #0x48]
        str r1, [r0, #0x4C]
        ldr r1, [r0, #0x50]
        and r1, r1, #0x00FFFFFF
        str r1, [r0, #0x50]

        ldr r0, AIPS2_CTRL_BASE_ADDR_W
        ldr r1, =0x0
        str r1, [r0, #0x40]
        str r1, [r0, #0x44]
        str r1, [r0, #0x48]
        str r1, [r0, #0x4C]
        ldr r1, [r0, #0x50]
        and r1, r1, #0x00FFFFFF
        str r1, [r0, #0x50]
    .endm /* init_aips */

    /* MAX (Multi-Layer AHB Crossbar Switch) setup */
    .macro init_max
        ldr r0, MAX_BASE_ADDR_W
        /* MPR - priority is M4 > M2 > M3 > M5 > M0 > M1 */
        ldr r1, MAX_PARAM1
        str r1, [r0, #0x000]        /* for S0 */
        str r1, [r0, #0x100]        /* for S1 */
        str r1, [r0, #0x200]        /* for S2 */
        str r1, [r0, #0x300]        /* for S3 */
        str r1, [r0, #0x400]        /* for S4 */
        /* SGPCR - always park on last master */
        ldr r1, =0x10
        str r1, [r0, #0x010]        /* for S0 */
        str r1, [r0, #0x110]        /* for S1 */
        str r1, [r0, #0x210]        /* for S2 */
        str r1, [r0, #0x310]        /* for S3 */
        str r1, [r0, #0x410]        /* for S4 */
        /* MGPCR - restore default values */
        ldr r1, =0x0
        str r1, [r0, #0x800]        /* for M0 */
        str r1, [r0, #0x900]        /* for M1 */
        str r1, [r0, #0xA00]        /* for M2 */
        str r1, [r0, #0xB00]        /* for M3 */
        str r1, [r0, #0xC00]        /* for M4 */
        str r1, [r0, #0xD00]        /* for M5 */
    .endm /* init_max */

    /* Clock setup */
    .macro    init_clock clock_speed
        ldr r0, IPU_CTRL_BASE_ADDR_W  // init IPU before CCM
        ldr r1, =0x40
        str r1, [r0]

        ldr r0, CCM_BASE_ADDR_W
        // default CLKO to 1/4 of the AHB clock
        mov r1, #0x00000280
        add r1, r1, #0x00000005
        str r1, [r0, #CLKCTL_COSR]

        ldr r2, CCM_CCMR_0x074B0BF5
        ldr r3, CCM_CCMR_0x074B0BFD
        ldr r4, CCM_CCMR_0x074B0B7D

        // Make sure to use CKIH
        ldr r1, [r0, #CLKCTL_CCMR]
        bic r1, r1, #0x8            // disable PLL first
        str r1, [r0, #CLKCTL_CCMR]
        str r2, [r0, #CLKCTL_CCMR]  // select CKIH (26MHz) as PLL ref clock
        ldr r1, =0x1000             // wait loop for PLL to settle
    1:
        subs r1, r1, #0x1
        bne 1b
// FIXME!!! BKG: loop above and enable of PLL below seem out of sequence to me.
// Also the loop is very slow because the ARM is running at the 26MHz reference
// clock rate while the PLL is disabled.
        str r3, [r0, #CLKCTL_CCMR]  // enable PLL
        str r4, [r0, #CLKCTL_CCMR]  // switch to PLL (SPLL for FIR)

        // MCU PLL & Clock Divider to clock_speed
        ldr r1, CCM_PDR0_\clock_speed
        str r1, [r0, #CLKCTL_PDR0]
        ldr r1, MPCTL_PARAM_\clock_speed
        str r1, [r0, #CLKCTL_MPCTL]

        /* Set UPLL=240MHz, USB=60MHz */
        ldr r1, CCM_PDR1_0x49FCFE7F
        str r1, [r0, #CLKCTL_PDR1]
        ldr r1, CCM_UPCTL_PARAM_240
        str r1, [r0, #CLKCTL_UPCTL]
    .endm /* init_clock */

    /* M3IF setup */
    .macro init_m3if
        /* Configure M3IF registers */
        ldr r1, M3IF_BASE_W
        /*
        * M3IF Control Register (M3IFCTL)
        * MRRP[0] = L2CC0 not on priority list (0 << 0)        = 0x00000000
        * MRRP[1] = L2CC1 not on priority list (0 << 0)        = 0x00000000
        * MRRP[2] = MBX not on priority list (0 << 0)        = 0x00000000
        * MRRP[3] = MAX1 not on priority list (0 << 0)        = 0x00000000
        * MRRP[4] = SDMA not on priority list (0 << 0)        = 0x00000000
        * MRRP[5] = MPEG4 not on priority list (0 << 0)       = 0x00000000
        * MRRP[6] = IPU1 on priority list (1 << 6)             = 0x00000040
        * MRRP[7] = IPU2 not on priority list (0 << 0)   = 0x00000000
        *                                                       ------------
        *                                                       0x00000040
        */
        ldr r0, =0x00000040
        str r0, [r1]  /* M3IF control reg */
    .endm /* init_m3if */

    /* CS0 mode setup */
    .macro init_cs0 mode
        ldr r0, =WEIM_CTRL_CS0     // R0 to base address  for CS0
        ldr r1, CSCR0U_LOC_\mode   // Cicada Dev Board, PAGE or ASYNC mode
        str r1, [r0, #CSCRU]
        ldr r1, CSCR0L_LOC_\mode   // Cicada Dev Board, PAGE or ASYNC mode
        str r1, [r0, #CSCRL]
        ldr r1, CSCR0A_LOC_\mode   // Cicada Dev Board, PAGE or ASYNC mode
        str r1, [r0, #CSCRA]
    .endm /* init_cs0 mode */

    /* CS1 mode setup */
    .macro init_cs1 mode
        ldr r0, =WEIM_CTRL_CS1     // R0 to base address  for CS1
        ldr r1, CSCR1U_LOC_\mode   // Cicada MDOC, ASYNC or BURST mode
        str r1, [r0, #CSCRU]
        ldr r1, CSCR1L_LOC_\mode   // Cicada MDOC, ASYNC or BURST mode
        str r1, [r0, #CSCRL]
        ldr r1, CSCR1A_LOC_\mode   // Cicada MDOC, ASYNC or BURST mode
        str r1, [r0, #CSCRA]
    .endm /* init_cs1 mode */

    /* FPGA Internal on CS4 setup */
    .macro init_cs4
        ldr r0, =WEIM_CTRL_CS4     // R0 to base address  for CS4
        ldr r1, =CSCR4U_CIC_CS4    // Cicada Dev Board, FPGA Internal
        str r1, [r0, #CSCRU]
        ldr r1, =CSCR4L_CIC_CS4    // Cicada Dev Board, FPGA Internal
        str r1, [r0, #CSCRL]
        ldr r1, =CSCR4A_CIC_CS4    // Cicada Dev Board, FPGA Internal
        str r1, [r0, #CSCRA]
    .endm /* init_cs4 */

    /* FPGA to Module on CS5 setup */
    .macro init_cs5
        ldr r0, =WEIM_CTRL_CS5     // R0 to base address  for CS5
        ldr r1, =CSCR5U_CIC_CS5    // Cicada Dev Board, FPGA External
        str r1, [r0, #CSCRU]
        ldr r1, =CSCR5L_CIC_CS5    // Cicada Dev Board, FPGA External
        str r1, [r0, #CSCRL]
        ldr r1, =CSCR5A_CIC_CS5    // Cicada Dev Board, FPGA External
        str r1, [r0, #CSCRA]
    .endm /* init_cs5 */

    /* IO Mux control and data for DEBUG LEDS (SDIO CARD LEDS) */
    .macro init_leds
        // set SCLK0 and SRST0 (GPIO 3_2 & 3_3) to GPIO
        ldr r0, =IOMUXC_BASE_ADDR  // R0 to base address  for IO Mux Control
        ldr r1, =MUX_ALL_GPIO
        str r1, [r0, #GPIO3_1_OFFSET]  

        // set SCLK0 and SRST0 (GPIO 3_2 & 3_3) to output
        ldr r0, =GPIO3_BASE_ADDR  // R0 to base address  for GPIO 2
        ldr r1, [r0, #0x4]  // Direction Register
        orr  r1, r1, #0x0000000C
        str r1, [r0, #0x04]

        // set both SCLK0 (SD1-LED) and SRST0 (SD2-LED) to 1
        ldr r1, [r0]       // Data Register
        orr  r1, r1, #0x0000000C
        str r1, [r0]
    .endm /* init_leds */

    /* Toggle an LED for DEBUG (SDIO CARD LEDS) */
    /* Legal arguments are LED1 and LED2 */
    .macro led_toggle xxled
        ldr r0, =GPIO3_BASE_ADDR  // R0 to base address  for GPIO 2
        ldr r1, [r0]
        ldr r2, BIT_\xxled
        eor r1, r1, r2
        str r1, [r0]
    .endm /* led_toggle */

    /* Turn on LED for DEBUG (SDIO CARD LEDS) */
    /* Legal arguments are LED1 and LED2 */
    .macro led_on xxled
        ldr r0, =GPIO3_BASE_ADDR  // R0 to base address  for GPIO 2
        ldr r1, [r0]
        ldr r2, BIT_\xxled
        orr r1, r1, r2
        str r1, [r0]
    .endm /* led_on */

    /* Turn off LED for DEBUG (SDIO CARD LEDS) */
    /* Legal arguments are LED1 and LED2 */
    .macro led_off xxled
        ldr r0, =GPIO3_BASE_ADDR  // R0 to base address  for GPIO 2
        ldr r1, [r0]
        ldr r2, BIT_\xxled
        bic r1, r1, r2
        str r1, [r0]
    .endm /* led_off */

    .macro setup_sdram, name, bus_width, mode, full_page
        /* It sets the "Z" flag in the CPSR at the end of the macro */
        ldr r0, ESDCTL_BASE_W
        mov r2, #SDRAM_BASE_ADDR
        ldr r1, ESDCFG_LOC_DDR
        str r1, [r0, #ESDCTL_ESDCFG0]
        ldr r1, =ESDMISC_RST            // reset
        str r1, [r0, #ESDCTL_ESDMISC]
        ldr r1, SDRAM_MISC1_\mode
        str r1, [r0, #ESDCTL_ESDMISC]

        // Wait for 200us SDRAM/LPDDR external memory wakeup period
0:        ldr  r1, [r0, #ESDCTL_ESDMISC]
        ands r1, r1, #ESDMISC_SDRAMDRY
        beq  0b

        // Hold for more than 200ns
        ldr r1, =0x10000
1:
        subs r1, r1, #0x1
        bne 1b

        // ESDCTL <- Enable Precharge Command, 13 rows x 9 columns
        ldr r1, SDRAM_PRECHARGE
        str r1, [r0, #ESDCTL_ESDCTL0]
        // Precharge all banks
        ldr r1, =0x0
        ldr r12, SDRAM_PARAM2_\mode
        str r1, [r12]
        // ESDCTL <- Enable Auto Refresh Command, 13 rows x 9 columns
        ldr r1, SDRAM_AUTO_REFRESH
        str r1, [r0, #ESDCTL_ESDCTL0]
        // repeat auto refresh 8 times
        ldr r1, =0x0
        ldr r12, =0x7
2:
        str r1, [r2]
        subs r12, r12, #1
        bne 2b

        // ESDCTL <- Enable Load Mode Register  Command
        ldr r1, SDRAM_LOAD_MODE
        str r1, [r0, #ESDCTL_ESDCTL0]
        // init memory mode register via address bits
        ldr r1, =0x0
        .if \full_page
        strb r1, [r2, #SDRAM_FULL_PAGE_MODE]
        .else
        strb r1, [r2, #SDRAM_BURST_MODE]
        .endif

        // init LP extended mode register via address bits.
        // WARNING: may be bank size dependent.
        ldr r1, =0xFF
        ldr r12, =0x81000000
        strb r1, [r12]

        // Form final ESDCTL based on mode, bus width and full page params
        ldr r3, SDRAM_NORMAL
        ldr r4, SDRAM_PARAM3_\mode
        add r3, r3, r4
        ldr r4, SDRAM_PARAM4_\bus_width
        add r3, r3, r4
        .if \full_page
        add r3, r3, #ESDCTL_FP   /* Force to full page mode */
        .endif
        str r3, [r0, #ESDCTL_ESDCTL0]        // load ESDCTL
        ldr r1, =0x0
        str r1, [r2]        // activate normal mode operation

        /* Below only for DDR */
        ldr r1, [r0, #ESDCTL_ESDMISC]
        ands r1, r1, #ESDMISC_MDDREN        // testing for DDR mode set
        ldrne r1, =(ESDMISC_MDDR_DL_RST + ESDMISC_MDDREN)  //0x0000000C
        strne r1, [r0, #ESDCTL_ESDMISC]        // reset LPDDR delay line???

        /* Testing if it is truly DDR */
        ldr r1, SDRAM_0x55555555
        ldr r0, =SDRAM_BASE_ADDR
        str r1, [r0]
        ldr r2, SDRAM_0xAAAAAAAA
        str r2, [r0, #0x4]
        ldr r2, [r0]
        cmp r1, r2
    .endm

    .macro nfc_cmd_input
        strh r3, [r12, #NAND_FLASH_CMD_REG_OFF]
        mov r3, #NAND_FLASH_CONFIG2_FCMD_EN;
        strh r3, [r12, #NAND_FLASH_CONFIG2_REG_OFF]
        bl do_wait_op_done
    .endm   // nfc_cmd_input

    .macro do_addr_input
        and r3, r3, #0xFF
        strh r3, [r12, #NAND_FLASH_ADD_REG_OFF]
        mov r3, #NAND_FLASH_CONFIG2_FADD_EN
        strh r3, [r12, #NAND_FLASH_CONFIG2_REG_OFF]
        bl do_wait_op_done
    .endm   // do_addr_input

    /* To support 133MHz DDR */
    .macro  init_drive_strength
        /*
         * Disable maximum drive strength SDRAM/DDR lines by clearing DSE1 bits
         * in SW_PAD_CTL registers
         */

        // SDCLK
        ldr r1, IOMUXC_BASE_ADDR_W
        add r1, r1, #0x200
        // Now r1 = (IOMUX_BASE_ADDR + 0x200)
        ldr r0, [r1, #0x6C]
        bic r0, r0, #(1 << 12)
        str r0, [r1, #0x6C]
        
        // CAS
        ldr r0, [r1, #0x70]
        bic r0, r0, #(1 << 22)
        str r0, [r1, #0x70]
        
        // RAS
        ldr r0, [r1, #0x74]
        bic r0, r0, #(1 << 2)
        str r0, [r1, #0x74]
        
        // CS2 (CSD0)
        ldr r0, [r1, #0x7C]
        bic r0, r0, #(1 << 22)
        str r0, [r1, #0x7C]
        
        // DQM3
        ldr r0, [r1, #0x84]
        bic r0, r0, #(1 << 22)
        str r0, [r1, #0x84]
        
        // DQM2, DQM1, DQM0, SD31-SD0, A25-A0, MA10 (0x288..0x2DC)
        ldr r2, =22     // (0x2E0 - 0x288) / 4 = 22
pad_loop:
        ldr r0, [r1, #0x88]
        bic r0, r0, #(1 << 22)
        bic r0, r0, #(1 << 12)
        bic r0, r0, #(1 << 2)
        str r0, [r1, #0x88]
        add r1, r1, #4
        subs r2, r2, #0x1
        bne pad_loop
    .endm /* init_drive_strength */

#if 0
#define PLATFORM_VECTORS         _platform_vectors
    .macro  _platform_vectors
        .globl  _board_BCR, _board_CFG
_board_BCR:   .long   0       // Board Control register shadow
_board_CFG:   .long   0       // Board Configuration (read at RESET)
    .endm
#endif
    
ARM_PPMRR:              .word   0x40000015
L2CACHE_PARAM:          .word   0x00030024
IIM_SREV_REG_VAL:       .word   IIM_BASE_ADDR + IIM_SREV_OFF
AIPS1_CTRL_BASE_ADDR_W: .word   AIPS1_CTRL_BASE_ADDR
AIPS2_CTRL_BASE_ADDR_W: .word   AIPS2_CTRL_BASE_ADDR
AIPS1_PARAM_W:          .word   0x77777777
MAX_BASE_ADDR_W:        .word   MAX_BASE_ADDR
MAX_PARAM1:             .word   0x00302154
CLKCTL_BASE_ADDR_W:     .word   CLKCTL_BASE_ADDR
ESDCTL_BASE_W:          .word   ESDCTL_BASE
M3IF_BASE_W:            .word   M3IF_BASE
SDRAM_MISC1_DDR:        .word   ESDMISC_MDDREN
SDRAM_MISC1_SDR:        .word   0x0
SDRAM_PARAM2_DDR:       .word   0x80000F00
SDRAM_PARAM2_SDR:       .word   0x80000400
SDRAM_PARAM3_DDR:       .word   ESDCTL_COL(ESD_COL_INCR)
SDRAM_PARAM3_SDR:       .word   0x0
SDRAM_PARAM4_X32:       .word   ESDCTL_DSIZ(ESD_DSIZ_32)
SDRAM_PARAM4_X16:       .word   ESDCTL_DSIZ(ESD_DSIZ_16_LOW)
SDRAM_0x55555555:       .word   0x55555555
SDRAM_0xAAAAAAAA:       .word   0xAAAAAAAA
SDRAM_PRECHARGE:        .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_PRECHARGE) + \
                                 ESDCTL_ROW(ESD_ROW_13) + \
                                 ESDCTL_COL(ESD_COL_9))
SDRAM_AUTO_REFRESH:     .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_AUTO_REFSH) + \
                                 ESDCTL_ROW(ESD_ROW_13) + \
                                 ESDCTL_COL(ESD_COL_9))
SDRAM_LOAD_MODE:        .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_LOAD_MODE) + \
                                 ESDCTL_ROW(ESD_ROW_13) + \
                                 ESDCTL_COL(ESD_COL_9))
SDRAM_NORMAL:           .word   (ESDCTL_SDE + \
                                 ESDCTL_SMODE(ESD_SMODE_NORMAL) + \
                                 ESDCTL_ROW(ESD_ROW_13) + \
                                 ESDCTL_COL(ESD_COL_9) + \
                                 ESDCTL_SREFR(ESD_SREFR_4_8192) + \
                                 ESDCTL_PWDT(ESD_PWDT_DISABLE) + \
                                 ESDCTL_BL_8 )
ESDCFG_LOC_DDR:         .word   ESDCFG_CIC_DDR
IOMUXC_BASE_ADDR_W:     .word   IOMUXC_BASE_ADDR
CCM_PDR0_532_133:       .word   PDR0_532_133_66
CCM_PDR0_480_120:       .word   PDR0_480_120_60
CCM_PDR0_399_133:       .word   PDR0_399_133_66
CCM_PDR0_360_120:       .word   PDR0_360_120_60
MPCTL_PARAM_532_133:    .word   MPCTL_PARAM_532
MPCTL_PARAM_399_133:    .word   MPCTL_PARAM_399
MPCTL_PARAM_360_120:    .word   MPCTL_PARAM_360
MPCTL_PARAM_480_120:    .word   MPCTL_PARAM_480
CCM_PDR1_0x49FCFE7F:    .word   0x49FCFE7F
CCM_UPCTL_PARAM_240:    .word   UPCTL_PARAM_240
AVIC_VECTOR0_ADDR_W:    .word   MXCBOOT_FLAG_REG
AVIC_VECTOR1_ADDR_W:    .word   MXCFIS_FLAG_REG
MXC_REDBOOT_ROM_START:  .word   SDRAM_BASE_ADDR + SDRAM_SIZE - 0x100000
CONST_0x0FFF:           .word   0x0FFF
CCM_BASE_ADDR_W:        .word   CCM_BASE_ADDR
IPU_CTRL_BASE_ADDR_W:   .word   IPU_CTRL_BASE_ADDR
CCM_CCMR_0x074B0BF5:    .word   0x074B0BF5
CCM_CCMR_0x074B0BFD:    .word   0x074B0BFD
CCM_CCMR_0x074B0B7D:    .word   0x074B0B7D
CSCR0U_LOC_ASYNC:       .word   CSCR0U_CIC_ASYNC
CSCR0L_LOC_ASYNC:       .word   CSCR0L_CIC_ASYNC
CSCR0A_LOC_ASYNC:       .word   CSCR0A_CIC_ASYNC
CSCR0U_LOC_PAGE:        .word   CSCR0U_CIC_PAGE
CSCR0L_LOC_PAGE:        .word   CSCR0L_CIC_PAGE
CSCR0A_LOC_PAGE:        .word   CSCR0A_CIC_PAGE
CSCR1U_LOC_ASYNC:       .word   CSCR1U_CIC_ASYNC
CSCR1L_LOC_ASYNC:       .word   CSCR1L_CIC_ASYNC
CSCR1A_LOC_ASYNC:       .word   CSCR1A_CIC_ASYNC
CSCR1U_LOC_BURST:       .word   CSCR1U_CIC_BURST
CSCR1L_LOC_BURST:       .word   CSCR1L_CIC_BURST
CSCR1A_LOC_BURST:       .word   CSCR1A_CIC_BURST
BIT_LED1:               .word   (1 << 2)
BIT_LED2:               .word   (1 << 3)

/*---------------------------------------------------------------------------*/
/* end of hal_platform_setup.h                                               */
#endif /* CYGONCE_HAL_PLATFORM_SETUP_H */
