// Copyright (c) (2011) Fluke Corporation. All rights reserved.
//
// $Id: MouseKeySerialize.h 2013-01-22 jzhu9 $
// $URL: https://engsvnhost.tc.fluke.com/repos/fnet.pit.cicada/trunk/cicada-user/libtarget/fnet-FLOSS/uitouch/TouchScreenSerialize.h $

/*! \file TouchScreenSerialize.h 
  * \brief Helper header for serializing touch data
  *
  * This file provides serialization definition for touch point related
  * data for replay functionality.
  */

#ifndef TOUCHSCREENSERIALIZE_H_
#define TOUCHSCREENSERIALIZE_H_


#include <cstddef>

#include <QDataStream>
#include <QFile>
#include <QIODevice>
#include <iostream>

#include <QDateTime>
#include <QTouchEvent>
#include <QListIterator>
#include <QRectF>
#include <QPointF>


QDataStream &operator<<(QDataStream &s, const Qt::TouchPointState& state) {
	int stateIn = int(state);
	s << stateIn;

	return s;
}

QDataStream &operator>>(QDataStream& s, Qt::TouchPointState& state) {
	int stateOut;
	s >> stateOut;
	state = Qt::TouchPointState(stateOut);

	return s;
}

QDataStream &operator<<(QDataStream &s, const QTouchEvent::TouchPoint& touchpoint) {
	// list of functions obtained from here: http://docs.autodesk.com/MUD/2013/ENU/Mudbox-SDK-Documentation/index.html?url=cpp_ref/class_q_touch_event_1_1_touch_point.html,topicNumber=cpp_ref_class_q_touch_event_1_1_touch_point_html,hash=ae0c063cfb2479f614be43b9d2f92ba63
	s << touchpoint.id();
	s << touchpoint.state();
	s << touchpoint.isPrimary();
	s << touchpoint.rect();			// The rect values used to be later in the order
	s << touchpoint.sceneRect();	//
	s << touchpoint.screenRect();	//
	s << touchpoint.pos();
	s << touchpoint.startPos();
	s << touchpoint.lastPos();
	s << touchpoint.scenePos();
	s << touchpoint.startScenePos();
	s << touchpoint.lastScenePos();
	s << touchpoint.screenPos();
	s << touchpoint.startScreenPos();
	s << touchpoint.lastScreenPos();
	s << touchpoint.normalizedPos();
	s << touchpoint.startNormalizedPos();
	s << touchpoint.lastNormalizedPos();
//	s << touchpoint.rect();
//	s << touchpoint.sceneRect();
//	s << touchpoint.screenRect();
	s << touchpoint.pressure();

	return s;
}

QDataStream &operator>>(QDataStream& s, QTouchEvent::TouchPoint& touchpoint) {
	int id;
	Qt::TouchPointState state;		// Qt::TouchPointState is an enum in hex: http://doc.qt.nokia.com/4.7/qt.html#TouchPointState-enum
	bool isPrimary;
	QRectF rect;		// The rect values used to be later in the order
	QRectF sceneRect;
	QRectF screenRect;
	QPointF pos;
	QPointF startPos;
	QPointF lastPos;
	QPointF scenePos;
	QPointF startScenePos;
	QPointF lastScenePos;
	QPointF screenPos;
	QPointF startScreenPos;
	QPointF lastScreenPos;
	QPointF normalizedPos;
	QPointF startNormalizedPos;
	QPointF lastNormalizedPos;
//	QRectF rect;
//	QRectF sceneRect;
//	QRectF screenRect;
	qreal pressure;

	s >> id >> state >> isPrimary >> rect >> sceneRect >> screenRect >> pos >> startPos >> lastPos >> scenePos >> startScenePos >> lastScenePos >> screenPos >> startScreenPos >>
		 lastScreenPos >> normalizedPos >> startNormalizedPos >> lastNormalizedPos >> pressure;

	touchpoint.setId(id);
	touchpoint.setState(state);
	// touchpoint.setPrimary(isPrimary);	// there is not function to set isPrimary... how do we do it?
	touchpoint.setRect(rect);				// The rect values used to be later in the list
	touchpoint.setSceneRect(sceneRect);
	touchpoint.setScreenRect(screenRect);
	touchpoint.setPos(pos);
	touchpoint.setStartPos(startPos);
	touchpoint.setLastPos(lastPos);
	touchpoint.setScenePos(scenePos);
	touchpoint.setStartScenePos(startScenePos);
	touchpoint.setLastScenePos(lastScenePos);
	touchpoint.setScreenPos(screenPos);
	touchpoint.setStartScreenPos(startScreenPos);
	touchpoint.setLastScreenPos(lastScreenPos);
	touchpoint.setNormalizedPos(normalizedPos);
	touchpoint.setStartNormalizedPos(startNormalizedPos);
	touchpoint.setLastNormalizedPos(lastNormalizedPos);
//	touchpoint.setRect(rect);
//	touchpoint.setSceneRect(sceneRect);
//	touchpoint.setScreenRect(screenRect);
	touchpoint.setPressure(pressure);

	return s;
}

QDataStream &operator<<(QDataStream &s, const QList<QTouchEvent::TouchPoint>& list) {
	s << list.length();
	QListIterator<QTouchEvent::TouchPoint> i(list);
	while (i.hasNext()) {
		s << i.next();
	}

	return s;
}

QDataStream &operator>>(QDataStream& s, QList<QTouchEvent::TouchPoint>& list) {
	int size;
	s >> size;
	for (int i=0; i<size; i++) {
		QTouchEvent::TouchPoint temp;
		s >> temp;
		list.append(temp);
	}

	return s;
}



class TouchPointLogItem {
public:
	QList<QTouchEvent::TouchPoint> sentTouchList;
	int timestamp;

	TouchPointLogItem();
	TouchPointLogItem(QList<QTouchEvent::TouchPoint> sTL, int ts);
};

TouchPointLogItem::TouchPointLogItem() {
	QList<QTouchEvent::TouchPoint> templist;
	TouchPointLogItem(templist, 0);
}

TouchPointLogItem::TouchPointLogItem(QList<QTouchEvent::TouchPoint> sTL, int ts) {
	sentTouchList = sTL;
	timestamp = ts;
}

QDataStream &operator<<(QDataStream& s, const TouchPointLogItem& log) {
	s << log.timestamp;
	s << log.sentTouchList;

	return s;
}

QDataStream &operator>>(QDataStream& s, TouchPointLogItem& log) {
	s >> log.timestamp;
	s >> log.sentTouchList;

	return s;
}

QDataStream &operator<<(QDataStream& s, const QList<TouchPointLogItem>& list) {
	s << list.length();
	QListIterator<TouchPointLogItem> i(list);
	while (i.hasNext()) {
		s << i.next();
	}

	return s;
}

QDataStream &operator>>(QDataStream& s, QList<TouchPointLogItem>& list) {
	int size;
	s >> size;
	for (int i=0; i<size; i++) {
		TouchPointLogItem temp;
		s >> temp;
		list.append(temp);
	}

	return s;
}

#endif /* TOUCHSCREENSERIALIZE_H_ */
