// Copyright (c) (2011) Fluke Corporation. All rights reserved.
//
// $Id: MantisTouchScreen.h 4212 2010-09-10 21:01:00Z campbell $
// $URL: https://engsvnhost.tc.fluke.com/repos/fnet.pit.cicada/trunk/FLOSS/fnet/uitouch/MantisTouchScreen.h $

/*! \file  MantisTouchScreen.h
 *  \brief Raw touch event translation class declaration.
 *
 *  This file declares an object used to manage the translation of raw input
 *  from the kernel touch screen driver into QTouchEvents in the UI event queue.
 *
 *  Its implementation is based on code within the following files of the
 *  qt-everywhere-opensource-src-x.y.z hierarchy:
 *      src/gui/kernel/qapplication_x11.cpp
 *      src/testlib/qtesttouch.h
 */
/****************************************************************************
**
** Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#ifndef MANTISTOUCHSCREEN_H
#define MANTISTOUCHSCREEN_H

#include <QObject>
#include <QTouchEvent> // For QTouchEvent::TouchPoint.
#include <QDateTime>

class QScreen;
class QSocketNotifier;
class TouchPointLogItem;

/*! \brief Raw touch event to gesture translation.
 *
 *  This object "listens" (using a QSocketNotifier) on the file descriptor
 *  associated with touch input, translating it into a series of
 *  QTouchEvent::TouchPoint items. These items are then passed to the
 *  qt_translateRawTouchEvent() function, which results in the generation of
 *  QTouchEvents in the UI event queue.
 */

class MantisTouchScreen : public QObject
{
    Q_OBJECT

    public:

        MantisTouchScreen(const char* deviceName,QObject* parent = 0);
        ~MantisTouchScreen();

    private:

        /**
         * Management object for the touch screen device.
         *
         * This item is created in the constructor. Its purpose is to "listen"
         * for activity on the file descriptor associated with the touch screen.
         * When an event occurs, it emits the activated() signal, which is
         * connected to the sendTouchEvents() slot.
         */

        QSocketNotifier* _socketNotifier;

        // Screen dimension parameters.
        //
        // These items are set in the constructor using data read from the
        // device and applied in readTouchEvents().

        int _minX;
        int _maxX;
        int _scaleX;

        int _minY;
        int _maxY;
        int _scaleY;

        int _minZ;
        int _maxZ;

        // Touch event recording members
        QList<TouchPointLogItem> _touchlog; // List for recording or replaying touch info
        bool _saveTouchInfo;                // Bool controlling the saving of touch info
        int _replayCounter;                 // Counter for replay index
        QDateTime _myClock;                 // Used for timing
        qint64 _startTime;                  // Start time for the recording
        qint64 _replayStartTime;            // Start time for replays
        int _repeatCounter;                 // Repeat count index
        int _repeatNum;                     // Total number of replays we will run
        QTimer *_touchTimer;                // Timer used for kicking off touch replays

        /**
         * Items used to store the raw touch event data from the touch device
         * into a Qt specific structure.
         *
         * They are updated each time readTouchEvents() is called. Copies are
         * managed in sendTouchEvents() for qt_translateRawTouchEvent().
         */

        QMap<unsigned int, QTouchEvent::TouchPoint> _readTouchMap;
        QList<QTouchEvent::TouchPoint> _readTouchList;

        QMap<int, QTouchEvent::TouchPoint> _sendTouchMap;
        QList<QTouchEvent::TouchPoint> _sendTouchList;

        /**
         * Translate the socket event information into QTouchEvent::TouchPoint
         * data.
         *
         * This method is called from the sendTouchEvents() slot. The socket
         * event data is used to configure the corresponding items in the
         * _touchEventList.
         */

        void readTouchEvents();

        // This macro disables the default copy constructor and "=" operator.

        Q_DISABLE_COPY(MantisTouchScreen);


    signals:

        /**
         * Signals for recording/replaying macros as controlled by the 
         * UI 
         */

        // finished the current replay run (with more repeats to go)
		void touchRunFinished();

        // finished all macro replays, we're done
		void endTouchReplay();

        // finished serializing recorded touch events to file
        void touchEventsSerialized();

        // finished deserializing recorded touch events from file
        void touchEventsDeserialized();

    public slots:

        /**
         * Slots used for recording/replaying macros as dictated by cwiz 
         * and the UI 
         */

        // start recording touch input info at the offset of startTime
    	void startRecord(quint64 startTime);

        // stop recording touch input, and serialize (appending) out to filename
    	void stopRecord(QString filename);

        // start a macro replay by deserializing filename at pos and running for repeatCount
        // number of repeats, starting at startTime as a base time
    	void deserializeTouch(QString filename, int repeatCount, quint64 pos, quint64 startTime);

        // stop all replay activities and clean up
    	void stopTouchReplay();

        // start a macrto replay repetition. we need to be externally notified, as other items
        // may have a longer replay time than our list
        void repeatSetup(quint64 startTime);

    private slots:

        /**
         * Process a touch device event.
         *
         * This slot is connected to the activated() signal of the
         * QSocketNotifier object created in the constructor. The signal is
         * emitted when an event occurs on the socket associated with the
         * object.
         */

        void sendTouchEvents();

        /**
         * This slot drives the replay of touch events for macro replay 
         *  
         * We index our way through the touch point list, looking at 
         * when to play the next touch info, effectively recreating the 
         * info saved in the macro file for touch data. 
         */
        void replayTouchEvents();
};

#endif // MANTISTOUCHSCREEN_H

// Use 4 spaces for indentation and no tabs.
// -*- mode: c++; c-basic-offset: 4; indent-tabs-mode: nil -*-
