// Copyright (c) (2011) Fluke Corporation. All rights reserved.
//
// $Id: MantisTouchScreen.cpp 5094 2011-01-14 22:41:28Z campbell $
// $URL: https://engsvnhost.tc.fluke.com/repos/fnet.pit.cicada/trunk/FLOSS/fnet/uitouch/MantisTouchScreen.cpp $

/*! \file  MantisTouchScreen.cpp
 *  \brief Raw touch event translation class definition.
 *
 *  This file defines an object and functions used to translate raw input from
 *  the kernel touch screen driver into QTouchEvents in the UI event queue.
 *
 *  This implementation is based on code within the following files of the
 *  qt-everywhere-opensource-src-x.y.z hierarchy:
 *      src/gui/kernel/qapplication_x11.cpp
 *      src/testlib/qtesttouch.h
 */

/****************************************************************************
**
** Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <linux/input.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <QScreen>
#include <QSocketNotifier>

#include "MantisTouchScreen.h"

using namespace std;
#include <QTimer>
#include "TouchScreenSerialize.h"

extern Q_GUI_EXPORT void qt_translateRawTouchEvent(QWidget* window,
                                                   QTouchEvent::DeviceType deviceType,
                                                   const QList<QTouchEvent::TouchPoint>& touchPoints);

/*! \brief Determines if the specified bit is set in the indicated mask.
 *
 *  This function is called from openTouchDevice().
 */

static inline int testBit(const char* mask,int bit)
{
    return (mask[bit / 8] & (1 << (bit % 8)));
}

/*! \brief Open the specified (touch) device and determine if the desciptor can
 *  or will provide the necessary data.
 *
 *  This function is called from the MantisTouchScreen constructor.
 */

static int openTouchDevice(const char* deviceName)
{
    int fd = open(deviceName,O_RDONLY | O_NONBLOCK);

    if (fd == -1)
    {
        qWarning("(openTouchDevice) open call failed for %s",deviceName);
        return fd;
    }

    // Fetch the event type mask and check that the device reports absolute
    // coordinates.

    char eventTypeMask[(EV_MAX + sizeof(char) - 1) * sizeof(char) + 1];

    memset(eventTypeMask,0,sizeof(eventTypeMask));

    if (ioctl(fd,EVIOCGBIT(0,sizeof(eventTypeMask)),eventTypeMask) < 0)
    {
        qWarning("(openTouchDevice) event mask ioctl() call failed");
        close(fd);
        return -1;
    }

    if (!testBit(eventTypeMask,EV_ABS))
    {
        qWarning("(openTouchDevice) event mask testBit() call failed");
        close(fd);
        return -1;
    }

    // Make sure that we can get the absolute X and Y positions from the device.

    char absMask[(ABS_MAX + sizeof(char) - 1) * sizeof(char) + 1];

    memset(absMask,0,sizeof(absMask));

    if (ioctl(fd,EVIOCGBIT(EV_ABS,sizeof(absMask)),absMask) < 0)
    {
        qWarning("(openTouchDevice) abs mask ioctl() call failed");
        close(fd);
        return -1;
    }

    if (!testBit(absMask,ABS_X) || !testBit(absMask,ABS_Y))
    {
        qWarning("(openTouchDevice) abs mask testBit() call failed");
        close(fd);
        return -1;
    }

    return fd;
}

// Constructor.

MantisTouchScreen::MantisTouchScreen(const char* deviceName,QObject* parent) :
    QObject(parent),
    _socketNotifier(NULL),
    _minX(0),_maxX(0),_scaleX(0),
    _minY(0),_maxY(0),_scaleY(0),
    _minZ(0),_maxZ(0),
    _saveTouchInfo(false),_replayCounter(0),
    _myClock(),_startTime(0),_replayStartTime(0),
    _repeatCounter(1),_repeatNum(1),_touchTimer(new QTimer(this))
{
    qDebug("(MantisTouchScreen::MantisTouchScreen) this: %10p",this);

    QScreen* screenData = QScreen::instance();
    int      fd         = openTouchDevice(deviceName);

    if (screenData && (fd >= 0))
    {
        qDebug("screenData - width: %d height: %d device width: %d device height: %d transform orientation: %d interlaced: %d",
               screenData->width(),screenData->height(),
               screenData->deviceWidth(),screenData->deviceHeight(),
               screenData->transformOrientation(),screenData->isInterlaced());

        qDebug("screenData (color support) depth: %d pixmap depth: %d color count: %d pixel format: %d",
               screenData->depth(),screenData->pixmapDepth(),
               screenData->colorCount(),screenData->pixelFormat());

        qDebug("screenData (depth support) 1: %d 4: %d 8: %d 12: %d 15: %d 16: %d 18: %d 24: %d 32: %d",
               screenData->supportsDepth(1),screenData->supportsDepth(4),
               screenData->supportsDepth(8),screenData->supportsDepth(12),
               screenData->supportsDepth(15),screenData->supportsDepth(16),
               screenData->supportsDepth(18),screenData->supportsDepth(24),
               screenData->supportsDepth(32));

        struct input_absinfo abs_x,abs_y,abs_z;

        ioctl(fd,EVIOCGABS(ABS_X),&abs_x);
        ioctl(fd,EVIOCGABS(ABS_Y),&abs_y);
        ioctl(fd,EVIOCGABS(ABS_Z),&abs_z);

        _minX   = abs_x.minimum;
        _maxX   = 1023;//abs_x.maximum;
        _scaleX = screenData->width();

        _minY   = abs_y.minimum;
        _maxY   = 1023;//abs_y.maximum;
        _scaleY = screenData->height();

        _minZ   = abs_z.minimum;
        _maxZ   = abs_z.maximum;

        qDebug("_minX: %d _maxX: %d _scaleX: %d",_minX,_maxX,_scaleX);
        qDebug("_minY: %d _maxY: %d _scaleY: %d",_minY,_maxY,_scaleY);
        qDebug("_minZ: %d _maxZ: %d",_minZ,_maxZ);
        qDebug("abs_x.maximum: %d abs_y.maximum: %d",
               abs_x.maximum,abs_y.maximum);

        _socketNotifier = new QSocketNotifier(fd,QSocketNotifier::Read,this);

        connect(_socketNotifier,SIGNAL(activated(int)),
                this,SLOT(sendTouchEvents()));

        // Connect event-recording info             
    	connect(_touchTimer, SIGNAL(timeout()), this, SLOT(replayTouchEvents()));
    	_touchTimer->setSingleShot(true);

    }

    else
    {
        qWarning("(MantisTouchScreen::MantisTouchScreen) Failed to open touch device.");
    }
}

// Destructor.

MantisTouchScreen::~MantisTouchScreen()
{
    qDebug("(MantisTouchScreen::~MantisTouchScreen) this: %10p",this);

    if (_socketNotifier) close(_socketNotifier->socket());

    // clear touch containers
    _readTouchMap.clear();
    _readTouchList.clear();
    _sendTouchMap.clear();
    _sendTouchList.clear();
}


// readTouchEvents debug macro
#define rDebug(...) //qDebug("(readTouchEvents) " __VA_ARGS__)

// sendTouchEvents debug macro
#define sDebug(...) //qDebug("(sendTouchEvents) " __VA_ARGS__)
//#define sDebug(...) Cicada::Logger::log(Cicada::FAC_UI, Cicada::SEV_ERR, __VA_ARGS__)

// readTouchEvents: Translate the socket event information into
//                  QTouchEvent::TouchPoint data.

void MantisTouchScreen::readTouchEvents()
{
    assert(_socketNotifier);
    const int fd = _socketNotifier->socket();

    static bool sync = true;
    bool done = true, data = false;
    unsigned int event = 0;
    forever
    {
        // read event reports
        struct input_event inputEvent[256];
        const int bytesRead = read(fd, &inputEvent, sizeof(inputEvent));
        if (bytesRead > 0)
        {
            if (bytesRead % sizeof(&inputEvent))
            {
                qWarning("(readTouchEvents) read size error!");
                break;
            }

            // process each event report
            for (unsigned int i = 0; i < bytesRead/sizeof(inputEvent[0]); ++i)
            {
                const struct input_event* pEvent = &inputEvent[i];
                done = false;

                // EV_KEY
                if (sync && (pEvent->type == EV_KEY))
                {
                    rDebug("EV_KEY key %u: %s", pEvent->code, (pEvent->value) ? "PRESS" : "RELEASE");

                    // cicada_ti driver provides BTN_TOUCH for mouse events
                    if (pEvent->code != BTN_TOUCH)
                    {
                        if (_readTouchMap.contains(event) || (pEvent->value))
                        {
                            // create/update touchpoint
                            QTouchEvent::TouchPoint &touchPoint = _readTouchMap[event];
                            touchPoint.setId(pEvent->code);
                            touchPoint.setState((pEvent->value) ? Qt::TouchPointPressed : Qt::TouchPointReleased);
                            data = true;    // have current data
                        }
                    }
                }

                // EV_ABS
                else if (sync && (pEvent->type == EV_ABS))
                {
                    rDebug("EV_ABS code %u", pEvent->code);

                    if (_readTouchMap.contains(event))
                    {
                        if (pEvent->value >= 0)
                        {
                            // retrieve touchpoint positions
                            QTouchEvent::TouchPoint &touchPoint = _readTouchMap[event];
                            QPointF screenPos = touchPoint.screenPos();
                            QPointF normalizedPos = touchPoint.normalizedPos();

                            // calculate new X and assign new position
                            if (pEvent->code == ABS_X)
                            {
                                const qreal newNormalizedPos = qreal(pEvent->value - _minX) / qreal(_maxX - _minX);
                                const qreal newScreenPos = newNormalizedPos * _scaleX;

                                rDebug("  ABS_X key %u: %d, old:(%3d,%3d), new X: %3.1f %1.3f",
                                    touchPoint.id(), pEvent->value, screenPos.toPoint().x(), screenPos.toPoint().y(),
                                    newScreenPos, newNormalizedPos);

                                screenPos.rx() = newScreenPos;
                                normalizedPos.rx() = newNormalizedPos;
                            }

                            // calculate new Y and assign new position
                            else if (pEvent->code == ABS_Y)
                            {
                                const qreal newNormalizedPos = qreal(pEvent->value - _minY) / qreal(_maxY - _minY);
                                const qreal newScreenPos = newNormalizedPos * _scaleY;

                                rDebug("  ABS_Y key %u: %d, old:(%3d,%3d), new Y: %3.1f %1.3f",
                                    touchPoint.id(), pEvent->value, screenPos.toPoint().x(), screenPos.toPoint().y(),
                                    newScreenPos, newNormalizedPos);

                                screenPos.ry() = newScreenPos;
                                normalizedPos.ry() = newNormalizedPos;
                            }

                            // update touchpoint
                            touchPoint.setScreenPos(screenPos);
                            touchPoint.setNormalizedPos(normalizedPos);
                        }
                        data = true;    // have current data
                    }
                }

                // EV_SYN
                else if (pEvent->type == EV_SYN)
                {
                    rDebug("EV_SYN code %u", pEvent->code);

                    // event report complete
                    if (pEvent->code == SYN_REPORT)
                    {
                        event = 0;    // reset event index
                        sync = true;  // recovered from out-of-sync condition
                        done = true;  // allow method to exit if no new reports
                    }

                    // one multi-touch event report complete
                    else if (sync && (pEvent->code == SYN_MT_REPORT))
                    {
                        if (data && _readTouchMap.contains(event))
                        {
                            // append touchpoint onto touch event list
                            QTouchEvent::TouchPoint &touchPoint = _readTouchMap[event];
                            _readTouchList.append(touchPoint);

                            // remove released touchpoint
                            if (touchPoint.state() == Qt::TouchPointReleased)
                                _readTouchMap.remove(event);

                            // change touchpoint state for next EV_ABS event
                            else
                                touchPoint.setState(Qt::TouchPointMoved);
                        }
                        event++;    // next event index
                    }

                    // one or more event reports dropped: out-of-sync condition
                    else if (pEvent->code == SYN_DROPPED)
                    {
                        qWarning("(readTouchEvents) SYN_DROPPED!");

                        // release current touchpoints
                        QMapIterator<unsigned int, QTouchEvent::TouchPoint> i(_readTouchMap);
                        while (i.hasNext())
                        {
                            QTouchEvent::TouchPoint touchPoint = i.next().value();
                            touchPoint.setState(Qt::TouchPointReleased);
                            _readTouchList.append(touchPoint);
                        }

                        // clear current touchpoints
                        _readTouchMap.clear();

                        event = 0;    // reset event index
                        sync = false; // out-of-sync condition
                        done = true;  // allow method to exit if no new reports
                    }
                    data = false;     // have no current data
                }

                // un-handled event report
                else
                    qWarning("(readTouchEvents) event %u ignored!", pEvent->type);
            }
        }

        // event report read error
        else if ((bytesRead != -1) || (errno != EAGAIN))
        {
            qWarning("(readTouchEvents) read error %u!", errno);
            break;
        }

        // no event report to read and done
        else if (done)
            break;
    }
}


// sendTouchEvents: Process a touch device event.

void MantisTouchScreen::sendTouchEvents()
{
    readTouchEvents();
    while (!_readTouchList.isEmpty())
    {
        // Move events are collected and sent in groups of 3 for single finger
        // touches and groups of 2 (1 for each finger) for multi-touch.  Press
        // and Release events must be sent separately in order for gestures to
        // be recognized as "started" and "finished".

        const QTouchEvent::TouchPoint& touchPoint = _readTouchList.first();
        sDebug("send key %u: %u(%3d,%3d)", touchPoint.id(), touchPoint.state(),
            touchPoint.screenPos().toPoint().x(), touchPoint.screenPos().toPoint().y());

        bool send = true;
        if (_sendTouchList.isEmpty())
        {
            _sendTouchList.append(touchPoint);
            _sendTouchMap[touchPoint.id()] = touchPoint;
            if (touchPoint.state() == Qt::TouchPointPressed)
                _sendTouchMap[touchPoint.id()].setState(Qt::TouchPointMoved);
            else if (touchPoint.state() == Qt::TouchPointReleased)
                _sendTouchMap.remove(touchPoint.id());
            else
                send = false;
            _readTouchList.removeFirst();
        }
        else if (touchPoint.state() == Qt::TouchPointMoved)
        {
            // reject relative small changes
            const QPointF diff = _sendTouchMap[touchPoint.id()].screenPos() - touchPoint.screenPos();
            if (diff.toPoint().manhattanLength() < _readTouchList.size())
            {
                _readTouchList.removeFirst();
                send = false;
            }

            // single touch
            else if (_sendTouchMap.size() < 2)
            {
                _sendTouchList.append(touchPoint);
                _sendTouchMap[touchPoint.id()] = touchPoint;
                _readTouchList.removeFirst();
                send = !(_sendTouchList.size() < 3);
            }

            // multi-touch (place points in consistent order)
            else
            {
                if (_sendTouchList.first().id() != touchPoint.id())
                {
                    _sendTouchMap[touchPoint.id()] = touchPoint;
                    _readTouchList.removeFirst();
                }
                _sendTouchList.clear();
                QMapIterator<int, QTouchEvent::TouchPoint> i(_sendTouchMap);
                _sendTouchList.append(i.next().value());
                _sendTouchList.append(i.next().value());
            }
        }
        else    // flush touch points
        {
            sDebug("FLUSH");
            _sendTouchList.clear();
            send = false;
        }


        // send collected group of touchpoints
        if (send)
        {
        	if (_saveTouchInfo) {
				qint64 timeDifference = _myClock.currentMSecsSinceEpoch() - _startTime;

				TouchPointLogItem currentTouch(_sendTouchList, timeDifference);		// using the constructor
				_touchlog.append(currentTouch);
        	}

			sDebug("events sent: %u remaining: %u\n", _sendTouchList.count(), _readTouchList.count());
			qt_translateRawTouchEvent(0, QTouchEvent::TouchScreen, _sendTouchList);
			_sendTouchList.clear();
        }
    }
}


void MantisTouchScreen::startRecord(quint64 startTime)
{
//	sDebug("MANTISTOUCHSCREEN START!\n");
	_saveTouchInfo = true;
	_touchlog.clear();

	_startTime = startTime;
}

void MantisTouchScreen::stopRecord(QString filename)
{
	_saveTouchInfo = false;

	QFile touchwrite(filename);
	touchwrite.open(QIODevice::Append);
	QDataStream thisismyout(&touchwrite);

	thisismyout << _touchlog;
	_touchlog.clear();
	touchwrite.close();

    emit touchEventsSerialized();
}

void MantisTouchScreen::deserializeTouch(QString filename, int repeatCount, quint64 pos, quint64 startTime)
{
	sDebug("Start Replay: Begin replaying test sequence...\n");

	_touchlog.clear();

	QFile touchread(filename);
	touchread.open(QIODevice::ReadOnly);
	touchread.seek(pos);
	QDataStream thisismyin (&touchread);
	thisismyin >> _touchlog;
	touchread.close();

	_replayStartTime = startTime;

    emit touchEventsDeserialized();

	_repeatNum = repeatCount;
	_repeatCounter = 1;

	sDebug("STARTING REPLAY\n");
	_replayCounter = 0;

	if (_touchlog.length() > 0) {
		int tempsl = _touchlog.at(_replayCounter).timestamp;
		if (tempsl > 0) {
			_touchTimer->start(tempsl);
		} else {
			replayTouchEvents();
		}
	} else {
		emit touchRunFinished();
		emit endTouchReplay();
		_replayCounter = 0;
		_touchlog.clear();
		sDebug("No Recorded Gestures...\n");
	}
}

void MantisTouchScreen::replayTouchEvents()
{
	qt_translateRawTouchEvent(0, QTouchEvent::TouchScreen, _touchlog.at(_replayCounter).sentTouchList);
	_replayCounter++;
	if (_replayCounter >= _touchlog.length()) {
		_replayCounter = 0;
		if (_repeatCounter == _repeatNum) {
			emit endTouchReplay();
		} else {
			emit touchRunFinished();
		}
	} else {
		qint64 currenttime = _myClock.currentMSecsSinceEpoch() - _replayStartTime;
		qint64 sleeptime = _touchlog.at(_replayCounter).timestamp - currenttime;
		if (sleeptime > 0) {
			_touchTimer->start(sleeptime);
		} else {
			replayTouchEvents();
		}
	}
}

void MantisTouchScreen::stopTouchReplay()
{
	_touchTimer->stop();

	_replayCounter = 0;
	_touchlog.clear();
	sDebug("EXITING REPLAY\n");
}

void MantisTouchScreen::repeatSetup(quint64 startTime)
{
	_repeatCounter++;
	sDebug("Starting Repeat #:  %d\n", _repeatCounter);

	if (_touchlog.length() > 0) {
		_replayStartTime = startTime;
		qint64 tempsl = _touchlog.at(_replayCounter).timestamp;
		if (tempsl > 0) {
			_touchTimer->start(tempsl);
		} else {
			replayTouchEvents();
		}
	} else {
		_replayCounter = 0;
		_touchlog.clear();
		sDebug("No Recorded Gestures...\n");
	}
}
