/*
 * Copyright (c) 2015,2016,2017 Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/i2c.h>
#include <linux/of_platform.h>
#include <linux/clk.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/delay.h>
#include <linux/input.h>
#include <linux/gpio.h>
#include <linux/fsl_devices.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/timekeeping.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <sound/initval.h>

#include "../codecs/tlv320aic3x.h"
#include "imx-ssi.h"
#include "imx-audmux.h"

/*=====================================================*/
struct cicada_snd_priv {
	struct platform_device *pdev;

	unsigned clk_frequency;

	int amp_gpio;

	struct snd_soc_codec *codec;

	int opened;
	int mic_val;
	int hp_val;
	int spk_val;
	int jack_val;
	int jack_status;
	int jack_detect;

	int wakeup;

	/* headset jack detect */
	unsigned int headset_det_gpio;
	unsigned int headset_det_enabled;
	unsigned int headset_det_key_code;
	struct delayed_work headset_det_work;

	/* headset mic switch */
	unsigned int key_gpio;
	unsigned int key_enabled;
	unsigned int key_code;
	struct input_dev *idev;

	/* headset 3 or 4 pole detect */
	unsigned int p3_gpio;

	/* headset mic enable */
	unsigned int mic_ena_gpio;
};

struct cicada_snd_data {
	struct snd_soc_dai_link dai;
	struct snd_soc_card card;
	struct cicada_snd_priv priv;
};

static ssize_t cicada_jack_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	struct snd_soc_card *card = dev_get_drvdata(dev);
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);

	return sprintf(buf, "%u\n", priv->jack_status);
}
DEVICE_ATTR(jack, 0400, cicada_jack_show, NULL);

static ssize_t cicada_detect_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	struct snd_soc_card *card = dev_get_drvdata(dev);
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);

	return sprintf(buf, "%u\n", gpio_get_value(priv->headset_det_gpio));
}
DEVICE_ATTR(detect, 0400, cicada_detect_show, NULL);

static void cicada_ext_control(struct snd_soc_dapm_context *dapm)
{
	struct snd_soc_card *card = dapm->card;
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);

	dev_dbg(&priv->pdev->dev, "%s set: speaker %s, headphone %s, mic %s, detect %s\n", __func__,
		(priv->spk_val ? "on" : "off"),
		(priv->hp_val ? "on" : "off"),
		(priv->mic_val ? "on" : "off"),
		(priv->jack_detect ? "on" : "off"));

	snd_soc_dapm_mutex_lock(dapm);

	if (priv->spk_val) {
		mdelay(15);
		snd_soc_dapm_enable_pin_unlocked(dapm, "Speaker");
	} else {
		snd_soc_dapm_disable_pin_unlocked(dapm, "Speaker");
	}

	if (priv->hp_val)
		snd_soc_dapm_enable_pin_unlocked(dapm, "Headphone Jack");
	else
		snd_soc_dapm_disable_pin_unlocked(dapm, "Headphone Jack");

	if (priv->mic_val)
		snd_soc_dapm_enable_pin_unlocked(dapm, "Mic Jack");
	else
		snd_soc_dapm_disable_pin_unlocked(dapm, "Mic Jack");

	if (priv->mic_val || priv->jack_detect)
		snd_soc_dapm_enable_pin_unlocked(dapm, "Mic Bias");
	else
		snd_soc_dapm_disable_pin_unlocked(dapm, "Mic Bias");

	snd_soc_dapm_sync_unlocked(dapm);

	snd_soc_dapm_mutex_unlock(dapm);
}

static void jack_det_worker(struct work_struct *work)
{
	struct cicada_snd_priv *priv = container_of(work, struct cicada_snd_priv,
						    headset_det_work.work);
	struct cicada_snd_data *data = container_of(priv, struct cicada_snd_data, priv);
	struct snd_soc_card *card = &data->card;
	static int count;
	static int three_pole;
	static int quick_poll;

	if (gpio_get_value(priv->headset_det_gpio)) {
		/* remove */
		dev_dbg(&priv->pdev->dev, "%s, jack remove\n", __func__);
		count = 0;
		three_pole = 0;
		gpio_set_value(priv->mic_ena_gpio, 0);

		if (!priv->headset_det_enabled) {
			dev_dbg(&priv->pdev->dev, "%s, headset detect enabled\n", __func__);
			priv->headset_det_enabled = 1;
			enable_irq(gpio_to_irq(priv->headset_det_gpio));
		}

		if (priv->key_enabled) {
			dev_dbg(&priv->pdev->dev, "%s, mic key disabled\n", __func__);
			priv->key_enabled = 0;
			disable_irq(gpio_to_irq(priv->key_gpio));
		}

		/* headset jack removed */
		priv->spk_val = 1;
		priv->hp_val = 0;
		priv->mic_val = 0;
		priv->jack_detect = 1;
		cicada_ext_control(&card->dapm);

		priv->jack_val = 0;
		priv->jack_status = 0;
		if (priv->opened) {
			input_report_key(priv->idev, priv->headset_det_key_code,
					 priv->jack_val);
			input_sync(priv->idev);
		}
	} else if (!priv->jack_val) {
		/* insert */
		if (gpio_get_value(priv->p3_gpio) == three_pole) {
			dev_dbg(&priv->pdev->dev, "%s, %s pole, cnt %d\n", __func__, (three_pole ? "3":"4"), count);
			count++;
			if (count > 5) {
				/* headset jack present */
				if (three_pole) {
					/* 3 pole */
					dev_dbg(&priv->pdev->dev, "%s, 3 pole\n", __func__);

					priv->jack_status = 1;
					gpio_set_value(priv->mic_ena_gpio, 0);

					priv->mic_val = 0;

					if (priv->key_enabled) {
						dev_dbg(&priv->pdev->dev, "%s, mic key disabled\n", __func__);
						priv->key_enabled = 0;
						disable_irq(gpio_to_irq(priv->key_gpio));
					}
				} else {
					/* 4 pole */
					dev_dbg(&priv->pdev->dev, "%s, 4 pole\n", __func__);

					priv->jack_status = 2;
					gpio_set_value(priv->mic_ena_gpio, 1);

					priv->mic_val = 1;

					if (!priv->key_enabled) {
						dev_dbg(&priv->pdev->dev, "%s, mic key enabled\n", __func__);
						priv->key_enabled = 1;
						enable_irq(gpio_to_irq(priv->key_gpio));
					}
				}

				priv->jack_val = 1;

				priv->spk_val = 0;
				priv->hp_val = 1;
				priv->jack_detect = 0;
				cicada_ext_control(&card->dapm);

				count = 0;
				three_pole = 0;
				quick_poll = 1;
				if (!priv->headset_det_enabled) {
					dev_dbg(&priv->pdev->dev, "%s, headset detect enabled\n", __func__);
					priv->headset_det_enabled = 1;
					enable_irq(gpio_to_irq(priv->headset_det_gpio));
				}

				if (priv->opened) {
					input_report_key(priv->idev,
							 priv->headset_det_key_code,
							 priv->jack_val);
					input_sync(priv->idev);
				}
			} else if (quick_poll || ktime_to_ms(ktime_get()) > 10000) {
				schedule_delayed_work(&priv->headset_det_work, msecs_to_jiffies(200));
			} else {
				schedule_delayed_work(&priv->headset_det_work, msecs_to_jiffies(2000));
			}
		} else {
			dev_dbg(&priv->pdev->dev, "%s, poles changed\n", __func__);
			count = 0;
			three_pole = gpio_get_value(priv->p3_gpio);
			if (!three_pole) quick_poll = 1;  /* detected non-default setting */
			schedule_delayed_work(&priv->headset_det_work, msecs_to_jiffies(200));
		}
	} else {
		if (!priv->headset_det_enabled) {
			dev_dbg(&priv->pdev->dev, "%s, headset detect enabled\n", __func__);
			priv->headset_det_enabled = 1;
			enable_irq(gpio_to_irq(priv->headset_det_gpio));
		}
	}
}

static irqreturn_t cicada_detect_isr(int irqno, void *data)
{
	struct cicada_snd_priv *priv = data;

	/* dev_dbg(&priv->pdev->dev, "%s, %d\n", __func__, gpio_get_value(priv->headset_det_gpio)); */

	priv->headset_det_enabled = 0;
	disable_irq_nosync(gpio_to_irq(priv->headset_det_gpio));

	schedule_delayed_work(&priv->headset_det_work, msecs_to_jiffies(200));

	return IRQ_HANDLED;
}

static irqreturn_t cicada_mic_key_isr(int irqno, void *data)
{
	struct cicada_snd_priv *priv = data;

	if (priv->opened) {
		input_report_key(priv->idev, priv->key_code, gpio_get_value(priv->key_gpio));
		input_sync(priv->idev);
	}

	return IRQ_HANDLED;
}

static int cicada_jack_open(struct input_dev *dev)
{
	struct cicada_snd_priv *priv = input_get_drvdata(dev);

	dev_dbg(&dev->dev, "%s\n", __func__);

	priv->opened++;

	schedule_delayed_work(&priv->headset_det_work, msecs_to_jiffies(200));

	return 0;
}

static void cicada_jack_close(struct input_dev *dev)
{
	struct cicada_snd_priv *priv = input_get_drvdata(dev);

	dev_dbg(&dev->dev, "%s\n", __func__);

	priv->opened--;
}

static int cicada_jack_device_register(struct cicada_snd_priv *priv)
{
	struct platform_device *pdev = priv->pdev;
	struct input_dev *input;
	int ret = 0;

	input = input_allocate_device();
	if (input == NULL)
		return -ENOMEM;

	ret = device_create_file(&pdev->dev, &dev_attr_jack);
	ret = device_create_file(&pdev->dev, &dev_attr_detect);
	if (ret)
		return ret;

	input_set_drvdata(input, priv);

	input->name = "cicada-mic";
	input->phys = "cicada-mic";
	input->open = cicada_jack_open;
	input->close = cicada_jack_close;
	input->evbit[0] = BIT_MASK(EV_KEY);
	input->keybit[BIT_WORD(priv->key_code)] = BIT_MASK(priv->key_code);
	input->keybit[BIT_WORD(priv->headset_det_key_code)] |= BIT_MASK(priv->headset_det_key_code);

	ret = input_register_device(input);
	if (ret) {
		dev_err(&priv->pdev->dev, "input_register_device failed\n");
		goto err;
	}

	priv->idev = input;

	return 0;

err:
	device_remove_file(&pdev->dev, &dev_attr_jack);
	device_remove_file(&pdev->dev, &dev_attr_detect);
	return ret;
}

/*=====================================================*/
static int cicada_hw_params(struct snd_pcm_substream *substream,
				struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct snd_soc_card *card = rtd->card;
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);
	int ret = 0;
	u32 dai_format;

	if (priv->spk_val)
		gpio_set_value(priv->amp_gpio, 1);
	else
		gpio_set_value(priv->amp_gpio, 0);

	dai_format = SND_SOC_DAIFMT_I2S | SND_SOC_DAIFMT_NB_NF |
			SND_SOC_DAIFMT_CBM_CFM;

	/* set codec DAI configuration */
	ret = snd_soc_dai_set_fmt(codec_dai, dai_format);
	if (ret) {
		dev_err(&priv->pdev->dev, "%s, failed set codec_dai format\n", __func__);
		return ret;
	}

	/* set cpu DAI configuration */
	ret = snd_soc_dai_set_fmt(cpu_dai, dai_format);
	if (ret) {
		dev_err(&priv->pdev->dev, "%s, failed set cpu_dai format\n", __func__);
		return ret;
	}

	/* set the codec system clock as output */
	ret = snd_soc_dai_set_sysclk(codec_dai, 0,
			priv->clk_frequency, SND_SOC_CLOCK_OUT);
	if (ret) {
		dev_err(&priv->pdev->dev, "%s, failed setting codec clk_frequency\n", __func__);
		return ret;
	}

	/* dev_dbg(&priv->pdev->dev, "%s, clk %d, rate %d, channels %d\n", __func__, priv->clk_frequency, params_rate(params), params_channels(params)); */

	/* set the SSI system clock as input (unused) */
	ret = snd_soc_dai_set_sysclk(cpu_dai, IMX_SSP_SYS_CLK,
			0, SND_SOC_CLOCK_IN);
	if (ret) {
		dev_err(&priv->pdev->dev, "can't set CPU system clock IMX_SSP_SYS_CLK\n");
		return ret;
	}

	return 0;
}

static int get_spk(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_card *card = snd_kcontrol_chip(kcontrol);
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);

	dev_dbg(&priv->pdev->dev, "%s, %d\n", __func__, priv->spk_val);
	ucontrol->value.integer.value[0] = priv->spk_val;
	return 0;
}

static int set_spk(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_card *card = snd_kcontrol_chip(kcontrol);
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);

	if (priv->spk_val == ucontrol->value.integer.value[0])
		return 0;

	priv->spk_val = ucontrol->value.integer.value[0];
	dev_dbg(&priv->pdev->dev, "%s, %d\n", __func__, priv->spk_val);

	cicada_ext_control(&card->dapm);
	return 1;
}

static int get_hp(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_card *card = snd_kcontrol_chip(kcontrol);
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);

	dev_dbg(&priv->pdev->dev, "%s, %d\n", __func__, priv->hp_val);
	ucontrol->value.integer.value[0] = priv->hp_val;
	return 0;
}

static int set_hp(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_card *card = snd_kcontrol_chip(kcontrol);
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);

	if (priv->hp_val == ucontrol->value.integer.value[0])
		return 0;

	priv->hp_val = ucontrol->value.integer.value[0];
	dev_dbg(&priv->pdev->dev, "%s, %d\n", __func__, priv->hp_val);

	/* leave the hp on so that sounds are not queued up if speaker is off */
	/* return do_ctrl(kcontrol, ucontrol, 1, &hp_val); */
#if 1
	cicada_ext_control(&card->dapm);
#endif
	return 1;
}

static int get_mic(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_card *card = snd_kcontrol_chip(kcontrol);
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);

	dev_dbg(&priv->pdev->dev, "%s, %d\n", __func__, priv->mic_val);
	ucontrol->value.integer.value[0] = priv->mic_val;
	return 0;
}

static int set_mic(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_card *card = snd_kcontrol_chip(kcontrol);
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);

	if (priv->mic_val == ucontrol->value.integer.value[0])
		return 0;

	priv->mic_val = ucontrol->value.integer.value[0];
	dev_dbg(&priv->pdev->dev, "%s, %d\n", __func__, priv->mic_val);

	cicada_ext_control(&card->dapm);
	return 1;
}

static int amp_event(struct snd_soc_dapm_widget *w,
	struct snd_kcontrol *control, int event)
{
	struct snd_soc_dapm_context *dapm = w->dapm;
	struct snd_soc_card *card = dapm->card;
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);

	dev_dbg(&priv->pdev->dev, "%s, %s\n", __func__,
		(SND_SOC_DAPM_EVENT_ON(event) ? "on" : "off"));

	gpio_set_value(priv->amp_gpio, SND_SOC_DAPM_EVENT_ON(event));

	return 0;
}

static int hp_event(struct snd_soc_dapm_widget *w,
	struct snd_kcontrol *control, int event)
{
	struct snd_soc_dapm_context *dapm = w->dapm;
	struct snd_soc_card *card = dapm->card;
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);
	struct snd_soc_codec *codec = priv->codec;

	int regL;
	int regR;

	dev_dbg(&priv->pdev->dev, "%s, %s\n", __func__,
		(SND_SOC_DAPM_EVENT_ON(event)  ? "on" : "off"));

	regL = snd_soc_read(codec, HPLOUT_CTRL);
	regR = snd_soc_read(codec, HPROUT_CTRL);

	if (SND_SOC_DAPM_EVENT_ON(event)) {
		mdelay(15);
		snd_soc_write(codec, HPLOUT_CTRL, regL | UNMUTE);
		snd_soc_write(codec, HPROUT_CTRL, regR | UNMUTE);
	} else {
		snd_soc_write(codec, HPLOUT_CTRL, regL & ~UNMUTE);
		snd_soc_write(codec, HPROUT_CTRL, regR & ~UNMUTE);
	}

	return 0;
}

/* cicada specific control */
static const char * const offOn_function[] = {"Off", "On"};

static const struct soc_enum cicada_enum[] = {
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(offOn_function), offOn_function),
};

static const struct snd_kcontrol_new cicada_controls[] = {
	SOC_ENUM_EXT("Speaker", cicada_enum[0], get_spk, set_spk),
	SOC_ENUM_EXT("Headphone Jack", cicada_enum[0], get_hp, set_hp),
	SOC_ENUM_EXT("Mic Jack", cicada_enum[0], get_mic, set_mic),
};

/* machine dapm widgets */
static const struct snd_soc_dapm_widget cicada_dapm_widgets[] = {
	SND_SOC_DAPM_SPK("Speaker", amp_event),
	SND_SOC_DAPM_HP("Headphone Jack", hp_event),
	SND_SOC_DAPM_MIC("Mic Jack", NULL),
	SND_SOC_DAPM_LINE("Mic Bias", NULL),
};

/* audio mapings */
static const struct snd_soc_dapm_route audio_map[] = {
	/* Input */

	/* L mic input to L adc */
	{"MIC3L", NULL, "Mic Jack"},
	{"Left PGA Mixer", NULL, "MIC3L"},
	{"Left ADC", NULL, "Left PGA Mixer"},

	/* R mic input with bias to R adc */
	{"MIC3R", NULL, "Mic Jack"},
	{"Right PGA Mixer", NULL, "MIC3R"},
	{"Right ADC", NULL, "Right PGA Mixer"},

	/* output */

	/* left DAC Mixers to Speaker */
	{"Speaker", NULL, "LLOUT"},

	/* Left and Right hp to Headphone */
	{"Headphone Jack", NULL, "HPROUT"},
	{"Headphone Jack", NULL, "HPLOUT"},

	/* Mic bias for jack detect */
	{"Detection", NULL, "Mic Bias"},
};

/* Logic for a aic3x as connected on a cicada */
static int cicada_snd_init(struct snd_soc_pcm_runtime *rtd)
{
	struct snd_soc_codec *codec = rtd->codec;
	struct snd_soc_dapm_context *dapm = &codec->dapm;
	struct snd_soc_card *card = rtd->card;
	struct device_node *np = card->dev->of_node;
	struct cicada_snd_priv *priv = snd_soc_card_get_drvdata(card);
	int ret;

	dev_dbg(&priv->pdev->dev, "%s\n", __func__);

	priv->codec = codec;

	/* NOTE: muxes, mixers and switches
	 * The hardware regs must be set the way you want them
	 * and the routing must be in the audio_map, even if
	 * the route is already specified in the codec code,
	 * before calling snd_soc_dapm_add_routes
	 */

	/* R & L DAC volume (0x7f max)
	 * name='PCM Playback Volume' */
	snd_soc_write(codec, LDAC_VOL, INVERT_VOL(0x7f));
	snd_soc_write(codec, RDAC_VOL, INVERT_VOL(0x7f));

	/* speaker volume (0x7f max)
	 * name='Line DAC Playback Volume' */
	snd_soc_write(codec, DACL1_2_LLOPM_VOL, INVERT_VOL(0x7f) | ROUTE_ON);
	snd_soc_write(codec, DACR1_2_RLOPM_VOL, INVERT_VOL(0x7f) | ROUTE_ON);

	/* speaker gain (0x9 max)
	 * name='Line Playback Volume' */
/*	snd_soc_write(codec, LLOPM_CTRL, (0x9 << 4) | UNMUTE); */

	/* head phone volume (0x7f max)
	 * name='HP DAC Playback Volume' */
	snd_soc_write(codec, DACL1_2_HPLOUT_VOL, INVERT_VOL(0x45) | ROUTE_ON);
	snd_soc_write(codec, DACR1_2_HPROUT_VOL, INVERT_VOL(0x45) | ROUTE_ON);

	/* MIC3L/R input level contol for left ADC PGA mix
	 * (MIC3L attenuation | MIC3R not connected to Left PGA)
	 * attenuation is 0 = min, 0x08 = max, 0xf = not connected
	 * name='Left PGA Mixer Mic3L Switch' */
	snd_soc_write(codec, MIC3LR_2_LADC_CTRL, (0 << 4) | (0xf << 0));

	/* MIC3L/R input level contol for right ADC PGA mix
	 * (MIC3L not connected to Right PGA | MIC3R attenuation)
	 * attenuation is 0 = min, 0x08 = max, 0xf = not connected
	 * name='Right PGA Mixer Mic3R Switch' */
	snd_soc_write(codec, MIC3LR_2_RADC_CTRL, (0xf << 4) | (0 << 0));

	/* Mic gain
	 * ADC PGA gain control (0x7f max)
	 * unmute &= ~(unsigned char)MUTE_ON
	 * name='PGA Capture Volume' */
	snd_soc_write(codec, LADC_VOL, ~(unsigned char)MUTE_ON & 0x70);
	snd_soc_write(codec, RADC_VOL, ~(unsigned char)MUTE_ON & 0x70);

	/* Set up specific audio path audio_map */
	if (np) {
		ret = snd_soc_of_parse_audio_routing(card, "audio-routing");
		if (ret)
			return ret;
	} else {
		/* Set up davinci-evm specific audio path audio_map */
		snd_soc_dapm_add_routes(dapm, audio_map, ARRAY_SIZE(audio_map));
	}

	/* not connected */
	snd_soc_dapm_nc_pin(dapm, "RLOUT");
	snd_soc_dapm_nc_pin(dapm, "MONO_LOUT");
	snd_soc_dapm_nc_pin(dapm, "HPLCOM");
	snd_soc_dapm_nc_pin(dapm, "HPRCOM");
	snd_soc_dapm_nc_pin(dapm, "LINE1R");
	snd_soc_dapm_nc_pin(dapm, "LINE1L");
	snd_soc_dapm_nc_pin(dapm, "LINE2R");
	snd_soc_dapm_nc_pin(dapm, "LINE2L");

	snd_soc_dapm_enable_pin(&card->dapm, "Mic Bias");

	snd_soc_dapm_sync(dapm);

	return 0;
}

static struct snd_soc_ops cicada_snd_ops = {
	.hw_params = cicada_hw_params,
};

static int cicada_snd_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct platform_device *cpu_pdev;
	struct i2c_client *codec_dev;
	struct device_node *np = pdev->dev.of_node;
	struct device_node *cpu_np;
	struct device_node *codec_np;
	struct cicada_snd_data *data;
	struct cicada_snd_priv *priv;
	struct snd_soc_card *card;
	struct snd_soc_dai_link *dai;
	int int_port, ext_port;
	int retval = 0;

	dev_dbg(dev, "%s\n", __func__);

	if (!np)
		return -EINVAL;

	if (!of_device_is_available(np))
		return -ENODEV;

	data = devm_kzalloc(dev, sizeof(*data), GFP_KERNEL);
	if (!data) {
		retval = -ENOMEM;
		goto done;
	}
	card = &data->card;
	dai = &data->dai;
	priv = &data->priv;

	priv->pdev = pdev;

	/* ssi-controler */
	cpu_np = of_parse_phandle(pdev->dev.of_node, "cpu-dai", 0);
	if (!cpu_np) {
		dev_err(dev, "phandle cpu-dai for missing or invalid\n");
		retval = -EINVAL;
		goto done;
	}

	cpu_pdev = of_find_device_by_node(cpu_np);
	if (!cpu_pdev) {
		dev_err(dev, "failed to find ssi platform device\n");
		retval = -EPROBE_DEFER;
		goto done;
	}

	codec_np = of_parse_phandle(pdev->dev.of_node, "audio-codec", 0);
	if (!codec_np) {
		dev_err(dev, "phandle for audio-codec missing or invalid\n");
		retval = -EINVAL;
		goto done;
	}

	codec_dev = of_find_i2c_device_by_node(codec_np);
	if (!codec_dev) {
		dev_err(dev, "failed to find codec platform device\n");
		retval = -EPROBE_DEFER;
		goto done;
	}

	retval = of_property_read_u32(np, "codec-clock-rate", &priv->clk_frequency);
	if (retval < 0) {
		dev_err(dev, "codec-clock-rate missing or invalid\n");
		retval = -EINVAL;
		goto done;
	}

	retval = of_property_read_u32(np, "mux-int-port", &int_port);
	if (retval) {
		dev_err(dev, "mux-int-port missing or invalid\n");
		retval = -EINVAL;
		goto done;
	}
	retval = of_property_read_u32(np, "mux-ext-port", &ext_port);
	if (retval) {
		dev_err(dev, "mux-ext-port missing or invalid\n");
		retval = -EINVAL;
		goto done;
	}

	/*
	 * The port numbering in the hardware manual starts at 1, while
	 * the audmux API expects it starts at 0.
	 */
	int_port--;
	ext_port--;
	/* audmux, port 0, internal cpu dai */
	imx_audmux_v2_configure_port(int_port,
		IMX_AUDMUX_V2_PTCR_SYN |		/* 4wire mode */
		IMX_AUDMUX_V2_PTCR_TFSEL(ext_port) |	/* TxFS (frame sync) source */
		IMX_AUDMUX_V2_PTCR_TCSEL(ext_port) |	/* TxClk source */
		IMX_AUDMUX_V2_PTCR_TFSDIR |		/* TxFS is output */
		IMX_AUDMUX_V2_PTCR_TCLKDIR,		/* TxClk  is output */
		IMX_AUDMUX_V2_PDCR_RXDSEL(ext_port));	/* Rx data source*/

	dev_dbg(dev, "audmux src ssi port %d, ptcr = %lx, pdcr = %lx\n",
		int_port,
		(long)IMX_AUDMUX_V2_PTCR_SYN |
		IMX_AUDMUX_V2_PTCR_TFSEL(ext_port) |
		IMX_AUDMUX_V2_PTCR_TCSEL(ext_port) |
		IMX_AUDMUX_V2_PTCR_TFSDIR |
		IMX_AUDMUX_V2_PTCR_TCLKDIR,
		(long)IMX_AUDMUX_V2_PDCR_RXDSEL(ext_port));

	/* audmux, port 3, external codec dai */
	imx_audmux_v2_configure_port(ext_port,
		IMX_AUDMUX_V2_PTCR_SYN,			/* 4wire mode */
		IMX_AUDMUX_V2_PDCR_RXDSEL(int_port));	/* Rx data source*/

	dev_dbg(dev, "audmux ext ssi port %d, ptcr = %lx, pdcr = %lx\n",
		ext_port,
		(long)IMX_AUDMUX_V2_PTCR_SYN,
		(long)IMX_AUDMUX_V2_PDCR_RXDSEL(int_port));

	/* GPIO2_12, SPK-ON */
	/* speaker amp on, output, active high */
	priv->amp_gpio = of_get_named_gpio(np, "amp-en-gpio", 0);
	if (!gpio_is_valid(priv->amp_gpio)) {
		dev_err(dev, "failed to get amp-gpio\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, priv->amp_gpio,
			GPIOF_OUT_INIT_LOW, "amp-en-gpio");
	if (retval) {
		dev_err(dev, "failed to request gpio amp-en-gpio\n");
		return retval;
	}

	dai->name = "tlv320aic3x";
	dai->stream_name = "AIC3X";
	dai->codec_dai_name = "tlv320aic3x-hifi";
	dai->codec_of_node = codec_np;
	dai->cpu_dai_name = dev_name(&cpu_pdev->dev);
	dai->cpu_of_node = cpu_np;
	dai->platform_of_node = cpu_np;
	dai->init = cicada_snd_init;
	dai->ops = &cicada_snd_ops;

	card->dev = dev;
	retval = snd_soc_of_parse_card_name(card, "model");
	if (retval)
		goto done;

	card->owner = THIS_MODULE;
	card->dai_link = dai;
	card->num_links = 1;
	card->controls = cicada_controls;
	card->num_controls = ARRAY_SIZE(cicada_controls);
	card->dapm_widgets = cicada_dapm_widgets;
	card->num_dapm_widgets = ARRAY_SIZE(cicada_dapm_widgets);

	platform_set_drvdata(pdev, card);
	snd_soc_card_set_drvdata(card, priv);

	/*headset -------------------------------------- */

	device_init_wakeup(dev, priv->wakeup);

	priv->mic_val = 0;
	priv->hp_val = 0;
	priv->spk_val = 1;
	/* jack status unknown till jack_det_worker */
	priv->jack_status = -1;

	/* headset microphone switch enable, output, active high */
	priv->mic_ena_gpio = of_get_named_gpio(np, "hset-mic-en-gpio", 0);
	if (!gpio_is_valid(priv->mic_ena_gpio)) {
		dev_err(dev, "failed to get hset-mic-en-gpio\n");
		retval = -EINVAL;
		goto done;
	}
	gpio_direction_output(priv->mic_ena_gpio, 0);

	/* headset detection, input, active low */
	priv->headset_det_gpio = of_get_named_gpio(np, "hset-detect-gpio", 0);
	if (!gpio_is_valid(priv->headset_det_gpio)) {
		dev_err(dev, "failed to get hset-detect-gpio\n");
		retval = -EINVAL;
		goto done;
	}
	gpio_direction_input(priv->headset_det_gpio);

	of_property_read_u32(np, "hset-detect-key-code", &priv->headset_det_key_code);
	if (!priv->headset_det_key_code) {
		dev_err(dev, "failed to get hset-detect-key-code\n");
		retval = -EINVAL;
		goto done;
	}

	INIT_DELAYED_WORK(&priv->headset_det_work, jack_det_worker);

	retval = devm_request_irq(&pdev->dev, gpio_to_irq(priv->headset_det_gpio), cicada_detect_isr,
			IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING, "hset-detect", priv);
	if (retval) {
		dev_dbg(dev, "%s, request_irq hset-detect-gpio failed\n", __func__);
		retval = -EINVAL;
		goto done;
	}
	disable_irq(gpio_to_irq(priv->headset_det_gpio));

	/* headset type (3-pole vs 4-pole), input, 4-pole == active low */
	priv->p3_gpio = of_get_named_gpio(np, "hset-3pole-gpio", 0);
	if (!gpio_is_valid(priv->p3_gpio)) {
		dev_err(dev, "failed to get hset-3pole-gpio\n");
		retval = -EINVAL;
		goto done;
	}
	gpio_direction_input(priv->p3_gpio);

	/* headset microphone keypress, input, active high */
	priv->key_gpio = of_get_named_gpio(np, "hset-mic-key-gpio", 0);
	if (!gpio_is_valid(priv->key_gpio)) {
		dev_err(dev, "failed to get hset-mic-key-gpio\n");
		retval = -EINVAL;
		goto done;
	}
	gpio_direction_input(priv->key_gpio);

	of_property_read_u32(np, "hset-mic-key-code", &priv->key_code);
	if (!priv->key_code) {
		dev_err(dev, "failed to get hset-mic-key-code\n");
		retval = -EINVAL;
		goto done;
	}

	retval = devm_request_irq(&pdev->dev, gpio_to_irq(priv->key_gpio), cicada_mic_key_isr,
			IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING, "cicada-mic-key", priv);
	if (retval) {
		dev_dbg(dev, "%s, request_irq hset-mic-key-gpio failed\n", __func__);
		retval = -EINVAL;
		goto done;
	}
	priv->key_enabled = 0;
	disable_irq(gpio_to_irq(priv->key_gpio));

	retval = cicada_jack_device_register(priv);
	if (retval) {
		dev_dbg(dev, "%s, cicada_jack_device_register failed\n", __func__);
		goto done;
	}

	priv->headset_det_enabled = 0;
	schedule_delayed_work(&priv->headset_det_work, 0);

	retval = snd_soc_register_card(card);
	if (retval) {
		dev_err(dev, "snd_soc_register_card failed (%d)\n", retval);
		goto done;
	}

done:
	if (cpu_np)
		of_node_put(cpu_np);
	if (codec_np)
		of_node_put(codec_np);

	return retval;
}

static int cicada_snd_remove(struct platform_device *pdev)
{
	struct snd_soc_card *card = platform_get_drvdata(pdev);

	snd_soc_unregister_card(card);
	return 0;
}

static const struct of_device_id cicada_snd_dt_ids[] = {
	{
		.compatible = "fnet,cicada-audio",
	},
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, cicada_snd_dt_ids);

static struct platform_driver cicada_snd_driver = {
	.driver = {
		.name = "cicada-tlv320aic3x-snd",
		.owner = THIS_MODULE,
		.pm	= &snd_soc_pm_ops,
		.of_match_table = cicada_snd_dt_ids,
	},
	.probe = cicada_snd_probe,
	.remove = cicada_snd_remove,
};
module_platform_driver(cicada_snd_driver);

MODULE_AUTHOR("Fluke Networks");
MODULE_DESCRIPTION("ALSA SoC tlv320aic3x cicada");
MODULE_LICENSE("GPL");
