/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/err.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/of.h>
#include <video/of_display_timing.h>
#include <linux/fb.h>
#include <linux/dma-mapping.h>
#include <linux/pinctrl/pinctrl.h>
#include <linux/platform_data/video-mx3fb.h>

#define NR_IRQS_LEGACY	16
#define MX35_INT_IPU_ERR	(NR_IRQS_LEGACY + 41)
#define MX35_INT_IPU_SYN	(NR_IRQS_LEGACY + 42)

#define MX35_AIPS2_BASE_ADDR		0x53f00000
#define MX35_AIPS2_SIZE			SZ_1M
#define MX35_IPU_CTRL_BASE_ADDR	(MX35_AIPS2_BASE_ADDR + 0xc0000)

extern int cicada_hw_id(void);

struct cicada_lcd_data {
	struct platform_device *pdev;

	struct platform_device *ipu;
	struct platform_device *fb;
	struct platform_device_info fbdev_info;
	struct mx3fb_platform_data mx3fb_pdata;
};

static int cicada_lcd_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct cicada_lcd_data *pdata;
	struct device *ipu_dev;
	struct platform_device *ipu_pdev;
	struct resource *ipu_res;
	struct resource fb_res;
	int retcode;

	dev_dbg(dev, "%s\n", __func__);

	if (cicada_hw_id() != 0)
		return -ENODEV;

	if (!np)
		return -EINVAL;

	if (!of_device_is_available(np))
		return -ENODEV;

	pdata = devm_kzalloc(dev, sizeof(*pdata), GFP_KERNEL);
	if (!pdata) {
		dev_err(dev, "could not allocate object\n");
		return -ENOMEM;
	}

	platform_set_drvdata(pdev, pdata);

	ipu_dev = bus_find_device_by_name(&platform_bus_type, NULL, "ipu-core");
	if (!ipu_dev) {
		pr_err("%s: failed to find ipu-core device\n", __func__);
		return -ENODEV;
	}

	ipu_pdev = to_platform_device(ipu_dev);
	ipu_res = platform_get_resource(ipu_pdev, IORESOURCE_MEM, 0);

	/* ipu registers
	 *	SDC (synchronous display controller) registers
	 *	ADC (asynchronous display controller) registers
	 *	DI (Display Interface) registers */
	fb_res.start = ipu_res->start + 0xb4;
	fb_res.end =  ipu_res->start + 0x1bf;
	fb_res.flags = IORESOURCE_MEM;

	pdata->mx3fb_pdata.dma_dev = ipu_dev;

	/* using only the native mode */
	pdata->mx3fb_pdata.num_modes = 1;

	pdata->mx3fb_pdata.mode = devm_kzalloc(dev,
			sizeof(struct fb_videomode), GFP_KERNEL);
	if (!pdata->mx3fb_pdata.mode) {
		return -ENOMEM;
	}

	retcode = of_property_read_string(np, "model", &pdata->mx3fb_pdata.name);
	if (retcode)
		pdata->mx3fb_pdata.name = NULL;

	retcode = of_get_fb_videomode(np, (struct fb_videomode *)pdata->mx3fb_pdata.mode, OF_USE_NATIVE_MODE);
	if (retcode) {
		dev_err(dev, "Failed to get videomode from DT\n");
		return retcode;
	}

	pdata->fbdev_info.name = "mx3_sdc_fb",
	pdata->fbdev_info.id = -1,
	pdata->fbdev_info.res = &fb_res,
	pdata->fbdev_info.num_res = 1,
	pdata->fbdev_info.data = &pdata->mx3fb_pdata,
	pdata->fbdev_info.size_data = sizeof(pdata->mx3fb_pdata),
	pdata->fbdev_info.dma_mask = DMA_BIT_MASK(32),
	
	pdata->fb = platform_device_register_full(&pdata->fbdev_info);

	if (IS_ERR_OR_NULL(pdata->fb))
		return -ENODEV;

	return 0;
}

static int cicada_lcd_remove(struct platform_device *pdev)
{
	dev_dbg(&pdev->dev, "%s\n", __func__);

	return 0;
}

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-lcd", .data = NULL },
	{}
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

static struct platform_driver cicada_lcd_driver = {
	.driver = {
		.name = "cicada-lcd",
		.owner = THIS_MODULE,
		.of_match_table = cicada_device_ids,
	},
	.probe	  = cicada_lcd_probe,
	.remove   = cicada_lcd_remove,
};
module_platform_driver(cicada_lcd_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("Cicada Lcd");


