/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* derived from drivers/acpi/sbs.c and drivers/power/ds2760_battery.c */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/param.h>
#include <linux/jiffies.h>
#include <linux/workqueue.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <linux/pm.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/cicada_gasgauge.h>
#include <linux/slab.h>
#include <linux/printk.h>

extern int cicada_gasgauge_init(void);
extern void cicada_gasgauge_exit(void);

static unsigned int cache_time = 3000;
module_param(cache_time, uint, 0644);
MODULE_PARM_DESC(cache_time, "cache time in milliseconds");

struct cicada_battery {
	struct platform_device *pdev;

	/* cicada data, valid after calling cicada_battery_read_status() */
	unsigned long update_time;	/* jiffies when data read */

	int charge_status;		/* POWER_SUPPLY_STATUS_* */
	int chg_disable_gpio;
	int chg_offline_gpio;
	int chg_charging_gpio;
	int chg_fault_gpio;
	
	struct cicada_battery_data data;

	struct power_supply bat;
	struct workqueue_struct *monitor_wqueue;
	struct delayed_work monitor_work;
};

int cicada_charger_status(struct cicada_battery *pdata)
{
	struct platform_device *pdev = pdata->pdev;
	int val = (!!gpio_get_value(pdata->chg_charging_gpio) << 1) |
		!!gpio_get_value(pdata->chg_fault_gpio);

	if (!!gpio_get_value(pdata->chg_offline_gpio) || !!gpio_get_value(pdata->chg_disable_gpio) || (val == 0))
		return CICADA_CHARGER_STATUS_NOTCHARGING;
	else if (val == 1)
		return CICADA_CHARGER_STATUS_BATTERYFAULT;
	else if (val == 2)
		return CICADA_CHARGER_STATUS_CHARGING;
	else if (val == 3)
		return CICADA_CHARGER_STATUS_TEMPFAULT;
	else
		dev_err(&pdev->dev, "unknown charger status\n");

	return CICADA_CHARGER_STATUS_UNKNOWN;
}

static ssize_t charger_enable_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	struct cicada_battery *pdata = dev_get_drvdata(dev->parent);
	return sprintf(buf, "%u\n", !gpio_get_value(pdata->chg_disable_gpio));
}

static ssize_t charger_enable_store(struct device *dev,
				    struct device_attribute *attr,
				    const char *buf, size_t n)
{
	struct cicada_battery *pdata = dev_get_drvdata(dev->parent);
	unsigned int value = simple_strtol(buf, NULL, 10);

	if (value != 0 && value != 1) {
		printk(KERN_ERR "charger_store: Invalid value\n");
		return -EINVAL;
	}

	gpio_set_value(pdata->chg_disable_gpio, (value ? 0:1));

	return n;
}

static DEVICE_ATTR_RW(charger_enable);

static ssize_t charger_status_show(struct device *dev,
				   struct device_attribute *attr,
				   char *buf)
{
	struct cicada_battery *pdata = dev_get_drvdata(dev->parent);
 	return sprintf(buf, "%u\n", cicada_charger_status(pdata));
}

static DEVICE_ATTR_RO(charger_status);

static ssize_t charger_fault_show(struct device *dev,
				  struct device_attribute *attr,
				  char *buf)
{
	struct cicada_battery *pdata = dev_get_drvdata(dev->parent);
 	return sprintf(buf, "%u\n", gpio_get_value(pdata->chg_fault_gpio));
}

static DEVICE_ATTR_RO(charger_fault);

static ssize_t charging_show(struct device *dev,
			     struct device_attribute *attr,
			     char *buf)
{
	struct cicada_battery *pdata = dev_get_drvdata(dev->parent);
	return sprintf(buf, "%u\n", gpio_get_value(pdata->chg_charging_gpio));
}

static DEVICE_ATTR_RO(charging);

static ssize_t ac_online_show(struct device *dev,
			      struct device_attribute *attr,
			      char *buf)
{
	struct cicada_battery *pdata = dev_get_drvdata(dev->parent);
	return sprintf(buf, "%u\n", !gpio_get_value(pdata->chg_offline_gpio));
}

static DEVICE_ATTR_RO(ac_online);

static inline int battery_scale(int log)
{
	int scale = 1;
	while (log--)
		scale *= 10;
	return scale;
}

static inline int cicada_battery_vscale(struct cicada_battery *pdata)
{
	return battery_scale((pdata->data.spec & 0x0f00) >> 8);
}

static inline int cicada_battery_ipscale(struct cicada_battery *pdata)
{
	return battery_scale((pdata->data.spec & 0xf000) >> 12);
}

static inline int cicada_battery_mode(struct cicada_battery *pdata)
{
	return pdata->data.mode & SB_MODE_CAPACITY_MODE;
}

static inline int cicada_battery_scale(struct cicada_battery *pdata)
{
	return (cicada_battery_mode(pdata) ? 10 : 1) *
		cicada_battery_ipscale(pdata);
}

static int cicada_battery_technology(struct cicada_battery *pdata)
{
	if (!strcasecmp("NiCd", pdata->data.device_chemistry))
		return POWER_SUPPLY_TECHNOLOGY_NiCd;
	if (!strcasecmp("NiMH", pdata->data.device_chemistry))
		return POWER_SUPPLY_TECHNOLOGY_NiMH;
	if (!strcasecmp("LION", pdata->data.device_chemistry))
		return POWER_SUPPLY_TECHNOLOGY_LION;
	if (!strcasecmp("LiP", pdata->data.device_chemistry))
		return POWER_SUPPLY_TECHNOLOGY_LIPO;
	return POWER_SUPPLY_TECHNOLOGY_UNKNOWN;
}

static int cicada_battery_read_status(struct cicada_battery *pdata)
{
	struct platform_device *pdev = pdata->pdev;
	int present = pdata->data.present;

	if (pdata->update_time && time_before(jiffies, pdata->update_time +
					   msecs_to_jiffies(cache_time)))
		return 0;

	cicada_gasgauge_get(&pdata->data);

	if (!present) {
		if (pdata->data.present)
			dev_dbg(&pdev->dev, "battery connected\n");
	} else {
		if (!pdata->data.present)
			dev_dbg(&pdev->dev, "battery removed\n");
	}

	pdata->update_time = jiffies;

	return 0;
}

static void cicada_battery_update_status(struct cicada_battery *pdata)
{
	int old_charge_status = pdata->charge_status;

	cicada_battery_read_status(pdata);

	if (pdata->data.present) {
		if (pdata->data.status & SB_STATUS_FULLY_CHARGED)
			pdata->charge_status = POWER_SUPPLY_STATUS_FULL;
		else if (pdata->data.status & SB_STATUS_DISCHARGING)
			pdata->charge_status = POWER_SUPPLY_STATUS_DISCHARGING;
		else
			pdata->charge_status = POWER_SUPPLY_STATUS_CHARGING;
	} else
		pdata->charge_status = POWER_SUPPLY_STATUS_UNKNOWN;

	if (pdata->charge_status != old_charge_status)
		power_supply_changed(&pdata->bat);
}

static void cicada_battery_work(struct work_struct *work)
{
	struct cicada_battery *pdata = container_of(work,
		struct cicada_battery, monitor_work.work);

	cicada_battery_update_status(pdata);
	queue_delayed_work(pdata->monitor_wqueue, &pdata->monitor_work,
			msecs_to_jiffies(cache_time));
}

#define to_cicada_battery(x) container_of((x), struct cicada_battery, bat);

static void cicada_battery_external_power_changed(struct power_supply *psy)
{
	struct cicada_battery *pdata = to_cicada_battery(psy);

	dev_dbg(&pdata->pdev->dev, "%s\n", __func__);
}

/*
 * All voltages, currents, charges, energies, time and temperatures in uV,
 * µA, µAh, µWh, seconds and tenths of degree Celsius unless otherwise
 * stated.
 */
static int cicada_battery_get_property(struct power_supply *psy,
					   enum power_supply_property psp,
					   union power_supply_propval *val)
{
	int ret;
	struct cicada_battery *pdata = to_cicada_battery(psy);

	if ((!pdata->data.present) && psp != POWER_SUPPLY_PROP_PRESENT)
		return -ENODEV;

	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		val->intval = pdata->charge_status;
		break;
	case POWER_SUPPLY_PROP_HEALTH:
		ret = cicada_charger_status(pdata);
		if ((pdata->data.status & SB_STATUS_OVER_TEMP_ALARM) ||
		    (ret == CICADA_CHARGER_STATUS_TEMPFAULT))
			val->intval = POWER_SUPPLY_HEALTH_OVERHEAT;
		else if (ret == CICADA_CHARGER_STATUS_BATTERYFAULT)
			val->intval = POWER_SUPPLY_HEALTH_UNSPEC_FAILURE;
		else if (pdata->data.status & SB_STATUS_FULLY_DISCHARGED)
			val->intval = POWER_SUPPLY_HEALTH_DEAD;
		else
			val->intval = POWER_SUPPLY_HEALTH_GOOD;
		break;
	case POWER_SUPPLY_PROP_PRESENT:
		val->intval = pdata->data.present;
		break;
	case POWER_SUPPLY_PROP_TECHNOLOGY:
		val->intval = cicada_battery_technology(pdata);
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN:
		val->intval = pdata->data.design_voltage *
			cicada_battery_vscale(pdata) * 1000;
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
		val->intval = pdata->data.voltage_now *
				cicada_battery_vscale(pdata) * 1000;
		break;
	case POWER_SUPPLY_PROP_CURRENT_NOW:
		val->intval = pdata->data.current_now *
				cicada_battery_ipscale(pdata) * 1000;
		break;
	case POWER_SUPPLY_PROP_CURRENT_AVG:
		val->intval = pdata->data.current_avg *
				cicada_battery_ipscale(pdata) * 1000;
		break;
	case POWER_SUPPLY_PROP_CAPACITY:
		val->intval = pdata->data.state_of_charge;
		break;
	case POWER_SUPPLY_PROP_CHARGE_FULL_DESIGN:
	case POWER_SUPPLY_PROP_ENERGY_FULL_DESIGN:
		val->intval = pdata->data.design_capacity *
			cicada_battery_scale(pdata) * 1000;
		break;
	case POWER_SUPPLY_PROP_CHARGE_FULL:
	case POWER_SUPPLY_PROP_ENERGY_FULL:
		val->intval = pdata->data.full_charge_capacity *
			cicada_battery_scale(pdata) * 1000;
		break;
	case POWER_SUPPLY_PROP_CHARGE_NOW:
	case POWER_SUPPLY_PROP_ENERGY_NOW:
		val->intval = pdata->data.capacity_now *
				cicada_battery_scale(pdata) * 1000;
		break;
	case POWER_SUPPLY_PROP_CHARGE_COUNTER:
		val->intval = pdata->data.cycle_count;
		break;
	case POWER_SUPPLY_PROP_TIME_TO_EMPTY_NOW:
		val->intval = pdata->data.time_to_empty * 60;
		break;
	case POWER_SUPPLY_PROP_TIME_TO_EMPTY_AVG:
		val->intval = pdata->data.time_to_empty_avg * 60;
		break;
	case POWER_SUPPLY_PROP_TIME_TO_FULL_AVG:
		val->intval = pdata->data.time_to_full_avg * 60;
		break;
	case POWER_SUPPLY_PROP_TEMP:
		val->intval = pdata->data.temp_now - 2722;
		break;
	case POWER_SUPPLY_PROP_MODEL_NAME:
		val->strval = pdata->data.device_name;
		break;
	case POWER_SUPPLY_PROP_MANUFACTURER:
		val->strval = pdata->data.manufacturer_name;
		break;
	case POWER_SUPPLY_PROP_SERIAL_NUMBER:
		val->strval = pdata->data.serial_number_str;
		break;
	case POWER_SUPPLY_PROP_MANUFACTURER_DATE:
		val->strval = pdata->data.manufacturer_date_str;
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

/* from linux/power_supply.h */
static enum power_supply_property cicada_battery_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_HEALTH,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_TECHNOLOGY,
	POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
	POWER_SUPPLY_PROP_CURRENT_NOW,
	POWER_SUPPLY_PROP_CURRENT_AVG,
	POWER_SUPPLY_PROP_CAPACITY,
	POWER_SUPPLY_PROP_CHARGE_FULL_DESIGN,
	POWER_SUPPLY_PROP_CHARGE_FULL,
	POWER_SUPPLY_PROP_CHARGE_NOW,
	POWER_SUPPLY_PROP_CHARGE_COUNTER,
	POWER_SUPPLY_PROP_TIME_TO_EMPTY_NOW,
	POWER_SUPPLY_PROP_TIME_TO_EMPTY_AVG,
	POWER_SUPPLY_PROP_TIME_TO_FULL_AVG,
	POWER_SUPPLY_PROP_TEMP,
	POWER_SUPPLY_PROP_MODEL_NAME,
	POWER_SUPPLY_PROP_MANUFACTURER,
	POWER_SUPPLY_PROP_SERIAL_NUMBER,
	POWER_SUPPLY_PROP_MANUFACTURER_DATE,
};

static enum power_supply_property cicada_energy_battery_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_HEALTH,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_TECHNOLOGY,
	POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
	POWER_SUPPLY_PROP_CURRENT_NOW,
	POWER_SUPPLY_PROP_CURRENT_AVG,
	POWER_SUPPLY_PROP_CAPACITY,
	POWER_SUPPLY_PROP_ENERGY_FULL_DESIGN,
	POWER_SUPPLY_PROP_ENERGY_FULL,
	POWER_SUPPLY_PROP_ENERGY_NOW,
	POWER_SUPPLY_PROP_CHARGE_COUNTER,
	POWER_SUPPLY_PROP_TIME_TO_EMPTY_NOW,
	POWER_SUPPLY_PROP_TIME_TO_EMPTY_AVG,
	POWER_SUPPLY_PROP_TIME_TO_FULL_AVG,
	POWER_SUPPLY_PROP_TEMP,
	POWER_SUPPLY_PROP_MODEL_NAME,
	POWER_SUPPLY_PROP_MANUFACTURER,
	POWER_SUPPLY_PROP_SERIAL_NUMBER,
	POWER_SUPPLY_PROP_MANUFACTURER_DATE,
};

static struct attribute *attrs[] = {
	&dev_attr_charger_enable.attr,
	&dev_attr_charger_status.attr,
	&dev_attr_charger_fault.attr,
	&dev_attr_charging.attr,
	&dev_attr_ac_online.attr,
	NULL
};

static const struct attribute_group attr_group = {
	.attrs = attrs,
};

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-battery", .data = NULL },
	{}
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

static int cicada_battery_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct cicada_battery *pdata;
	int retval = 0;
/*	struct cicada_platform_data *pdata = pdev->dev.platform_data; */

	dev_dbg(dev, "%s:\n", __func__);

	if (!np)
		return -EINVAL;

	if (!of_device_is_available(np))
		return -ENODEV;

	pdata = devm_kzalloc(dev, sizeof(*pdata), GFP_KERNEL);
	if (!pdata) {
		return -ENOMEM;
	}

	platform_set_drvdata(pdev, pdata);

	pdata->pdev = pdev;
	pdata->bat.name = dev_name(dev);
	pdata->bat.type = POWER_SUPPLY_TYPE_BATTERY;
	if (!cicada_battery_mode(pdata)) {
		pdata->bat.properties = cicada_battery_props;
		pdata->bat.num_properties =
			ARRAY_SIZE(cicada_battery_props);
	} else {
		pdata->bat.properties = cicada_energy_battery_props;
		pdata->bat.num_properties =
			ARRAY_SIZE(cicada_energy_battery_props);
	}
	pdata->bat.get_property = cicada_battery_get_property;
	pdata->bat.external_power_changed = cicada_battery_external_power_changed;

	pdata->charge_status = POWER_SUPPLY_STATUS_UNKNOWN;

	/* charger disable gpio */
	pdata->chg_disable_gpio = of_get_named_gpio(np, "chg-disable-gpios", 0);
	if (!gpio_is_valid(pdata->chg_disable_gpio)) {
		dev_err(dev, "failed to get chg-disable-gpios\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, pdata->chg_disable_gpio,
			GPIOF_OUT_INIT_LOW, "chg-disable-gpios");
	if (retval) {
		dev_err(dev, "failed to request gpio chg-disable-gpios\n");
		return retval;
	}

	/* charger offline gpio */
	pdata->chg_offline_gpio = of_get_named_gpio(np, "chg-offline-gpios", 0);
	if (!gpio_is_valid(pdata->chg_offline_gpio)) {
		dev_err(dev, "failed to get chg-offline-gpios\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, pdata->chg_offline_gpio,
			GPIOF_IN, "chg-offline-gpios");
	if (retval) {
		dev_err(dev, "failed to request gpio chg-offline-gpios\n");
		return retval;
	}

	/* charger charging gpio */
	pdata->chg_charging_gpio = of_get_named_gpio(np, "chg-charging-gpios", 0);
	if (!gpio_is_valid(pdata->chg_charging_gpio)) {
		dev_err(dev, "failed to get chg-charging-gpios\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, pdata->chg_charging_gpio,
			GPIOF_IN, "chg-charging-gpios");
	if (retval) {
		dev_err(dev, "failed to request gpio chg-charging-gpios\n");
		return retval;
	}

	/* charger fault gpio */
	pdata->chg_fault_gpio = of_get_named_gpio(np, "chg-fault-gpios", 0);
	if (!gpio_is_valid(pdata->chg_fault_gpio)) {
		dev_err(dev, "failed to get chg-fault-gpios\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, pdata->chg_fault_gpio,
			GPIOF_IN, "chg-fault-gpios");
	if (retval) {
		dev_err(dev, "failed to request gpio chg-fault-gpios\n");
		return retval;
	}

	retval = power_supply_register(dev, &pdata->bat);
	if (retval) {
		dev_err(dev, "failed to register battery\n");
		goto error;
	}

	retval = sysfs_create_group(&pdata->bat.dev->kobj, &attr_group);
	if (retval) {
		dev_err(dev, "%s, sysfs_create_group failed\n", __func__);
		goto error_1;
	}

	INIT_DELAYED_WORK(&pdata->monitor_work, cicada_battery_work);
	pdata->monitor_wqueue = create_singlethread_workqueue(dev_name(dev));
	if (!pdata->monitor_wqueue) {
		dev_err(dev, "workqueue creation failed\n");
		retval = -ESRCH;
		goto error_2;
	}
	pdata->update_time = 0;
	queue_delayed_work(pdata->monitor_wqueue, &pdata->monitor_work,
			msecs_to_jiffies(cache_time));

	dev_info(dev, "probe succeded\n");

	return 0;

error_2:
	sysfs_remove_group(&pdata->bat.dev->kobj, &attr_group);
error_1:
	power_supply_unregister(&pdata->bat);
error:
	return retval;
}

static int cicada_battery_remove(struct platform_device *pdev)
{
	struct cicada_battery *pdata = platform_get_drvdata(pdev);

	dev_dbg(&pdev->dev, "%s:\n", __func__);

	cancel_delayed_work_sync(&pdata->monitor_work);
	destroy_workqueue(pdata->monitor_wqueue);

	sysfs_remove_group(&pdata->bat.dev->kobj, &attr_group);

	power_supply_unregister(&pdata->bat);
	kfree(pdata);

	cicada_gasgauge_exit();

	return 0;
}

static int cicada_battery_suspend(struct platform_device *pdev,
				  pm_message_t state)
{
	struct cicada_battery *pdata = platform_get_drvdata(pdev);

	dev_dbg(&pdev->dev, "%s:\n", __func__);

	cancel_delayed_work_sync(&pdata->monitor_work);

	pdata->charge_status = POWER_SUPPLY_STATUS_UNKNOWN;

	return 0;
}

static int cicada_battery_resume(struct platform_device *pdev)
{
	struct cicada_battery *pdata = platform_get_drvdata(pdev);

	dev_dbg(&pdev->dev, "%s:\n", __func__);

	pdata->charge_status = POWER_SUPPLY_STATUS_UNKNOWN;
	pdata->update_time = 0;

	queue_delayed_work(pdata->monitor_wqueue, &pdata->monitor_work,
			msecs_to_jiffies(cache_time));

	return 0;
}

static struct platform_driver cicada_battery_driver = {
	.driver = {
		.name = "cicada-battery",
		.owner = THIS_MODULE,
		.of_match_table = cicada_device_ids,
	},
	.probe	  = cicada_battery_probe,
	.remove   = cicada_battery_remove,
	.suspend  = cicada_battery_suspend,
	.resume	  = cicada_battery_resume,
};

module_platform_driver(cicada_battery_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("cicada battery driver");

