/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <linux/module.h>
#include <linux/err.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/platform_device.h>

extern int cicada_hw_id(void);

enum {
	RMT_LED_PASS	= 0x01,
	RMT_LED_TEST	= 0x02,
	RMT_LED_FAIL	= 0x04,
	RMT_LED_TALK	= 0x08,
	RMT_LED_TONE	= 0x10,
	RMT_LED_LOWBATT	= 0x20,
	RMT_LED_ALL	= 0x3F,
} rmt_led_t;

struct cicada_rmt_led_data {
	struct platform_device *pdev;

	int gpio_led_tone;
	int gpio_led_lowbat;
	int gpio_led_pass;
	int gpio_led_test;
	int gpio_led_fail;
	int gpio_led_talk;
};

static ssize_t pass_show(struct device *dev,
				 struct device_attribute *attr,
				 char *buf)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	int val = !!gpio_get_value(pdata->gpio_led_pass);

	return sprintf(buf, "%u\n", val);
}

static ssize_t pass_store(struct device *dev,
				  struct device_attribute *attr,
				  const char *buf, size_t n)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 10, &value) != 0) {
		dev_err(dev, "%s, failed to convert value\n", __func__);
		return -EINVAL;
	}

	if (value != 0 && value != 1) {
		printk(KERN_ERR "rmt_led_pass_store: Invalid value %d\n",
		       value);
		return -EINVAL;
	}

	gpio_set_value(pdata->gpio_led_pass, value);

	return n;
}

DEVICE_ATTR_RW(pass);


static ssize_t test_show(struct device *dev,
				 struct device_attribute *attr,
				 char *buf)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	int val = gpio_get_value(pdata->gpio_led_test) ? RMT_LED_TEST : 0;

	return sprintf(buf, "%u\n", val);
}

static ssize_t test_store(struct device *dev,
				  struct device_attribute *attr,
				  const char *buf, size_t n)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 10, &value) != 0) {
		dev_err(dev, "%s, failed to convert value\n", __func__);
		return -EINVAL;
	}

	if (value != 0 && value != 1) {
		printk(KERN_ERR "rmt_led_test_store: Invalid value %d\n",
		       value);
		return -EINVAL;
	}

	gpio_set_value(pdata->gpio_led_test, value);

	return n;
}

DEVICE_ATTR_RW(test);


static ssize_t fail_show(struct device *dev,
				 struct device_attribute *attr,
				 char *buf)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	int val = gpio_get_value(pdata->gpio_led_fail) ? RMT_LED_FAIL : 0;

	return sprintf(buf, "%u\n", val);
}

static ssize_t fail_store(struct device *dev,
				  struct device_attribute *attr,
				  const char *buf, size_t n)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 10, &value) != 0) {
		dev_err(dev, "%s, failed to convert value\n", __func__);
		return -EINVAL;
	}

	if (value != 0 && value != 1) {
		printk(KERN_ERR "rmt_led_fail_store: Invalid value %d\n",
		       value);
		return -EINVAL;
	}

	gpio_set_value(pdata->gpio_led_fail, value);

	return n;
}

DEVICE_ATTR_RW(fail);


static ssize_t talk_show(struct device *dev,
				 struct device_attribute *attr,
				 char *buf)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	int val = gpio_get_value(pdata->gpio_led_talk) ? RMT_LED_TALK : 0;

	return sprintf(buf, "%u\n", val);
}

static ssize_t talk_store(struct device *dev,
				  struct device_attribute *attr,
				  const char *buf, size_t n)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 10, &value) != 0) {
		dev_err(dev, "%s, failed to convert value\n", __func__);
		return -EINVAL;
	}

	if (value != 0 && value != 1) {
		printk(KERN_ERR "rmt_led_talk_store: Invalid value %d\n",
		       value);
		return -EINVAL;
	}

	gpio_set_value(pdata->gpio_led_talk, value);

	return n;
}

DEVICE_ATTR_RW(talk);


static ssize_t tone_show(struct device *dev,
				 struct device_attribute *attr,
				 char *buf)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	int val = gpio_get_value(pdata->gpio_led_tone) ? RMT_LED_TONE : 0;

	return sprintf(buf, "%u\n", val);
}

static ssize_t tone_store(struct device *dev,
				  struct device_attribute *attr,
				  const char *buf, size_t n)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 10, &value) != 0) {
		dev_err(dev, "%s, failed to convert value\n", __func__);
		return -EINVAL;
	}

	if (value != 0 && value != 1) {
		printk(KERN_ERR "rmt_led_tone_store: Invalid value %d\n",
		       value);
		return -EINVAL;
	}

	gpio_set_value(pdata->gpio_led_tone, value);

	return n;
}

DEVICE_ATTR_RW(tone);


static ssize_t lowbatt_show(struct device *dev,
				    struct device_attribute *attr,
				    char *buf)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	int val = gpio_get_value(pdata->gpio_led_lowbat) ? RMT_LED_LOWBATT : 0;

	return sprintf(buf, "%u\n", val);
}

static ssize_t lowbatt_store(struct device *dev,
				     struct device_attribute *attr,
				     const char *buf, size_t n)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 10, &value) != 0) {
		dev_err(dev, "%s, failed to convert value\n", __func__);
		return -EINVAL;
	}

	if (value != 0 && value != 1) {
		printk(KERN_ERR "rmt_led_lowbatt_store: Invalid value %d\n",
		       value);
		return -EINVAL;
	}

	gpio_set_value(pdata->gpio_led_lowbat, value);

	return n;
}

DEVICE_ATTR_RW(lowbatt);


static ssize_t all_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	static char *led_text[] = {
		"PASS", "TEST", "FAIL", "TALK", "TONE", "LOWBATT"
	};
	int val = 0;
	int ccount = 0;
	int i;

	val |= gpio_get_value(pdata->gpio_led_pass) ? RMT_LED_PASS : 0;
	val |= gpio_get_value(pdata->gpio_led_test) ? RMT_LED_TEST : 0;
	val |= gpio_get_value(pdata->gpio_led_fail) ? RMT_LED_FAIL : 0;
	val |= gpio_get_value(pdata->gpio_led_talk) ? RMT_LED_TALK : 0;
	val |= gpio_get_value(pdata->gpio_led_tone) ? RMT_LED_TONE : 0;
	val |= gpio_get_value(pdata->gpio_led_lowbat) ? RMT_LED_LOWBATT : 0;

	ccount += sprintf(buf + ccount, "0x%02x ", val);
	for (i = sizeof(led_text); i >= 0; i--) {
		if (val & (1 << i))
			ccount += sprintf(buf + ccount, "%s ", led_text[i]);
	}

	ccount += sprintf(buf + ccount, "\n");

	return ccount;
}

static ssize_t all_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t n)
{
	struct cicada_rmt_led_data *pdata = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 10, &value) != 0) {
		if (kstrtouint(buf, 16, &value) != 0) {
			dev_err(dev, "%s, failed to convert value\n", __func__);
			return -EINVAL;
		}
	}

	if (value > RMT_LED_ALL || value < 0) {
		printk(KERN_ERR "rmt_led_all_store: Invalid value %d\n",
		       value);
		return -EINVAL;
	}

	gpio_set_value(pdata->gpio_led_tone, !!(value & RMT_LED_TONE));
	gpio_set_value(pdata->gpio_led_lowbat, !!(value & RMT_LED_LOWBATT));
	gpio_set_value(pdata->gpio_led_pass, !!(value & RMT_LED_PASS));
	gpio_set_value(pdata->gpio_led_test, !!(value & RMT_LED_TEST));
	gpio_set_value(pdata->gpio_led_fail, !!(value & RMT_LED_FAIL));
	gpio_set_value(pdata->gpio_led_talk, !!(value & RMT_LED_TALK));

	return n;
}

DEVICE_ATTR_RW(all);


static struct attribute *attrs[] = {
	&dev_attr_pass.attr,
	&dev_attr_test.attr,
	&dev_attr_fail.attr,
	&dev_attr_talk.attr,
	&dev_attr_tone.attr,
	&dev_attr_lowbatt.attr,
	&dev_attr_all.attr,
	NULL
};

static const struct attribute_group attr_group = {
	.attrs = attrs,
};

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-rmt-led", .data = NULL },
	{}
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

static int cicada_rmt_led_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct cicada_rmt_led_data *pdata;
	int retval = 0;

	if (cicada_hw_id() != 1)
		return -ENODEV;

	pr_info("Cicada Platform Hardware Id: remote led display\n");

	if (!of_device_is_available(np))
		return -ENODEV;

	pdata = devm_kzalloc(dev, sizeof(*pdata), GFP_KERNEL);
	if (!pdata) {
		dev_err(dev, "could not allocate object\n");
		return -ENOMEM;
	}

	platform_set_drvdata(pdev, pdata);

	pdata->gpio_led_tone = of_get_named_gpio(np, "led-tone-gpios", 0);
	if (!gpio_is_valid(pdata->gpio_led_tone)) {
		dev_err(dev, "failed to get led-tone-gpios\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, pdata->gpio_led_tone,
			GPIOF_OUT_INIT_HIGH, "led-tone-gpios");
	if (retval) {
		dev_err(dev, "failed to request gpio led-tone-gpios\n");
		return retval;
	}

	pdata->gpio_led_lowbat = of_get_named_gpio(np, "led-lowbat-gpios", 0);
	if (!gpio_is_valid(pdata->gpio_led_lowbat)) {
		dev_err(dev, "failed to get led-lowbat-gpios\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, pdata->gpio_led_lowbat,
			GPIOF_OUT_INIT_HIGH, "led-lowbat-gpios");
	if (retval) {
		dev_err(dev, "failed to request gpio led-lowbat-gpios\n");
		return retval;
	}

	pdata->gpio_led_pass = of_get_named_gpio(np, "led-pass-gpios", 0);
	if (!gpio_is_valid(pdata->gpio_led_pass)) {
		dev_err(dev, "failed to get led-pass-gpios\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, pdata->gpio_led_pass,
			GPIOF_OUT_INIT_HIGH, "led-pass-gpios");
	if (retval) {
		dev_err(dev, "failed to request gpio led-pass-gpios\n");
		return retval;
	}

	pdata->gpio_led_test = of_get_named_gpio(np, "led-test-gpios", 0);
	if (!gpio_is_valid(pdata->gpio_led_test)) {
		dev_err(dev, "failed to get led-test-gpios\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, pdata->gpio_led_test,
			GPIOF_OUT_INIT_HIGH, "led-test-gpios");
	if (retval) {
		dev_err(dev, "failed to request gpio led-test-gpios\n");
		return retval;
	}

	pdata->gpio_led_fail = of_get_named_gpio(np, "led-fail-gpios", 0);
	if (!gpio_is_valid(pdata->gpio_led_fail)) {
		dev_err(dev, "failed to get led-fail-gpios\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, pdata->gpio_led_fail,
			GPIOF_OUT_INIT_HIGH, "led-fail-gpios");
	if (retval) {
		dev_err(dev, "failed to request gpio led-fail-gpios\n");
		return retval;
	}

	pdata->gpio_led_talk = of_get_named_gpio(np, "led-talk-gpios", 0);
	if (!gpio_is_valid(pdata->gpio_led_talk)) {
		dev_err(dev, "failed to get led-talk-gpios\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, pdata->gpio_led_talk,
			GPIOF_OUT_INIT_HIGH, "led-talk-gpios");
	if (retval) {
		dev_err(dev, "failed to request gpio led-talk-gpios\n");
		return retval;
	}

	retval = sysfs_create_group(&dev->kobj, &attr_group);
	if (retval) {
		dev_err(dev, "%s, sysfs_create_group failed\n", __func__);
		goto error;
	}

	dev_info(&pdev->dev, "success\n");

	return 0;

error:
	return retval;
}

static int cicada_rmt_led_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;

	dev_dbg(dev, "%s\n", __func__);

	sysfs_remove_group(&dev->kobj, &attr_group);

	return 0;
}

static struct platform_driver cicada_rmt_led_driver = {
	.driver = {
		.name = "cicada-rmt-led",
		.owner = THIS_MODULE,
		.of_match_table = cicada_device_ids,
	},
	.probe	  = cicada_rmt_led_probe,
	.remove   = cicada_rmt_led_remove,
};

module_platform_driver(cicada_rmt_led_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("Cicada Remote LED");

