/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define DEBUG
#include <linux/printk.h>
#include <linux/module.h>
#include <linux/pm.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>

struct cicada_module_if {
	struct platform_device *pdev;

	int special_enable_gpios;
	int special_gpios;
	int vbat_enable_gpios;
	int vbat_ok_gpios;
};

struct cicada_module_if *cicada_module_if = NULL;

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-module-if", .data = NULL },
	{}
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

/*=================================================*/
void cicada_set_special_enable(int enable)
{
	if (!cicada_module_if) {
		pr_err("%s, no device\n", __func__);
		return;
	}

	if (enable != 0 && enable != 1) {
		pr_err("%s, Invalid value\n", __func__);
		return;
	}

	pr_debug("%s, %s\n", __func__, (enable ? "enabled" : "disabled"));

	gpio_set_value(cicada_module_if->special_enable_gpios, (enable ? 1 : 0));
}
EXPORT_SYMBOL(cicada_set_special_enable);

void cicada_set_special(int val)
{
	if (!cicada_module_if) {
		pr_err("%s, no device\n", __func__);
		return;
	}

	if (val != 0 && val != 1) {
		pr_err("%s, Invalid value\n", __func__);
		return;
	}

	pr_debug("%s, %s\n", __func__, (val ? "high" : "low"));

	gpio_set_value(cicada_module_if->special_gpios, (val ? 1 : 0));
}
EXPORT_SYMBOL(cicada_set_special);

int cicada_get_special(void)
{
	int val;

	if (!cicada_module_if) {
		pr_err("%s, no device\n", __func__);
		return -ENODEV;
	}

	val = !!gpio_get_value(cicada_module_if->special_gpios);
	pr_debug("%s, %s\n", __func__, (val ? "high" : "low"));

	return val;
}
EXPORT_SYMBOL(cicada_get_special);

void cicada_set_module_vbat(int enable)
{
	if (!cicada_module_if) {
		pr_err("%s, no device\n", __func__);
		return;
	}

	pr_debug("%s, %s\n", __func__, (enable ? "enabled" : "disabled"));

	gpio_set_value(cicada_module_if->vbat_enable_gpios, (enable ? 1 : 0));
}
EXPORT_SYMBOL(cicada_set_module_vbat);

#define CICADA_VBAT_ENABLE	0x01
#define CICADA_VBAT_OC		0x02
int cicada_get_module_vbat(void)
{
	int val = 0;

	if (!cicada_module_if) {
		pr_err("%s, no device\n", __func__);
		return -ENODEV;
	}

	val |= !!gpio_get_value(cicada_module_if->vbat_enable_gpios) ? CICADA_VBAT_ENABLE : 0;
	val |= !!gpio_get_value(cicada_module_if->vbat_ok_gpios) ? 0 : CICADA_VBAT_OC;

	pr_debug("%s, %s %s\n", __func__,
		((val & CICADA_VBAT_ENABLE) ? "enabled" : "disabled"),
		((val & CICADA_VBAT_OC) ? "oc" : ""));

	return val;
}
EXPORT_SYMBOL(cicada_get_module_vbat);

/*=================================================*/
static ssize_t special_enable_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	struct cicada_module_if *pdata = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", !!gpio_get_value(pdata->special_enable_gpios));
}

static ssize_t special_enable_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t n)
{
	struct cicada_module_if *pdata = dev_get_drvdata(dev);
	unsigned int val = simple_strtol(buf, NULL, 10);

	if (val != 0 && val != 1) {
		dev_err(dev, "Invalid value\n");
		return -EINVAL;
	}

	gpio_set_value(pdata->special_enable_gpios, val);

	return n;
}

static DEVICE_ATTR_RW(special_enable);

static ssize_t special_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	struct cicada_module_if *pdata = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", !!gpio_get_value(pdata->special_gpios));
}

static ssize_t special_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t n)
{
	struct cicada_module_if *pdata = dev_get_drvdata(dev);
	unsigned int val = simple_strtol(buf, NULL, 10);

	if (val != 0 && val != 1) {
		dev_err(dev, "Invalid value\n");
		return -EINVAL;
	}

	gpio_set_value(pdata->special_gpios, val);

	return n;
}

static DEVICE_ATTR_RW(special);

static ssize_t vbat_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	return sprintf(buf, "%u\n", cicada_get_module_vbat());
}

static ssize_t vbat_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t n)
{
	unsigned int val = simple_strtol(buf, NULL, 10);

	if (val != 0 && val != 1) {
		dev_err(dev, "Invalid value\n");
		return -EINVAL;
	}

	cicada_set_module_vbat(val);

	return n;
}

static DEVICE_ATTR_RW(vbat);

static struct attribute *module_if_attrs[] = {
	&dev_attr_special_enable.attr,
	&dev_attr_special.attr,
	&dev_attr_vbat.attr,
	NULL,
};

static const struct attribute_group module_if_group = {
	.attrs = module_if_attrs,
};

/*=================================================*/
static int cicada_module_if_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct cicada_module_if *pdata;
	int ret = 0;

	dev_dbg(dev, "%s\n", __func__);

	if (!np)
		return -EINVAL;

	if (!of_device_is_available(np))
		return -ENODEV;

	pdata = devm_kzalloc(dev, sizeof(*pdata),
			GFP_KERNEL);
	if (!pdata) {
		dev_err(dev, "could not allocate object\n");
		return -ENOMEM;
	}
	
	platform_set_drvdata(pdev, pdata);
	cicada_module_if = pdata;

	/* enable special, output, active high */
	pdata->special_enable_gpios = of_get_named_gpio(np, "special-enable-gpios", 0);
	ret = gpio_request(pdata->special_enable_gpios, "special-enable-gpios");
	if (ret < 0) {
		dev_dbg(dev, "request special-enable-gpios pin failed\n");
		goto fail;
	}
	gpio_direction_output(pdata->special_enable_gpios, 0);

	/* special data, bi-directional (open-drain output), active low */
	pdata->special_gpios = of_get_named_gpio(np, "special-gpios", 0);
	ret = gpio_request(pdata->special_gpios, "special-gpios");
	if (ret < 0) {
		dev_err(dev, "request special-gpios pin failed\n");
		goto fail;
	}
	gpio_direction_output(pdata->special_gpios, 1);

	/* v bat enable, output, active high */
	pdata->vbat_enable_gpios = of_get_named_gpio(np, "vbat-enable-gpios", 0);
	ret = gpio_request(pdata->vbat_enable_gpios, "vbat-enable-gpios");
	if (ret < 0) {
		dev_err(dev, "request vbat-enable-gpios pin failed\n");
		goto fail;
	}
	gpio_direction_output(pdata->vbat_enable_gpios, 0);

	/* v bat over current, input, active low */
	pdata->vbat_ok_gpios = of_get_named_gpio(np, "vbat-ok-gpios", 0);
	ret = gpio_request(pdata->vbat_ok_gpios, "vbat-ok-gpios");
	if (ret < 0) {
		dev_err(dev, "request vbat-ok-gpios pin failed\n");
		goto fail;
	}
	gpio_direction_input(pdata->vbat_ok_gpios);

	ret = sysfs_create_group(&dev->kobj, &module_if_group);
	if (ret) {
		dev_err(dev, "%s, sysfs_create_group failed\n", __func__);
		return -ENODEV;
	}

	return 0;

fail:
	dev_err(dev, "failed\n");
	return ret;
}

static int cicada_module_if_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;

	sysfs_remove_group(&dev->kobj, &module_if_group);

	return 0;
}

static struct platform_driver cicada_module_if_driver = {
	.driver = {
		.name = "fnet,cicada-module-if",
		.owner = THIS_MODULE,
		.of_match_table = cicada_device_ids,
	},
	.probe	  = cicada_module_if_probe,
	.remove   = cicada_module_if_remove,
};

module_platform_driver(cicada_module_if_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("Cicada Module Interface");




