/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/* #define DEBUG */

#include <linux/module.h>
#include <linux/err.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_gpio.h>

extern unsigned int system_rev;
extern int cicada_is_cpu_imx353(void);
extern u64 cicada_cpu_uid(void);

struct cicada_hw_type {
	struct platform_device *pdev;

	int gpio_hwid_2;
	int gpio_hwid_1;
	int gpio_hwid_0;

	int gpio_hwrev_3;
	int gpio_hwrev_2;
	int gpio_hwrev_1;
	int gpio_hwrev_0;
};
static struct cicada_hw_type *cicada_hw_type = NULL;

/*
 * Hardware ID -- used to distinguish between Main and Remote units
 * where major blocks of hardware may not be installed on a Remote, or
 * an interface is initialized and used differently (like the display).
 * This function returns the following values:
 *	 0    -> Main
 *	 1    -> Remote
 *	 2..7 -> undefined
 */
int cicada_hw_id(void)
{
	int val = 0;

	if (!cicada_hw_type) {
		pr_err("Cicada Platform Hardware Id: driver not ready\n");
		return -ENODEV;
	}

	val |= !!gpio_get_value(cicada_hw_type->gpio_hwid_0);
	val |= !!gpio_get_value(cicada_hw_type->gpio_hwid_1) << 1;
	val |= !!gpio_get_value(cicada_hw_type->gpio_hwid_2) << 2;
	val = ~val & 0x7;

	return val;
}
EXPORT_SYMBOL(cicada_hw_id);

/*
 * Hardware Rev -- used to designate significant changes to the hardware
 * on a board rev that must be detectable at run time by the firmware. A
 * possible example would be a new board rev with more LPDDR or Flash memory.
 */
#define BOARD_REV_1 (0 << 8)
#define BOARD_REV_2 (1 << 8)
#define BOARD_REV_3 (2 << 8)
#define BOARD_REV_4 (3 << 8)
#define BOARD_REV_5 (4 << 8)
#define BOARD_REV_6 (5 << 8)
#define BOARD_REV_7 (6 << 8)
#define BOARD_REV_8 (7 << 8)
#define BOARD_REV_9 (8 << 8)
#define BOARD_REV_10 (9 << 8)
#define BOARD_REV_11 (10 << 8)
#define BOARD_REV_12 (11 << 8)
#define BOARD_REV_13 (12 << 8)
#define BOARD_REV_14 (13 << 8)
#define BOARD_REV_15 (14 << 8)
#define BOARD_REV_16 (15 << 8)

int cicada_hw_rev(void)
{
	int val = 0;

	if (!cicada_hw_type) {
		pr_err("Cicada Platform Hardware Rev: driver not ready\n");
		return -ENODEV;
	}

	val |= !!gpio_get_value(cicada_hw_type->gpio_hwrev_0);
	val |= !!gpio_get_value(cicada_hw_type->gpio_hwrev_1) << 1;
	val |= !!gpio_get_value(cicada_hw_type->gpio_hwrev_2) << 2;
	val |= !!gpio_get_value(cicada_hw_type->gpio_hwrev_3) << 3;
	val = ~val & 0xf;

	return val;
}

static ssize_t hw_id_show(struct device *dev,
			  struct device_attribute *attr,
			  char *buf)
{
	return sprintf(buf, "%u\n", cicada_hw_id());
}

DEVICE_ATTR(hw_id, 0444, hw_id_show, NULL);

static ssize_t hw_rev_show(struct device *dev,
			   struct device_attribute *attr,
			   char *buf)
{
	return sprintf(buf, "%u\n", cicada_hw_rev());
}

DEVICE_ATTR(hw_rev, 0444, hw_rev_show, NULL);

static ssize_t cpu_type_show(struct device *dev,
			     struct device_attribute *attr,
			     char *buf)
{
	return sprintf(buf, "%s\n", (cicada_is_cpu_imx353() ? "mx353" : "mx357"));
}

DEVICE_ATTR(cpu_type, 0444, cpu_type_show, NULL);

static ssize_t cpu_uid_show(struct device *dev,
			   struct device_attribute *attr,
			   char *buf)
{
	return sprintf(buf, "0x%llx\n", cicada_cpu_uid());
}

DEVICE_ATTR(cpu_uid, 0444, cpu_uid_show, NULL);

static struct attribute *attrs[] = {
	&dev_attr_hw_id.attr,
	&dev_attr_hw_rev.attr,
	&dev_attr_cpu_type.attr,
	&dev_attr_cpu_uid.attr,
	NULL
};

static const struct attribute_group attr_group = {
	.attrs = attrs,
};

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-hw-type", .data = NULL },
	{}
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

static int cicada_hw_type_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct cicada_hw_type *pdata;
	enum of_gpio_flags flags;
	unsigned int hw_rev;
	unsigned int hw_id;
	int retval = 0;

	dev_dbg(dev, "%s\n", __func__);

	if (!np)
		return -EINVAL;

	if (!of_device_is_available(np))
		return -ENODEV;

	pdata = devm_kzalloc(dev, sizeof(*pdata), GFP_KERNEL);
	if (!pdata) {
		dev_err(dev, "could not allocate object\n");
		return -ENOMEM;
	}

	cicada_hw_type = pdata;
	platform_set_drvdata(pdev, pdata);

	/*
	* Hardware Id -- used to indicate different loading of parts on the same
	* board revision.  An example would be the Remote for Copper cable testing
	* which probably has parts NOLOADed to reduce cost.  The Ethernet port, USB
	* Host port, display backlight supply, microSD card socket and touch panel
	* interface are likely candidates to NOLOAD.
	*/
	/* HWID-2 (GPIO2_24) */
	pdata->gpio_hwid_2 = of_get_named_gpio_flags(np, "hwid-2-gpios", 0, &flags);
	if (!gpio_is_valid(pdata->gpio_hwid_2)) {
		dev_err(dev, "failed to get hwid-2\n");
		return -EINVAL;
	}
	gpio_direction_input(pdata->gpio_hwid_2);

	/* HWID-1 (GPIO1_31) */
	pdata->gpio_hwid_1 = of_get_named_gpio_flags(np, "hwid-1-gpios", 0, &flags);
	if (!gpio_is_valid(pdata->gpio_hwid_1)) {
		dev_err(dev, "failed to get hwid-1\n");
		return -EINVAL;
	}
	gpio_direction_input(pdata->gpio_hwid_1);

	/* HWID-0 (GPIO1_28) */
	pdata->gpio_hwid_0 = of_get_named_gpio_flags(np, "hwid-0-gpios", 0, &flags);
	if (!gpio_is_valid(pdata->gpio_hwid_0)) {
		dev_err(dev, "failed to get hwid-0\n");
		return -EINVAL;
	}
	gpio_direction_input(pdata->gpio_hwid_0);

	/*
	* Hardware Rev -- used to designate significant changes to the hardware
	* on a board rev that must be detectable at run time by the firmware. A
	* possible example would be a new board rev with more LPDDR or Flash memory.
	*/
	/* HWREV-3 (GPIO3_13) */
	pdata->gpio_hwrev_3 = of_get_named_gpio_flags(np, "hwrev-3-gpios", 0, &flags);
	if (!gpio_is_valid(pdata->gpio_hwrev_3)) {
		dev_err(dev, "failed to get hwrev-3\n");
		return -EINVAL;
	}
	gpio_direction_input(pdata->gpio_hwrev_3);

	/* HWREV-2 (GPIO3_12) */
	pdata->gpio_hwrev_2 = of_get_named_gpio_flags(np, "hwrev-2-gpios", 0, &flags);
	if (!gpio_is_valid(pdata->gpio_hwrev_2)) {
		dev_err(dev, "failed to get hwrev-2\n");
		return -EINVAL;
	}
	gpio_direction_input(pdata->gpio_hwrev_2);

	/* HWREV-1 (GPIO3_11) */
	pdata->gpio_hwrev_1 = of_get_named_gpio_flags(np, "hwrev-1-gpios", 0, &flags);
	if (!gpio_is_valid(pdata->gpio_hwrev_1)) {
		dev_err(dev, "failed to get hwrev-1\n");
		return -EINVAL;
	}
	gpio_direction_input(pdata->gpio_hwrev_1);

	/* HWREV-0 (GPIO3_10) */
	pdata->gpio_hwrev_0 = of_get_named_gpio_flags(np, "hwrev-0-gpios", 0, &flags);
	if (!gpio_is_valid(pdata->gpio_hwrev_0)) {
		dev_err(dev, "failed to get hwrev-0\n");
		return -EINVAL;
	}
	gpio_direction_input(pdata->gpio_hwrev_0);

	/* read and set the hardware rev, to query use:
	   board_is_rev(rev) macro */
	hw_rev = cicada_hw_rev();

	hw_id = cicada_hw_id();

	pr_info("Cicada Platform Hardware Rev %d, Id %d\n", hw_rev, hw_id);

	system_rev &= ~(0x0f00);
	system_rev |= (hw_rev << 8);

	retval = sysfs_create_group(&dev->kobj, &attr_group);
	if (retval) {
		dev_err(dev, "%s, sysfs_create_group failed\n", __func__);
		return -ENODEV;
	}

	return 0;
}

static int cicada_hw_type_remove(struct platform_device *pdev)
{
	dev_dbg(&pdev->dev, "%s\n", __func__);

	return 0;
}

static struct platform_driver cicada_hw_type_driver = {
	.driver = {
		.name = "cicada-hw-type",
		.owner = THIS_MODULE,
		.of_match_table = cicada_device_ids,
	},
	.remove   = cicada_hw_type_remove,
};

static int __init cicada_hw_init(void)
{
	return platform_driver_probe(&cicada_hw_type_driver, cicada_hw_type_probe);
}
subsys_initcall(cicada_hw_init);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("Cicada Hardware Type");

